/*
 * newsiconmgr.cpp
 *
 * Copyright (c) 2001 Frerich Raabe <raabe@kde.org>
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. For licensing and distribution details, check the
 * accompanying file 'COPYING'.
 */
#include "newsiconmgr.h"


#include <kiconloader.h>
#include <kstandarddirs.h>
#include <QtDBus>
#include <qbuffer.h>
#include <qfile.h>
#include <qimage.h>
//Added by qt3to4:
#include <QPixmap>

#define FAVICONINTERFACE "org.kde.KonqFavIcon"

NewsIconMgr *NewsIconMgr::m_instance = 0;

NewsIconMgr *NewsIconMgr::self()
{
	if (!m_instance)
		m_instance = new NewsIconMgr();

	return m_instance;
}

NewsIconMgr::NewsIconMgr(QObject *parent, const char *name)
	: QObject(parent),
	m_stdIcon(SmallIcon(QLatin1String("news")))
{
   QDBusConnection::sessionBus().registerObject("/NewsIconMgr", this, QDBusConnection::ExportAllSlots);
	setObjectName (name);
  m_favIconsModule = new QDBusInterface("org.kde.kded", "/modules/favicons",
                                      FAVICONINTERFACE);
  connect( m_favIconsModule, SIGNAL(iconChanged(bool,QString,QString)),
           this, SLOT(notifyChange(bool,QString,QString)));

}

NewsIconMgr::~NewsIconMgr()
{
	delete m_instance;
        delete m_favIconsModule;
}

void NewsIconMgr::getIcon(const KUrl &url)
{
	if (url.isEmpty()) {
		emit gotIcon(url, m_stdIcon);
		return;
	}

	if (url.isLocalFile()) {
		if (QFile::exists(url.encodedPathAndQuery())) {
			QPixmap icon(url.encodedPathAndQuery());
			if (!icon.isNull()) {
				if (icon.size() != QSize(16, 16)) {
					icon.scaled(16, 16, Qt::KeepAspectRatio, Qt::SmoothTransformation);
					if (icon.isNull()) {
						emit gotIcon(url, m_stdIcon);
						return;
					}
				}
				emit gotIcon(url, icon);
				return;
			}
		}
		emit gotIcon(url, m_stdIcon);
		return;
	}

	if (url.encodedPathAndQuery() == "/favicon.ico") {
		if (favicon(url).isNull()) {
                    m_favIconsModule->call( "downloadHostIcon", url.url() );
		} else {
			emit gotIcon(url, QPixmap(KGlobal::dirs()->findResource("cache",
							QString::fromLatin1("favicons/%1.png").arg(url.host()))));
		}
	} else {
		KIO::Job *job = KIO::get(url, KIO::Reload, KIO::HideProgressInfo);
		connect(job, SIGNAL(data(KIO::Job *, const QByteArray &)),
				SLOT(slotData(KIO::Job *, const QByteArray &)));
		connect(job, SIGNAL(result(KJob *)), SLOT(slotResult(KJob *)));

		KIODownload download;
		download.url = url;
		download.dataOffset = 0;
		m_kioDownload.insert(job, download);
	}
}

bool NewsIconMgr::isStdIcon(const QPixmap &pixmap) const
{
	if (!pixmap.isNull())
		return pixmap.toImage() == m_stdIcon.toImage();
	else
		return false;
}

void NewsIconMgr::slotData(KIO::Job *job, const QByteArray &data)
{
	QBuffer buf(&m_kioDownload[job].data);
	buf.open(QIODevice::WriteOnly);
	buf.seek(m_kioDownload[job].dataOffset);
	buf.write(data);
	m_kioDownload[job].dataOffset = buf.pos();
}

void NewsIconMgr::slotResult(KJob *job)
{
	emit gotIcon(m_kioDownload[job].url, QPixmap(m_kioDownload[job].data));
	m_kioDownload.remove(job);
}

void NewsIconMgr::slotGotIcon(bool isHost, const QString &hostOrURL, const QString &iconName)
{
	KUrl url = KUrl(hostOrURL);
	if (!isHost)
		url.setProtocol(QLatin1String("http"));

	if (iconName.isNull())
		emit gotIcon(url, m_stdIcon);
	else
		emit gotIcon(url, QPixmap(KGlobal::dirs()->findResource("cache",
						QString::fromLatin1("favicons/%1.png").arg(url.host()))));
}

QString NewsIconMgr::favicon(const KUrl &url) const
{
    QDBusReply<QString> reply = m_favIconsModule->call( "iconForURL", url.url() );

    if (reply.isValid()) {
        QString result = reply;
        return result;
    }

    return QString();
}

#include "newsiconmgr.moc"
