/*
**
** Copyright (C) 1998-2001 by Matthias Hölzer-Klüpfel <hoelzer@kde.org>
**	Maintainence has ceased - send questions to kde-devel@kde.org.
**
*/

/*
** This program is free software; you can redistribute it and/or modify
** it under the terms of the GNU General Public License as published by
** the Free Software Foundation; either version 2 of the License, or
** (at your option) any later version.
**
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
** GNU General Public License for more details.
**
** You should have received a copy of the GNU General Public License
** along with this program in a file called COPYING; if not, write to
** the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
** MA 02110-1301, USA.
*/

/*
** Bug reports and questions can be sent to kde-devel@kde.org
*/
#include <stdlib.h>
#include <time.h>


#include <qlabel.h>
#include <qlayout.h>
#include <qdatetime.h>
#include <qtimer.h>
#include <qcombobox.h>
#include <qlineedit.h>
#include <q3popupmenu.h>
//Added by qt3to4:
#include <QMouseEvent>
#include <QFrame>
#include <QHBoxLayout>
#include <QEvent>


#include <kglobal.h>
#include <klocale.h>
#include <kconfig.h>


#include "clock.h"
#include "cities.h"
#include "zoneclock.moc"
#include <kdebug.h>

ZoneClock::ZoneClock(const QString &zone, const QString &name, QWidget *parent)
  : QFrame(parent), _zone(zone), _name(name)
{
  setFrameStyle(QFrame::Panel | QFrame::Raised);
  QHBoxLayout *hbox = new QHBoxLayout(this);
  hbox->setSpacing(2);
  hbox->setMargin(2);

  _name.append(":");
  _nameLabel = new QLabel(_name, this);
  hbox->addWidget(_nameLabel, 1);
  hbox->addSpacing(4);

  _timeLabel = new QLabel(this);
  hbox->addWidget(_timeLabel, 0, Qt::AlignRight);

  _popup = new QMenu(this);
  _popup->addAction(i18n("&Edit..."), this, SLOT(editClock()));
  _popup->addAction(i18n("&Add..."), this, SLOT(slotAddClock()));
  _popup->addAction(i18n("&Remove"), this, SLOT(slotRemoveClock()));

  _nameLabel->installEventFilter(this);
  _timeLabel->installEventFilter(this);

  updateTime();
}


void ZoneClock::slotRemoveClock()
{
  // Note: this is stupid, but we can't get us deleted
  // from this slot, as we would return and crash.
  // So instead we fire up an idle event triggering the delete
  // after the return.

  QTimer::singleShot(0, this, SLOT(removeTimeout()));
}


void ZoneClock::removeTimeout()
{
  emit removeMe(this);
}


void ZoneClock::slotAddClock()
{
  emit addClock(_zone);
}


void ZoneClock::editClock()
{
  ClockDialog *_dlg = new ClockDialog(this);
  _dlg->setModal(true);
  CityList cities;
  QStringList timezones = cities.timezones();
  for (QStringList::iterator it = timezones.begin(); it != timezones.end(); ++it)
    _dlg->ClockZone->addItem(i18n((*it).toUtf8()));

  _dlg->ClockCaption->setText(_nameLabel->text().left(_nameLabel->text().length()-1));
  for (int i=0; i<_dlg->ClockZone->count(); ++i)
    if (_dlg->ClockZone->itemText(i) == i18n(_zone.toUtf8()))
      {
        _dlg->ClockZone->setCurrentIndex(i);
        break;
      }

  if (_dlg->exec() == QDialog::Accepted)
    {
      _zone = timezones[_dlg->ClockZone->currentIndex()];
      _name = _dlg->ClockCaption->text().append(":");
      _nameLabel->setText(_dlg->ClockCaption->text().append(":"));
      updateTime();
      layout()->invalidate();
      emit changed();
    }

  delete _dlg;
}


bool ZoneClock::eventFilter(QObject *obj, QEvent *ev)
{
  if (ev->type() == QEvent::MouseButtonPress)
    {
      QMouseEvent *e = (QMouseEvent*)ev;
      if (e->button() == Qt::RightButton)
	_popup->exec(e->globalPos());
    }

  return QFrame::eventFilter(obj, ev);
}


void ZoneClock::updateTime()
{
  char *initial_TZ = getenv("TZ");
  setenv("TZ", _zone.toLatin1(), 1);
  tzset();

  time_t t = time(NULL);
  QDateTime dt;
  dt.setTime_t(t);
  _timeLabel->setText(QString("%1, %2").arg(KGlobal::locale()->formatTime(dt.time(), true)).arg(KGlobal::locale()->formatDate(dt.date(), true)));

  if (initial_TZ != 0)
    setenv("TZ", initial_TZ, 1);
  else
    unsetenv("TZ");
  tzset();
}


ZoneClockPanel::ZoneClockPanel(QWidget *parent)
  : QFrame(parent), _dlg(0)
{
#ifdef __GNUC__
#warning It used to implement its own Layout class - FIXME
#endif		
  //_flow = new SimpleFlow(this,1,1);

  QTimer *t = new QTimer(this);

  connect(t, SIGNAL(timeout()), this, SLOT(updateTimer()));
  t->start(500);

}

ZoneClockPanel::~ZoneClockPanel()
{
	qDeleteAll(_clocks);
	_clocks.clear();
}

void ZoneClockPanel::createDialog()
{
  if (!_dlg)
    {
      _dlg = new ClockDialog(this);
      _dlg->setModal(true);
      CityList cities;
      QStringList timezones = cities.timezones();
      for (QStringList::iterator it = timezones.begin(); it != timezones.end(); ++it)
        _dlg->ClockZone->addItem(i18n((*it).toUtf8()));
    }
}


void ZoneClockPanel::addClock(const QString &zone, const QString &name)
{
  // add the clocks
  ZoneClock *zc = new ZoneClock(zone, name, this);
  //_flow->add(zc);
  _clocks.append(zc);
  zc->show();

  realign();

  connect(zc, SIGNAL(addClock(const QString &)), this, SLOT(addClock(const QString &)));
  connect(zc, SIGNAL(changed()), this, SLOT(realign()));
  connect(zc, SIGNAL(removeMe(ZoneClock *)), this, SLOT(removeClock(ZoneClock *)));
}


void ZoneClockPanel::removeClock(ZoneClock *clock)
{
  _clocks.removeAll(clock);
  realign();
}


void ZoneClockPanel::realign()
{
  // realign the labels
  int w = 0;
  QListIterator<ZoneClock*> it(_clocks);
  while(it.hasNext())
  {
	ZoneClock *item = it.next();
    if (item->sizeHint().width() > w)
      w = item->sizeHint().width();
  }
  it.toFront();
  while(it.hasNext())
    it.next()->setFixedWidth(w);
}


void ZoneClockPanel::updateTimer()
{
  QListIterator<ZoneClock*> it(_clocks);
  while(it.hasNext())
    it.next()->updateTime();
}

void ZoneClockPanel::addClock(const QString &zone)
{
  createDialog();

  _dlg->ClockCaption->setText(i18n(zone.toUtf8()).section('/', -1));
  for (int i=0; i<_dlg->ClockZone->count(); ++i)
    if (_dlg->ClockZone->itemText(i) == i18n(zone.toUtf8()))
      {
        _dlg->ClockZone->setCurrentIndex(i);
        break;
      }
  if (_dlg->exec() == QDialog::Accepted)
    {
      CityList cities;
      QStringList timezones = cities.timezones();
      QString newzone = timezones[_dlg->ClockZone->currentIndex()];
      addClock(newzone, _dlg->ClockCaption->text());
      update();
    }
}


void ZoneClockPanel::save(KConfig *config)
{
  config->writeEntry("Clocks", _clocks.count());

  QListIterator<ZoneClock*> it(_clocks);
  int cnt=0;
  while(it.hasNext())
    {
	  ZoneClock *item = it.next();
      QString n = item->name();
      n = n.left(n.length()-1);
      config->writeEntry(QString("Clock_%1_Name").arg(cnt), n);
      config->writeEntry(QString("Clock_%1_Zone").arg(cnt), item->zone());
      cnt++;
    }
}


void ZoneClockPanel::load(KConfig *config)
{
  _clocks.clear();

  int num = config->readEntry("Clocks", 0);

  for (int i=0; i<num; ++i)
    {
      addClock(config->readEntry(QString("Clock_%1_Zone").arg(i),QString()), config->readEntry(QString("Clock_%1_Name").arg(i),QString()));
    }
}

