/*
 *  This file is part of the KDE libraries
 *  Copyright (C) 2003 Apple Computer, Inc.
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Library General Public
 *  License as published by the Free Software Foundation; either
 *  version 2 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Library General Public License for more details.
 *
 *  You should have received a copy of the GNU Library General Public License
 *  along with this library; see the file COPYING.LIB.  If not, write to
 *  the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 *  Boston, MA 02111-1307, USA.
 *
 */

#ifndef KJS_IDENTIFIER_H
#define KJS_IDENTIFIER_H

#include "ustring.h"

namespace KJS {

    class Identifier {
        friend class PropertyMap;
    public:
        Identifier() { }
        Identifier(const char *s) : _ustring(add(s)) { }
        Identifier(const UChar *s, int length) : _ustring(add(s, length)) { }
        explicit Identifier(const UString &s) : _ustring(add(s.rep)) { }

        const UString &ustring() const { return _ustring; }
        DOM::DOMString string() const;
        QString qstring() const;

        const UChar *data() const { return _ustring.data(); }
        int size() const { return _ustring.size(); }

        const char *ascii() const { return _ustring.ascii(); }

        static Identifier from(unsigned y) { return Identifier(UString::from(y)); }

        bool isNull() const { return _ustring.isNull(); }
        bool isEmpty() const { return _ustring.isEmpty(); }

        unsigned long toULong(bool *ok) const { return _ustring.toULong(ok); }
        unsigned toStrictUInt32(bool *ok) const { return _ustring.toStrictUInt32(ok); }
        unsigned toArrayIndex(bool *ok) const { return _ustring.toArrayIndex(ok); }

        double toDouble() const { return _ustring.toDouble(); }

        static const Identifier &null();

        friend bool operator==(const Identifier &, const Identifier &);
        friend bool operator!=(const Identifier &, const Identifier &);

        friend bool operator==(const Identifier &, const char *);

        static void remove(UString::Rep *);

    private:
        UString _ustring;

        static bool equal(UString::Rep *, const char *);
        static bool equal(UString::Rep *, const UChar *, int length);
        static bool equal(UString::Rep *, UString::Rep *);

        static bool equal(const Identifier &a, const Identifier &b)
            { return a._ustring.rep == b._ustring.rep; }
        static bool equal(const Identifier &a, const char *b)
            { return equal(a._ustring.rep, b); }

        static UString::Rep *add(const char *);
        static UString::Rep *add(const UChar *, int length);
        static UString::Rep *add(UString::Rep *);

        static void insert(UString::Rep *);

        static void rehash(int newTableSize);
        static void expand();
        static void shrink();

	// TODO: move into .cpp file
        static UString::Rep **_table;
        static int _tableSize;
        static int _tableSizeMask;
        static int _keyCount;
    };

    inline bool operator==(const Identifier &a, const Identifier &b)
        { return Identifier::equal(a, b); }

    inline bool operator!=(const Identifier &a, const Identifier &b)
        { return !Identifier::equal(a, b); }

    inline bool operator==(const Identifier &a, const char *b)
        { return Identifier::equal(a, b); }

    extern const Identifier argumentsPropertyName;
    extern const Identifier calleePropertyName;
    extern const Identifier constructorPropertyName;
    extern const Identifier lengthPropertyName;
    extern const Identifier messagePropertyName;
    extern const Identifier namePropertyName;
    extern const Identifier prototypePropertyName;
    extern const Identifier specialPrototypePropertyName;
    extern const Identifier toLocaleStringPropertyName;
    extern const Identifier toStringPropertyName;
    extern const Identifier valueOfPropertyName;

}

#endif
