/***************************************************************************
 *   This file is part of KDevelop                                         *
 *   Copyright 2008 Andreas Pakulat <apaku@gmx.de>                         *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU Library General Public License as       *
 *   published by the Free Software Foundation; either version 2 of the    *
 *   License, or (at your option) any later version.                       *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU Library General Public     *
 *   License along with this program; if not, write to the                 *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.         *
 ***************************************************************************/

#include "snippetcompletionmodel.h"

#include <KTextEditor/Document>
#include <KTextEditor/View>

#include <kdeversion.h>

#if KDE_VERSION > KDE_MAKE_VERSION(4, 4, 4)
    #define HAVE_HIGHLIGHT_IFACE
    #include <KTextEditor/HighlightInterface>
#endif

#include "snippetstore.h"
#include "snippetrepository.h"
#include "snippet.h"
#include "snippetcompletionitem.h"

SnippetCompletionModel::SnippetCompletionModel()
    : KTextEditor::CodeCompletionModel(0)
{
    setHasGroups(false);
}

SnippetCompletionModel::~SnippetCompletionModel()
{
}

QVariant SnippetCompletionModel::data( const QModelIndex& idx, int role ) const
{
    if( !idx.isValid() || idx.row() < 0 || idx.row() >= rowCount() ) {
        return QVariant();
    } else {
        return m_snippets.at( idx.row() )->data(idx, role, 0);
    }
}

void SnippetCompletionModel::executeCompletionItem( KTextEditor::Document* doc, const KTextEditor::Range& w, int row ) const
{
    m_snippets.at( row )->execute(doc, w);
}

void SnippetCompletionModel::completionInvoked(KTextEditor::View *view, const KTextEditor::Range &range, InvocationType invocationType)
{
    Q_UNUSED( range );
    Q_UNUSED( invocationType );
    initData(view);
}

void SnippetCompletionModel::initData(KTextEditor::View* view)
{
    QString mode;
    #ifdef HAVE_HIGHLIGHT_IFACE
        if ( KTextEditor::HighlightInterface* iface = qobject_cast<KTextEditor::HighlightInterface*>(view->document()) ) {
            mode = iface->highlightingModeAt(view->cursorPosition());
        }
    #endif // HAVE_HIGHLIGHT_IFACE

    if ( mode.isEmpty() ) {
        mode = view->document()->mode();
    }

    m_snippets.clear();
    SnippetStore* store = SnippetStore::self();
    for(int i = 0; i < store->rowCount(); i++ )
    {
        if ( store->item(i, 0)->checkState() != Qt::Checked ) {
            continue;
        }
        SnippetRepository* repo = dynamic_cast<SnippetRepository*>( store->item( i, 0 ) );
        if( repo && (repo->fileTypes().isEmpty() || repo->fileTypes().contains(mode)) )
        {
            for ( int j = 0; j < repo->rowCount(); ++j ) {
                if ( Snippet* snippet = dynamic_cast<Snippet*>(repo->child(j)) ) {
                    m_snippets << new SnippetCompletionItem(snippet);
                }
            }
        }
    }
    reset();
    setRowCount( m_snippets.count() );
}

#include "snippetcompletionmodel.moc"
