/***************************************************************************
 *   Copyright (C) 2004, 2005 Thomas Nagy                                  *
 *   tnagy2^8@yahoo.fr                                                     *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License version 2        *
 *   as published by the Free Software Foundation (see COPYING)            *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 ***************************************************************************/

#ifndef _DCANVASVIEW_H
#define _DCANVASVIEW_H

#include <qcanvas.h>
#include <qvaluelist.h>

#include "DCanvasRef.h"
#include "DGuiView.h"
#include "DGuiItem.h"
#include "DItem.h"
#include "DCanvasView.h"
#include "DCanvasTip.h"


class QPoint;
class DCanvasItem;
class DCanvasPopup;
class QRect;
class DCanvasRef;

/**
 * This class is a view for the mindmap
 *
 * @short Mindmap view
 * @author Thomas Nagy <tnagy2^8@yahoo.fr>
 * @version 0.3.1
 */
class DCanvasView : public QCanvasView, public DGuiView
{
	Q_OBJECT

	friend class DCanvasTip;
	public:
		enum ActionType {
			act_point,
			act_link,
			act_sort,
			act_scroll
		};

		DCanvasView( QWidget* parent=0, const char* name=0 );
		~DCanvasView();

		bool oneItemIsSelected() const;
		bool itemsAreSelected() const;

		/* 
		 * clear the selection
		 */
		void deselectAll();

		/*
		 * This method is used to change the action type
		 * triggered by the user
		 */
		void setActionType(ActionType action);

		/* 
		 * Use this to avoid casting Item() all the time
		 */
		DCanvasItem* canvasItem(int);

		/*
		 * This method is used to return the minimum window containing
		 * all mindmap items
		 */
		QRect canvasSize();

		inline const DCanvasRef* selectedRef() {return m_selectedref;}
		inline const DCanvasLink* selectedLink() {return m_selectedlink;}

		// action to request an existing color theme
		void applyColorScheme(int sel);
		// action to request a custom color
		void applyCustomColor();

	signals:
		/*
		 * redrawDone is emitted after the redrawing of the canvas is done.  It is used
		 * to detect when the complete loading sequence, including the initial draw and the
		 * data manipulations resulting from that, is finished. At that point, the changed
		 * flags can be reset to false, so real document changes can be detected.
		 */
		void redrawDone();
		
		/*
		 * selectionChanged is emitted after the object selected have changed,
		 */
		void selectionChanged();

	protected:
		void contentsMouseReleaseEvent(QMouseEvent* me);
		void contentsMouseMoveEvent(QMouseEvent* me);
		void contentsMouseDoubleClickEvent(QMouseEvent* me);
		void contentsMousePressEvent(QMouseEvent* me);
		void contentsWheelEvent( QWheelEvent* );
		void keyPressEvent( QKeyEvent* );
		void keyReleaseEvent( QKeyEvent* );

		void drawContents( QPainter*, int, int, int, int);
		void dragEnterEvent(QDragEnterEvent *event);
		void dropEvent(QDropEvent *event);

	private:

		// global tooltip (TODO : improve it to display pictures, .. like in konqueror)
		DCanvasTip* m_tooltip;

		///
		DCanvasItem* m_startItem;

		/// last item selected
		int m_lastitem;

		/// last item created or selected
		int m_lastitemtoselect;

		/// rubber band for link mode
		QCanvasLine* m_rubberLine;

		/// rubber band for selecting items
		QCanvasRectangle* m_rubberRect;

		/// start of a movement (necessary for rubberbands)
		QPoint m_moveStart;


		/// popup menu for item properties
		DCanvasPopup* m_menu;

		/// necessary for connecting the view
		void plug();
		void unplug();

		int m_itemtosort;
		int m_sortcount;

		ActionType m_currentAction;
		ActionType m_previousAction;

		void addItemToSelection( int );
		void removeSelectedItems();

		// on mouse press, locate the corresponding item
		int locateItem( const QPoint& );

		// same as above, but used to change the sort order of an item
		int locateItemPos( const QPoint& );

		/// center the view on a particular mindmap item
		void centerOnObject( int obj );

		void makeObjectVisible( int obj );

		void updateObjLinks( int obj );

		// is the mouse left button currently pressed ?
		bool m_pressed;

		/// the user is trying to select multiple items with the shift key (multiselect)
		bool m_multiSelect;

		bool m_ctrlIsPressed;

		/// the user is trying to select multiple items with the rubberband
		bool m_canSelect;

		/// has the user selected something ? he may want to keep the selection
		bool m_justSelectedSomething;

		DCanvasRef* m_selectedref;
		DCanvasLink* m_selectedlink;

		// selected items - by id
		QValueList<int> m_selectedList;
		QValueList<int>::iterator m_it;

	public slots:
		void popupMenuSel(int sel);

		void tidyClean();
		void tidyCanvas();
		void tidyUnclutter();

		/// focus on the root of the biggest tree
		void focusOnRoot();

		/// cycle the focus on the roots of trees
		void focusOnNextRoot();

		/// change the view
		void selectObjUp();
		void selectObjDown();
		void selectObjLeft();
		void selectObjRight();

		/// move objects
		void moveSelObjectsUp();
		void moveSelObjectsDown();
		void moveSelObjectsLeft();
		void moveSelObjectsRight();

		/// add objects with an action
		void addChild();
		void addSibling();


	protected slots:
		void settingsChanged();
		void updateItem(int id);
		void createItem(int);
		void removeItem(int);
		void updateSelectedItem(int, DGuiView*);
		void changeRef(int, int, bool add=true);
};

#endif // _DCANVASVIEW_H

