/*
 * linux/kernel/power/swsusp.c
 *
 * This file is to realize architecture-independent
 * machine suspend feature using pretty near only high-level routines
 *
 * Copyright (C) 1998-2001 Gabor Kuti <seasons@fornax.hu>
 * Copyright (C) 1998,2001-2004 Pavel Machek <pavel@suse.cz>
 *
 * This file is released under the GPLv2.
 *
 * I'd like to thank the following people for their work:
 * 
 * Pavel Machek <pavel@ucw.cz>:
 * Modifications, defectiveness pointing, being with me at the very beginning,
 * suspend to swap space, stop all tasks. Port to 2.4.18-ac and 2.5.17.
 *
 * Steve Doddi <dirk@loth.demon.co.uk>: 
 * Support the possibility of hardware state restoring.
 *
 * Raph <grey.havens@earthling.net>:
 * Support for preserving states of network devices and virtual console
 * (including X and svgatextmode)
 *
 * Kurt Garloff <garloff@suse.de>:
 * Straightened the critical function in order to prevent compilers from
 * playing tricks with local variables.
 *
 * Andreas Mohr <a.mohr@mailto.de>
 *
 * Alex Badea <vampire@go.ro>:
 * Fixed runaway init
 *
 * More state savers are welcome. Especially for the scsi layer...
 *
 * For TODOs,FIXMEs also look in Documentation/power/swsusp.txt
 */

#include <linux/module.h>
#include <linux/mm.h>
#include <linux/suspend.h>
#include <linux/utsname.h>
#include <linux/version.h>
#include <linux/kernel.h>
#include <linux/major.h>
#include <linux/swap.h>
#include <linux/pm.h>
#include <linux/syscalls.h>
#include <linux/console.h>
#include <linux/init.h>
#include <linux/moduleparam.h>
#include <linux/kdev_t.h>
#include <linux/blkdev.h>
#include <linux/buffer_head.h>
#include <linux/swapops.h>
#include <linux/cpumask.h>
#include <linux/slab.h>
#include <linux/fs.h>
#include <linux/string.h>
#include <linux/pagemap.h>

#include "power.h"
#include "swsusp.h"

unsigned char software_suspend_enabled = 0;

static int noresume __initdata;
static char resume_file[256] __initdata;

/*
 * Saving part...
 */

static int prepare_suspend_processes(void)
{
	sys_sync();	/* Syncing needs pdflushd, so do it before stopping processes */
	if (freeze_processes()) {
		printk( KERN_ERR "Suspend failed: Not all processes stopped!\n" );
		thaw_processes();
		return 1;
	}
	return 0;
}

/*
 * Try to free as much memory as possible, but do not OOM-kill anyone
 *
 * Notice: all userland should be stopped at this point, or livelock is possible.
 */
static void free_some_memory(void)
{
	printk("Freeing memory: ");
	while (shrink_all_memory(10000))
		printk(".");
	printk("|\n");
}

/*
 * This is main interface to the outside world. It needs to be
 * called from process context.
 */
static int software_suspend(void)
{
	int res;
	if (!software_suspend_enabled)
		return -EAGAIN;

	software_suspend_enabled = 0;
	might_sleep();

	if (arch_prepare_suspend()) {
		printk("%sArchitecture failed to prepare\n", name_suspend);
		return -EPERM;
	}		
	if (pm_prepare_console())
		printk( "%sCan't allocate a console... proceeding\n", name_suspend);
	if (!prepare_suspend_processes()) {

		/* At this point, all user processes and "dangerous"
                   kernel threads are stopped. Free some memory, as we
                   need half of memory free. */

		free_some_memory();
		
		/* Save state of all device drivers, and stop them. */		   
		if ((res = device_suspend(4))==0)
			/* If stopping device drivers worked, we proceed basically into
			 * suspend_save_image.
			 *
			 * do_magic(0) returns after system is resumed.
			 *
			 * do_magic() copies all "used" memory to "free" memory, then
			 * unsuspends all device drivers, and writes memory to disk
			 * using normal kernel mechanism.
			 */
			do_magic(0);
		thaw_processes();
	} else
		res = -EBUSY;
	software_suspend_enabled = 1;
	MDELAY(1000);
	pm_restore_console();
	return res;
}

/* More restore stuff */

/* FIXME: Why not memcpy(to, from, 1<<pagedir_order*PAGE_SIZE)? */
static void copy_pagedir(suspend_pagedir_t *to, suspend_pagedir_t *from)
{
	int i;
	char *topointer=(char *)to, *frompointer=(char *)from;

	for(i=0; i < 1 << pagedir_order; i++) {
		copy_page(topointer, frompointer);
		topointer += PAGE_SIZE;
		frompointer += PAGE_SIZE;
	}
}

#define does_collide(addr) does_collide_order(pagedir_nosave, addr, 0)

/*
 * Returns true if given address/order collides with any orig_address 
 */
static int does_collide_order(suspend_pagedir_t *pagedir, unsigned long addr,
		int order)
{
	int i;
	unsigned long addre = addr + (PAGE_SIZE<<order);
	
	for(i=0; i < nr_copy_pages; i++)
		if((pagedir+i)->orig_address >= addr &&
			(pagedir+i)->orig_address < addre)
			return 1;

	return 0;
}

/*
 * We check here that pagedir & pages it points to won't collide with pages
 * where we're going to restore from the loaded pages later
 */
static int check_pagedir(void)
{
	int i;

	for(i=0; i < nr_copy_pages; i++) {
		unsigned long addr;

		do {
			addr = get_zeroed_page(GFP_ATOMIC);
			if(!addr)
				return -ENOMEM;
		} while (does_collide(addr));

		(pagedir_nosave+i)->address = addr;
	}
	return 0;
}

static int relocate_pagedir(void)
{
	/*
	 * We have to avoid recursion (not to overflow kernel stack),
	 * and that's why code looks pretty cryptic 
	 */
	suspend_pagedir_t *new_pagedir, *old_pagedir = pagedir_nosave;
	void **eaten_memory = NULL;
	void **c = eaten_memory, *m, *f;

	printk("Relocating pagedir");

	if(!does_collide_order(old_pagedir, (unsigned long)old_pagedir, pagedir_order)) {
		printk("not necessary\n");
		return 0;
	}

	while ((m = (void *) __get_free_pages(GFP_ATOMIC, pagedir_order))) {
		memset(m, 0, PAGE_SIZE);
		if (!does_collide_order(old_pagedir, (unsigned long)m, pagedir_order))
			break;
		eaten_memory = m;
		printk( "." ); 
		*eaten_memory = c;
		c = eaten_memory;
	}

	if (!m)
		return -ENOMEM;

	pagedir_nosave = new_pagedir = m;
	copy_pagedir(new_pagedir, old_pagedir);

	c = eaten_memory;
	while(c) {
		printk(":");
		f = *c;
		c = *c;
		if (f)
			free_pages((unsigned long)f, pagedir_order);
	}
	printk("|\n");
	return 0;
}

/*
 * Sanity check if this image makes sense with this kernel/swap context
 * I really don't think that it's foolproof but more than nothing..
 */

static int sanity_check_failed(char *reason)
{
	printk(KERN_ERR "%s%s\n", name_resume, reason);
	return -EPERM;
}

static int sanity_check(struct suspend_header *sh)
{
	if (sh->version_code != LINUX_VERSION_CODE)
		return sanity_check_failed("Incorrect kernel version");
	if (sh->num_physpages != num_physpages)
		return sanity_check_failed("Incorrect memory size");
	if (strncmp(sh->machine, system_utsname.machine, 8))
		return sanity_check_failed("Incorrect machine type");
	if (strncmp(sh->version, system_utsname.version, 20))
		return sanity_check_failed("Incorrect version");
	if (sh->num_cpus != num_online_cpus())
		return sanity_check_failed("Incorrect number of cpus");
	if (sh->page_size != PAGE_SIZE)
		return sanity_check_failed("Incorrect PAGE_SIZE");
	return 0;
}

static int bdev_read_page(struct block_device *bdev, long pos, void *buf)
{
	struct buffer_head *bh;
	BUG_ON (pos%PAGE_SIZE);
	bh = __bread(bdev, pos/PAGE_SIZE, PAGE_SIZE);
	if (!bh || (!bh->b_data)) {
		return -1;
	}
	memcpy(buf, bh->b_data, PAGE_SIZE);	/* FIXME: may need kmap() */
	BUG_ON(!buffer_uptodate(bh));
	brelse(bh);
	return 0;
} 

static int bdev_write_page(struct block_device *bdev, long pos, void *buf)
{
#if 0
	struct buffer_head *bh;
	BUG_ON (pos%PAGE_SIZE);
	bh = __bread(bdev, pos/PAGE_SIZE, PAGE_SIZE);
	if (!bh || (!bh->b_data)) {
		return -1;
	}
	memcpy(bh->b_data, buf, PAGE_SIZE);	/* FIXME: may need kmap() */
	BUG_ON(!buffer_uptodate(bh));
	generic_make_request(WRITE, bh);
	if (!buffer_uptodate(bh))
		printk(KERN_CRIT "%sWarning %s: Fixing swap signatures unsuccessful...\n", name_resume, resume_file);
	wait_on_buffer(bh);
	brelse(bh);
	return 0;
#endif
	printk(KERN_CRIT "%sWarning %s: Fixing swap signatures unimplemented...\n", name_resume, resume_file);
	return 0;
}

static int __init __read_suspend_image(struct block_device *bdev, union diskpage *cur, int noresume)
{
	swp_entry_t next;
	int i, nr_pgdir_pages;

#define PREPARENEXT \
	{	next = cur->link.next; \
		next.val = swp_offset(next) * PAGE_SIZE; \
        }

	if (bdev_read_page(bdev, 0, cur)) return -EIO;

	if ((!memcmp("SWAP-SPACE",cur->swh.magic.magic,10)) ||
	    (!memcmp("SWAPSPACE2",cur->swh.magic.magic,10))) {
		printk(KERN_ERR "%sThis is normal swap space\n", name_resume );
		return -EINVAL;
	}

	PREPARENEXT; /* We have to read next position before we overwrite it */

	if (!memcmp("S1",cur->swh.magic.magic,2))
		memcpy(cur->swh.magic.magic,"SWAP-SPACE",10);
	else if (!memcmp("S2",cur->swh.magic.magic,2))
		memcpy(cur->swh.magic.magic,"SWAPSPACE2",10);
	else {
		if (noresume)
			return -EINVAL;
		panic("%sUnable to find suspended-data signature (%.10s - misspelled?\n", 
			name_resume, cur->swh.magic.magic);
	}
	if (noresume) {
		/* We don't do a sanity check here: we want to restore the swap
		   whatever version of kernel made the suspend image;
		   We need to write swap, but swap is *not* enabled so
		   we must write the device directly */
		printk("%s: Fixing swap signatures %s...\n", name_resume, resume_file);
		bdev_write_page(bdev, 0, cur);
	}

	printk( "%sSignature found, resuming\n", name_resume );
	MDELAY(1000);

	if (bdev_read_page(bdev, next.val, cur)) return -EIO;
	if (sanity_check(&cur->sh)) 	/* Is this same machine? */	
		return -EPERM;
	PREPARENEXT;

	pagedir_save = cur->sh.suspend_pagedir;
	nr_copy_pages = cur->sh.num_pbes;
	nr_pgdir_pages = SUSPEND_PD_PAGES(nr_copy_pages);
	pagedir_order = get_bitmask_order(nr_pgdir_pages);

	pagedir_nosave = (suspend_pagedir_t *)__get_free_pages(GFP_ATOMIC, pagedir_order);
	if (!pagedir_nosave)
		return -ENOMEM;

	PRINTK( "%sReading pagedir, ", name_resume );

	/* We get pages in reverse order of saving! */
	for (i=nr_pgdir_pages-1; i>=0; i--) {
		BUG_ON (!next.val);
		cur = (union diskpage *)((char *) pagedir_nosave)+i;
		if (bdev_read_page(bdev, next.val, cur)) return -EIO;
		PREPARENEXT;
	}
	BUG_ON (next.val);

	if (relocate_pagedir())
		return -ENOMEM;
	if (check_pagedir())
		return -ENOMEM;

	printk( "Reading image data (%d pages): ", nr_copy_pages );
	for(i=0; i < nr_copy_pages; i++) {
		swp_entry_t swap_address = (pagedir_nosave+i)->swap_address;
		if (!(i%100))
			printk( "." );
		/* You do not need to check for overlaps...
		   ... check_pagedir already did this work */
		if (bdev_read_page(bdev, swp_offset(swap_address) * PAGE_SIZE, (char *)((pagedir_nosave+i)->address)))
			return -EIO;
	}
	printk( "|\n" );
	return 0;
}

static int __init read_suspend_image(const char * specialfile, int noresume)
{
	union diskpage *cur;
	unsigned long scratch_page = 0;
	int error;
	char b[BDEVNAME_SIZE];
#ifdef MODULE
	char *p;

	swsusp_resume_device =
		new_decode_dev(simple_strtoul(specialfile, &p, 16));
	if (*p)
		swsusp_resume_device = 0;
#else
	extern dev_t __init name_to_dev_t(const char *line);

	swsusp_resume_device = name_to_dev_t(specialfile);
#endif
	if (!swsusp_resume_device) {
		printk(KERN_ERR "%s%s: Invalid device\n", name_resume,
		       specialfile);
		error = -EINVAL;
		goto out;
	}
	scratch_page = get_zeroed_page(GFP_ATOMIC);
	cur = (void *) scratch_page;
	if (cur) {
		struct block_device *bdev;
		printk("Resuming from device %s\n",
				__bdevname(swsusp_resume_device, b));
		bdev = open_by_devnum(swsusp_resume_device, FMODE_READ);
		if (IS_ERR(bdev)) {
			error = PTR_ERR(bdev);
		} else {
			set_blocksize(bdev, PAGE_SIZE);
			error = __read_suspend_image(bdev, cur, noresume);
			blkdev_put(bdev);
		}
	} else error = -ENOMEM;

	if (scratch_page)
		free_page(scratch_page);
	switch (error) {
		case 0:
			PRINTK("Reading resume file was successful\n");
			break;
		case -EINVAL:
			break;
		case -EIO:
			printk( "%sI/O error\n", name_resume);
			break;
		case -ENOENT:
			printk( "%s%s: No such file or directory\n", name_resume, specialfile);
			break;
		case -ENOMEM:
			printk( "%sNot enough memory\n", name_resume);
			break;
		default:
			printk( "%sError %d resuming\n", name_resume, error );
	}
out:
	MDELAY(1000);
	return error;
}

/**
 *	software_resume - Resume from a saved image.
 *
 *	Called as a late_initcall (so all devices are discovered and 
 *	initialized), we call swsusp to see if we have a saved image or not.
 *	If so, we quiesce devices, then restore the saved image. We will 
 *	return above (in pm_suspend_disk() ) if everything goes well. 
 *	Otherwise, we fail gracefully and return to the normally 
 *	scheduled program.
 *
 */
static int __init software_resume(void)
{
	if (num_online_cpus() > 1) {
		printk(KERN_WARNING "Software Suspend has malfunctioning SMP support. Disabled :(\n");	
		return -EINVAL;
	}
	/* We enable the possibility of machine suspend */
	software_suspend_enabled = 1;

	down(&software_suspend_sem);
	software_suspend_module = THIS_MODULE;
	software_suspend_hook = software_suspend;
	up(&software_suspend_sem);

	if (!noresume && !resume_file[0])
		return 0;

	printk( "%s", name_resume );
	if (noresume) {
		if(resume_file[0])
			read_suspend_image(resume_file, 1);
		printk( "disabled\n" );
		return 0;
	}
	MDELAY(1000);

	if (pm_prepare_console())
		printk("swsusp: Can't allocate a console... proceeding\n");

	printk( "resuming from %s\n", resume_file);
	if (read_suspend_image(resume_file, 0))
		goto read_failure;
	device_suspend(4);
	do_magic(1);
	panic("This never returns");

read_failure:
	pm_restore_console();
	return 0;
}

static void __exit software_resume_exit(void)
{
	down(&software_suspend_sem);
	software_suspend_module = 0;
	up(&software_suspend_sem);
}

late_initcall(software_resume);
module_exit(software_resume_exit);

module_param(noresume, bool, 0);
module_param_string(resume, resume_file, sizeof(resume_file), 0);

MODULE_LICENSE("GPL");

EXPORT_SYMBOL(software_suspend_enabled);
