/*
 * This file is part of the KFTPGrabber project
 *
 * Copyright (C) 2003-2004 by the KFTPGrabber developers
 * Copyright (C) 2003-2004 Jernej Kos <kostko@jweb-network.net>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * is provided AS IS, WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE, and
 * NON-INFRINGEMENT.  See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Steet, Fifth Floor, Boston,
 * MA 02110-1301, USA.
 *
 * In addition, as a special exception, the copyright holders give
 * permission to link the code of portions of this program with the
 * OpenSSL library under certain conditions as described in each
 * individual source file, and distribute linked combinations
 * including the two.
 * You must obey the GNU General Public License in all respects
 * for all of the code used other than OpenSSL.  If you modify
 * file(s) with this exception, you may extend this exception to your
 * version of the file(s), but you are not obligated to do so.  If you
 * do not wish to do so, delete this exception statement from your
 * version.  If you delete this exception statement from all source
 * files in the program, then also delete it here.
 */

#ifndef KFTPNETWORKSOCKET_H
#define KFTPNETWORKSOCKET_H

#include <stdio.h>
#include <sys/types.h>

#include <sys/socket.h>
#include <sys/stat.h>
#include <utime.h>
#include <fcntl.h>
#include <netinet/in.h>
#include <arpa/inet.h>

#include <unistd.h>
#include <netdb.h>
#include <stdlib.h>

#include <qobject.h>
#include <qmap.h>
#include <qptrlist.h>
#include <qvaluestack.h>

#include <kurl.h>
#include <kextsock.h>
#include <kremoteencoding.h>

#include "kftpextensions.h"
#include "errorhandler.h"

namespace KFTPQueue {
  class Transfer;
}

enum SockFeatures {
  SF_FXP_TRANSFER = 1,
  SF_RAW_COMMAND  = 2
};

namespace KFTPNetwork {

/**
 * This class enables per-socket speed limiting. Every underlying protocol implementation
 * should use the SpeedLimit object provided by KFTPNetwork::Socket.
 *
 * @author Jernej Kos
 */
class SpeedLimit {
public:
    SpeedLimit() {};
    
    /**
     * Set the desired speed limit (in bytes/sec).
     *
     * @param speed The speed limit
     */
    void setLimit(int speed) { m_speedLimit = speed; }
    
    /**
     * Should be called by the socket when transfer starts.
     */
    void transferStart();
    
    /**
     * Should be called by the socket before reading. It will return the number
     * of bytes the socket can read.
     *
     * @return The number of bytes the socket is allowed to read
     */
    int allowedBytes();
private:
    int m_speedLimit;
    struct timeval m_lastPoll;
    
    static long _timeSince(struct timeval *then);
};

enum SocketState {
  S_CONNECT,
  S_LOGIN,
  S_ABORT,
  S_TRANSFER,
  S_LIST,
  S_SCAN,
  S_REMOVE,
  S_IDLE
};

/**
 * This class provides socket state handling. The underlying Socket implementation should
 * use it when entering different states, so the current socket state is known to the other
 * parts of the application.
 *
 * @author Jernej Kos
 */
class SocketStateInfo
{
public:
    SocketStateInfo(Socket *socket);
    
    /**
     * This method returns the current socket state.
     *
     * @return The current SocketState
     */
    SocketState getSocketState();
    
    /**
     * This method enters a new socket state. It should be called whenever the current
     * operation changes. This method will also call abortFinished() if the socket is
     * currently aborting.
     *
     * @param state The new SocketState
     * @param compositeSate Set this to true when doing larger operations that can have substates
     */
    void enterSocketState(SocketState state, bool compositeState = false);
    
    /**
     * This method sets the abort marker, so the current operation will be aborted.
     */
    void abortCurrentState() { m_abortInProgress = true; }
    
    /**
     * The underlying socket should use this method to check for abortion when doing
     * longer operations.
     *
     * @return True if the current operation must be aborted
     */
    bool abortInProgress() { return m_abortInProgress; }
    
    /**
     * This method clears the abort marker.
     */
    void abortFinished();
    
    /**
     * This will completely reset the state queue and set the current state to S_IDLE.
     */
    void resetState();
private:
    Socket *m_socket;
    
    QValueStack<SocketState> m_stateStack;
    bool m_abortInProgress;
    bool m_compositeState;
    SocketState m_sockState;
};

/**
 * This class represents a network protocol abstract. Actual protocols should
 * inherit this class and implement the virtual methods.
 *
 * @author Jernej Kos
 */
class Socket : public QObject
{
Q_OBJECT
friend class ErrorHandler;
friend class SocketStateInfo;
public:
    Socket(QObject *parent, const QString &protocol);
    ~Socket();
    
    void setConfig(const QString &key, int val) { m_sockConfig[key] = QString::number(val); }
    void setConfig(const QString &key, const QString &val) { m_sockConfig[key] = val; }
    int getConfig(const QString &key) { return m_sockConfig[key].toInt(); }
    QString getConfigStr(const QString &key) { return m_sockConfig[key]; }
    
    /**
     * This sets the site's url.
     *
     * @param url The url
     */
    void setSiteUrl(KURL url);
    
    /**
     * This establishes a socket connection to the given server. All proper
     * login methods will be called.
     *
     * @param url KURL that will be used for connection
     */
    void connect(KURL url);
    
    /**
     * Drop current connection and disconnect from the server. All KSSL objects
     * are cleared if this is a TLS connection.
     */
    void disconnect();
    
    /**
     * Abort current FTP operation.
     */
    void abort() { m_stateInfo.abortCurrentState(); }
    
    /**
     * Gets (downloads) a file from the remote server.
     *
     * @param source Source (remote) path
     * @param destination Destination (local) path
     */
    virtual void get(KURL source, KURL destination) = 0;
    
    /**
     * Puts (uploads) a file to the remote server.
     *
     * @param source Source (local) path
     * @param destination Destination (remote) path
     */
    virtual void put(const KURL &source, const KURL &destination) = 0;
    
    /**
     * Removes a file or directory from the server. The filetype is determined on
     * the spot using "checkIsDir" method.
     *
     * @see checkIsDir
     * @param url Path of the file/directory that should be removed
     */
    virtual void remove(const KURL &url) = 0;
    
    /**
     * Renames a file on the remote server.
     *
     * @param source File path to be renamed
     * @param destination The new name (must be complete path)
     */
    virtual void rename(const KURL &source, const KURL &destination) = 0;
    
    /**
     * Change remote file's permissions.
     *
     * @param url File path of which permissions should be changed
     * @param mode New file mode in decimal form (eg. 644)
     */
    virtual void chmod(const KURL &url, int mode) = 0;
    
    /**
     * Creates a directory on the remote server.
     *
     * @param url Path of the directory to be created
     */
    virtual void mkdir(const KURL &url) = 0;
    
    /**
     * Starts a site-to-site transfer between two remote hosts. This function
     * should do nothing if the protocol doesn't support it.
     *
     * @param source Source url
     * @param destination Destination url
     * @param client The other KFTPNetwork::Socket
     */
    virtual void fxpTransfer(const KURL &source, const KURL &destination, Socket *client) = 0;
    
    /**
     * Deletes a directory recursively.
     *
     * @param url Url of the directory to be deleted
     */
    void recursiveDelete(const KURL &url);
    
    /**
     * Recursively scans a directory and queues all files/directories found.
     *
     * @param parent Parent transfer to be scanned
     */
    void recursiveScan(KFTPQueue::Transfer *parent);
    
    /**
     * Get dir listing for a specific remote URL. This does a LIST command on the
     * FTP server and then parses the output from the data connection. Cache is
     * first checked if apropriate option is set.
     *
     * @param url URL to get the listing for
     * @param ignoreCache If set to true, the cache will not be checked
     * @return A FTPDirList object which contains parsed data
     */
    FTPDirList *dirList(const KURL &url, bool ignoreCache = false);
    
    /**
     * Retrieve as much information about a file/directory as possible. The
     * sigStatResult signal will be emitted when successfull.
     *
     * @param url File for which to retreieve stats for
     */
    virtual void stat(const KURL &url) = 0;
    
    /**
     * Return the features of this socket.
     *
     * @return Features of the current socket
     */
    virtual int getFeatures() = 0;
    
    /**
     * Return this socket's protocol.
     *
     * @return This socket's protocol identifier
     */
    QString getProtocol() { return m_protocol; }
    
    /**
     * Return current connection info as a KURL.
     *
     * @return KURL that has all parameters set the same it was given to connect
     */
    KURL getClientInfoUrl() { return m_currentUrl; }
    
    /**
     * Function that returns the last successfull directory listing outputed
     * by the dirList method.
     *
     * @see dirList
     * @return A FTPDirList object which contains parsed data
     */
    FTPDirList *getLastDirList() { return &m_lastDirList; }
    
    /**
     * Function that returns the currently processed (downloaded, uploaded, ...)
     * filesize.
     *
     * @return Currently processed filesize
     */
    filesize_t getProcessed() { return m_processedSize; }
    
    /**
     * Get current transfer speed.
     *
     * @return Transfer speed
     */
    filesize_t getSpeed();
    
    /**
     * Set the desired speed limit (in bytes/sec).
     *
     * @param speed The speed limit
     */
    void setSpeedLimit(filesize_t speed) { m_speedLimiter.setLimit(speed); }
    
    /**
     * Is this client connected ?
     *
     * @return True if the connection is currently established, false otherwise
     */
    bool isConnected() { return m_isLoggedIn; }
    
    /**
     * Is the current connection encrypted or not. This method should be reimplemented
     * by underlying socket implementation. The default allways returns false.
     *
     * @return True if the connection is encrypted, false otherwise
     */
    virtual bool isEncrypted() { return false; }
    
    /**
     * Is this client currently busy with operation ?
     *
     * @return True if the client is busy, false otherwise
     */
    bool isBusy() { return m_stateInfo.getSocketState() != S_IDLE; }
    
    /**
     * Checks if the specified path is a file or a directory.
     *
     * @param url Path to be checked
     * @return True if the path is a directory
     */
    bool checkIsDir(const KURL &url);
    
    /**
     * Resets current connection settings to default values.
     */
    void initConfig();
    
    /**
     * Returs the current response buffer.
     *
     * @return Current response buffer
     */
    QString getRespBuffer() { return QString(m_responseBuf); }
    
    /**
     * Sends a raw command to the server and reads the response. Note: this method should
     * only be used OUTSIDE this class. For internal purpuses there is protoSendCommand!
     * This method does nothing if the destination socket doesn't support the
     * SW_RAW_COMMAND feature.
     *
     * @param cmd Raw command to send
     */
    void rawCommand(const QString &cmd);
    
    /**
     * Changes the current encoding for this socket. The change will be visible in the
     * next command parsed.
     *
     * @param encoding The new encoding
     */
    void changeEncoding(const QString &encoding);
    
    /**
     * Returns the default directory (usualy user's home directory). This method should
     * be overriden by the underlying socket class. By default, this method always returns
     * the root dir (/).
     *
     * @return Default directory
     */
    virtual QString getDefaultDir() { return QString("/"); }
    
    /**
     * Returns the directory last returned by CWD. This method should be overriden by the
     * underlying socket class. By default, this method always returns the root dir (/).
     *
     * @return Last directory
     */
   virtual QString getLastDir() { return QString("/"); }
protected:
    /* Helper classes */
    SocketStateInfo m_stateInfo;
    ErrorHandler *m_errorHandler;
    SpeedLimit m_speedLimiter;
    
    /* FTP Client status */
    bool m_isLoggedIn;
    
    /* Buffer */
    char m_responseBuf[1024];
    
    /* Latest directory listing */
    FTPDirList m_lastDirList;
    
    /* Remote encoding engine */
    KRemoteEncoding *m_remoteEncoding;
    
    virtual int protoConnect() = 0;
    virtual void protoDisconnect() = 0;
    
    virtual int protoLogin() = 0;
    virtual bool protoDirList(const KURL &url) = 0;
    virtual int protoCwd(const QString &dir) = 0;
    virtual int protoRmdir(const QString &dir) = 0;
    virtual int protoDelete(const QString &path) = 0;
    virtual void protoSendRawCommand(const QString &command) = 0;
    
    void processedSize(filesize_t size);
    void setCompleteSize(filesize_t size);
    void setOffset(filesize_t offset);
private:
    int m_reconnectTimeout;
    bool m_busy[2];
    
    QMap<QString, QString> m_sockConfig;
    QString m_protocol;
    KURL m_currentUrl;
    
    /* Current operation status */
    filesize_t m_processedOffset;
    filesize_t m_processedSize;
    int m_stallCount;
    time_t m_lastTime;
    time_t m_operationStart;
    filesize_t m_lastXfered;
    
    bool recRecursiveScan(KFTPQueue::Transfer *parent);
private slots:
    void slotReconnect();
signals:
    void sigRawReply(const QString &reply);
    void sigLogUpdate(int type, const QString &text);
    void sigDisconnectDone();
    void sigLoginComplete(bool success);
    void sigStatResult(FTPEntry e);
    void sigResumedOffset(filesize_t bytes);
    void sigError(KFTPNetwork::Error error);
    void sigRetrySuccess();
    
    void sigStatusChanged(bool busy);
    void sigStateChanged(KFTPNetwork::SocketState state);
};

/**
 * This class represents the "socket abstraction layer". It is able to switch
 * between various protocols depending on the provided @ref KURL.
 *
 * @author Jernej Kos
 */
class SocketManager : public QObject
{
Q_OBJECT
public:
    SocketManager(QObject *parent);
    
    void setProtocol(KURL url);
    Socket *socket() { return m_socket; }
private:
    QPtrList<Socket> m_socketList;
    QString m_protocol;
    Socket *m_socket;
    
    void addProtocol(Socket *socket);
};

}

#endif
