/*
 * This file is part of the KFTPGrabber project
 *
 * Copyright (C) 2003-2004 by the KFTPGrabber developers
 * Copyright (C) 2003-2004 Jernej Kos <kostko@jweb-network.net>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * is provided AS IS, WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE, and
 * NON-INFRINGEMENT.  See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Steet, Fifth Floor, Boston,
 * MA 02110-1301, USA.
 *
 * In addition, as a special exception, the copyright holders give
 * permission to link the code of portions of this program with the
 * OpenSSL library under certain conditions as described in each
 * individual source file, and distribute linked combinations
 * including the two.
 * You must obey the GNU General Public License in all respects
 * for all of the code used other than OpenSSL.  If you modify
 * file(s) with this exception, you may extend this exception to your
 * version of the file(s), but you are not obligated to do so.  If you
 * do not wish to do so, delete this exception statement from your
 * version.  If you delete this exception statement from all source
 * files in the program, then also delete it here.
 */

#include "kftptransferdir.h"
#include "kftpqueue.h"
#include "kftpsession.h"

namespace KFTPQueue {

TransferDir::TransferDir(QObject *parent)
  : Transfer(parent, Transfer::Directory),
    m_childIterator(QPtrListIterator<QueueObject>(m_children))
{
  m_dirScanDone = false;
}

TransferDir::~TransferDir()
{
}

void TransferDir::execute()
{
  // Let's allocate some sessions for this transfer (actually for all child transfers)
  if (!m_srcSession) {
    m_srcSession = FTPSessionManager->spawnRemoteSession(IGNORE_SIDE, m_sourceUrl, 0, true);
    m_dstSession = FTPSessionManager->spawnRemoteSession(OPPOSITE_SIDE(m_srcSession->getSide()), m_destUrl, 0, true);
    
    if (!m_srcSession->isFreeConnection() || !m_dstSession->isFreeConnection()) {
      // There are no free connections
      // FIXME what to do ?
      return;
    }
  }
  
  if (!m_srcSession->isConnected() || !m_dstSession->isConnected()) {
    // The sessions are not yet connected, wait for connections
    connect(m_srcSession->getClient(), SIGNAL(loginComplete(bool)), this, SLOT(slotSessionConnected(bool)));
    connect(m_dstSession->getClient(), SIGNAL(loginComplete(bool)), this, SLOT(slotSessionConnected(bool)));
    
    m_status = Connecting;
    return;
  }
  
  if (!m_dirScanDone && !hasParentTransfer() && getRemainingChildren() == 0) {
    // We need to scan the directory if it contains some files =)
    m_srcSession->scanDirectory(this);
    m_dirScanDone = true;
    
    connect(m_srcSession, SIGNAL(dirScanDone()), this, SLOT(slotDirScanDone()));
    return;
  }
  
  m_status = Running;
  
  if (!m_childIterator.toFirst()) {
    // We are finished
    m_deleteMe = true;
    resetTransfer();
    
    emit transferComplete(m_id);
    KFTPQueue::Manager::self()->doEmitUpdate();
    return;
  }
  
  slotExecuteChild();
}

void TransferDir::abort()
{
  // If not running, just return
  if (!isRunning()) return;
  
  Transfer::abort();
  
  // Signal abort to all child transfers
  if (!m_deleteMe) {
    // Only if we actualy have children
    if (getRemainingChildren() > 0) {
      QueueObject *i;
  
      for (i = m_children.first(); i; i = m_children.next()) {
        if (i->isRunning() && !i->isAborting())
          i->abort();
      }
    }
  }
  
  m_dirScanDone = false;
  resetTransfer();
  update();
}

void TransferDir::slotExecuteChild()
{
  if (m_aborting)
    return;
    
  Transfer *child = static_cast<Transfer*>(m_childIterator.current());

  if (!child)
    return;
  
  connect(child, SIGNAL(transferComplete(long)), this, SLOT(slotChildFinished()));
  connect(child, SIGNAL(transferAbort(long)), this, SLOT(slotChildAborted()));
  connect(child, SIGNAL(transferStart(long)), this, SLOT(slotChildStarted()));
  
  child->delayedExecute();
}

void TransferDir::slotChildStarted()
{
  if (m_aborting)
    return;
    
  if (!m_srcSession || !m_dstSession)
    return;
    
  // Check if there are any connections left free in this session
  if (m_srcSession->isFreeConnection() && m_dstSession->isFreeConnection()) {
    // There are free connections, continue to the next transfer
    QTimer::singleShot(100, this, SLOT(slotChildFinished()));
  } else {
    // No more free connections, wait for a child transfer to complete
  }
}

void TransferDir::slotChildAborted()
{
  if (m_aborting) {
    // We are aborting as well, so we shouldn't take any further actions
    return;
  } else {
    // The child has aborted, abort the whole directory
    abort();
  }
}

void TransferDir::slotChildFinished()
{
  // Our child is finished -- if there are no children left, then we are finished
  // as well.
  if (!++m_childIterator) {
    // Check if there are any more child transfers left (since there can
    // be multiple transfers running at the same time). If there are transfers
    // we should wait until they are completed.
    
    if (m_children.count() <= 1) {
      // There are no more transfers, so we are finished
      showTransCompleteBalloon();
      m_deleteMe = true;
      resetTransfer();
      
      emit transferComplete(m_id);
      KFTPQueue::Manager::self()->doEmitUpdate();
    }
  } else {
    // Move to the next
    slotExecuteChild();
  }
}

void TransferDir::slotDirScanDone()
{
  // Directory scanning has been completed, execute the transfer
  delayedExecute();
}

void TransferDir::slotSessionConnected(bool success)
{
  if (success) {
    if (!m_srcSession->isConnected() || !m_dstSession->isConnected())
      return;
      
    delayedExecute();
  } else {
    // Failed to connect one of the sessions, transfer must abort
  }
}

}

#include "kftptransferdir.moc"
