/*
 * This file is part of the KFTPGrabber project
 *
 * Copyright (C) 2003-2004 by the KFTPGrabber developers
 * Copyright (C) 2003-2004 Jernej Kos <kostko@jweb-network.net>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * is provided AS IS, WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE, and
 * NON-INFRINGEMENT.  See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Steet, Fifth Floor, Boston,
 * MA 02110-1301, USA.
 *
 * In addition, as a special exception, the copyright holders give
 * permission to link the code of portions of this program with the
 * OpenSSL library under certain conditions as described in each
 * individual source file, and distribute linked combinations
 * including the two.
 * You must obey the GNU General Public License in all respects
 * for all of the code used other than OpenSSL.  If you modify
 * file(s) with this exception, you may extend this exception to your
 * version of the file(s), but you are not obligated to do so.  If you
 * do not wish to do so, delete this exception statement from your
 * version.  If you delete this exception statement from all source
 * files in the program, then also delete it here.
 */

#include "kftptransferfile.h"
#include "kftpsystemtray.h"
#include "kftpsession.h"
#include "errorhandler.h"
#include "statistics.h"
#include "scheduler.h"

#include "misc/config.h"

#include <kmessagebox.h>
#include <klocale.h>
#include <kio/renamedlg.h>
#include <kdiskfreesp.h>

#include <qtimer.h>
#include <qfileinfo.h>

#define REMOTE_CONNECTION ( m_srcSession->isRemote() ? m_srcConnection : m_dstConnection )

namespace KFTPQueue {

TransferFile::TransferFile(QObject *parent)
  : Transfer(parent, Transfer::File),
    m_srcConnection(0),
    m_dstConnection(0),
    m_updateTimer(0),
    m_dfTimer(0)
{
}

TransferFile::~TransferFile()
{
}

KFTPSessionConnection *TransferFile::getOppositeConnection(KFTPSessionConnection *conn)
{
  return m_srcConnection == conn ? m_dstConnection : m_srcConnection;
}

KFTPSessionConnection *TransferFile::remoteConnection()
{
  return REMOTE_CONNECTION;
}

void TransferFile::slotConnectionAvailable()
{
  if (getStatus() != Waiting || !m_srcSession->isFreeConnection() || !m_dstSession->isFreeConnection())
    return;
    
  m_srcSession->QObject::disconnect(this, SLOT(slotConnectionAvailable()));
  m_dstSession->QObject::disconnect(this, SLOT(slotConnectionAvailable()));
    
  // Connection has become available, grab it now
  execute();
}

void TransferFile::execute()
{
  // Failed transfers aren't allowed to be executed until they are readded to
  // the queue and the Failed status is changed to Stopped.
  if (getStatus() == Failed)
    return;
    
  if (!m_srcSession || getStatus() == Waiting) {
    // If we have just been waiting we should have a free connection, so no need to reinitialize sessions
    if (getStatus() != Waiting) {
      m_srcSession = FTPSessionManager->spawnRemoteSession(IGNORE_SIDE, m_sourceUrl, 0, true);
      m_dstSession = FTPSessionManager->spawnRemoteSession(OPPOSITE_SIDE(m_srcSession->getSide()), m_destUrl, 0, true);
      
      if (!m_srcSession->isFreeConnection() || !m_dstSession->isFreeConnection()) {
        // We should wait for a connection to come
        connect(m_srcSession, SIGNAL(freeConnectionAvailable()), this, SLOT(slotConnectionAvailable()));
        connect(m_dstSession, SIGNAL(freeConnectionAvailable()), this, SLOT(slotConnectionAvailable()));
        
        m_status = Waiting;
        emit objectUpdated();
        return;
      }
    }
    
    // We need to get some connections
    m_srcConnection = m_srcSession->assignConnection();
    m_dstConnection = m_dstSession->assignConnection();
    
    // Acquire connections
    m_srcConnection->acquire(this);
    m_dstConnection->acquire(this);
        
    m_completed = 0;
    m_resumed = 0;
  
    // Connect events
    connect(REMOTE_CONNECTION->getClient(), SIGNAL(finished(CommandType)), this, SLOT(slotCommandFinished(CommandType)));
    connect(REMOTE_CONNECTION->getClient(), SIGNAL(resumedOffset(filesize_t)), this, SLOT(slotResumedOffset(filesize_t)));
    
    if (!m_sourceUrl.isLocalFile()) {
      connect(m_srcConnection->getClient(), SIGNAL(loginComplete(bool)), this, SLOT(slotClientConnected(bool)));
      connect(m_srcConnection->getClient(), SIGNAL(errorHandler(KFTPNetwork::Error)), this, SLOT(slotErrorHandler(KFTPNetwork::Error)));
      connect(m_srcConnection, SIGNAL(aborting()), this, SLOT(slotSessionAborting()));
    }
    
    if (!m_destUrl.isLocalFile()) {
      connect(m_dstConnection->getClient(), SIGNAL(loginComplete(bool)), this, SLOT(slotClientConnected(bool)));
      connect(m_dstConnection->getClient(), SIGNAL(errorHandler(KFTPNetwork::Error)), this, SLOT(slotErrorHandler(KFTPNetwork::Error)));
      connect(m_dstConnection, SIGNAL(aborting()), this, SLOT(slotSessionAborting()));
    }
  }
  
  // If any session is not yet connected, wait for the connection
  if (!m_srcConnection->isConnected() || !m_dstConnection->isConnected()) {
    m_status = Connecting;
    return;
  }
  
  // We are running now
  m_status = Running;
  
  // Init timer to follow the update
  if (!m_updateTimer) {
    m_updateTimer = new QTimer(this);
    connect(m_updateTimer, SIGNAL(timeout()), this, SLOT(slotTimerUpdate()));
    m_updateTimer->start(1000); // update status every 1 sec
  }
  
  // Should we check for free space ?
  if (KFTPCore::Config::diskCheckSpace() && !m_dfTimer) {
    m_dfTimer = new QTimer(this);
    connect(m_dfTimer, SIGNAL(timeout()), this, SLOT(slotTimerDiskFree()));
    m_dfTimer->start(KFTPCore::Config::diskCheckInterval() * 1000);
  }
  
  emit transferStart(m_id);
  
  switch(m_transferType) {
    case Download: {
      Scheduler::self()->registerTransfer(this);
      m_finishCmd = TCMD_GET;
      m_srcConnection->getClient()->get(m_sourceUrl, m_destUrl);
      break;
    }
    case Upload: {
      Scheduler::self()->registerTransfer(this);
      m_finishCmd = TCMD_PUT;
      m_dstConnection->getClient()->put(m_sourceUrl, m_destUrl);
      break;
    }
    case FXP: {
      // Start the timer to extrapolate transfer rate
      m_elapsedTime.start();

      m_finishCmd = TCMD_FXP;
      m_srcConnection->getClient()->fxpTransfer(m_sourceUrl, m_destUrl, m_dstConnection->getClient());
      break;
    }
  }
}

void TransferFile::slotTimerUpdate()
{
  // Update the current stats
  if (!m_srcSession) {
    m_updateTimer->stop();
    m_updateTimer->QObject::disconnect();
    return;
  }

  // Get speed from connection, or use FXP extrapolation.
  if (getTransferType() == FXP) {
    double lastFxpSpeed = Statistics::self()->getSite(m_sourceUrl)->lastFxpSpeed();
    
    if (lastFxpSpeed != 0.0) {
      setSpeed(lastFxpSpeed);
      
      if (m_completed < m_size)
        addCompleted(getSpeed());
    }
  } else {
    addCompleted(REMOTE_CONNECTION->getClient()->getClient()->getProcessed() - m_completed);
    setSpeed(REMOTE_CONNECTION->getClient()->getClient()->getSpeed());
  }

  update();
}

void TransferFile::slotTimerDiskFree()
{
  // Check for disk usage
  if (KFTPCore::Config::diskCheckSpace()) {
    KDiskFreeSp *df = KDiskFreeSp::findUsageInfo((getDestUrl().path()));
    connect(df, SIGNAL(foundMountPoint(const QString&, unsigned long, unsigned long, unsigned long)), this, SLOT(slotDiskFree(const QString&, unsigned long, unsigned long, unsigned long)));
  }
}

void TransferFile::slotDiskFree(const QString &mountPoint, unsigned long, unsigned long, unsigned long kBAvail)
{
  if (KFTPCore::Config::diskCheckSpace()) {
    // Is there enough free space ?
    if (kBAvail < (unsigned long) KFTPCore::Config::diskMinFreeSpace()) {
      QString p_transAbortStr = i18n("Transfer of the following files <b>has been aborted</b> because there is not enough free space left on '%1':").arg(mountPoint);
      p_transAbortStr += "<br><i>";
      p_transAbortStr += getSourceUrl().fileName();
      p_transAbortStr += "</i>";
      s_sysTray->showBalloon(p_transAbortStr);
      
      // Abort the transfer
      abort();
    }
  }
}

void TransferFile::slotErrorHandler(KFTPNetwork::Error error)
{
  KFTPNetwork::ErrorHandler *handler = error.handler();
  
  // We are handling the error
  long id = handler->handlingError(error);

  switch (error.code()) {
    case KFTPNetwork::EC_TIMEOUT: {
      // The client connection has timed out. We have to abort the current transfer
      // and restart it. The signals must be blocked so parent transfer doesn't abort
      // as well.
      blockSignals(true);
      abort();
      blockSignals(false);
      
      // Now it should be aborted, let's restart
      delayedExecute();
      break;
    }
    case KFTPNetwork::EC_FILE_EXISTS_DOWNLOAD: {
      int ret = Manager::fileExistsAction(this, Download, m_sourceUrl, m_destUrl, error.data().stat1());

      if (ret != -1) {
        handler->handlerReturnCode(id, ret);
        
        delayedExecute();
      }
      break;
    }
    case KFTPNetwork::EC_FILE_EXISTS_UPLOAD: {
      int ret = Manager::fileExistsAction(this, Upload, m_sourceUrl, m_destUrl, FTPEntry(), error.data().stat1());
      
      if (ret != -1) {
        handler->handlerReturnCode(id, ret);
        
        delayedExecute();
      }
      break;
    }
    case KFTPNetwork::EC_FILE_EXISTS_FXP: {
      int ret = Manager::fileExistsAction(this, FXP, m_sourceUrl, m_destUrl, error.data().stat2(), error.data().stat1());
      
      if (ret != -1) {
        handler->handlerReturnCode(id, ret);
        
        delayedExecute();
      }
      break;
    }
    case KFTPNetwork::EC_FXP_SSL_ERR: {
      // FXP transfer is not possible due to incompatible SSL modes, transfer failed.
      FailedTransfer::fail(this, i18n("Incompatible SSL modes on source and destination server."));
      break;
    }
    case KFTPNetwork::EC_FILE_NOT_FOUND: {
      // The source file can't be found, transfer failed.
      FailedTransfer::fail(this, i18n("Source file cannot be found."));
      break;
    }
    case KFTPNetwork::EC_DATA_CONN_ERR: {
      FailedTransfer::fail(this, i18n("There was a problem establishing the data connection."));
      break;
    }
    case KFTPNetwork::EC_FD_ERR: {
      FailedTransfer::fail(this, i18n("Error reading file."));
      break;
    }
    default: {
      // We haven't handled the error
      handler->abandonHandler(id);
      break;
    }
  }
}

void TransferFile::slotCommandFinished(CommandType cmd)
{
  if (cmd == m_finishCmd) {

    // Calculate transfer rate for last transfer, and save to site's statistics
    if (getTransferType() == FXP) {
      if (m_elapsedTime.elapsed() > 10000) {
        double speed = (m_size - m_resumed) / (double) m_elapsedTime.elapsed();
        Statistics::self()->getSite(m_sourceUrl)->setLastFxpSpeed(speed * 1024);
      }
    }

    m_updateTimer->stop();
    m_updateTimer->QObject::disconnect();
    
    if (m_openAfterTransfer && m_transferType == Download) {
      Manager::self()->openAfterTransfer(this);
    } else {
      showTransCompleteBalloon();
    }
    
    m_deleteMe = true;
    addActualSize(-m_size);
    
    resetTransfer();
    emit transferComplete(m_id);
    
    KFTPQueue::Manager::self()->doEmitUpdate();
  }
}

void TransferFile::slotResumedOffset(filesize_t offset)
{
  m_resumed = offset;
}

void TransferFile::slotClientConnected(bool success)
{
  if (success) {
    if (!m_srcConnection->isConnected() || !m_dstConnection->isConnected()) return;
    
    // Ok, both sessions are ready -- start the transfer
    m_status = Running;
    delayedExecute();
  } else {
    // Unable to login
    FailedTransfer::fail(this, i18n("Unable to connect with server."));
  }
}

void TransferFile::resetTransfer()
{
  // Unlock the sessions (they should be unlocked automaticly when the transferComplete signal
  // is emitted, but when a transfer is a child transfer, the next transfer may need the session
  // sooner). Also sessions should be unlocked when transfer aborts.
  if (m_srcSession && m_dstSession && getStatus() != Waiting) {
    // Disconnect signals
    REMOTE_CONNECTION->getClient()->QObject::disconnect(this, SLOT(slotCommandFinished(CommandType)));
    REMOTE_CONNECTION->getClient()->QObject::disconnect(this, SLOT(slotResumedOffset(filesize_t)));
    
    if (!m_sourceUrl.isLocalFile()) {
      m_srcConnection->getClient()->QObject::disconnect(this, SLOT(slotClientConnected(bool)));
      m_srcConnection->getClient()->QObject::disconnect(this, SLOT(slotErrorHandler(KFTPNetwork::Error)));
      m_srcConnection->QObject::disconnect(this, SLOT(slotSessionAborting()));
      
      m_srcConnection->remove();
    }
    
    if (!m_destUrl.isLocalFile()) {
      m_dstConnection->getClient()->QObject::disconnect(this, SLOT(slotClientConnected(bool)));
      m_dstConnection->getClient()->QObject::disconnect(this, SLOT(slotErrorHandler(KFTPNetwork::Error)));
      m_dstConnection->QObject::disconnect(this, SLOT(slotSessionAborting()));
      
      m_dstConnection->remove();
    }
  }
  
  m_srcConnection = 0L;
  m_dstConnection = 0L;
  
  m_finishCmd = TCMD_NONE;
  
  Transfer::resetTransfer();
}

void TransferFile::slotSessionAborting()
{
  if (!m_aborting) {
    abort();
  }
}

void TransferFile::abort()
{
  // If not running, just return
  if (!isRunning()) return;
  
  Transfer::abort();
  
  if (getStatus() == Waiting) {
    m_srcSession->QObject::disconnect(this, SLOT(slotConnectionAvailable()));
    m_dstSession->QObject::disconnect(this, SLOT(slotConnectionAvailable()));
  }
  
  if (m_updateTimer) {
    m_updateTimer->stop();
    m_updateTimer->QObject::disconnect();
    
    delete m_updateTimer;
    m_updateTimer = 0L;
  }
  
  if (m_dfTimer) {
    m_dfTimer->stop();
    m_dfTimer->QObject::disconnect();
    
    delete m_dfTimer;
    m_dfTimer = 0L;
  }

  // Abort any transfers
  if (m_srcConnection && !m_sourceUrl.isLocalFile()) {
    m_srcConnection->abort();
  }
  
  if (m_dstConnection && !m_destUrl.isLocalFile()) {
    m_dstConnection->abort();
  }
  
  // Update everything
  resetTransfer();
  
  if (!hasParentTransfer())
    update();
  
  if (hasParentObject() && parentObject()->isAborting())
    disconnect(parent());
}

}


#include "kftptransferfile.moc"
