/*
 * This file is part of the KFTPGrabber project
 *
 * Copyright (C) 2003-2004 by the KFTPGrabber developers
 * Copyright (C) 2003-2004 Jernej Kos <kostko@jweb-network.net>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * is provided AS IS, WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE, and
 * NON-INFRINGEMENT.  See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Steet, Fifth Floor, Boston,
 * MA 02110-1301, USA.
 *
 * In addition, as a special exception, the copyright holders give
 * permission to link the code of portions of this program with the
 * OpenSSL library under certain conditions as described in each
 * individual source file, and distribute linked combinations
 * including the two.
 * You must obey the GNU General Public License in all respects
 * for all of the code used other than OpenSSL.  If you modify
 * file(s) with this exception, you may extend this exception to your
 * version of the file(s), but you are not obligated to do so.  If you
 * do not wish to do so, delete this exception statement from your
 * version.  If you delete this exception statement from all source
 * files in the program, then also delete it here.
 */

#include "browser/listview.h"
#include "browser/treeview.h"
#include "browser/view.h"
#include "browser/actions.h"

#include "misc.h"
#include "misc/config.h"
#include "kftpqueue.h"
#include "errorhandler.h"

#include <klocale.h>
#include <kmessagebox.h>
#include <kiconeffect.h>
#include <kcombobox.h>
#include <kpopupmenu.h>
#include <kio/job.h>

#include <qdir.h>
#include <qimage.h>
#include <qapplication.h>
#include <qregexp.h>

using namespace KFTPGrabberBase;

namespace KFTPWidgets {

namespace Browser {

/*
 *                      History class
 */

History::History()
  : m_historyPos(0)
{
}
 
void History::newEntry(const KURL &url)
{
  // Add an entry to the list
  if (m_historyPos != m_history.fromLast() && m_history.count() > 1) {
    while (m_historyPos != m_history.fromLast()) {
      m_history.pop_back();
    }
  }

  if (m_history.empty() || (*m_historyPos).first != url) {
    // Add the currently selected URL if there is some history
    if (!m_history.empty())
      (*m_historyPos).second = url.fileName();

    m_history.append(QPair<KURL, QString>(url, QString::null));
  }

  // Set the new history position
  m_historyPos = m_history.fromLast();
}

KURL History::getCurrent()
{
  return (*m_historyPos).first;
}

QString History::getSelected()
{
  return (*m_historyPos).second;
}

void History::goBack()
{
  if (m_historyPos != m_history.begin())
    m_historyPos--;
}

void History::goForward()
{
  if (m_historyPos != m_history.fromLast())
    m_historyPos++;
}

/*
 *                        ListViewItem class
 */

ListViewItem::ListViewItem(ListView *parent)
  : KListViewItem(parent)
{
  m_paintOffline = false;
}

ListViewItem::ListViewItem(ListViewItem *parent)
  : KListViewItem(parent)
{
  m_paintOffline = false;
}

ListViewItem::ListViewItem(ListView *parent, KFileItem *item)
  : KListViewItem(parent)
{
  m_paintOffline = false;
  readFromFileItem(item);
}

ListViewItem::ListViewItem(ListViewItem *parent, KFileItem *item)
  : KListViewItem(parent)
{
  m_paintOffline = false;
  readFromFileItem(item);
}

ListViewItem::ListViewItem(ListView *parent, const FTPDirectoryItem &item, const KURL &url)
  : KListViewItem(parent)
{
  // Generate URL
  m_url = url;
  m_paintOffline = false;

  // Copy the dirObject
  m_dirObject = item;
  m_directory = false;

  // Create the columns
  createColumns();
}

void ListViewItem::readFromFileItem(KFileItem *item)
{
  // Transfer all item's properties to our class
  m_url = item->url();

  // Permissions
  m_dirObject.m_ftpEntry.permissions = item->permissionsString();

  // Ownership
  m_dirObject.m_ftpEntry.owner = item->user();
  m_dirObject.m_ftpEntry.group = item->group();

  // Type
  m_dirObject.m_ftpEntry.type = (item->isLink()) ? 'l' : 'f';
  m_directory = item->isDir();

  // Symlink destination
  m_dirObject.m_ftpEntry.link = item->linkDest();

  // Name
  m_dirObject.m_ftpEntry.name = item->name();

  // Time
  m_dirObject.m_ftpEntry.date = item->time(KIO::UDS_MODIFICATION_TIME);

  // Size
  m_dirObject.m_ftpEntry.size = item->size();

  // FileItem ptr (just for identification)
  m_filePtr = item;

  // Create the columns
  createColumns();
}

void ListViewItem::createColumns()
{
  // Create the columns from m_dirObject
  setText(0, m_dirObject.name());
  setText(1, KFTPCore::Config::showSizeInBytes() ? KIO::number(m_dirObject.size()) : KIO::convertSize(m_dirObject.size()));
  setText(2, m_dirObject.dateAsString());
  setText(3, m_dirObject.permissions());
  setText(4, m_dirObject.owner());
  setText(5, m_dirObject.group());

  QString iconText;
  if (!m_url.isLocalFile()) {
    // Guess file type
    if (m_dirObject.type() == 'd') {
      iconText = "folder";
      m_directory = true;
    } else if (m_dirObject.type() == 'l') {
      // We can't know if the sym-linked file is realy a directory, but most of
      // the time it is. So if we can't determine the MIME type, set it to directory.
      KMimeType::Ptr mimeType = KMimeType::findByURL(m_url, 0, false, true);
      if (mimeType->name() == KMimeType::defaultMimeType()) {
        iconText = "folder";
        m_directory = true;
      } else {
        iconText = mimeType->icon(QString::null, false);
      }
    } else {
      KMimeType::Ptr mimeType = KMimeType::findByURL(m_url, 0, false, true);
      iconText = mimeType->icon(QString::null, false);
    }
  }

  // Get the pixmap
  KMimeType::Ptr mimeType = KMimeType::findByURL(m_url, 0, m_url.isLocalFile(), !m_url.isLocalFile());
  QPixmap filePixmap = m_url.isLocalFile() ? loadSmallPixmap(mimeType->icon(QString::null, false)) : loadSmallPixmap(iconText);

  if (m_dirObject.type() == 'l') {
    // Do the link icon overlay for symlinks
    QImage linkImage = loadSmallPixmap("link").convertToImage();
    QImage fileImage = filePixmap.convertToImage();
    KIconEffect::overlay(fileImage, linkImage);
    filePixmap.convertFromImage(fileImage);
  }

  setPixmap(0, filePixmap);
}

int ListViewItem::compare(QListViewItem *i, int col, bool asc) const
{
  ListViewItem *cmp = static_cast<ListViewItem*>(i);
  
  if (m_directory != cmp->m_directory)
    return m_directory ? -1 : 1;
    
  bool isHidden1 = m_dirObject.name().at(0) == '.';
  bool isHidden2 = cmp->m_dirObject.name().at(0) == '.';
    
  if (isHidden1 != isHidden2)
    return isHidden1 ? -1 : 1;
  
  switch (col) {
    case 2: {
      // Sorting by date
      time_t t1 = m_dirObject.date();
      time_t t2 = cmp->m_dirObject.date();
      return (t1 > t2) ? 1 : (t1 < t2) ? -1 : 0;
    }
  }

  return text(col).lower().localeAwareCompare(cmp->text(col).lower());
}

void ListViewItem::paintCell(QPainter *p, const QColorGroup &cg, int column, int width, int alignment)
{
    QColorGroup _cg( cg );
    QColor c = _cg.text();

    QColor n_color = KFTPCore::Config::self()->colorizer()->getColor(text(0));
    if (n_color.isValid())
      _cg.setColor(QColorGroup::Text, n_color);

    // TODO make this offline color be read from the config
    if (m_paintOffline)
      _cg.setColor(QColorGroup::Text, QColor("#E3E3E3"));

    QListViewItem::paintCell( p, _cg, column, width, alignment );

    _cg.setColor( QColorGroup::Text, c );
}

/*
 *                          ListView class
 */

ListView::ListView(QWidget *parent, KFTPClientThr *p_ftpClient)
 : KFTPWidgets::ListView(parent)
{
  // Map the ftp client
  m_ftpClient = p_ftpClient;

  // Create the columns
  addColumn(i18n("Name"));
  addColumn(i18n("Size"));
  addColumn(i18n("Date"));
  addColumn(i18n("Permissions"));

  // Set column width
  setColumnWidthMode(0, QListView::Manual);
  setColumnWidthMode(1, QListView::Manual);
  setColumnWidthMode(2, QListView::Manual);
  setColumnWidthMode(3, QListView::Manual);
  
  if (KFTPCore::Config::showOwnerGroup()) {
    // Only show owner/group if set in the config
    addColumn(i18n("Owner"));
    addColumn(i18n("Group"));
    
    setColumnWidthMode(4, QListView::Manual);
    setColumnWidthMode(5, QListView::Manual);
  } else {
    setFullWidth(true);
  }
  
  setColumnWidth(0, 140);
  setColumnWidth(2, 100);

  // Other settings
  setDragEnabled(true);
  setAcceptDrops(true);
  setAllColumnsShowFocus(true);
  setSelectionModeExt(Extended);
  setEmptyListText("This directory is empty or the contents\n could not be displayed.");
  setItemsRenameable(true);

  // Create the dir lister for local files and connect signals
  m_dirLister = new KDirLister();

  connect(m_dirLister, SIGNAL(newItems(const KFileItemList&)), this, SLOT(slotDirListerNewItems(const KFileItemList&)));
  connect(m_dirLister, SIGNAL(deleteItem(KFileItem*)), this, SLOT(slotDirListerDeleteItem(KFileItem*)));
  connect(m_dirLister, SIGNAL(clear()), this, SLOT(slotDirListerClear()));
  connect(m_dirLister, SIGNAL(refreshItems(const KFileItemList&)), this, SLOT(slotDirListerRefreshItems(const KFileItemList&)));

  // Connect the ftp-related signals
  connect(m_ftpClient, SIGNAL(loginComplete(bool)), this, SLOT(slotFtpConnect(bool)));
  connect(m_ftpClient, SIGNAL(disconnectDone()), this, SLOT(slotFtpDisconnect()));
  connect(m_ftpClient, SIGNAL(finished(CommandType)), this, SLOT(slotFtpFinished(CommandType)));
  connect(m_ftpClient, SIGNAL(errorHandler(KFTPNetwork::Error)), this, SLOT(slotFtpErrorHandler(KFTPNetwork::Error)));

  // Connect the list-view related signals
  connect(this, SIGNAL(executed(QListViewItem*)), this, SLOT(slotListClicked()));
  connect(this, SIGNAL(contextMenuRequested(QListViewItem*, const QPoint&, int)), this, SLOT(slotContextMenu(QListViewItem*, const QPoint&)));
  connect(this, SIGNAL(itemRenamed(QListViewItem*, const QString&, int)), this, SLOT(slotItemRenamed(QListViewItem*, const QString&, int)));

  // Create the timer for drag & drop
  m_dropItem = 0L;
  m_dragOpenTimer = new QTimer(this);
  connect(m_dragOpenTimer, SIGNAL(timeout()), this, SLOT(slotDragTimeout()));

  // Set the default home url
  setHomeURL(KURL(KFTPCore::Config::defLocalDir()));
  m_offlineMode = false;

  // Reset companion
  setCompanion(0);
}

ListView::~ListView()
{
  delete m_dirLister;
}

void ListView::setCompanion(ListView *companion)
{
  // Set this list's companion
  m_companion = companion;

  /* == About companions:
   *
   * - they are used to share current connection status, so they know
   *   if transfer of files is possible
   * - the operation mode can be detected automaticly (normal ftp or
   *   fxp operation)
   * - if the companion is not set, all transfer functions will be
   *   disabled
   */

}

void ListView::setTreeView(TreeView *tree)
{
  // Set this list's tree view
  m_tree = tree;

  // Connect the signals
  connect(m_tree, SIGNAL(pathChanged(const KURL)), this, SLOT(slotTreeClicked(const KURL)));
}

void ListView::setHomeURL(const KURL url)
{
  // Set the home url
  m_homeURL = url;
}

void ListView::setFilter(const QString &filter)
{
  // Set the file filter
  m_filter = filter;

  // Reload the view
  moveTo(Reload);
}

void ListView::resetView(const KURL url, const QString &rootText)
{
  // Clear all the views
  m_tree->resetView(url, rootText);
  clear();

  // Clear history
  m_history.m_history.clear();
  m_history.m_historyPos = m_history.m_history.fromLast();
  static_cast<View*>(parent()->parent())->m_historyCombo->clear();
}

bool ListView::canTransfer(ListView *caller)
{
  // First check that we are transfer capable
  if (!canLocalActions())
    return false;

  // If there is no companion there shall be no transfers
  if (!m_companion && !caller)
    return false;

  // If both (this and the companion) are local, no transfer is possible
  if (m_companion->m_curURL.isLocalFile() && m_curURL.isLocalFile())
    return false;
    
  // If both are remote and protocols differ or one protocol doesn't support site-to-site
  // transfers, no transfer is possible
  if (!m_companion->m_curURL.isLocalFile() && !m_curURL.isLocalFile() &&
      (
        m_companion->m_ftpClient->getClient()->getProtocol() != m_ftpClient->getClient()->getProtocol() ||
        !(m_companion->m_ftpClient->getClient()->getFeatures() & SF_FXP_TRANSFER) ||
        !(m_ftpClient->getClient()->getFeatures() & SF_FXP_TRANSFER)
      ))
    return false;

  // Check if the other end is ready
  if (caller) {
    // This has already been called by our companion
    if (m_curURL.isLocalFile())
      return true;
    else if (m_ftpClient->getClient()->isConnected())
      return true;
  } else {
    // We check the companion first
    if (m_companion->canTransfer(this)) {
      // We are transfer capable
      return true;
    }
  }

  return false;
}

bool ListView::canLocalActions()
{
  if (m_curURL.isLocalFile())
    return true;
  else if (m_ftpClient->getClient()->isConnected())
    return true;

  return false;
}

int ListView::transferMode()
{
  // Check what kind of url does our companion have
  bool c_local = m_companion->m_curURL.isLocalFile();
  bool w_local = m_curURL.isLocalFile();

  // Now analize :)
  if (c_local && w_local) /* Two local lists ?!? WIERD */
    return -1;

  if (c_local && !w_local || w_local && !c_local) /* One local and one non-local */
    return 1;

  if (!c_local && !w_local) /* Two remotes - fxp transfer mode */
    return 2;

  // This should not happen
  return -1;
}

KURL::List ListView::getCurrentSelection()
{
  // Get all the files that are currently selected
  KURL::List list;

  // Go trough all items and add them to the list
  for (unsigned int i = 0; i < KListView::selectedItems().count(); i++) {
    ListViewItem *item = (ListViewItem*) KListView::selectedItems().at(i);
    list.append(item->m_url);
  }

  return list;
}

ListItems ListView::getCurrentSelectionLI()
{
  // Get all the files that are currently selected
  ListItems list;

  // Go trough all items and add them to the list
  for (unsigned int i = 0; i < KListView::selectedItems().count(); i++) {
    ListViewItem *item = (ListViewItem*) KListView::selectedItems().at(i);
    list.append(item);
  }

  return list;
}

void ListView::slotFtpErrorHandler(KFTPNetwork::Error error)
{
  KFTPNetwork::ErrorHandler *handler = error.handler();
  
  // We are handling the error
  long id = handler->handlingError(error);
  
  switch (error.code()) {
    case KFTPNetwork::EC_UNABLE_TO_ENTER_DIR: {
      // We are unable to enter a directory, display an error message
      KComboBox *historyCombo = static_cast<View*>(parent()->parent())->m_historyCombo;
      historyCombo->setCurrentItem(m_curURL.path());
      
      // Remove this url from KHistoryCombo
      for (int i = 0; i < historyCombo->count(); i++) {
        if (historyCombo->text(i) == error.data().filename())
          historyCombo->removeItem(i);
      }
      
      KMessageBox::error(0, i18n("Unable to open directory '%1'.").arg(error.data().filename()));
      break;
    }
    default: {
      // We haven't handled the error
      handler->abandonHandler(id);
      break;
    }
  }
}

void ListView::slotFtpFinished(CommandType cmdType)
{
  switch (cmdType) {
    case TCMD_LIST: {
      if (!m_ftpClient->getClient()->isConnected() && !m_offlineMode) {
        qDebug("WARNING: Tried to open remote url, but client is not connected!");

        // Open local URL
        resetView(KURL(KFTPCore::Config::defLocalDir()), i18n("Root directory"));
        openURL(KURL(KFTPCore::Config::defLocalDir()));
        return;
      }

      KURL url = m_reqURL;

      // Disable dirlister for the time of remote connections ;)
      m_dirLister->setAutoUpdate(false);

      // Request the directory :)
      KListView::clear();
      FTPDirList *p_tmp;

      if (m_offlineMode)
        p_tmp = m_offlineList;
      else {
        p_tmp= m_ftpClient->getClient()->getLastDirList();
        
        url.setPath(m_ftpClient->getClient()->getLastDir());
      }

      if (!p_tmp) {
        if (m_offlineMode)
          KMessageBox::error(0, i18n("Directory is not in cache."));
        else
          KMessageBox::error(0, i18n("Unable to open directory."));

        openURL(m_curURL);
        return;
      }

      FTPDirList p_dirList = *p_tmp;

      // Check the file count
      if (p_dirList.count() > 2000) {
        // If there are more than 2000 files in the dir, ask the user if the files
        // should be displayed
        QString warnText;
        warnText  = i18n("This directory contains 1 file.", "This directory contains %n files.", p_dirList.count());
        warnText += " ";
        warnText += i18n("Displaying all might take some time.");
        warnText += "\n\n";
        warnText += i18n("Are you sure?");

        if (KMessageBox::warningContinueCancel(0, warnText, QString::null, KStdGuiItem::cont(), "ManyFilesWarning") == KMessageBox::Cancel)
        {
          // Nope, don't display them ;)
          if (canMoveTo(Back)) {
            moveTo(Back);
          } else if (url.path() != "/") {
            KURL newURL = m_curURL;
            newURL.setPath("/");
            openURL(newURL);
          } else {
            // Nothing left to do :(
          }
          return;
        }
      }
      
      // Create the folder if it doesn't exist yet
      m_tree->createFolder(url, loadSmallPixmap("folder"));

      // Generate file url
      KURL fileURL = m_offlineMode ? m_offlineURL : m_ftpClient->getClient()->getClientInfoUrl();

      // Regexp for filtering
      QRegExp filter(m_filter);
      filter.setWildcard(true);

      // Process all the files we have got
      int curNum = 0;
      FTPDirList::iterator end( p_dirList.end() );
      for (FTPDirList::iterator i( p_dirList.begin() ); i != end; ++i) {
        fileURL.setPath(url.path(1) + (*i).name());
        ListViewItem *item = new ListViewItem(this, (*i).m_ftpEntry, fileURL);

        if (!(*i).name().isEmpty() && (m_filter.isEmpty() || filter.exactMatch((*i).name()) || item->m_directory) &&
            (KFTPCore::Config::showHiddenFiles() || (*i).name().at(0) != '.')) {
          // If we have a tree view and this is a directory
          if (item->m_directory && m_tree)
            m_tree->createFolder(fileURL, *item->pixmap(0));
        } else {
          delete item;
          continue;
        }

        if (item->m_directory && m_offlineMode && !FTPCache().listGetFromCache(item->m_url)) {
          // Directory is not available in offline mode, make it look like it
          item->m_paintOffline = true;
        } else {
          item->m_paintOffline = false;
        }

        // Process events every 500 files (show progress)
        if (++curNum >= 500) {
          curNum = 0;
          qApp->processEvents();
        }
      }
      
      m_tree->endUpdate(url);

      // Set the correct url
      fileURL.setPath(url.path(0));

      // Update the history
      if (m_addHistory) {
        m_history.newEntry(fileURL);

        // Get the KHistoryCombo
        KComboBox *histCombo = static_cast<View*>(parent()->parent())->m_historyCombo;

        // Remove duplicates from KHistoryCombo
        for(int i = 0;i<histCombo->count();i++) {
          if (histCombo->text(i) == fileURL.path(-1))
            histCombo->removeItem(i);
        }

        // Add to KHistoryCombo
        histCombo->insertItem(loadSmallPixmap("folder"), fileURL.path(-1), 0);
        histCombo->setCurrentItem(0);
      }

      // Set the current url
      m_curURL = fileURL;
      
      // Reselect the correct folder
      smartFolderReselect();

      // Request update
      static_cast<View*>(parent()->parent())->updateActions();

      // Open the tree
      m_tree->openURL(fileURL);
      break;
    }
    default: break;
  }
}

void ListView::openURL(const KURL url, bool addHistory)
{
  KURL fileURL;

  // Detect site change
  KURL tmpURL = m_curURL;
  tmpURL.setPath(url.path());
  if (tmpURL != url) {
    // Site change, resetting views
    if (url.isLocalFile()) {
      tmpURL = KURL(KFTPCore::Config::defLocalDir());
      setHomeURL(KURL(KFTPCore::Config::defLocalDir()));
    } else {
      tmpURL = m_ftpClient->getClient()->getClientInfoUrl();
      tmpURL.setPath(m_homeURL.path());
    }

    resetView(tmpURL, url.isLocalFile() ? i18n("Root directory") : url.host());
  }

  if (url.isLocalFile()) {
    // Local file
    
    // We must set this url or the dir lister won't list anything
    m_curURL = url;
    
    m_dirLister->setAutoUpdate(true);
    m_dirLister->setShowingDotFiles(KFTPCore::Config::showHiddenFiles());
    m_dirLister->setNameFilter(m_filter);
    m_dirLister->openURL(url);
    
    // Do it anyway ;)
    fileURL = url;
  } else if (m_ftpClient) {
    // Request the dir listing and return
    m_reqURL = url;
    m_addHistory = addHistory;

    if (m_offlineMode) {
      if (FTPCache().listGetFromCache(m_reqURL)) {
        FTPCacheItem *item = FTPCache().listGetFromCache(m_reqURL);
        m_offlineList = &item->m_dirList;

        slotFtpFinished(TCMD_LIST);
      } else {
        m_offlineList = 0L;

        slotFtpFinished(TCMD_LIST);
      }
    } else {
      m_ftpClient->dirList(url);
    }

    return;
  }

  // Update the history
  if (addHistory) {
    m_history.newEntry(fileURL);
  }
  
  // Get the KHistoryCombo
  KComboBox *histCombo = static_cast<View*>(parent()->parent())->m_historyCombo;

  // Remove duplicates from KHistoryCombo
  for(int i = 0;i<histCombo->count();i++) {
    if (histCombo->text(i) == fileURL.path(-1))
      histCombo->removeItem(i);
  }

  // Add to KHistoryCombo
  histCombo->insertItem(loadSmallPixmap("folder"), fileURL.path(-1), 0);
  histCombo->setCurrentItem(0);

  // Set the current url
  m_curURL = fileURL;
  
  // Reselect the correct folder
  smartFolderReselect();

  // Request update
  static_cast<View*>(parent()->parent())->updateActions();

  // Open the tree
  m_tree->openURL(fileURL);
}

void ListView::smartFolderReselect()
{
  // Scroll to the item that was previously selected
  if (m_history.getSelected() != QString::null) {
    QListViewItem *i = findItem(m_history.getSelected(), 0);
    
    if (i) {
      setSelected(i, true);
      ensureItemVisible(i);
    }
  } else if (canMoveTo(Back)) {
    // Did we just go up ?
    m_history.goBack();
    
    if (m_curURL == m_history.getCurrent().upURL()) {
      QListViewItem *i = findItem(m_history.getCurrent().fileName(), 0);
    
      if (i) {
        setSelected(i, true);
        ensureItemVisible(i);
      }
    }
    
    m_history.goForward();
  }
}

void ListView::moveTo(LocationInfo location)
{
  switch (location) {
    case Up: {
      // Just get the one directory up and reload
      openURL(m_curURL.upURL());
      break;
    }
    case Back: {
      // Use the history and go back
      m_history.goBack();
      openURL(m_history.getCurrent(), false);
      break;
    }
    case Forward: {
      // Use the history and go forward
      m_history.goForward();
      openURL(m_history.getCurrent(), false);
      break;
    }
    case Reload: {
      // Just reload the current url
      openURL(m_curURL, false);
      break;
    }
    case Home: {
      // Go home =)
      openURL(m_homeURL);
      break;
    }
  }
}

bool ListView::canMoveTo(LocationInfo location)
{
  switch (location) {
    case Up: {
      if (m_curURL.path() == "/")
        return false;
      break;
    }
    case Back: {
      if (m_history.m_historyPos == m_history.m_history.begin())
        return false;
      break;
    }
    case Forward: {
      if (m_history.m_historyPos == m_history.m_history.fromLast())
        return false;
      break;
    }
    case Home:
    case Reload: {
      if (!m_curURL.isLocalFile() && !m_ftpClient->getClient()->isConnected())
        return false;
      break;
    }
  }

  return true;
}

void ListView::slotContextMenu(QListViewItem *i, const QPoint &p)
{
  // Request update
  static_cast<View*>(parent()->parent())->updateActions();

  // Create the popup menu
  KPopupMenu *contextMenu = new KPopupMenu(this);
  Actions *actions = static_cast<View*>(parent()->parent())->m_actions;

  // Always show create directory
  actions->m_createDirAction->plug(contextMenu);
  contextMenu->insertSeparator();

  // If nothing is selected, show the navigation menus
  if (!i) {
    actions->m_goUpAction->plug(contextMenu);
    actions->m_goBackAction->plug(contextMenu);
    actions->m_goForwardAction->plug(contextMenu);
    actions->m_reloadAction->plug(contextMenu);
  } else {
    actions->m_transferAction->plug(contextMenu);
    actions->m_queueTransferAction->plug(contextMenu);
    actions->m_renameAction->plug(contextMenu);
    actions->m_deleteAction->plug(contextMenu);
    actions->m_shredAction->plug(contextMenu);
    actions->m_fileEditAction->plug(contextMenu);
    actions->m_verifyAction->plug(contextMenu);
    contextMenu->insertSeparator();
    actions->m_copyAction->plug(contextMenu);
    actions->m_pasteAction->plug(contextMenu);
  }

  // Always show properties and filters
  contextMenu->insertSeparator();
  actions->m_setFilterAction->plug(contextMenu);
  actions->m_clearFilterAction->plug(contextMenu);
  contextMenu->insertSeparator();
  actions->m_propsAction->plug(contextMenu);

  contextMenu->exec(p);
}

void ListView::slotItemRenamed(QListViewItem *item, const QString &str, int)
{
  // Item was renamed, commit the change to the actual filesystem
  ListViewItem *o_item = static_cast<ListViewItem*>(item);
  
  // Get the new file URL
  KURL oldName = o_item->m_url.url();
  KURL newName = oldName;
  newName.setPath(oldName.directory(false) + str);
  
  // Do the actual renaming
  if (oldName.isLocalFile()) {
    KIO::rename(oldName, newName, false);
    
    // Reload the listing
    moveTo(Reload);
  } else if (oldName != newName) {
    m_ftpClient->rename(oldName, newName);
  }
}

void ListView::slotListClicked()
{
  // First, get all selected files
  for (unsigned int i = 0; i < KListView::selectedItems().count(); i++) {
    ListViewItem *item = static_cast<ListViewItem*>(KListView::selectedItems().at(i));

    // Check if this is a directory
    if (item->m_directory) {
      // Open the new url :)
      openURL(item->m_url);
      return;
    } else {
      // File clicked
      // FIXME what to do with files ?
    }
  }
}

void ListView::slotTreeClicked(const KURL url)
{
  // Tree was clicked and url was changed, refresh
  openURL(url);
}

void ListView::slotFtpConnect(bool success)
{
  // FTP client has logged in
  if (success) {
    // If there is no default URL specified, use the default dir got from the
    // socket.
    if (m_homeURL.isEmpty())
      m_homeURL = m_ftpClient->getClient()->getDefaultDir();

    KURL newURL = m_ftpClient->getClient()->getClientInfoUrl();
    newURL.setPath(m_homeURL.path());
    m_homeURL = newURL;
    m_curURL = newURL;
    
    resetView(newURL, newURL.host());
    openURL(newURL);
    
    if (m_companion && m_companion->m_curURL.isLocalFile() && !m_ftpClient->getClient()->getConfigStr("local_dir").isEmpty())
      m_companion->openURL(m_ftpClient->getClient()->getConfigStr("local_dir"));
  }
}

void ListView::slotFtpDisconnect()
{
  // FTP client has disconnected
  if (!m_curURL.isLocalFile()) {
    // Change to local root
    openURL(KURL(KFTPCore::Config::defLocalDir()));
  }
}

void ListView::slotDirListerNewItems(const KFileItemList &items)
{
  if (!m_curURL.isLocalFile())
    return;

  // Add local files receieved from the dir lister
  KFileItemList p_items = items; // FIXME any better way ?
  for (KFileItem *item = p_items.first(); item; item = p_items.next()) {
    new ListViewItem(this, item);

    // If we have a tree view and this is a directory
    if (item->isDir() && m_tree)
      m_tree->createFolder(item->url(), item->pixmap(16));
  }
}

void ListView::slotDirListerDeleteItem(KFileItem *item)
{
  if (!m_curURL.isLocalFile())
    return;

  // Find the item and delete it - search by URL
  QListViewItem *i = firstChild();
  while (i) {
    // Search by filePtr (unique identification for this KFileItem)
    if (static_cast<ListViewItem*>(i)->m_filePtr == item) {
      if (item->isDir()) {
        // Clear current url from treeview
        m_tree->removeFolder(item->url());
      }

      delete i;
      break;
    }

    i = i->itemBelow();
  }
}

void ListView::slotDirListerClear()
{
  if (!m_curURL.isLocalFile())
    return;

  // Clear all items
  clear();
}

void ListView::slotDirListerRefreshItems(const KFileItemList &items)
{
  if (!m_curURL.isLocalFile())
    return;
    
  // Refresh the items by deleting and recreating
  KFileItemList p_items = items; // FIXME any better way ?
  for (KFileItem *item = p_items.first(); item; item = p_items.next()) {
    slotDirListerDeleteItem(item);
    new ListViewItem(this, item);
    
    // If we have a tree view and this is a directory
    if (item->isDir() && m_tree)
      m_tree->createFolder(item->url(), item->pixmap(16));
  }
}

/* Drag & drop support below */
void ListView::startDrag()
{
  dragObject()->drag();
}

bool ListView::acceptDrag(QDropEvent *e)
{
  return KURLDrag::canDecode(e) &&
         ( e->action() == QDropEvent::Copy
           || e->action() == QDropEvent::Move
           || e->action() == QDropEvent::Link )
         && acceptDrops()
         && dragEnabled()
         && e->source() != this;
}

QDragObject *ListView::dragObject()
{
  KURL::List p_selection = getCurrentSelection();
  ListItems p_li = getCurrentSelectionLI();

  // Set the correct pixmap
  QPixmap pix;
  if (p_selection.count() > 1) {
    // More than one object selected
    pix = DesktopIcon("kmultiple", 16);
  } else {
    // One object selected, get it's icon from the only item that is
    // selected
    pix = *p_li.at(0)->pixmap(0);
  }

  // Add some metadata for all urls
  KIO::MetaData p_meta;
  ListViewItem *i;
  for (i = p_li.first(); i; i = p_li.next()) {
    char type = i->m_directory ? 'D' : 'F';
    QString data;
    data.sprintf("%c:%s", type, KIO::number(i->m_dirObject.size()).ascii());
    p_meta.insert(i->m_url.htmlURL().local8Bit(), data);
  }

  m_dragObject = new KURLDrag(p_selection, p_meta, this, name());
  m_dragObject->setPixmap(pix, QPoint(pix.width() / 2, pix.height() / 2));
  
  return m_dragObject;
}

void ListView::slotDragTimeout()
{
  m_dragOpenTimer->stop();
  if (!m_dropItem)
    return;

  QListViewItem *i = firstChild();
  while (i) {
    ListViewItem *item = static_cast<ListViewItem*>(i);
    if (i->text(0) == m_dropItem->text(0)) {
      if (item->m_directory) {
        openURL(item->m_url);
      }

      return;
    }

    i = i->itemBelow();
  }
}

void ListView::contentsDragEnterEvent(QDragEnterEvent *e)
{
  if (!acceptDrag(e)) {
    e->accept(false);
    return;
  }

  e->acceptAction();
  QListViewItem *i = itemAt(contentsToViewport(e->pos()));
  if (i) {
    m_dropItem = i;
    m_dragOpenTimer->start(1500);
  }
}

void ListView::contentsDragMoveEvent(QDragMoveEvent *e)
{
  if (!acceptDrag(e)) {
    e->accept(false);
    return;
  }

  e->acceptAction();
  QListViewItem *i = itemAt(contentsToViewport(e->pos()));
  if (i) {
    if (i != m_dropItem) {
      m_dragOpenTimer->stop();
      m_dropItem = i;
      m_dragOpenTimer->start(1500);
    }
  } else {
    m_dragOpenTimer->stop();
  }
}

void ListView::contentsDragLeaveEvent(QDragLeaveEvent*)
{
  m_dragOpenTimer->stop();
  m_dropItem = 0L;
}

void ListView::contentsDropEvent(QDropEvent *e)
{
  m_dragOpenTimer->stop();
  m_dropItem = 0L;

  if (!acceptDrag(e)) {
    e->acceptAction(false);
    return;
  }
  e->acceptAction();

  // Decode the data and try to init transfer
  KIO::MetaData p_meta;
  KURL::List p_urls;
  KURLDrag::decode(e, p_urls, p_meta);

  // Add destination url and call the QueueManager
  p_meta.insert("DestURL", m_curURL.url());
  KURLDrag *drag = new KURLDrag(p_urls, p_meta, this, name());
  KFTPQueue::Manager::self()->insertTransfer(drag);
}

}

}

#include "listview.moc"
