/*
 * This file is part of the KFTPGrabber project
 *
 * Copyright (C) 2003-2005 by the KFTPGrabber developers
 * Copyright (C) 2003-2005 Jernej Kos <kostko@jweb-network.net>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * is provided AS IS, WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE, and
 * NON-INFRINGEMENT.  See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Steet, Fifth Floor, Boston,
 * MA 02110-1301, USA.
 *
 * In addition, as a special exception, the copyright holders give
 * permission to link the code of portions of this program with the
 * OpenSSL library under certain conditions as described in each
 * individual source file, and distribute linked combinations
 * including the two.
 *
 * You must obey the GNU General Public License in all respects
 * for all of the code used other than OpenSSL.  If you modify
 * file(s) with this exception, you may extend this exception to your
 * version of the file(s), but you are not obligated to do so.  If you
 * do not wish to do so, delete this exception statement from your
 * version.  If you delete this exception statement from all source
 * files in the program, then also delete it here.
 */

#include "configfilter.h"

#include "misc/config.h"
#include "misc.h"

#include <klistview.h>
#include <klocale.h>
#include <kpushbutton.h>
#include <kmessagebox.h>
#include <klineeditdlg.h>

#include <klineedit.h>
#include <kcolorbutton.h>
#include <kdialogbase.h>

#include <qcheckbox.h>
#include <qlayout.h>

using namespace KFTPGrabberBase;

namespace KFTPWidgets {

ConfigFilterAddPatternDialog::ConfigFilterAddPatternDialog(QWidget *parent, const char *name)
 : KDialogBase(parent, name, true, i18n("Add New Pattern"), Ok|Cancel, Ok)
{
  m_addLayout = new KFTPFilterAddPatternLayout(this);

  // Dialog options
  setMainWidget(m_addLayout);
  setInitialSize(QSize(300, 75));
}

ConfigFilter::ConfigFilter(QWidget *parent, const char *name)
 : QWidget(parent, name)
{
  // Create the main widget
  QVBoxLayout *t = new QVBoxLayout(this, 0, 0);
  m_editorLayout = new KFTPFilterEditorLayout(this);
  t->addWidget(m_editorLayout);

  // Set the listview options
  m_editorLayout->patternList->setAllColumnsShowFocus(true);
  m_editorLayout->patternList->setColumnWidth(0, 100);

  loadSettings();

  // Connect the slots
  connect(m_editorLayout->addPatternButton, SIGNAL(clicked()), this, SLOT(slotAddHighlight()));
  connect(m_editorLayout->removePatternButton, SIGNAL(clicked()), this, SLOT(slotRemoveHighlight()));
  connect(m_editorLayout->editPatternButton, SIGNAL(clicked()), this, SLOT(slotEditHighlight()));

  connect(m_editorLayout->addExtButton, SIGNAL(clicked()), this, SLOT(slotAddAscii()));
  connect(m_editorLayout->removeExtButton, SIGNAL(clicked()), this, SLOT(slotRemoveAscii()));

  connect(m_editorLayout->addPrioButton, SIGNAL(clicked()), this, SLOT(slotAddPrio()));
  connect(m_editorLayout->removePrioButton, SIGNAL(clicked()), this, SLOT(slotRemovePrio()));
}

void ConfigFilter::loadSettings()
{
  // Set the options
  m_editorLayout->enabledCheck->setChecked(KFTPCore::Config::enableHighlight());

  m_editorLayout->prioEnable->setChecked(KFTPCore::Config::enablePrioList());
  m_editorLayout->skipEmptyFiles->setChecked(KFTPCore::Config::skipEmptyFiles());
  m_editorLayout->skipEmptyDirs->setChecked(KFTPCore::Config::skipEmptyDirs());
  m_editorLayout->prioApplyToDirs->setChecked(KFTPCore::Config::applyPrioToDirs());
  m_editorLayout->skipNoQueue->setChecked(KFTPCore::Config::skipNoQueue());
  m_editorLayout->queueDirsFirst->setChecked(KFTPCore::Config::queueDirsFirst());

  highlightLoadPatterns();
  asciiLoadExtensions();
  prioLoadList();
}

void ConfigFilter::saveSettings()
{
  // Save the settings
  KFTPCore::Config::setEnableHighlight(m_editorLayout->enabledCheck->isChecked());
  KFTPCore::Config::setFilterList(highlightToStringList());
  KFTPCore::Config::self()->refreshColorizer();

  KFTPCore::Config::setAsciiList(asciiToStringList());

  KFTPCore::Config::setEnablePrioList(m_editorLayout->prioEnable->isChecked());
  KFTPCore::Config::setSkipEmptyFiles(m_editorLayout->skipEmptyFiles->isChecked());
  KFTPCore::Config::setSkipEmptyDirs(m_editorLayout->skipEmptyDirs->isChecked());
  KFTPCore::Config::setApplyPrioToDirs(m_editorLayout->prioApplyToDirs->isChecked());
  KFTPCore::Config::setSkipNoQueue(m_editorLayout->skipNoQueue->isChecked());
  KFTPCore::Config::setQueueDirsFirst(m_editorLayout->queueDirsFirst->isChecked());
  
  prioSaveList();
}

void ConfigFilter::slotAddPrio()
{
  if (!m_editorLayout->newPrioPat->text().stripWhiteSpace().isEmpty()) {
    int prio = 0;
    if (m_editorLayout->priorityHigh->isChecked())
      prio = m_editorLayout->prioritySetting->value();
    else if (m_editorLayout->priorityLow->isChecked())
      prio -= m_editorLayout->prioritySetting->value();

    if (!m_editorLayout->prioList->findItem(m_editorLayout->newPrioPat->text().stripWhiteSpace(),0,0)) {
      new QListViewItem(m_editorLayout->prioList, m_editorLayout->newPrioPat->text().stripWhiteSpace(), QString::number(prio));
      m_editorLayout->newPrioPat->clear();
    }
  }
}

void ConfigFilter::slotRemovePrio()
{
  if (!m_editorLayout->prioList->selectedItem())
    return;

  delete m_editorLayout->prioList->selectedItem();
}

void ConfigFilter::prioLoadList()
{
  // Load the skiplist
  m_editorLayout->prioList->clear();
  QValueList<QPair<QString, int> > p_prioList = *KFTPCore::Config::self()->priorityList();

  QValueList<QPair<QString, int> >::iterator end(p_prioList.end());
  for (QValueList<QPair<QString, int> >::iterator i(p_prioList.begin()); i != end; ++i) {
    new QListViewItem(m_editorLayout->prioList, (*i).first, QString::number((*i).second));
  }
}

void ConfigFilter::prioSaveList()
{
  QValueList<QPair<QString, int> > *prioList = KFTPCore::Config::self()->priorityList();
  QListViewItem *i = m_editorLayout->prioList->firstChild();
  
  prioList->clear();
  while (i) {
    prioList->append(QPair<QString, int>(i->text(0),i->text(1).toInt()));

    i = i->itemBelow();
  }
}

void ConfigFilter::slotAddAscii()
{
  if (!m_editorLayout->newExtension->text().stripWhiteSpace().isEmpty()) {
    new QListViewItem(m_editorLayout->extensionList, m_editorLayout->newExtension->text().stripWhiteSpace());
    m_editorLayout->newExtension->clear();
  }
}

void ConfigFilter::slotRemoveAscii()
{
  if (!m_editorLayout->extensionList->selectedItem())
    return;

  delete m_editorLayout->extensionList->selectedItem();
}

void ConfigFilter::slotAddHighlight()
{
  ConfigFilterAddPatternDialog *p_addPatternDialog = new ConfigFilterAddPatternDialog(this, "add pattern dialog");
  if (p_addPatternDialog->exec() == KDialogBase::Accepted) {
    // FIXME replace if name exists
    QString pat = p_addPatternDialog->m_addLayout->patName->text();
    QString col = p_addPatternDialog->m_addLayout->patColor->color().name();

    QListViewItem *p_item = new QListViewItem(m_editorLayout->patternList, pat, " " + col);
    p_item->setPixmap(1, createColorPixmap(col));
  }
}

void ConfigFilter::slotRemoveHighlight()
{
  if (!m_editorLayout->patternList->selectedItem())
    return;

  if (KMessageBox::warningYesNo(0, i18n("Are you sure that you want to remove this pattern?")) == KMessageBox::Yes)
  {
    delete m_editorLayout->patternList->selectedItem();
  }
}

void ConfigFilter::slotEditHighlight()
{
  if (!m_editorLayout->patternList->selectedItem())
    return;

  ConfigFilterAddPatternDialog *p_addPatternDialog = new ConfigFilterAddPatternDialog(this, "add pattern dialog");

  QListViewItem *p_item = m_editorLayout->patternList->selectedItem();
  p_addPatternDialog->m_addLayout->patName->setText(p_item->text(0));
  p_addPatternDialog->m_addLayout->patColor->setColor(QColor(p_item->text(1).stripWhiteSpace()));

  if (p_addPatternDialog->exec() == KDialogBase::Accepted) {
    p_item->setText(0, p_addPatternDialog->m_addLayout->patName->text());
    p_item->setText(1, " " + p_addPatternDialog->m_addLayout->patColor->color().name());
    p_item->setPixmap(1, createColorPixmap(p_addPatternDialog->m_addLayout->patColor->color().name()));
  }
}

void ConfigFilter::asciiLoadExtensions()
{
  // Load the ascii extensions
  m_editorLayout->extensionList->clear();
  QStringList p_extensions = KFTPCore::Config::asciiList();

  QStringList::iterator end( p_extensions.end() );
  for (QStringList::iterator i( p_extensions.begin() ); i != end; ++i) {
    new QListViewItem(m_editorLayout->extensionList, (*i));
  }

}

QStringList ConfigFilter::asciiToStringList()
{
  QStringList p_extensions;
  QListViewItem *i = m_editorLayout->extensionList->firstChild();
  while (i) {
    p_extensions.append(i->text(0));

    i = i->itemBelow();
  }

  return p_extensions;
}

void ConfigFilter::highlightLoadPatterns()
{
  // Load the patterns
  m_editorLayout->patternList->clear();
  QStringList p_patterns = KFTPCore::Config::filterList();

  QStringList::Iterator end( p_patterns.end() );
  for (QStringList::Iterator i( p_patterns.begin() ); i != end; ++i) {
    QString pat = (*i).section(':', 0, 0);
    QString col = (*i).section(':', 1, 1);

    QListViewItem *p_item = new QListViewItem(m_editorLayout->patternList, pat, " " + col);
    p_item->setPixmap(1, createColorPixmap(col));
  }
}

QStringList ConfigFilter::highlightToStringList()
{
  QStringList p_patterns;
  QListViewItem *i = m_editorLayout->patternList->firstChild();
  while (i) {
    p_patterns.append(i->text(0)+":"+i->text(1).stripWhiteSpace());

    i = i->itemBelow();
  }

  return p_patterns;
}

}
#include "configfilter.moc"
