/*
 * This file is part of the KFTPGrabber project
 *
 * Copyright (C) 2003-2005 by the KFTPGrabber developers
 * Copyright (C) 2003-2005 Jernej Kos <kostko@jweb-network.net>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * is provided AS IS, WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE, and
 * NON-INFRINGEMENT.  See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Steet, Fifth Floor, Boston,
 * MA 02110-1301, USA.
 *
 * In addition, as a special exception, the copyright holders give
 * permission to link the code of portions of this program with the
 * OpenSSL library under certain conditions as described in each
 * individual source file, and distribute linked combinations
 * including the two.
 *
 * You must obey the GNU General Public License in all respects
 * for all of the code used other than OpenSSL.  If you modify
 * file(s) with this exception, you may extend this exception to your
 * version of the file(s), but you are not obligated to do so.  If you
 * do not wish to do so, delete this exception statement from your
 * version.  If you delete this exception statement from all source
 * files in the program, then also delete it here.
 */
#include "threadview.h"
#include "kftpapi.h"

#include "listview.h"

#include <qlayout.h>

#include <klocale.h>

namespace KFTPWidgets {

ThreadViewItem::ThreadViewItem(KFTPSession *session, QListView *parent)
  : QObject(parent), QListViewItem(parent), m_connection(0), m_session(session)
{
  refresh();
}

ThreadViewItem::ThreadViewItem(KFTPSessionConnection *conn, QListViewItem *parent, int id)
  : QObject(), QListViewItem(parent), m_id(id), m_connection(conn), m_session(0)
{
  connect(conn, SIGNAL(connectionRemoved()), this, SLOT(slotConnectionRemoved()));
  
  // Connect the transfer signals if the transfer is already present
  if (m_connection->getTransfer()) {
    connect(m_connection->getTransfer(), SIGNAL(objectUpdated()), this, SLOT(slotUpdateItemRequested()));
  } else {
    connect(conn, SIGNAL(connectionAcquired()), this, SLOT(slotConnectionAcquired()));
  }
  
  refresh();
}

void ThreadViewItem::slotConnectionAcquired()
{
  if (!m_connection->getTransfer())
    return;
    
  connect(m_connection->getTransfer(), SIGNAL(objectUpdated()), this, SLOT(slotUpdateItemRequested()));
}

void ThreadViewItem::slotConnectionRemoved()
{
  if (!m_connection->getTransfer())
    return;
    
  m_connection->getTransfer()->disconnect(this);
  
  // Refresh the item
  refresh();
}

void ThreadViewItem::refresh()
{
  if (m_session) {
    // Set the columns
    setText(0, i18n("Site session [%1]").arg(m_session->getClient()->getClient()->getClientInfoUrl().host()));
    setPixmap(0, loadSmallPixmap("ftp"));
  } else if (m_connection) {
    // Speed
    QString speed;
    
    if (m_connection->getTransfer() && m_connection->getTransfer()->isRunning()) {
      filesize_t c_speed = m_connection->getTransfer()->getSpeed();
      
      speed.sprintf( "%lld KB/s", (c_speed / 1024) );
      
      if (c_speed > 1024*1024)
        speed.sprintf("%lld MB/s", (c_speed / 1024) / 1024);
    
      if (c_speed == 0)
        speed = i18n("stalled");
    }
  
    setText(0, i18n("Thread %1").arg(m_id));
    setPixmap(0, loadSmallPixmap("socket"));
    
    if (m_connection->getTransfer() && m_connection->getTransfer()->isRunning()) {
      if (m_connection->getTransfer()->getStatus() == KFTPQueue::Transfer::Connecting) {
        setText(1, i18n("connecting"));
      } else {
        setText(1, i18n("transferring"));
      }
    } else {
      setText(1, m_connection->isConnected() ? i18n("idle") : i18n("disconnected"));
    }
    
    if (m_connection->getTransfer() && m_connection->getTransfer()->getTransferType() == KFTPQueue::FXP && m_connection->getTransfer()->getSpeed() == 0 && m_connection->getTransfer()->isRunning()) {
      KFTPSessionConnection *c = static_cast<KFTPQueue::TransferFile*>(m_connection->getTransfer())->getOppositeConnection(m_connection);
      
      setText(2, i18n("FXP - [%1]").arg(c->getUrl().host()));
    } else {
      setText(2, speed);
    }
  }
}

void ThreadViewItem::paintCell(QPainter *p, const QColorGroup &cg, int column, int width, int alignment)
{
  QListViewItem::paintCell(p, cg, column, width, alignment);
}

void ThreadViewItem::slotUpdateItemRequested()
{
  refresh();
}

KActionCollection *ThreadView::actionCollection()
{
  return KFTPAPI::getInstance()->mainWindow()->actionCollection();
}

ThreadView::ThreadView(QWidget *parent, const char *name)
 : QWidget(parent, name)
{
  QVBoxLayout *layout = new QVBoxLayout(this);

  // Create the list view
  m_threads = new KFTPWidgets::ListView(this);
  
  // Create the columns
  m_threads->addColumn(i18n("Name"), 400);
  m_threads->addColumn(i18n("Status"), 120);
  m_threads->addColumn(i18n("Speed"), 70);
  
  // Text when there are no threads
  m_threads->setEmptyListText(i18n("There are no threads currently running."));

  // Multi-select
  m_threads->setSelectionModeExt(KListView::FileManager);
  m_threads->setAllColumnsShowFocus(true);
  m_threads->setRootIsDecorated(true);

  layout->addWidget(m_threads);
  
  connect(FTPSessionManager, SIGNAL(update()), this, SLOT(slotUpdateSessions()));
}

ThreadView::~ThreadView()
{
}

void ThreadView::slotUpdateSessions()
{
  KFTPSessionList *list = FTPSessionManager->getSessionList();
  KFTPSession *i;
  
  m_threads->clear();
  
  for (i = list->first(); i; i = list->next()) {
    if (i->isRemote()) {
      ThreadViewItem *site = new ThreadViewItem(i, m_threads);
      
      QPtrList<KFTPSessionConnection> *c_list = i->getConnectionList();
      
      if (c_list->count() > 0) {
        KFTPSessionConnection *conn;
        int id = 0;
        
        for (conn = c_list->first(); conn; conn = c_list->next()) {
          new ThreadViewItem(conn, site, ++id);
        }
        
        site->setOpen(true);
      }
    }
  }
}

}

#include "threadview.moc"

