/*
 * This file is part of the KFTPGrabber project
 *
 * Copyright (C) 2003-2005 by the KFTPGrabber developers
 * Copyright (C) 2003-2005 Jernej Kos <kostko@jweb-network.net>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * is provided AS IS, WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE, and
 * NON-INFRINGEMENT.  See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Steet, Fifth Floor, Boston,
 * MA 02110-1301, USA.
 *
 * In addition, as a special exception, the copyright holders give
 * permission to link the code of portions of this program with the
 * OpenSSL library under certain conditions as described in each
 * individual source file, and distribute linked combinations
 * including the two.
 *
 * You must obey the GNU General Public License in all respects
 * for all of the code used other than OpenSSL.  If you modify
 * file(s) with this exception, you may extend this exception to your
 * version of the file(s), but you are not obligated to do so.  If you
 * do not wish to do so, delete this exception statement from your
 * version.  If you delete this exception statement from all source
 * files in the program, then also delete it here.
 */

#include <qpoint.h>
#include <qtimer.h>
#include <qcursor.h>

#include <kwin.h>
#include <kdebug.h>
#include <kaction.h>
#include <klocale.h>
#include <kpopupmenu.h>
#include <kmessagebox.h>
#include <kactioncollection.h>

#include "qcut.h"
#include "kftpballoon.h"
#include "kftpsession.h"
#include "kftpclientthread.h"

#include "dialog.h"
#include "listview.h"

namespace KFTPWidgets {

namespace ServerManagement {

/////////////////////////////////////////////////////////////////////////////////
///////////////////     ListViewItem     ////////////////////
/////////////////////////////////////////////////////////////////////////////////

ListViewItem::ListViewItem(QListView* parent, const QString &text)
  : KListViewItem(parent, text)
{
  m_valueMin = 0;
  m_valueMax = 0;
}

ListViewItem::ListViewItem(QListViewItem* parent, const QString &text)
  : KListViewItem(parent, text)
{
}

void ListViewItem::setDescription(QString description)
{
  m_description = description;
  setText(1, m_description);
}

void ListViewItem::setValueRange(QString range)
{
  m_valueMinMax = range;
  m_valueMin = QCut().cut( "", "-", range, 0, 1 ).toInt();
  m_valueMax = QCut().cut( "-", "", range, 1, 0 ).toInt();
}

////////////////////////////////////////////////////////////////////////////
///////////////////     ListView     ///////////////////
////////////////////////////////////////////////////////////////////////////

ListView::ListView(KFTPServerManagement::Manager* manager, QWidget* parent, const char *name)
  : KFTPWidgets::ListView(parent, name),
    m_item(0),
    m_timer1(0),
    m_timer2(0),
    m_balloon(0),
    m_session(0),
    m_client(0),
    m_manager(manager),
    m_dialog(0),
    m_popupMenu(0)
{
  // Debug the Constructor
  init();
}

void ListView::init()
{
  m_actionCollection = new KActionCollection(this);

  initActions();

  // Add columns
  addColumn(i18n("Command"), 200);
  addColumn(i18n("Description"));

  // expand root item
  setRootIsDecorated(true);

  connect(this, SIGNAL(doubleClicked(QListViewItem*, const QPoint&, int )),
          this, SLOT  (slotDoubleClickedItem(QListViewItem*, const QPoint&, int)));

  connect(this, SIGNAL(contextMenuRequested(QListViewItem*, const QPoint&, int )),
          this, SLOT  (slotContextMenu(QListViewItem*, const QPoint&, int )));
}

void ListView::initActions()
{
  m_actionWhatsThis = new KAction(i18n("What's this?"), "contexthelp", KShortcut(), this, SLOT(slotWhatsThis()), m_actionCollection, "actionWhatsThis");
}

void ListView::slotDoubleClickedItem(QListViewItem* item, const QPoint &pos, int)
{
  // get current selected item;
  m_item = dynamic_cast<ListViewItem*>(item);

  // Ignore categories
  if (item->childCount() > 0)
    return;

  // save clicked item position
  m_itemPos = pos;

  // clear balloon text
  m_balloonText = QString::null;

  // Get the active side
  m_session = FTPSessionManager->getActiveSide();

  // no active side
  if (m_session == NULL)
    return;

  // get current client
  m_client = m_session->getClient();

  // show the dialog if the user must input something
  showDialog();
}

void ListView::showDialog()
{
  // if the dialog was already shown
  if (m_dialog) {
    delete m_dialog;
    m_dialog = 0;
  }

  // create new Dialog
  m_dialog = new Dialog();

  connect(m_dialog, SIGNAL(command(bool,QString)), this, SLOT(slotSendCommand(bool,QString)));

  // show a dialog if the user must input something
  m_dialog->showDialog(m_item);
}

void ListView::slotSendCommand(bool execute, QString command)
{
  if (!execute) {
    KMessageBox::sorry(0, command),
    showDialog();
  } else {
//     kdDebug() << command << endl;

//   don't execute command ( test )
//     return;

    // send the command
    m_client->rawCommand(command);

    connect(m_client, SIGNAL(rawReply(const QString&)), this, SLOT(slotCollectText(const QString&)));
  }
}

void ListView::slotCollectText(const QString& text)
{
  // append text
  m_balloonText += text+"<br>";

  // timer already exists from a previous instance, delete it
  if (m_timer1) {
    m_timer1->stop();
    delete m_timer1;
    m_timer1 = 0;
  }

  m_timer1 = new QTimer();
  connect(m_timer1, SIGNAL(timeout()), this, SLOT(slotDisconnectBalloon()));

  // wait 500 milli seconds before showing the balloon
  // this is if more lines comes from the output
  m_timer1->start(500, true);
}

void ListView::slotDisconnectBalloon()
{
//   kdDebug() << "slotDisconnectBalloon() called" << endl;

  // disconnect the signal, 500 milli seconds expired from m_timer1
  disconnect(m_client, SIGNAL(rawReply(const QString&)), this, SLOT(slotCollectText(const QString&)));

  // show a balloon
  showBalloon(m_balloonText);
}

void ListView::showBalloon(const QString &text)
{
  // Balloon already exists - user clicked a new site command and didn't wait until the old balloon hides
  if (m_balloon) {
    m_balloon->hide();
    delete m_balloon;
    m_balloon = 0;
  }

  // customize ballon show up, depends on the text length
  int time = 0;
  time = text.length() / 20;

  // new Balloon
  m_balloon = new KFTPBalloon("<qt><nobr><b>KFTPGrabber "+i18n("Server Management")+":</b><br>"+text+"</nobr></qt>", "info");
  // set Position to current Cursor position
//   m_balloon->setAnchor(QCursor::pos());
  m_balloon->setAnchor(m_itemPos);
  // show the balloon
  m_balloon->show();
  // show the balloon on every desktop
  KWin::setOnAllDesktops(m_balloon->winId() , true);

  // timer already exists from a previous instance
  if(m_timer2) {
    // stop timer and delete it
    m_timer2->stop();
    delete m_timer2;
    m_timer2 = 0;
  }

  // new timer
  m_timer2 = new QTimer();

  //    hide the balloon if the timer ends
  // or if the user clicked the balloon
  connect(m_timer2, SIGNAL(timeout()), this, SLOT(slotHideBalloon()));
  connect(m_balloon, SIGNAL(signalBalloonClicked()), this, SLOT(slotHideBalloon()));

  // Hide the balloon after 5 sec
  m_timer2->start(time*1000, true);
}

void ListView::slotHideBalloon()
{
  if(m_balloon) {
    // hide the balloon
    m_balloon->hide();
    // set balloon text empty
    m_balloonText = QString::null;
  }
}

void ListView::slotSiteStat(const QString& stat)
{
  bool ok = false;

  static double download = 0;
  static double upload = 0;
  static double credits = 0;
  static QString ratio = 0;

  if (stat.contains("Ul:"))
  {
    upload = QCut().cut( "Ul: ", "MB", stat, 1, 1 ).toDouble();
    download = QCut().cut( "Dl: ", "MB", stat, 1, 2 ).toDouble();
//     kdDebug() << upload << endl;
//     kdDebug() << download << endl;
  }

  if (stat.contains("Credits:"))
  {
    credits = QCut().cut( "Credits: ", "MB", stat, 1, 1 ).toDouble();
    ratio = QCut().cut( "Ratio: ", "]", stat, 1, 3 );
//     kdDebug() << credits << endl;
//     kdDebug() << ratio << endl;
    ok = true;
  }

  if (ok) {
    emit setStatus(i18n("Download: %1 MB Upload: %2 MB<br>Credits: %3 MB Ratio: %4").arg(download).arg(upload).arg(credits).arg(ratio));

    disconnect(m_client, SIGNAL(rawReply(const QString&)), this, SLOT(slotSiteStat(const QString&)));
  }
}

void ListView::slotContextMenu(QListViewItem*, const QPoint &pos, int )
{
  // save item position
  m_itemPos = pos;

  m_popupMenu = new KPopupMenu(this, "popupMenu");

  m_actionWhatsThis->plug(m_popupMenu);

  m_popupMenu->popup(m_itemPos);
}

void ListView::slotWhatsThis()
{
  ListViewItem* item = dynamic_cast<ListViewItem*>(currentItem());
  showBalloon(item->whatsThis());
//   kdDebug() << item->whatsThis() << endl;
}

}

}

#include "listview.moc"
