/*
 * This file is part of the KFTPGrabber project
 *
 * Copyright (C) 2003-2005 by the KFTPGrabber developers
 * Copyright (C) 2003-2005 Jernej Kos <kostko@jweb-network.net>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * is provided AS IS, WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE, and
 * NON-INFRINGEMENT.  See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Steet, Fifth Floor, Boston,
 * MA 02110-1301, USA.
 *
 * In addition, as a special exception, the copyright holders give
 * permission to link the code of portions of this program with the
 * OpenSSL library under certain conditions as described in each
 * individual source file, and distribute linked combinations
 * including the two.
 *
 * You must obey the GNU General Public License in all respects
 * for all of the code used other than OpenSSL.  If you modify
 * file(s) with this exception, you may extend this exception to your
 * version of the file(s), but you are not obligated to do so.  If you
 * do not wish to do so, delete this exception statement from your
 * version.  If you delete this exception statement from all source
 * files in the program, then also delete it here.
 */
#include "scheduler.h"
#include "misc/config.h"
#include "kftpsession.h"
#include "engine/speedlimiter.h"

#include <kstaticdeleter.h>

namespace KFTPQueue {

Scheduler *Scheduler::m_self = 0;
static KStaticDeleter<Scheduler> staticSchedulerDeleter;

Scheduler *Scheduler::self()
{
  if (!m_self) {
    staticSchedulerDeleter.setObject(m_self, new Scheduler());
  }
  
  return m_self;
}

Scheduler::Scheduler()
 : QObject()
{
  // Be up to date with latest limits configuration
  connect(KFTPCore::Config::self(), SIGNAL(configChanged()), this, SLOT(slotConfigUpdate()));
}

Scheduler::~Scheduler()
{
  if (m_self == this)
    staticSchedulerDeleter.setObject(m_self, 0, false);
}

void Scheduler::registerTransfer(TransferFile *transfer)
{
  // Check if the transfer is already registred
  if (m_transfers.findRef(transfer) != -1)
    return;
    
  // Register the transfer
  connect(transfer, SIGNAL(objectUpdated()), this, SLOT(slotTransferUpdated()));
  connect(transfer, SIGNAL(transferComplete(long)), this, SLOT(slotTransferDone(long)));
  connect(transfer, SIGNAL(transferAbort(long)), this, SLOT(slotTransferDone(long)));
  connect(transfer, SIGNAL(destroyed(QObject*)), this, SLOT(slotTransferDestroyed(QObject*)));
  
  m_transfers.append(transfer);
  slotConfigUpdate();
}

int Scheduler::countTransfers(TransferType type)
{
  int count = 0;
  
  for (TransferFile *i = m_transfers.first(); i; i = m_transfers.next()) {
    if (i->getTransferType() == type)
      count++;
  }
  
  return count == 0 ? 1 : count;
}

filesize_t Scheduler::allocateUpload()
{
  filesize_t r = m_unallocatedUp;
  m_unallocatedUp = 0;
  return r;
}

filesize_t Scheduler::allocateDownload()
{
  filesize_t r = m_unallocatedDown;
  m_unallocatedDown = 0;
  return r;
}

void Scheduler::slotConfigUpdate()
{
  // Reset speed limits
  m_limits.clear();
  filesize_t limitDown = KFTPCore::Config::downloadSpeedLimit() * 1024 / countTransfers(Download);
  filesize_t limitUp = KFTPCore::Config::uploadSpeedLimit() * 1024 / countTransfers(Upload);
  
  for (TransferFile *i = m_transfers.first(); i; i = m_transfers.next()) {
    filesize_t limit = 0;
    
    switch (i->getTransferType()) {
      case Download: limit = limitDown; break;
      case Upload: limit = limitUp; break;
      case FXP: continue;
    }
    
    i->remoteConnection()->getClient()->socket()->getSpeedLimiter()->setLimit(limit);
    m_limits[i->getId()] = limit;
  }
  
  m_unallocatedUp = 0;
  m_unallocatedDown = 0;
}

void Scheduler::slotTransferUpdated()
{
  TransferFile *transfer = (TransferFile*) QObject::sender();
  
  if (m_limits[transfer->getId()] == 0 || transfer->getSpeed() <= 1024)
    return;
  
  if (transfer->getSpeed() < m_limits[transfer->getId()]) {
    // Deallocate some transfer bandwidth
    if (transfer->getTransferType() == Download)
      deallocateDownload(m_limits[transfer->getId()] - transfer->getSpeed());
    else
      deallocateUpload(m_limits[transfer->getId()] - transfer->getSpeed());
      
    m_limits[transfer->getId()] = transfer->getSpeed();
  } else {
    // This transfer needs some more bandwidth
    if (transfer->getTransferType() == Download)
      m_limits[transfer->getId()] += allocateDownload();
    else
      m_limits[transfer->getId()] += allocateUpload();
  }
  
  filesize_t limit = m_limits[transfer->getId()];
  transfer->remoteConnection()->getClient()->socket()->getSpeedLimiter()->setLimit(limit);
}

void Scheduler::slotTransferDone(long id)
{
  TransferFile *transfer = (TransferFile*) QObject::sender();
  
  if (transfer->getTransferType() == Download)
    deallocateDownload(m_limits[id]);
  else
    deallocateUpload(m_limits[id]);
  
  // Disconnect signals
  transfer->QObject::disconnect(this);
  
  m_transfers.remove(transfer);
}

void Scheduler::slotTransferDestroyed(QObject *object)
{
  TransferFile *transfer = static_cast<TransferFile*>(object);
  
  m_transfers.remove(transfer);
}

}

#include "scheduler.moc"
