/***************************************************************************
 *   Copyright (C) 2008 by Konstantinos Smanis                             *
 *   kon.smanis@gmail.com                                                  *
 *                                                                         *
 *   This file is part of KGRUBEditor.                                     *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin Street, Fifth Floor                                       *
 *   Boston, MA  02111-1307, USA.                                          *
 ***************************************************************************/

#ifndef FILETRANSACTIONS_H
#define FILETRANSACTIONS_H

//KDE
class KUrl;

//KGRUBEditor
#include "data.h"

namespace FileTransactions
{
	/**
	* Reads input from the 2 files specified and extracts information. File errors are handled through message boxes and debug messages.
	* Supports Netowrk access using KIO::NetAccess.
	*
	* @param menulst The GRUB's configuration file. Needed to extract GRUB Entry and Settings information. Usually this is /boot/grub/menu.lst.
	* @param devicemap The GRUB's map file. In conjuction with mtab, it retrieves  information about how GRUB names each device. Usually this
	*                  is /boot/grub/device.map.
	* @param settings A class which contains all GRUB Settings which were detected. Used to extract information from menulst.
	* @param entries A vector containing all GRUB entries. Used to extract information from menulst.
	* @param devices A class which contains all GRUB Device Naming information found in devicemap and mtab.
	*/
	void fileInput( const KUrl menulst, const KUrl devicemap, GRUB::ConfigFile::Settings *settings, QVector<GRUB::ConfigFile::Entry> *entries, QVector<GRUB::Misc::Device> *devices );
	/**
	* Writes information found in the settings and items parameters to the file menulst.
	* Supports Netowrk access using KIO::NetAccess.
	*
	* @param menulst The GRUB's configuration file. Needed to write GRUB Entry and Settings information. Usually this is /boot/grub/menu.lst.
	* @param settings A class which contains all GRUB Settings which were detected. Used to write information to menulst.
	* @param entries A vector containing all GRUB entries. Used to write information to menulst.
	*/
	void fileOutput( const KUrl menulst, const GRUB::ConfigFile::Settings *settings, const QVector<GRUB::ConfigFile::Entry> *entries );
	
	/**
	* Opens file and returns the contents.
	*
	* @return the contents of file file.
	*/
	QString view( const KUrl file );
	
	/**
	* Backups the file source to target.
	*
	* @param source The file to be backed-up.
	* @param target Where the file will be backed-up.
	*/
	void backup( const KUrl source, const KUrl target );
	/**
	* Restores the file source to target.
	*
	* @param source The file to be restored.
	* @param target Where the file will be restored.
	*/
	void restoreBackup( const KUrl source, const KUrl target );
	/**
	* Deletes the file backup.
	*
	* @param backup The file to be deleted.
	*/
	void deleteBackup( const KUrl backup );
	
	/**
	* Moves an entry from source to target in a QVector containing GRUB Entries.
	*
	* @param source The initial position of the entry in the QVector.
	* @param target The final position of the entry in the QVector.
	* @param menulst The GRUB's configuration file. Needed to write GRUB Entry and Settings information. Usually this is /boot/grub/menu.lst.
	* @param settings A class which contains all GRUB Settings which were detected. Used to write information to menulst.
	* @param entries A vector containing all GRUB entries. Used to write information to menulst.
	*/
	void moveEntry( const int source, const int target, const KUrl menulst, const GRUB::ConfigFile::Settings *settings, QVector<GRUB::ConfigFile::Entry> *entries );

	/**
	* Converts a generic path to one using GRUB standard naming.
	* Eg. if /media/linux is recognised by GRUB as (hd0,0)
	*     then the path /media/linux/boot/vmlinuz would be
	*     converted to (hd0,0)/boot/vmlinuz.
	*
	* @param path The path to be converted.
	* @param devices The device list to use for conversion.
	*
	* @return The result of the conversion.
	*/
	QString convertToGRUBPath( const QString path, const QVector<GRUB::Misc::Device> devices );
	/**
	* Converts a GRUB path to one using generic naming.
	* Eg. if /media/linux is recognised by GRUB as (hd0,0)
	*     then the path (hd0,0)/boot/vmlinuz would be
	*     converted to /media/linux/boot/vmlinuz.
	*
	* @param path The path to be converted.
	* @param devices The device list to use for conversion.
	* @param root The root of the file path if already known.
	*
	* @return The result of the conversion.
	*/
	QString convertToGenericPath( const QString path, const QVector<GRUB::Misc::Device> devices );
}
#endif
