/*
 *  Callback management functions
 *  Copyright (C) 2002 Tim Jansen <tim@tjansen.de>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#ifndef KDE_GST_CALLBACKMANAGER_H
#define KDE_GST_CALLBACKMANAGER_H

#include <qptrlist.h>

namespace KDE {
namespace GST {
	struct CallbackData {
		CallbackData(const QString &n,
			     void *wrapCb, 
			     void *origCb, 
			     void *c,
			     bool _unregisterSignal = true) :
			name(n),
			wrapCallback(wrapCb),
			origCallback(origCb),
			cookie(c),
			unregisterSignal(_unregisterSignal) {
		}
		QString name;
		void *wrapCallback;
		void *origCallback;
		void *cookie;
		// if true, contains a signal that must be removed when
		// the wrappers will be disassociated (and not a callback
		// that must be unregistered manually)
		bool unregisterSignal; 
	};

	class CallbackManager {
	private:
		void *reserved;
			
	public:
		CallbackManager() :
			callbackDataList(0) {
		}
		~CallbackManager() {
			if (callbackDataList)
				delete callbackDataList;
		}
		CallbackData *registerWrappedCallback(const QString &name,
						      void *wrapCallback, 
						      void *origCallback, 
						      void *cookie,
						      bool unregisterSignal = true) {
			if (!callbackDataList) {
				callbackDataList = new QPtrList<CallbackData>();
				callbackDataList->setAutoDelete(true);
			}
			
			CallbackData *cd;
			
			cd = findWrappedCallback(name, wrapCallback, 
						 origCallback, cookie);
			if (cd)
				return cd;

			cd = new CallbackData(name,
					      wrapCallback, 
					      origCallback, 
					      cookie,
					      unregisterSignal);
			callbackDataList->append(cd);
			return cd;
		}
		CallbackData *findWrappedCallback(const QString &name,
						  void *wrapCallback, 
						  void *origCallback, 
						  void *cookie) {
			if (!callbackDataList)
				return 0;
			int p = findCallbackIndex(name, 
						  wrapCallback, 
						  origCallback, 
						  cookie);
			if (p < 0)
				return 0;
			return callbackDataList->at(p);
		}
		void removeWrappedCallback(const QString &name,
					   void *wrapCallback, 
					   void *origCallback, 
					   void *cookie) {
			if (!callbackDataList)
				return;
			int p = findCallbackIndex(name,
						  wrapCallback, 
						  origCallback, 
						  cookie);
			if (p < 0)
				return;
			callbackDataList->remove(p);
		}
		// remove all callbacks of that name, regardless of refcount
		void removeAllWrappedCallbacks(const QString &name) {
			if (!callbackDataList)
				return;
			int n = callbackDataList->count();
			for (int i = 0; i < n; i++) {
				if (callbackDataList->at(i)->name == name)
					callbackDataList->remove(i);
			}
		}
		void removeWrappedCallback(CallbackData *cd) {
			if (!callbackDataList)
				return;
			callbackDataList->removeRef(cd);
		}
		int findCallbackIndex(const QString &name,
				      void *wrapCallback, 
				      void *origCallback, 
				      void *cookie) {
			if (!callbackDataList)
				return -1;
			int n = callbackDataList->count();
			for (int i = 0; i < n; i++) {
				CallbackData *cd = callbackDataList->at(i);
				if ((cd->name == name) &&
				    (cd->wrapCallback == wrapCallback) &&
				    (cd->origCallback == origCallback) && 
				    (cd->cookie == cookie))
					return i;
			}
			return -1;
		}

		QPtrList<CallbackData> *callbackDataList;
	};
}
}

#endif
