/*
 *  Interface/wrapper for GStreamer system functions
 *  Copyright (C) 2002 Tim Jansen <tim@tjansen.de>
 *  API Documentation
 *  Copyright (C) 1999,2000 Erik Walthinsen <omega@cse.ogi.edu>
 *                     2000 Wim Taymans <wtay@chello.be>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#ifndef KDE_GST_GSTREAMER_H
#define KDE_GST_GSTREAMER_H

namespace KDE {
namespace GST {
/**
 * GStreamer is a framework for constructing graphs of various filters
 * (termed elements here) that will handle streaming media.  Any discreet
 * (packetizable) media type is supported, with provisions for automatically
 * determining source type.  Metadata can be passed with all data to provide
 * formatting/framing information.  Plugins are heavily used to provide for
 * all elements, allowing one to construct plugins outside of the GST
 * library, even released binary-only if license require (please don't).
 *
 * GStreamer borrows heavily from both the OGI media pipeline and
 * Microsoft's DirectShow, hopefully taking the best of both and leaving the
 * cruft behind.  Its interface is still very fluid (I've redesigned the
 * metadata handling twice already), and thus can be changed to increase the
 * sanity/noise ratio.
 *
 * The GStreamer library should be initialized with Gstreamer::init() before
 * it can be used. You should pass a pointer to the main argc and argv 
 * variables so that GStreamer can process its own command line options, as 
 * shown in the following example.
 * 
 * <pre>
 *   int main (int argc, char *argv[])
 *   {
 *        // initialize the GStreamer library 
 *        KDE::GST::Gstreamer::init (&argc, &argv);
 *        ...
 *   }
 * </pre>
 *
 * Use version() to query the library version at runtime.
 */
	class GStreamer {
	private:
		static bool m_isInitialized;
	public:
/**
 * Initializes the GStreamer library, setting up internal path lists,
 * registering built-in elements, and loading standard plugins.
 * (Note that unlike the C API equivalent, it also initializes 
 *  the gstcontrol library that is used for DParams.)
 *
 * @param argc pointer to application's argc
 * @param argv pointer to application's argv
 */
		static void init(int *argc = 0, char **argv[] = 0);

/**
 * Gets the version number of the GStreamer library.
 *
 * @param major pointer to a uint to store the major version number
 * @param minor pointer to a uint to store the minor version number
 * @param micro pointer to a uint to store the micro version number
 */
		static void version(unsigned int *major,
				    unsigned int *minor,
				    unsigned int *micro);

/**
 * Instructs the core to turn on/off threading. When threading
 * is turned off, all thread operations such as mutexes and conditionals
 * are turned into NOPs. Use this if you want absolute minimal overhead
 * and you don't use any threads in the pipeline.
 * It is not required to call this function in order to use threads.
 *
 * @param useThreads flag indicating threads should be used
 * @see hasThreads()
 */
		static void useThreads(bool useThreads);

/**
 * Query if GStreamer has threads enabled.
 *
 * @return true if threads are enabled.
 * @see useThreads()
 */
		static bool hasThreads();

/**
 * Enter the main GStreamer processing loop. Will not return before
 * you call @ref #mainQuit from another thread. Usually you do not want 
 * to call this, but @ref QApplication#exec.
 */
		static void main();

/**
 * Exits the main GStreamer processing loop that has been started 
 * using @ref #main.
 */
		static void mainQuit();
	};
	
}
}

#endif
