/*
 *  Interface/wrapper for GStreamer GstObject
 *  Copyright (C) 2002 Tim Jansen <tim@tjansen.de>
 *  API Documentation
 *  Copyright (C) 1999,2000 Erik Walthinsen <omega@cse.ogi.edu>
 *                     2000 Wim Taymans <wtay@chello.be>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#ifndef KDE_GST_OBJECT_H
#define KDE_GST_OBJECT_H

#include <kde/gst/wrapper.h>

#include <qobject.h>
#include <qmap.h>

namespace KDE {
namespace GST {
/**
 * Object provides a root for the object hierarchy tree filed in by the
 * KDE::GST wrappers, and is used to wrap GStreamer's GstObject.
 * Object gives us basic refcounting, parenting functionality and locking. 
 * @ref #setName() and @ref #getName() are used to set/get the name of the 
 * object.
 *
 * @short Basis for the KDE::GST Object hierarchy.
 */

	class Object : public Wrapper {
	Q_OBJECT 
        private:
                friend class GStreamer;
                static Wrapper* wrapperFactory(void *real);
		QMap<QString,unsigned long> *m_signalMap;
		void *reserved;
		
	public:
		enum Flags {
			FLAG_DESTROYED   = 0,
			FLAG_FLOATING,
			FLAG_LAST   = 4
		};
	
/**
 * Creates a new Object that wrapps the given GstObject. Usually
 * you really dont want to call this, use @ref #wrap instead.
 * You must never create a object of this class on the stack, always 
 * use new.
 * @param real the GstObject to be wrapped
 */
	        Object(void *real);
	        virtual ~Object();

/**
 * Creates a new Object that wrapps the given GstObject. Unlike the
 * Object constructor this returns the appropriate sub-class for the 
 * GstObject, so when you give a GstElement you will get a KDE::GST::Element 
 * instead of a KDE::GST::Object. And if there is already a wrapper
 * object it will be re-used.
 * @param real the GstObject to be wrapped
 * @return the wrapped Object (or derivative)
 */
		static Object* wrap(void *real);

/**
 * Set the name of the object.
 * @param name new name of object
 */
		void setName(const QString &name);

/**
 * Get the name of the object.
 *
 * @return name of the object
 */
		QString name() const;
		
/**
 * This function returns the entire set of flags for the object.
 */
		unsigned int flags() const;

/**
 * This function checks to see if the given flag is set.
 */
		bool flagIsSet(unsigned int flag);

/**
 * This function sets the given bits.
 */
		void flagSet(unsigned int flag);

/**
 * This function usets the given bits.
 */
		void flagUnset(unsigned int flag);

/**
 * Check if the object is floating.
 */
		bool floating() const;

/**
 * Check if the object has been destroyed.
 */
		bool destroyed() const;


/**
 * This function will obtain a lock on the object, making serialization possible.
 */
		void lock();

/**
 * This function releases a lock on the object.
 */
		void unlock();

/**
 * This function will try to obtain a lock on the object, but will return with 
 * FALSE if it can't get it immediately.
 */
		bool tryLock();


/**
 * Get the parent of this object
 */
		Object* parent() const;

/**
 * Set the parent of the object. The object's reference count is incremented.
 */
		void setParent(Object *parent);

/**
 * Clear the parent of the object, removing the associated reference.
 */
		void unparent();

/**
 * Generates a string describing the path of the object in
 * the object hierarchy. Only useful (or used) for debugging
 *
 * @return a string describing the path of the object
 */
		QString getPathString() const;

/**
 * Increments the refence count on the object.
 */
		void ref();

/**
 * Decrements the refence count on the object.  If reference count hits
 * zero, destroy the object.
 */
		void unref();

/**
 * Removes floating reference on an object.  Any newly created object has
 * a refcount of 1 and is FLOATING.  This function should be used when
 * creating a new object to symbolically 'take ownership of' the object.
 * Use @ref #gst_object_set_parent to have this done for you.
 */
		void sink();
	};
}
}

#endif
