/*
 *  Widget-independent player class for video and audio
 *  Copyright (C) 2002-2003 Tim Jansen <tim@tjansen.de>
 *  Based on gst-player's play.h
 *  Copyright (C) 1999,2000,2001,2002 Erik Walthinsen <omega@cse.ogi.edu>
 *                     2000,2001,2002 Wim Taymans <wtay@chello.be>
 *                               2002 Steve Baker <steve@stevebaker.org>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#ifndef KDE_GSTPLAY_PLAY_H
#define KDE_GSTPLAY_PLAY_H

#include <kde/gst/element.h>
#include <qobject.h>
#include <qmutex.h>
#include <qmap.h>
#include <qtimer.h>

class KDE::GST::Bin;

namespace KDE {
namespace GSTPlay {
	
	class PlayPrivate;

/*
 * Play is a simple class for audio and video playback.
 * It's job is to get the media (supplied by a URI) played.  
 * More specific it should get the media from source to the output elements.
 * How that is done should not be relevant for developers using this class. 
 * A user using this class should not have to know very much about how
 * GStreamer works, other than that it plays back media.
 * Additionally it supplies signals to get information about the current
 * playing state.
 * @see VideoScreen
 */
	class Play : public QObject {
	Q_OBJECT
	public:
	        enum PipeType {
			PIPE_AUDIO_THREADED,
			PIPE_AUDIO_BUFFER_THREADED,
			PIPE_VIDEO
		};
	protected:

	        KDE::GSTPlay::PlayPrivate *d;
		
		static void callbackAudioSinkEos(KDE::GST::Element *element);
		static void callbackVideoHaveXid(void *element, int xid, Play *p);
		static void callbackVideoHaveSize(void *e, int w, int h, Play *p);
		static void callbackBinPreIterate(KDE::GST::Bin*,void*);
		static void callbackBinPostIterate(KDE::GST::Bin*,void*);
		
		bool setupForAudioThreaded();
		bool setupForAudioBufferThreaded();
		bool setupForVideoNormal();
		bool setupForVideoThreadsafe();

		bool setAudioForAudioThreaded(KDE::GST::Element *audioSink);
		bool setAutoForAudioThreaded(KDE::GST::Element *autoplugger);
		bool setAudioForAudioBufferThreaded(KDE::GST::Element *audioSink);
		bool setAutoForAudioBufferThreaded(KDE::GST::Element *autoplugger);
		bool setAutoForVideoNormal(KDE::GST::Element *autoplugger);
		bool setVideoForVideoNormal(KDE::GST::Element *videoSink);
		bool setAudioForVideoNormal(KDE::GST::Element *audioSink);

		void customEvent(QCustomEvent *e);
		
	private slots:
		void slotStateChange(KDE::GST::Element::State old, 
				     KDE::GST::Element::State state);
	        void slotGetLength();
		void slotTick();
		
	public:
/**
 * Constructs a new Play object of the given type. 
 * @param pipeType the type of the pipeline. Unless you have very special needs,
 *                 use PIPE_VIDEO for videos and PIPE_AUDIO_THREADED for audio 
 *                 only.
 * @param parent the parent widget
 * @param name the name of the widget
 */
	         Play(PipeType pipeType, 
		      QObject *parent = 0,
		      const char *name = 0);
		 ~Play();

/**
 * Seeks to the specified position
 * @param timeNanos the position to seek to in ns
 */		 
		 void seekToTime(long long timeNanos);
		 
/**
 * Replaces the video sink with a custom video sink. Should
 * be called before you start playing.
 * @param element the new video sink
 */
		 bool setVideoSink(KDE::GST::Element *element);

/**
 * Replaces the video sink with a custom audio sink. Should be
 * called before you start playing.
 * @param element the new audio sink
 */
		 bool setAudioSink(KDE::GST::Element *element);

		 void needNewVideoWindow();

/**
 * Sets the state of the pipeline. Use this function to start playing,
 * to pause etc.
 * @param state the new state.
 * @return the new state
 */		 
		 KDE::GST::Element::State setState(KDE::GST::Element::State state);
	    
/**
 * Retrieves the current state of the pipeline.
 * @return the current state
 */
		 KDE::GST::Element::State getState();
		 
/**
 * Sets the location of the file to play. If a file has been set, the 
 * pipeline's state will be resetted.
 */
		 bool setLocation(const QString &location);

/**
 * Retrieves the current location.
 */
		 QString getLocation();
		 
/**
 * Not implemented yet.
 */
		 void setVolume(float volume);
/**
 * Not implemented yet.
 */
		 float getVolume();
	
/**
 * Not implemented yet.
 */
		 void setMute(bool mute);
/**
 * Not implemented yet.
 */
		 bool getMute();
	signals:
/**
 * Called when the Window Id for video is known. You should then
 * embed it. If you use @ref VideoScreen you do not have to care
 * for this is VideoScreen will do the work.
 * @param wId the window id
 * @see VideoScreen
 */
	         void haveXId(int wId);

/**
 * Called when the video's size is known or changed
 * @param w width of the video
 * @param h height of the video
 */
		 void haveVideoSize(int w, int h);

/**
 * Called when the pipeline's state changed.
 * @param oldState old state 
 * @param newState new state
 */
		 void stateChange(KDE::GST::Element::State oldState, 
				  KDE::GST::Element::State newState);

/**
 * Called once a second with the current position
 * @param time the position of the stream in ns
 */		 
		 void timeTick(long long time);

/**
 * Called when the stream's length is known or has changed. When a new
 * location is set streamLength() will be called with 0 as length.
 * @param lengthNs the new length (0 when unknown)
 */
		 void streamLength(long long lengthNs);

/**
 * Called when playing ended.
 */
		 void streamEnd();
	};
}
}

#endif
