/**
 * \file id3form.cpp
 * GUI for kid3, originally generated by QT Designer.
 *
 * \b Project: Kid3
 * \author Urs Fleisch
 * \date 8 Apr 2003
 */

#include <qcheckbox.h>
#include <qpushbutton.h>
#include <qspinbox.h>
#include <qlayout.h>
#include <qtooltip.h>
#include <qsplitter.h>
#include <qdir.h>
#include <qframe.h>

#if QT_VERSION >= 0x040000
#include <Q3DragObject>
#include <Q3ListBox>
#include <Q3VBox>
#include <QVBoxLayout>
#include <QDropEvent>
#include <QDragEnterEvent>
#include <QLabel>
#include <QGridLayout>
#include <Q3GroupBox>
#else
#include <qdragobject.h>
#include <qlistbox.h>
#include <qvbox.h>
#include <qgroupbox.h>
#endif

/** Shortcut for pointer to parent (application) widget. */
#define theApp ((Kid3App *)parentWidget())

#include "filelist.h"
#include "framelist.h"
#include "genres.h"
#include "kid3.h"
#include "miscconfig.h"
#include "formatconfig.h"
#include "id3form.h"

/**
 * A QScrollView which returns the sizeHint of its child.
 * This is necessary to get a reasonable default size of the window.
 */
class Kid3ScrollView: public Q3ScrollView {
public:
	/**
	 * Constructor.
	 * @param parent parent widget
	 * @param name   name
	 * @param f      window flags
	 */
	Kid3ScrollView(QWidget* parent = 0, const char* name = 0, Qt::WFlags f = 0);
	/**
	 * Constructor.
	 * @param parent  parent widget
	 * @param _client client widget
	 */
	Kid3ScrollView(QWidget* parent, QWidget *_client);
	/**
	 * Get size hint.
	 * @return size hint.
	 */
	virtual QSize sizeHint() const;
	/**
	 * Add child widget.
	 * @param child child widget
	 * @param x     x-coordinate
	 * @param y     y-coordinate
	 */
	virtual void addChild(QWidget* child, int x = 0, int y = 0);
private:
	QWidget* client;
};

/**
 * Constructor.
 *
 * @param parent parent widget
 * @see QScrollView
 */
Kid3ScrollView::Kid3ScrollView(QWidget* parent, const char* name, Qt::WFlags f)
	: Q3ScrollView(parent, name, f), client(0) {}

/**
 * Returns the recommended size for the widget by using the sizeHint of
 * the child.
 *
 * @return recommended size.
 */
QSize Kid3ScrollView::sizeHint() const
{
	return client ? client->sizeHint() : Q3ScrollView::sizeHint();
}

/**
 * Add a single widget to the ScrollView.
 * The widget's parent should be the ScrollView's viewport.
 *
 * @param child child widget
 */
void Kid3ScrollView::addChild(QWidget* child, int x, int y)
{
	client = child;
	Q3ScrollView::addChild(child, x, y);
}

/** 
 * Constructs an Id3Form as a child of 'parent', with the 
 * name 'name' and widget flags set to 'f'.
 * @param parent parent widget
 * @param name   Qt name
 */
Id3Form::Id3Form(QWidget* parent, const char* name)
	: QSplitter(parent, name)
{
#if QT_VERSION >= 0x040200
	const int margin = 6;
	const int spacing = 2;
#elif QT_VERSION >= 0x040000
	const int margin = 12;
	const int spacing = 2;
#else
	const int margin = 16;
	const int spacing = 6;
#endif

	setAcceptDrops(true);
	setCaption(i18n("Kid3"));

	m_vSplitter = new QSplitter(Qt::Vertical, this, "vSplitter");
	m_fileListBox = new FileList(m_vSplitter, "mp3ListBox");
	m_dirListBox = new DirList(m_vSplitter, "dirListBox");

	Kid3ScrollView* scrollView = new Kid3ScrollView(this);
	scrollView->setResizePolicy(Q3ScrollView::AutoOneFit);
	scrollView->setFrameStyle(QFrame::NoFrame);
	m_rightHalfVBox = new Q3VBox(scrollView->viewport());
	m_rightHalfVBox->setSpacing(2);
	m_rightHalfVBox->setMargin(2);


	Q3GroupBox* filenameGroupBox = new Q3GroupBox(m_rightHalfVBox, "filenameGroupBox");
	filenameGroupBox->setTitle(i18n("File&name"));
	QGridLayout* filenameGroupBoxLayout = new QGridLayout(filenameGroupBox, 3, 3, margin, spacing);

	QLabel* nameLabel = new QLabel(filenameGroupBox, "nameLabel");
	nameLabel->setText(i18n("Name:"));
	filenameGroupBoxLayout->addWidget(nameLabel, 0, 0);

	m_nameLineEdit = new QLineEdit(filenameGroupBox, "nameLineEdit");
	filenameGroupBoxLayout->addWidget(m_nameLineEdit, 0, 1);

	m_fnV1Button = new QPushButton(filenameGroupBox, "fnV1Button");
	m_fnV1Button->setText(i18n("From Tag 1"));
	filenameGroupBoxLayout->addWidget(m_fnV1Button, 0, 2);

	QLabel* formatLabel = new QLabel(filenameGroupBox, "formatLabel");
	formatLabel->setText(i18n("Format:"));
	filenameGroupBoxLayout->addWidget(formatLabel, 1, 0);

	m_formatComboBox = new QComboBox(false, filenameGroupBox, "formatComboBox");
	filenameGroupBoxLayout->addWidget(m_formatComboBox, 1, 1);

	QPushButton* fnV2Button = new QPushButton(filenameGroupBox, "fnV2Button");
	fnV2Button->setText(i18n("From Tag 2"));
	filenameGroupBoxLayout->addWidget(fnV2Button, 1, 2);

	QLabel* infoLabel = new QLabel(filenameGroupBox, "infoLabel");
	infoLabel->setText(i18n("Info:"));
	filenameGroupBoxLayout->addWidget(infoLabel, 2, 0);

	m_detailsLabel = new QLabel(filenameGroupBox, "detailsLabel");
	filenameGroupBoxLayout->addMultiCellWidget(m_detailsLabel, 2, 2, 1, 2);

	m_idV1GroupBox = new Q3GroupBox(m_rightHalfVBox, "idV1GroupBox");
	m_idV1GroupBox->setTitle(i18n("Tag &1"));
	QGridLayout* idV1GroupBoxLayout = new QGridLayout(m_idV1GroupBox, 6, 3, margin, spacing);

	m_titleV1CheckBox = new QCheckBox(m_idV1GroupBox, "titleV1CheckBox");
	m_titleV1CheckBox->setText(i18n("Title:"));
	idV1GroupBoxLayout->addWidget(m_titleV1CheckBox, 0, 0);

	m_titleV1LineEdit = new QLineEdit(m_idV1GroupBox, "titleV1LineEdit");
	m_titleV1LineEdit->setMaxLength(30);
	idV1GroupBoxLayout->addWidget(m_titleV1LineEdit, 0, 1);

	QPushButton* filenameV1PushButton = new QPushButton(
		m_idV1GroupBox, "filenameV1PushButton");
	filenameV1PushButton->setText(i18n("From Filename"));
	idV1GroupBoxLayout->addWidget(filenameV1PushButton, 0, 2);

	m_artistV1CheckBox = new QCheckBox(m_idV1GroupBox, "artistV1CheckBox");
	m_artistV1CheckBox->setText(i18n("Artist:"));
	idV1GroupBoxLayout->addWidget(m_artistV1CheckBox, 1, 0);

	m_artistV1LineEdit = new QLineEdit(m_idV1GroupBox, "artistV1LineEdit");
	m_artistV1LineEdit->setMaxLength(30);
	idV1GroupBoxLayout->addWidget(m_artistV1LineEdit, 1, 1);

	QPushButton* id3V1PushButton = new QPushButton(m_idV1GroupBox, "id3V1PushButton");
	id3V1PushButton->setText(i18n("From Tag 2"));
	idV1GroupBoxLayout->addWidget(id3V1PushButton, 1, 2);

	m_albumV1CheckBox = new QCheckBox(m_idV1GroupBox, "albumV1CheckBox");
	m_albumV1CheckBox->setText(i18n("Album:"));
	idV1GroupBoxLayout->addWidget(m_albumV1CheckBox, 2, 0);

	m_albumV1LineEdit = new QLineEdit(m_idV1GroupBox, "albumV1LineEdit");
	m_albumV1LineEdit->setMaxLength(30);
	idV1GroupBoxLayout->addWidget(m_albumV1LineEdit, 2, 1);

	QPushButton* copyV1PushButton = new QPushButton(m_idV1GroupBox, "copyV1PushButton");
	copyV1PushButton->setText(i18n("Copy"));
	idV1GroupBoxLayout->addWidget(copyV1PushButton, 2, 2);

	QPushButton* removeV1PushButton = new QPushButton(m_idV1GroupBox, "removeV1PushButton");
	removeV1PushButton->setText(i18n("Remove"));
	idV1GroupBoxLayout->addWidget(removeV1PushButton, 4, 2);

	m_commentV1CheckBox = new QCheckBox(m_idV1GroupBox, "commentV1CheckBox");
	m_commentV1CheckBox->setText(i18n("Comment:"));
	idV1GroupBoxLayout->addWidget(m_commentV1CheckBox, 3, 0);

	m_commentV1LineEdit = new QLineEdit(m_idV1GroupBox, "commentV1LineEdit");
	m_commentV1LineEdit->setMaxLength(28);
	idV1GroupBoxLayout->addWidget(m_commentV1LineEdit, 3, 1);

	QPushButton* pasteV1PushButton = new QPushButton(m_idV1GroupBox, "pasteV1PushButton");
	pasteV1PushButton->setText(i18n("Paste"));
	idV1GroupBoxLayout->addWidget(pasteV1PushButton, 3, 2);

	m_yearV1CheckBox = new QCheckBox(m_idV1GroupBox, "yearV1CheckBox");
	m_yearV1CheckBox->setText(i18n("Year:"));
	idV1GroupBoxLayout->addWidget(m_yearV1CheckBox, 4, 0);

	Q3HBox* trackV1HBox = new Q3HBox(m_idV1GroupBox, "trackV1HBox");
	trackV1HBox->setSpacing(6);
	m_yearV1SpinBox = new QSpinBox(trackV1HBox, "yearV1SpinBox");
	m_yearV1SpinBox->setMaxValue(9999);
	m_trackV1CheckBox = new QCheckBox(trackV1HBox, "trackV1CheckBox");
	m_trackV1CheckBox->setText(i18n("Track:"));
	m_trackV1SpinBox = new QSpinBox(trackV1HBox, "trackV1SpinBox");
	m_trackV1SpinBox->setMaxValue(255);
	idV1GroupBoxLayout->addWidget(trackV1HBox, 4, 1);

	m_genreV1CheckBox = new QCheckBox(m_idV1GroupBox, "genreV1CheckBox");
	m_genreV1CheckBox->setText(i18n("Genre:"));
	idV1GroupBoxLayout->addWidget(m_genreV1CheckBox, 5, 0);

	m_genreV1ComboBox = new QComboBox(false, m_idV1GroupBox, "genreV1ComboBox");
	idV1GroupBoxLayout->addWidget(m_genreV1ComboBox, 5, 1);

	m_idV2GroupBox = new Q3GroupBox(m_rightHalfVBox, "idV2GroupBox");
	m_idV2GroupBox->setTitle(i18n("Tag &2"));

	QGridLayout* idV2GroupBoxLayout = new QGridLayout(m_idV2GroupBox, 7, 3, margin, spacing);
	idV2GroupBoxLayout->setAlignment(Qt::AlignTop);

	m_titleV2CheckBox = new QCheckBox(m_idV2GroupBox, "titleV2CheckBox");
	m_titleV2CheckBox->setText(i18n("Title:"));
	idV2GroupBoxLayout->addWidget(m_titleV2CheckBox, 0, 0);

	m_titleV2LineEdit = new QLineEdit(m_idV2GroupBox, "titleV2LineEdit");
	idV2GroupBoxLayout->addWidget(m_titleV2LineEdit, 0, 1);

	QPushButton* filenameV2PushButton = new QPushButton(m_idV2GroupBox,
										   "filenameV2PushButton");
	filenameV2PushButton->setText(i18n("From Filename"));
	idV2GroupBoxLayout->addWidget(filenameV2PushButton, 0, 2);

	m_artistV2CheckBox = new QCheckBox(m_idV2GroupBox, "artistV2CheckBox");
	m_artistV2CheckBox->setText(i18n("Artist:"));
	idV2GroupBoxLayout->addWidget(m_artistV2CheckBox, 1, 0);

	m_artistV2LineEdit = new QLineEdit(m_idV2GroupBox, "artistV2LineEdit");
	idV2GroupBoxLayout->addWidget(m_artistV2LineEdit, 1, 1);

	m_id3V2PushButton = new QPushButton(m_idV2GroupBox, "id3V2PushButton");
	m_id3V2PushButton->setText(i18n("From Tag 1"));
	idV2GroupBoxLayout->addWidget(m_id3V2PushButton, 1, 2);

	m_albumV2CheckBox = new QCheckBox(m_idV2GroupBox, "albumV2CheckBox");
	m_albumV2CheckBox->setText(i18n("Album:"));
	idV2GroupBoxLayout->addWidget(m_albumV2CheckBox, 2, 0);

	m_albumV2LineEdit = new QLineEdit(m_idV2GroupBox, "albumV2LineEdit");
	idV2GroupBoxLayout->addWidget(m_albumV2LineEdit, 2, 1);

	QPushButton* copyV2PushButton = new QPushButton(m_idV2GroupBox, "copyV2PushButton");
	copyV2PushButton->setText(i18n("Copy"));
	idV2GroupBoxLayout->addWidget(copyV2PushButton, 2, 2);

	m_commentV2CheckBox = new QCheckBox(m_idV2GroupBox, "commentV2CheckBox");
	m_commentV2CheckBox->setText(i18n("Comment:"));
	idV2GroupBoxLayout->addWidget(m_commentV2CheckBox, 3, 0);

	m_commentV2LineEdit = new QLineEdit(m_idV2GroupBox, "commentV2LineEdit");
	idV2GroupBoxLayout->addWidget(m_commentV2LineEdit, 3, 1);

	QPushButton* pasteV2PushButton = new QPushButton(m_idV2GroupBox, "pasteV2PushButton");
	pasteV2PushButton->setText(i18n("Paste"));
	idV2GroupBoxLayout->addWidget(pasteV2PushButton, 3, 2);

	m_yearV2CheckBox = new QCheckBox(m_idV2GroupBox, "yearV2CheckBox");
	m_yearV2CheckBox->setText(i18n("Year:"));
	idV2GroupBoxLayout->addWidget(m_yearV2CheckBox, 4, 0);

	Q3HBox* trackV2HBox = new Q3HBox(m_idV2GroupBox, "trackV2HBox"); 
	trackV2HBox->setSpacing(6);
	m_yearV2SpinBox = new QSpinBox(trackV2HBox, "yearV2SpinBox");
	m_yearV2SpinBox->setMaxValue(9999);
	m_trackV2CheckBox = new QCheckBox(trackV2HBox, "trackV2CheckBox");
	m_trackV2CheckBox->setText(i18n("Track:"));
	m_trackV2SpinBox = new QSpinBox(trackV2HBox, "trackV2SpinBox");
	m_trackV2SpinBox->setMaxValue(999);
	idV2GroupBoxLayout->addWidget(trackV2HBox, 4, 1);

	QPushButton* removeV2PushButton = new QPushButton(m_idV2GroupBox, "removeV2PushButton");
	removeV2PushButton->setText(i18n("Remove"));
	idV2GroupBoxLayout->addWidget(removeV2PushButton, 4, 2);

	m_genreV2CheckBox = new QCheckBox(m_idV2GroupBox, "genreV2CheckBox");
	m_genreV2CheckBox->setText(i18n("Genre:"));
	idV2GroupBoxLayout->addWidget(m_genreV2CheckBox, 5, 0);

	m_genreV2ComboBox = new QComboBox(false, m_idV2GroupBox, "genreV2ComboBox");
	m_genreV2ComboBox->setEditable(true);
	m_genreV2ComboBox->setAutoCompletion(true);
	m_genreV2ComboBox->setDuplicatesEnabled(false);
	idV2GroupBoxLayout->addWidget(m_genreV2ComboBox, 5, 1);

	QLabel* framesTextLabel = new QLabel(m_idV2GroupBox, "framesTextLabel");
	framesTextLabel->setText(i18n("Frames:"));

	idV2GroupBoxLayout->addWidget(framesTextLabel, 6, 0, Qt::AlignTop);

	m_framesListBox = new Q3ListBox(m_idV2GroupBox, "framesListBox");
	idV2GroupBoxLayout->addWidget(m_framesListBox, 6, 1);

	QVBoxLayout* frameButtonLayout = new QVBoxLayout;
	frameButtonLayout->setSpacing(spacing);
	QPushButton* editFramesPushButton =
		new QPushButton(i18n("Edit"), m_idV2GroupBox);
	frameButtonLayout->addWidget(editFramesPushButton);
	QPushButton* framesAddPushButton =
		new QPushButton(i18n("Add"), m_idV2GroupBox);
	frameButtonLayout->addWidget(framesAddPushButton);
	QPushButton* deleteFramesPushButton =
		new QPushButton(i18n("Delete"), m_idV2GroupBox);
	frameButtonLayout->addWidget(deleteFramesPushButton);
	frameButtonLayout->addItem(
		new QSpacerItem(0, 0, QSizePolicy::Minimum, QSizePolicy::Expanding));
	idV2GroupBoxLayout->addLayout(frameButtonLayout, 6, 2);


	scrollView->addChild(m_rightHalfVBox);

	// signals and slots connections
	connect(filenameV1PushButton, SIGNAL(clicked()), this,
			SLOT(fromFilenameV1()));
	connect(id3V1PushButton, SIGNAL(clicked()), this, SLOT(fromID3V1()));
	connect(copyV1PushButton, SIGNAL(clicked()), this, SLOT(copyV1()));
	connect(pasteV1PushButton, SIGNAL(clicked()), this, SLOT(pasteV1()));
	connect(removeV1PushButton, SIGNAL(clicked()), this, SLOT(removeV1()));
	connect(filenameV2PushButton, SIGNAL(clicked()), this,
			SLOT(fromFilenameV2()));
	connect(m_id3V2PushButton, SIGNAL(clicked()), this, SLOT(fromID3V2()));
	connect(copyV2PushButton, SIGNAL(clicked()), this, SLOT(copyV2()));
	connect(pasteV2PushButton, SIGNAL(clicked()), this, SLOT(pasteV2()));
	connect(removeV2PushButton, SIGNAL(clicked()), this, SLOT(removeV2()));
	connect(m_fileListBox, SIGNAL(selectionChanged()), this,
			SLOT(fileSelected()));
#if QT_VERSION >= 0x040000
	connect(m_framesListBox, SIGNAL(doubleClicked(Q3ListBoxItem*)), this,
			SLOT(editFrame()));
	connect(m_framesListBox, SIGNAL(returnPressed(Q3ListBoxItem*)), this,
			SLOT(editFrame()));
#else
	connect(m_framesListBox, SIGNAL(doubleClicked(QListBoxItem*)), this,
			SLOT(editFrame()));
	connect(m_framesListBox, SIGNAL(returnPressed(QListBoxItem*)), this,
			SLOT(editFrame()));
#endif
	connect(framesAddPushButton, SIGNAL(clicked()), this, SLOT(addFrame()));
	connect(deleteFramesPushButton, SIGNAL(clicked()), this,
			SLOT(deleteFrame()));
	connect(editFramesPushButton, SIGNAL(clicked()), this, SLOT(editFrame()));
	connect(m_fnV1Button, SIGNAL(clicked()), this, SLOT(fnFromID3V1()));
	connect(fnV2Button, SIGNAL(clicked()), this, SLOT(fnFromID3V2()));
	connect(m_nameLineEdit, SIGNAL(textChanged(const QString&)), this,
			SLOT(nameLineEditChanged(const QString&)));
	connect(m_albumV1LineEdit, SIGNAL(textChanged(const QString&)), this,
			SLOT(albumV1LineEditChanged(const QString&)));
	connect(m_artistV1LineEdit, SIGNAL(textChanged(const QString&)), this,
			SLOT(artistV1LineEditChanged(const QString&)));
	connect(m_titleV1LineEdit, SIGNAL(textChanged(const QString&)), this,
			SLOT(titleV1LineEditChanged(const QString&)));
	connect(m_albumV2LineEdit, SIGNAL(textChanged(const QString&)), this,
			SLOT(albumV2LineEditChanged(const QString&)));
	connect(m_artistV2LineEdit, SIGNAL(textChanged(const QString&)), this,
			SLOT(artistV2LineEditChanged(const QString&)));
	connect(m_titleV2LineEdit, SIGNAL(textChanged(const QString&)), this,
			SLOT(titleV2LineEditChanged(const QString&)));
#if QT_VERSION >= 0x040000
	connect(m_dirListBox, SIGNAL(doubleClicked(Q3ListBoxItem *)), this,
			SLOT(dirSelected(Q3ListBoxItem *)));
	connect(m_dirListBox, SIGNAL(returnPressed(Q3ListBoxItem *)), this,
			SLOT(dirSelected(Q3ListBoxItem *)));
#else
	connect(m_dirListBox, SIGNAL(doubleClicked(QListBoxItem *)), this,
			SLOT(dirSelected(QListBoxItem *)));
	connect(m_dirListBox, SIGNAL(returnPressed(QListBoxItem *)), this,
			SLOT(dirSelected(QListBoxItem *)));
#endif
	connect(m_fileListBox, SIGNAL(selectedFilesRenamed()),
					SIGNAL(selectedFilesRenamed()));

	// tab order
	setTabOrder(m_fileListBox, m_dirListBox);
	setTabOrder(m_dirListBox, filenameGroupBox);
	setTabOrder(filenameGroupBox, m_nameLineEdit);
	setTabOrder(m_nameLineEdit, m_formatComboBox);
	setTabOrder(m_formatComboBox, m_fnV1Button);
	setTabOrder(m_fnV1Button, fnV2Button);
	setTabOrder(fnV2Button, m_idV1GroupBox);
	setTabOrder(m_idV1GroupBox, m_titleV1CheckBox);
	setTabOrder(m_titleV1CheckBox, m_titleV1LineEdit);
	setTabOrder(m_titleV1LineEdit, m_artistV1CheckBox);
	setTabOrder(m_artistV1CheckBox, m_artistV1LineEdit);
	setTabOrder(m_artistV1LineEdit, m_albumV1CheckBox);
	setTabOrder(m_albumV1CheckBox, m_albumV1LineEdit);
	setTabOrder(m_albumV1LineEdit, m_commentV1CheckBox);
	setTabOrder(m_commentV1CheckBox, m_commentV1LineEdit);
	setTabOrder(m_commentV1LineEdit, m_yearV1CheckBox);
	setTabOrder(m_yearV1CheckBox, m_yearV1SpinBox);
	setTabOrder(m_yearV1SpinBox, m_trackV1CheckBox);
	setTabOrder(m_trackV1CheckBox, m_trackV1SpinBox);
	setTabOrder(m_trackV1SpinBox, m_genreV1CheckBox);
	setTabOrder(m_genreV1CheckBox, m_genreV1ComboBox);
	setTabOrder(m_genreV1ComboBox, filenameV1PushButton);
	setTabOrder(filenameV1PushButton, id3V1PushButton);
	setTabOrder(id3V1PushButton, copyV1PushButton);
	setTabOrder(copyV1PushButton, pasteV1PushButton);
	setTabOrder(pasteV1PushButton, removeV1PushButton);
	setTabOrder(removeV1PushButton, m_idV2GroupBox);
	setTabOrder(m_idV2GroupBox, m_titleV2CheckBox);
	setTabOrder(m_titleV2CheckBox, m_titleV2LineEdit);
	setTabOrder(m_titleV2LineEdit, m_artistV2CheckBox);
	setTabOrder(m_artistV2CheckBox, m_artistV2LineEdit);
	setTabOrder(m_artistV2LineEdit, m_albumV2CheckBox);
	setTabOrder(m_albumV2CheckBox, m_albumV2LineEdit);
	setTabOrder(m_albumV2LineEdit, m_commentV2CheckBox);
	setTabOrder(m_commentV2CheckBox, m_commentV2LineEdit);
	setTabOrder(m_commentV2LineEdit, m_yearV2CheckBox);
	setTabOrder(m_yearV2CheckBox, m_yearV2SpinBox);
	setTabOrder(m_yearV2SpinBox, m_trackV2CheckBox);
	setTabOrder(m_trackV2CheckBox, m_trackV2SpinBox);
	setTabOrder(m_trackV2SpinBox, m_genreV2CheckBox);
	setTabOrder(m_genreV2CheckBox, m_genreV2ComboBox);
	setTabOrder(m_genreV2ComboBox, filenameV2PushButton);
	setTabOrder(filenameV2PushButton, m_id3V2PushButton);
	setTabOrder(m_id3V2PushButton, copyV2PushButton);
	setTabOrder(copyV2PushButton, pasteV2PushButton);
	setTabOrder(pasteV2PushButton, removeV2PushButton);
	setTabOrder(removeV2PushButton, m_framesListBox);
	setTabOrder(m_framesListBox, framesAddPushButton);
	setTabOrder(framesAddPushButton, deleteFramesPushButton);

	// go to next line edit when pressing enter
	connect(m_titleV1LineEdit, SIGNAL(returnPressed()), m_artistV1LineEdit, SLOT(setFocus()));
	connect(m_artistV1LineEdit, SIGNAL(returnPressed()), m_albumV1LineEdit, SLOT(setFocus()));
	connect(m_albumV1LineEdit, SIGNAL(returnPressed()), m_commentV1LineEdit, SLOT(setFocus()));
	connect(m_commentV1LineEdit, SIGNAL(returnPressed()), m_yearV1SpinBox, SLOT(setFocus()));
	connect(m_titleV2LineEdit, SIGNAL(returnPressed()), m_artistV2LineEdit, SLOT(setFocus()));
	connect(m_artistV2LineEdit, SIGNAL(returnPressed()), m_albumV2LineEdit, SLOT(setFocus()));
	connect(m_albumV2LineEdit, SIGNAL(returnPressed()), m_commentV2LineEdit, SLOT(setFocus()));
	connect(m_commentV2LineEdit, SIGNAL(returnPressed()), m_yearV2SpinBox, SLOT(setFocus()));
}

/**
 * Button ID3v1 From Filename.
 */
void Id3Form::fromFilenameV1()
{
	theApp->getTagsFromFilenameV1();
}

/**
 * Button ID3v2 From Filename.
 */
void Id3Form::fromFilenameV2()
{
	theApp->getTagsFromFilenameV2();
}

/**
 * Button ID3v2 From ID3v1.
 */
void Id3Form::fromID3V2()
{
	theApp->copyV1ToV2();
}

/**
 * Button ID3v1 From ID3v2.
 */
void Id3Form::fromID3V1()
{
	theApp->copyV2ToV1();
}

/**
 * Button ID3v1 Copy.
 */
void Id3Form::copyV1()
{
	StandardTags st;
	getStandardTagsV1(&st);
	theApp->copyTags(&st);
}

/**
 * Button ID3v2 Copy.
 */
void Id3Form::copyV2()
{
	StandardTags st;
	getStandardTagsV2(&st);
	theApp->copyTags(&st);
}

/**
 * Button ID3v2 Remove.
 */
void Id3Form::removeV2()
{
	theApp->removeTagsV2();
}

/**
 * Button ID3v1 Paste.
 */
void Id3Form::pasteV1()
{
	theApp->pasteTagsV1();
}

/**
 * Button ID3v2 Paste.
 */
void Id3Form::pasteV2()
{
	theApp->pasteTagsV2();
}

/**
 * Button ID3v1 Remove.
 */
void Id3Form::removeV1()
{
	theApp->removeTagsV1();
}

/**
 * File list box file selected
 */
void Id3Form::fileSelected()
{
	theApp->fileSelected();
}

/**
 * Get standard tags from the ID3v1 controls.
 *
 * @param st standard tags to store result
 */
void Id3Form::getStandardTagsV1(StandardTags* st)
{
	st->title   = m_titleV1CheckBox->isChecked()   ? m_titleV1LineEdit->text()
		: QString::null;
	st->artist  = m_artistV1CheckBox->isChecked()  ? m_artistV1LineEdit->text()
		: QString::null;
	st->album   = m_albumV1CheckBox->isChecked()   ? m_albumV1LineEdit->text()
		: QString::null;
	st->comment = m_commentV1CheckBox->isChecked() ? m_commentV1LineEdit->text()
		: QString::null;
	st->year = m_yearV1CheckBox->isChecked()       ? m_yearV1SpinBox->value()
		: -1;
	st->track   = m_trackV1CheckBox->isChecked()   ? m_trackV1SpinBox->value()
		: -1;
	st->genre   = m_genreV1CheckBox->isChecked()   ? m_genreV1ComboBox->currentText()
		: QString::null;
}

/**
 * Get standard tags from the ID3v2 controls.
 *
 * @param st standard tags to store result
 */
void Id3Form::getStandardTagsV2(StandardTags* st)
{
	st->title   = m_titleV2CheckBox->isChecked()   ? m_titleV2LineEdit->text()
		: QString::null;
	st->artist  = m_artistV2CheckBox->isChecked()  ? m_artistV2LineEdit->text()
		: QString::null;
	st->album   = m_albumV2CheckBox->isChecked()   ? m_albumV2LineEdit->text()
		: QString::null;
	st->comment = m_commentV2CheckBox->isChecked() ? m_commentV2LineEdit->text()
		: QString::null;
	st->year    = m_yearV2CheckBox->isChecked()    ? m_yearV2SpinBox->value()
		: -1;
	st->track   = m_trackV2CheckBox->isChecked()   ? m_trackV2SpinBox->value()
		: -1;
	st->genre   = m_genreV2CheckBox->isChecked()   ? m_genreV2ComboBox->currentText()
		: QString::null;
}

/**
 * Set ID3v1 genre controls.
 *
 * @param genreStr genre string
 */
void Id3Form::setGenreV1(const QString& genreStr)
{
	m_genreV1CheckBox->setChecked(!genreStr.isNull());
	int genreIndex = genreStr.isNull() ? 0 :
		Genres::getIndex(Genres::getNumber(genreStr));
	m_genreV1ComboBox->setCurrentItem(
		Kid3App::s_miscCfg.m_onlyCustomGenres ? 0 : genreIndex);
	m_genreV1ComboBox->setCurrentText(genreStr);
}

/**
 * Set ID3v1 standard tags controls.
 *
 * @param st standard tags to set
 */
void Id3Form::setStandardTagsV1(const StandardTags* st)
{
	m_titleV1CheckBox->setChecked(!st->title.isNull());
	m_titleV1LineEdit->setText(st->title);
	m_artistV1CheckBox->setChecked(!st->artist.isNull());
	m_artistV1LineEdit->setText(st->artist);
	m_albumV1CheckBox->setChecked(!st->album.isNull());
	m_albumV1LineEdit->setText(st->album);
	m_commentV1CheckBox->setChecked(!st->comment.isNull());
	m_commentV1LineEdit->setText(st->comment);
	m_yearV1CheckBox->setChecked(st->year >= 0);
	m_yearV1SpinBox->setValue(st->year >= 0 ? st->year : 0);
	m_trackV1CheckBox->setChecked(st->track >= 0);
	m_trackV1SpinBox->setValue(st->track >= 0 ? st->track : 0);
	setGenreV1(st->genre);
}

/**
 * Set ID3v2 genre controls.
 *
 * @param genreStr genre string
 */
void Id3Form::setGenreV2(const QString& genreStr)
{
	m_genreV2CheckBox->setChecked(!genreStr.isNull());
	int genreIndex = genreStr.isNull() ? 0 :
		Genres::getIndex(Genres::getNumber(genreStr));
	m_genreV2ComboBox->setCurrentItem(
		Kid3App::s_miscCfg.m_onlyCustomGenres ? 0 :
		(genreIndex > 0 ? genreIndex : Genres::count + 1));
	m_genreV2ComboBox->setCurrentText(genreStr);
}

/**
 * Set ID3v2 standard tags controls.
 *
 * @param st standard tags to set
 */
void Id3Form::setStandardTagsV2(const StandardTags* st)
{
	m_titleV2CheckBox->setChecked(!st->title.isNull());
	m_titleV2LineEdit->setText(st->title);
	m_artistV2CheckBox->setChecked(!st->artist.isNull());
	m_artistV2LineEdit->setText(st->artist);
	m_albumV2CheckBox->setChecked(!st->album.isNull());
	m_albumV2LineEdit->setText(st->album);
	m_commentV2CheckBox->setChecked(!st->comment.isNull());
	m_commentV2LineEdit->setText(st->comment);
	m_yearV2CheckBox->setChecked(st->year >= 0);
	m_yearV2SpinBox->setValue(st->year >= 0 ? st->year : 0);
	m_trackV2CheckBox->setChecked(st->track >= 0);
	m_trackV2SpinBox->setValue(st->track >= 0 ? st->track : 0);
	setGenreV2(st->genre);
}

/**
 * Set all ID3v1 and ID3v2 check boxes on or off.
 *
 * @param val true to set check boxes on.
 */
void Id3Form::setAllCheckBoxes(bool val)
{
	m_titleV1CheckBox->setChecked(val);
	m_artistV1CheckBox->setChecked(val);
	m_albumV1CheckBox->setChecked(val);
	m_commentV1CheckBox->setChecked(val);
	m_yearV1CheckBox->setChecked(val);
	m_trackV1CheckBox->setChecked(val);
	m_genreV1CheckBox->setChecked(val);
	
	m_titleV2CheckBox->setChecked(val);
	m_artistV2CheckBox->setChecked(val);
	m_albumV2CheckBox->setChecked(val);
	m_commentV2CheckBox->setChecked(val);
	m_yearV2CheckBox->setChecked(val);
	m_trackV2CheckBox->setChecked(val);
	m_genreV2CheckBox->setChecked(val);
}

/**
 * Get number of files selected in file list box.
 *
 * @return number of files selected.
 */
int Id3Form::numFilesSelected()
{
	return m_fileListBox->numFilesSelected();
}

/**
 * Accept drag.
 *
 * @param ev drag event.
 */
void Id3Form::dragEnterEvent(QDragEnterEvent* ev)
{
	ev->accept(Q3TextDrag::canDecode(ev));
}

/**
 * Handle drop event.
 *
 * @param ev drop event.
 */
void Id3Form::dropEvent(QDropEvent* ev)
{
	QString text;
	if (Q3TextDrag::decode(ev, text)) {
		theApp->openDrop(text);
	}
}

/**
 * Frame list button Edit.
 */
void Id3Form::editFrame()
{
	theApp->editFrame();
}

/**
 * Frame list button Add.
 */
void Id3Form::addFrame()
{
	theApp->addFrame();
}

/**
 * Frame list button Delete.
 */
void Id3Form::deleteFrame()
{
	theApp->deleteFrame();
}

/**
 * Set filename according to ID3v1 tags.
 */

void Id3Form::fnFromID3V1()
{
	theApp->getFilenameFromTags(1);
}

/**
 * Set filename according to ID3v1 tags.
 */

void Id3Form::fnFromID3V2()
{
	theApp->getFilenameFromTags(2);
}

/**
 * Filename line edit is changed.
 * @param contents of line edit
 */
void Id3Form::nameLineEditChanged(const QString& txt)
{
	formatLineEdit(m_nameLineEdit, txt, &theApp->s_fnFormatCfg);
}

/**
 * Album V1 line edit is changed.
 * @param contents of line edit
 */
void Id3Form::albumV1LineEditChanged(const QString& txt)
{
	formatLineEdit(m_albumV1LineEdit, txt, &theApp->s_id3FormatCfg);
}

/**
 * Artist V1 line edit is changed.
 * @param contents of line edit
 */
void Id3Form::artistV1LineEditChanged(const QString& txt)
{
	formatLineEdit(m_artistV1LineEdit, txt, &theApp->s_id3FormatCfg);
}

/**
 * Title V1 line edit is changed.
 * @param contents of line edit
 */
void Id3Form::titleV1LineEditChanged(const QString& txt)
{
	formatLineEdit(m_titleV1LineEdit, txt, &theApp->s_id3FormatCfg);
}

/**
 * Album V2 line edit is changed.
 * @param contents of line edit
 */
void Id3Form::albumV2LineEditChanged(const QString& txt)
{
	formatLineEdit(m_albumV2LineEdit, txt, &theApp->s_id3FormatCfg);
}

/**
 * Artist V2 line edit is changed.
 * @param contents of line edit
 */
void Id3Form::artistV2LineEditChanged(const QString& txt)
{
	formatLineEdit(m_artistV2LineEdit, txt, &theApp->s_id3FormatCfg);
}

/**
 * Title V2 line edit is changed.
 * @param contents of line edit
 */
void Id3Form::titleV2LineEditChanged(const QString& txt)
{
	formatLineEdit(m_titleV2LineEdit, txt, &theApp->s_id3FormatCfg);
}

/**
 * Format string within line edit.
 *
 * @param le   line edit
 * @param txt  text in line edit
 * @param fcfg format configuration
 */
void Id3Form::formatLineEdit(QLineEdit* le, const QString& txt,
							 const FormatConfig* fcfg)
{
	if (fcfg->m_formatWhileEditing) {
		QString str(txt);
		fcfg->formatString(str);
		if (str != txt) {
			int curPos = le->cursorPosition();
			le->setText(str);
			le->setCursorPosition(curPos);
		}
	}
}

/**
 * Directory list box directory selected.
 *
 * @param item selected item
 */
void Id3Form::dirSelected(Q3ListBoxItem* item) {
	QDir dir(m_dirListBox->getDirname() + QDir::separator() +
					 item->text());
	m_dirListBox->setEntryToSelect(
		item->text() == ".." ? QDir(m_dirListBox->getDirname()).dirName() :
		QString::null);
	QString dirPath = dir.absPath();
	if (!dirPath.isEmpty()) {
		theApp->openDirectory(dirPath, true);
	}
}

/**
 * Get filter from ID3v1 check boxes.
 *
 * @return filter.
 */
StandardTagsFilter Id3Form::getFilterFromID3V1()
{
	StandardTagsFilter flt;
	flt.m_enableTitle   = m_titleV1CheckBox->isChecked();
	flt.m_enableArtist  = m_artistV1CheckBox->isChecked();
	flt.m_enableAlbum   = m_albumV1CheckBox->isChecked();
	flt.m_enableComment = m_commentV1CheckBox->isChecked();
	flt.m_enableYear    = m_yearV1CheckBox->isChecked();
	flt.m_enableTrack   = m_trackV1CheckBox->isChecked();
	flt.m_enableGenre   = m_genreV1CheckBox->isChecked();
	flt.allFalseToAllTrue();
	return flt;
}

/**
 * Get filter from ID3v2 check boxes.
 *
 * @return filter.
 */
StandardTagsFilter Id3Form::getFilterFromID3V2()
{
	StandardTagsFilter flt;
	flt.m_enableTitle   = m_titleV2CheckBox->isChecked();
	flt.m_enableArtist  = m_artistV2CheckBox->isChecked();
	flt.m_enableAlbum   = m_albumV2CheckBox->isChecked();
	flt.m_enableComment = m_commentV2CheckBox->isChecked();
	flt.m_enableYear    = m_yearV2CheckBox->isChecked();
	flt.m_enableTrack   = m_trackV2CheckBox->isChecked();
	flt.m_enableGenre   = m_genreV2CheckBox->isChecked();
	flt.allFalseToAllTrue();
	return flt;
}

/**
 * Enable or disable controls requiring ID3v1 tags.
 *
 * @param enable true to enable
 */
void Id3Form::enableControlsV1(bool enable)
{
	m_fnV1Button->setEnabled(enable);
	m_id3V2PushButton->setEnabled(enable);
	m_idV1GroupBox->setEnabled(enable);
}

/**
 * Display the format of tag 1.
 *
 * @param str string describing format, e.g. "ID3v1.1"
 */
void Id3Form::setTagFormatV1(const QString& str)
{
	QString txt = i18n("Tag &1");
	if (!str.isEmpty()) {
		txt += ": ";
		txt += str;
	}
	m_idV1GroupBox->setTitle(txt);
}

/**
 * Display the format of tag 2.
 *
 * @param str string describing format, e.g. "ID3v2.4"
 */
void Id3Form::setTagFormatV2(const QString& str)
{
	QString txt = i18n("Tag &2");
	if (!str.isEmpty()) {
		txt += ": ";
		txt += str;
	}
	m_idV2GroupBox->setTitle(txt);
}

/**
 * Adjust the size of the right half box.
 */
void Id3Form::adjustRightHalfBoxSize()
{
	m_rightHalfVBox->adjustSize();
}

/**
 * Hide or show tag 1 controls.
 *
 * @param hide true to hide, false to show
 */
void Id3Form::hideV1(bool hide)
{
	if (hide) {
		m_idV1GroupBox->hide();
	} else {
		m_idV1GroupBox->show();
	}
}

/**
 * Hide or show tag 2 controls.
 *
 * @param hide true to hide, false to show
 */
void Id3Form::hideV2(bool hide)
{
	if (hide) {
		m_idV2GroupBox->hide();
	} else {
		m_idV2GroupBox->show();
	}
}

/**
 * Set focus on filename controls.
 */
void Id3Form::setFocusFilename()
{
	m_nameLineEdit->setFocus();
}

/**
 * Set focus on tag 1 controls.
 */
void Id3Form::setFocusV1()
{
	m_titleV1LineEdit->setFocus();
}

/**
 * Set focus on tag 2 controls.
 */
void Id3Form::setFocusV2()
{
	m_titleV2LineEdit->setFocus();
}

/**
 * Set the custom genres configuration from the entries in the combo box.
 */
void Id3Form::customGenresComboBoxToConfig() const
{
	Kid3App::s_miscCfg.m_customGenres.clear();
	int idx, numGenres = m_genreV2ComboBox->count();
	for (idx = Kid3App::s_miscCfg.m_onlyCustomGenres ? 1 : Genres::count + 2;
			 idx < numGenres;
			 ++idx) {
		Kid3App::s_miscCfg.m_customGenres.append(m_genreV2ComboBox->text(idx));
	}
}

/**
 * Set the custom genres combo box from the configuration.
 */
void Id3Form::customGenresConfigToComboBox()
{
	QStringList strList;
	for (const char** sl = Genres::s_strList; *sl != 0; ++sl) {
		strList += *sl;
	}

	QString genreStr = m_genreV1CheckBox->isChecked() ?
		m_genreV1ComboBox->currentText() : QString::null;
	m_genreV1ComboBox->clear();
	if (Kid3App::s_miscCfg.m_onlyCustomGenres) {
		m_genreV1ComboBox->insertItem("");
	} else {
		m_genreV1ComboBox->QCM_addItems(strList);
	}
	for (QStringList::const_iterator it = Kid3App::s_miscCfg.m_customGenres.begin();
			 it != Kid3App::s_miscCfg.m_customGenres.end();
			 ++it) {
		if (Genres::getNumber(*it) != 255) {
			m_genreV1ComboBox->insertItem(*it);
		}
	}
	setGenreV1(genreStr);

	genreStr = m_genreV2CheckBox->isChecked() ?
		m_genreV2ComboBox->currentText() : QString::null;
	m_genreV2ComboBox->clear();
	if (Kid3App::s_miscCfg.m_onlyCustomGenres) {
		m_genreV2ComboBox->insertItem("");
	} else {
		m_genreV2ComboBox->QCM_addItems(strList);
	}
	m_genreV2ComboBox->insertStringList(Kid3App::s_miscCfg.m_customGenres);
	setGenreV2(genreStr);
}

/**
 * Save the local settings to the configuration.
 */
void Id3Form::saveConfig()
{
	Kid3App::s_miscCfg.m_splitterSizes = sizes();
	Kid3App::s_miscCfg.m_vSplitterSizes = m_vSplitter->sizes();
	Kid3App::s_miscCfg.m_formatItem = m_formatComboBox->currentItem();
	Kid3App::s_miscCfg.m_formatText = m_formatComboBox->currentText();
	customGenresComboBoxToConfig();
}

/**
 * Read the local settings from the configuration.
 */
void Id3Form::readConfig()
{
	if (!Kid3App::s_miscCfg.m_splitterSizes.empty()) {
		setSizes(Kid3App::s_miscCfg.m_splitterSizes);
	}
	if (!Kid3App::s_miscCfg.m_vSplitterSizes.empty()) {
		m_vSplitter->setSizes(Kid3App::s_miscCfg.m_vSplitterSizes);
	} else {
		// no values in configuration => set defaults
		// the window height is a bit too large, but works
		int sumSizes = size().height();
		if (sumSizes > 70) {
			Q3ValueList<int> sizes;
			sizes.append(sumSizes - 70);
			sizes.append(70);
			m_vSplitter->setSizes(sizes);
		}
	}
	m_formatComboBox->setCurrentItem(Kid3App::s_miscCfg.m_formatItem);
	m_formatComboBox->setCurrentText(Kid3App::s_miscCfg.m_formatText);
	customGenresConfigToComboBox();
}

/**
 * Init GUI.
 */
void Id3Form::initView()
{
	QStringList strList;
	m_formatComboBox->setEditable(true);
	for (const char** sl = MiscConfig::s_defaultFnFmtList; *sl != 0; ++sl) {
		strList += *sl;
	}
	m_formatComboBox->QCM_addItems(strList);

	FrameList::setListBox(m_framesListBox);
}

/**
 * Select all files.
 */
void Id3Form::selectAllFiles()
{
	m_fileListBox->selectAll(true);
}

/**
 * Select next file.
 */
void Id3Form::selectNextFile()
{
	m_fileListBox->selectNextFile();
}

/**
 * Select previous file.
 */
void Id3Form::selectPreviousFile()
{
	m_fileListBox->selectPreviousFile();
}

/**
 * Mark truncated ID3v1 fields.
 *
 * @param flags truncation flags
 * @see StandardTags::TruncationFlag.
 */
void Id3Form::markTruncatedFields(unsigned flags)
{
	if (flags & StandardTags::TF_Title) {
		m_titleV1LineEdit->setPaletteBackgroundColor(Qt::red);
	} else {
		m_titleV1LineEdit->unsetPalette();
	}
	if (flags & StandardTags::TF_Artist) {
		m_artistV1LineEdit->setPaletteBackgroundColor(Qt::red);
	} else {
		m_artistV1LineEdit->unsetPalette();
	}
	if (flags & StandardTags::TF_Album) {
		m_albumV1LineEdit->setPaletteBackgroundColor(Qt::red);
	} else {
		m_albumV1LineEdit->unsetPalette();
	}
	if (flags & StandardTags::TF_Comment) {
		m_commentV1LineEdit->setPaletteBackgroundColor(Qt::red);
	} else {
		m_commentV1LineEdit->unsetPalette();
	}
	if (flags & StandardTags::TF_Year) {
		m_yearV1SpinBox->setPaletteBackgroundColor(Qt::red);
	} else {
		m_yearV1SpinBox->unsetPalette();
	}
	if (flags & StandardTags::TF_Track) {
		m_trackV1SpinBox->setPaletteBackgroundColor(Qt::red);
	} else {
		m_trackV1SpinBox->unsetPalette();
	}
	if (flags & StandardTags::TF_Genre) {
		QPalette p = m_genreV1ComboBox->palette();
#ifdef WIN32
		p.setColor(QColorGroup::Base, Qt::red);
#else
		p.setColor(QColorGroup::Button, Qt::red);
#endif
		m_genreV1ComboBox->setPalette(p);
	} else {
		m_genreV1ComboBox->unsetPalette();
	}
}
