/**
 * \file fingerprintcalculator.cpp
 * Chromaprint fingerprint calculator.
 *
 * \b Project: Kid3
 * \author Urs Fleisch
 * \date 21 Jan 2012
 *
 * Copyright (C) 2012  Urs Fleisch
 *
 * This file is part of Kid3.
 *
 * Kid3 is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * Kid3 is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#define __STDC_CONSTANT_MACROS
#include "fingerprintcalculator.h"

#ifdef HAVE_CHROMAPRINT

#include <chromaprint.h>
#include "abstractfingerprintdecoder.h"

#ifdef HAVE_GSTREAMER
#include "gstfingerprintdecoder.h"
#elif defined HAVE_FFMPEG
#include "ffmpegfingerprintdecoder.h"
#elif QT_VERSION >= 0x050000
#include "qtfingerprintdecoder.h"
#endif

/**
 * Constructor.
 */
FingerprintCalculator::FingerprintCalculator(QObject* parent) : QObject(parent),
  m_chromaprintCtx(0),
  m_decoder(
#ifdef HAVE_GSTREAMER
    new GstFingerprintDecoder(this)
#elif defined HAVE_FFMPEG
    new FFmpegFingerprintDecoder(this)
#elif QT_VERSION >= 0x050000
    new QtFingerprintDecoder(this)
#endif
    )
{
  connect(m_decoder, SIGNAL(started(int,int)),
          this, SLOT(startChromaprint(int,int)));
  connect(m_decoder, SIGNAL(bufferReady(QByteArray)),
          this, SLOT(feedChromaprint(QByteArray)));
  connect(m_decoder, SIGNAL(error(int)),
          this, SLOT(receiveError(int)));
  connect(m_decoder, SIGNAL(finished(int)),
          this, SLOT(finishChromaprint(int)));
}

/**
 * Destructor.
 */
FingerprintCalculator::~FingerprintCalculator()
{
  if (m_chromaprintCtx) {
    ::chromaprint_free(m_chromaprintCtx);
  }
}

/**
 * Calculate audio fingerprint for audio file.
 * When the calculation is finished, finished() is emitted.
 *
 * @param fileName path to audio file
 */
void FingerprintCalculator::start(const QString& fileName) {
  if (!m_chromaprintCtx) {
    // Lazy intialization to save resources if not used
    m_chromaprintCtx = ::chromaprint_new(CHROMAPRINT_ALGORITHM_DEFAULT);
  }
  m_decoder->start(fileName);
}

/**
 * Stop decoder.
 */
void FingerprintCalculator::stop() {
  m_decoder->stop();
}

/**
 * Called when decoding starts.
 * @param sampleRate sample rate of the audio stream (in Hz)
 * @param channelCount numbers of channels in the audio stream (1 or 2)
 */
void FingerprintCalculator::startChromaprint(int sampleRate, int channelCount)
{
  ::chromaprint_start(m_chromaprintCtx, sampleRate, channelCount);
}

/**
 * Called when decoded data is available.
 * @param data 16-bit signed integers in native byte-order
 */
void FingerprintCalculator::feedChromaprint(QByteArray data)
{
  if (!::chromaprint_feed(m_chromaprintCtx, data.data(), data.size() / 2)) {
    m_decoder->stop();
    emit finished(QString(), 0, FingerprintCalculationFailed);
  }
}

/**
 * Called when an error occurs.
 * @param err error code, enum FingerprintCalculator::Error
 */
void FingerprintCalculator::receiveError(int err)
{
  emit finished(QString(), 0, err);
}

/**
 * Called when decoding finished successfully.
 * @param duration duration of stream in seconds
 */
void FingerprintCalculator::finishChromaprint(int duration)
{
  int err = Ok;
  QString fingerprint;
  char* fp;
  if (::chromaprint_finish(m_chromaprintCtx) &&
      ::chromaprint_get_fingerprint(m_chromaprintCtx, &fp)) {
    fingerprint = QString::fromLatin1(fp);
    ::chromaprint_dealloc(fp);
  } else {
    err = FingerprintCalculationFailed;
  }
  emit finished(fingerprint, duration, err);
}

#endif // HAVE_CHROMAPRINT
