/* $Id: triggers.c,v 1.37 2006/08/10 14:36:33 ekalin Exp $ */

/*
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston,
 * MA 02111-1307, USA.
 */

#ifdef HAVE_CONFIG_H
#  include <kcconfig.h>
#endif

#include <string.h>
#include <ctype.h>
#include <libintl.h>
#include <locale.h>
#include <gtk/gtk.h>
#include <glade/glade.h>

#include "kildclient.h"
#include "ansi.h"
#include "perlscript.h"


/***********************************************
 * Function prototypes - Test trigger function *
 ***********************************************/
static void       apply_highlight(World       *world,
                                  Trigger     *trigger,
                                  GtkTextIter *end_iter);
static GtkWidget *create_test_triggers_dialog(GtkWindow *parent, World *world);
static void       test_trigger_send_cb(GtkWidget *widget, gpointer data);
/* Glade signals */
G_MODULE_EXPORT void test_triggers_cb(GtkWidget *widget, gpointer data);



Trigger *
new_trigger(void)
{
  Trigger *trigger;

  trigger = g_new0(Trigger, 1);

  trigger->high_fg_color  = -1;
  trigger->high_bg_color  = -1;
  trigger->high_italic    = -1;
  trigger->high_strike    = -1;
  trigger->high_underline = -1;

  return trigger;
}


void
match_triggers(World     *world,
               guchar   **line,
               guchar   **colorline,
               gsize     *output_len,
               gboolean  *gag_output,
               gboolean  *gag_log,
               gint      *n_matches,
               GSList   **trigger_response)
{
  /* This function matches triggers against the line.

     If trigger_response is NULL, then it executes triggers, and also
     writes the line to the screen and/or log file if appropriate.

     If trigger_response points to a GSList, then this function does
     not do anything really, but stores the commands that would be
     executed in the list, for use in the Test Triggers function. */
  GSList      *trigptr;
  Trigger     *trigger;
  SV          *Pcolorline = NULL;
  int          i;
  gchar       *tmp;
  GSList      *highlights = NULL;
  gboolean     dont_flush = world->dont_flush;
  GtkTextIter  line_end;

  *n_matches  = 0;

  if (world->disable_triggers) {
    if (!trigger_response) {
      ansitextview_append_ansi_string(world->gui,
                                      (gchar *) *colorline, *output_len);
      if (world->log_file) {
        write_log_line(world, (gchar *) *line);
      }
    }

    return;
  }

  PERL_SET_CONTEXT(world->perl_interpreter);
  /*
   * First pass: rewriter triggers
   */
  trigptr = world->triggers;
  i = 0;
  while (trigptr) {
    trigger = (Trigger *) trigptr->data;
    if (trigger->rewriter
        && trigger->enabled
        && perl_match(world->perl_interpreter,
                      *line,
                      trigger->pattern_re,
                      NULL, NULL, 0)) {
      ++(*n_matches);
      if (!Pcolorline) {
        Pcolorline = get_sv("colorline", TRUE);
        sv_setpv(Pcolorline, (char *) *colorline);
      }

      if (debug_matches) {
        fprintf(stderr,
                "**Trigger %d '%s' (rewriter) matched for world %s\n"
                "  Line: %s"
                "  Pattern: %s\n"
                "  Action: %s\n",
                i,
                trigger->name ? trigger->name : "",
                world->name,
                *line,
                trigger->pattern,
                trigger->action);
      }

      parse_commands(world, trigger->action, strlen(trigger->action));

      g_free(*colorline);
      tmp = SvPV(Pcolorline, *output_len);
      *colorline = (guchar *) g_strdup(tmp);
      g_free(*line);
      *line = strip_ansi(*colorline, *output_len);
    }
    trigptr = trigptr->next;
    ++i;
  }

  /*
   * Second pass: see if one of the triggers is a gag
   */
  trigptr = world->triggers;
  while (trigptr) {
    trigger = (Trigger *) trigptr->data;
    if (!trigger->rewriter
        && trigger->enabled
        && perl_match(world->perl_interpreter,
                      *line,
                      trigger->pattern_re,
                      &trigger->match_start, &trigger->match_end,
                      trigger->high_target)) {
      *gag_output |= trigger->gag_output;
      *gag_log    |= trigger->gag_log;
      if (*gag_output && *gag_log) {
        break;
      }

      if (trigger->highlight) {
        highlights = g_slist_append(highlights, trigger);
      }

      if (!trigger->keepexecuting) {
        break;
      }
    }
    trigptr = trigptr->next;
  }

  if (!trigger_response) {
    if (!*gag_output) {
      GSList *cur_highlight;

      ansitextview_append_ansi_string(world->gui,
                                      (gchar *) *colorline, *output_len);
      gtk_text_buffer_get_iter_at_offset(world->gui->txtBuffer,
                                         &line_end,
                                         -1);

      /* Highlights */
      for (cur_highlight = highlights;
           cur_highlight;
           cur_highlight = cur_highlight->next) {
        apply_highlight(world, cur_highlight->data, &line_end);
      }
      g_slist_free(highlights);

      /* Move mark */
      gtk_text_buffer_move_mark(world->gui->txtBuffer,
                                world->gui->txtmark_linestart,
                                &line_end);
      /* Save color at end of line */
      world->gui->ta.saved_state = world->gui->ta.state;
    }
    if (world->log_file && !*gag_log) {
      write_log_line(world, (gchar *) *line);
    }
  }

  /*
   * Third pass: run normal triggers
   */
  world->dont_flush = TRUE;
  world_for_perl = world;
  trigptr = world->triggers;
  i = 0;
  while (trigptr) {
    trigger = (Trigger *) trigptr->data;
    if (!trigger->rewriter
        && trigger->enabled
        && perl_match(world->perl_interpreter,
                      *line,
                      trigger->pattern_re,
                      NULL, NULL, 0)) {
      ++(*n_matches);
      if (!Pcolorline) {
        Pcolorline = get_sv("colorline", TRUE);
        sv_setpv(Pcolorline, (char *) *colorline);
      }

      if (debug_matches) {
        fprintf(stderr,
                "**Trigger %d '%s' matched for world %s\n"
                "  Line: %s"
                "  Pattern: %s\n"
                "  Action: %s\n",
                i,
                trigger->name ? trigger->name : "",
                world->name,
                *line,
                trigger->pattern,
                trigger->action ? trigger->action : "");
      }

      if (trigger->action) {
        if (!trigger_response) {
          parse_commands(world, trigger->action, strlen(trigger->action));
        } else {
          *trigger_response = g_slist_append(*trigger_response,
                                             g_strdup(trigger->action));
        }
      }

      if (!trigger->keepexecuting) {
        break;
      }
    }
    trigptr = trigptr->next;
    ++i;
  }
  world_for_perl = currentWorld;
  world->dont_flush = dont_flush;
}


static
void
apply_highlight(World *world, Trigger *trigger, GtkTextIter *end_iter)
{
  GtkTextIter start;
  GtkTextIter end;
  int         i;

  start = *end_iter;
  gtk_text_iter_backward_line(&start);
  end = start;
  if (trigger->high_target != -1) {
    gtk_text_iter_set_line_offset(&start, trigger->match_start);
    gtk_text_iter_set_line_offset(&end,   trigger->match_end);
  } else {    /* The whole line */
    gtk_text_iter_forward_to_line_end(&end);
  }

  if (trigger->high_fg_color != -1) {
    /* Remove previous tags */
    for (i = 0; i < ANSI_TABLE_SIZE; ++i) {
      gtk_text_buffer_remove_tag(world->gui->txtBuffer,
                                 world->gui->ta.ansi_fore_tags[i],
                                 &start, &end);
    }
    gtk_text_buffer_apply_tag(world->gui->txtBuffer,
                              world->gui->ta.ansi_fore_tags[trigger->high_fg_color],
                              &start, &end);
  }

  if (trigger->high_bg_color != -1) {
    /* Remove previous tags */
    for (i = 0; i < ANSI_TABLE_SIZE; ++i) {
      gtk_text_buffer_remove_tag(world->gui->txtBuffer,
                                 world->gui->ta.ansi_back_tags[i],
                                 &start, &end);
    }
    gtk_text_buffer_apply_tag(world->gui->txtBuffer,
                              world->gui->ta.ansi_back_tags[trigger->high_bg_color],
                              &start, &end);
  }

  if (trigger->high_italic != -1) {
    gtk_text_buffer_remove_tag(world->gui->txtBuffer,
                               world->gui->ta.italics_tag,
                               &start, &end);
    if (trigger->high_italic) {
      gtk_text_buffer_apply_tag(world->gui->txtBuffer,
                                world->gui->ta.italics_tag,
                                &start, &end);
    }
  }

  if (trigger->high_strike != -1) {
    gtk_text_buffer_remove_tag(world->gui->txtBuffer,
                               world->gui->ta.strike_tag,
                               &start, &end);
    if (trigger->high_strike) {
      gtk_text_buffer_apply_tag(world->gui->txtBuffer,
                                world->gui->ta.strike_tag,
                                &start, &end);
    }
  }

  if (trigger->high_underline != -1) {
    gtk_text_buffer_remove_tag(world->gui->txtBuffer,
                               world->gui->ta.underline_tag,
                               &start, &end);
    gtk_text_buffer_remove_tag(world->gui->txtBuffer,
                               world->gui->ta.dblunderline_tag,
                               &start, &end);
    if (trigger->high_underline == 1) {
      gtk_text_buffer_apply_tag(world->gui->txtBuffer,
                                world->gui->ta.underline_tag,
                                &start, &end);
    }
    if (trigger->high_underline == 2) {
      gtk_text_buffer_apply_tag(world->gui->txtBuffer,
                                world->gui->ta.dblunderline_tag,
                                &start, &end);
    }
  }
}



void
remove_trigger(World *world, GSList *triggeritem)
{
  Trigger *trigger = (Trigger *) triggeritem->data;

  we_trigger_delete_trigger(world, trigger);

  world->triggers = g_slist_remove_link(world->triggers, triggeritem);
  if (!trigger->owner_plugin) {
    --world->trigger_pos;
  }
  free_trigger(trigger, NULL);
  g_slist_free(triggeritem);
}


gboolean
move_trigger(World *world, gint old_pos, gint new_pos)
{
  GSList   *triggeritem;
  gpointer  trigger;

  triggeritem = g_slist_nth(world->triggers, old_pos);
  if (!triggeritem) {
    return FALSE;
  }
  trigger = triggeritem->data;

  world->triggers = g_slist_delete_link(world->triggers, triggeritem);
  world->triggers = g_slist_insert(world->triggers, trigger, new_pos);

  we_trigger_delete_trigger(world, trigger);
  we_trigger_insert_trigger(world, trigger, new_pos);

  return TRUE;
}


void
free_trigger(Trigger *trigger, gpointer data)
{
  g_free(trigger->name);
  g_free(trigger->pattern);
  g_free(trigger->action);

  g_free(trigger);
}


void
list_triggers(World *world, Plugin *plugin)
{
  int      i;
  guint    total_width;
  guint    field_width;
  GSList  *trigptr;
  Trigger *trigger;

  /* The rows argument is not used. */
  ansitextview_get_size(world->gui, &field_width, &total_width);

  /* If the screen is really narrow, we can do nothing. */
  if (total_width < 49) {
    total_width = 49;
  }
  field_width = (total_width - 34)/2;

  ansitextview_append_stringf(world->gui,
                              _("Num Gag GLo Ena KeE ReW IgC Sty %-*.*s %-*.*s\n"),
                              field_width, field_width, _("Pattern"),
                              field_width, field_width, _("Action"));
  ansitextview_append_string(world->gui, "--- --- --- --- --- --- --- --- ");
  for (i = 0; i < field_width; ++i) {
    ansitextview_append_string(world->gui, "-");
  }
  ansitextview_append_string(world->gui, " ");
  for (i = 0; i < field_width; ++i) {
    ansitextview_append_string(world->gui, "-");
  }
  ansitextview_append_string(world->gui, "\n");

  i = 0;
  trigptr = world->triggers;
  while (trigptr) {
    trigger = (Trigger *) trigptr->data;

    if (trigger->owner_plugin == plugin) {
      ansitextview_append_stringf(world->gui,
                                  "%3d %-3.3s %-3.3s %-3.3s %-3.3s %-3.3s %-3.3s %-3.3s %-*.*s %-*.*s\n",
                                  i,
                                  trigger->gag_output    ? _("y") : _("n"),
                                  trigger->gag_log       ? _("y") : _("n"),
                                  trigger->enabled       ? _("y") : _("n"),
                                  trigger->keepexecuting ? _("y") : _("n"),
                                  trigger->rewriter      ? _("y") : _("n"),
                                  trigger->ignore_case   ? _("y") : _("n"),
                                  trigger->highlight     ? _("y") : _("n"),
                                  field_width, field_width,
                                  trigger->pattern,
                                  field_width, field_width,
                                  trigger->action ? trigger->action : "");
    }
    ++i;
    trigptr = trigptr->next;
  }
}


void
trigger_precompute_res(World *world)
{
  GSList  *triggerptr = world->triggers;
  Trigger *trigger;

  while (triggerptr) {
    trigger = (Trigger *) triggerptr->data;

    /* Plugins loaded from a script file already have the precomputed RE */
    if (!trigger->pattern_re) {
      trigger->pattern_re = precompute_re(world->perl_interpreter,
                                          trigger->pattern,
                                          trigger->ignore_case);
    }

    triggerptr = triggerptr->next;
  }
}


void
save_trigger(FILE *fp, Trigger *trigger)
{
  fprintf(fp, "    <trigger ");
  if (trigger->name) {
    fprintf_escaped(fp, "name=\"%s\" ", trigger->name);
  }
  fprintf(fp, "enabled=\"%d\" keepexecuting=\"%d\" rewriter=\"%d\" ignorecase=\"%d\"%s%s>\n",
          trigger->enabled,
          trigger->keepexecuting,
          trigger->rewriter,
          trigger->ignore_case,
          trigger->gag_output ? " gag=\"1\"" : "",
          trigger->gag_log    ? " gaglog=\"1\"": "");
  fprintf_escaped(fp, "      <pattern>%s</pattern>\n",
                  trigger->pattern);
  if (trigger->action) {
    fprintf_escaped(fp, "      <action>%s</action>\n",
                    trigger->action);
  }
  fprintf(fp, "      <highlight enabled=\"%d\" target=\"%d\" fg=\"%d\" bg=\"%d\" italics=\"%d\" strike=\"%d\" underline=\"%d\"/>\n",
          trigger->highlight, trigger->high_target,
          trigger->high_fg_color, trigger->high_bg_color,
          trigger->high_italic, trigger->high_strike, trigger->high_underline);
  fprintf(fp, "    </trigger>\n");
}


/*
 * Test trigger function.
 */
void
test_triggers_cb(GtkWidget *widget, gpointer data)
{
  if (currentWorld) {
    open_test_triggers_dialog(currentWorld);
  }
}


void
open_test_triggers_dialog(World *world)
{
  if (!world->dlgTestTriggers) {
    world->dlgTestTriggers =
      create_test_triggers_dialog(GTK_WINDOW(wndMain), world);
  }

  gtk_widget_show_all(world->dlgTestTriggers);
  gtk_window_present(GTK_WINDOW(world->dlgTestTriggers));
}


static
GtkWidget *
create_test_triggers_dialog(GtkWindow *parent, World *world)
{
  GladeXML         *gladexml;
  GtkWidget        *dlg;
  GtkWidget        *btnClose;
  GtkWidget        *btnSend;

  /* Create the dialog */
  gladexml = glade_xml_new(get_kildclient_installed_file("kildclient.glade"),
                           "dlgTestTriggers", NULL);
  dlg      = glade_xml_get_widget(gladexml, "dlgTestTriggers");

  /* Signals */
  glade_xml_signal_autoconnect(gladexml);
  btnClose = glade_xml_get_widget(gladexml, "btnClose");
  g_signal_connect_swapped(G_OBJECT(btnClose), "clicked",
                           G_CALLBACK(gtk_widget_hide), dlg);

  btnSend = glade_xml_get_widget(gladexml, "btnSend");
  g_signal_connect(G_OBJECT(btnSend), "clicked",
                   G_CALLBACK(test_trigger_send_cb), world);

  return dlg;
}


static
void
test_trigger_send_cb(GtkWidget *widget, gpointer data)
{
  World         *world = (World *) data;
  GladeXML      *gladexml;
  GtkWidget     *txtLine;
  GtkWidget     *lblMatches;
  GtkWidget     *txtCommands;
  GtkWidget     *txtOutput;
  GtkWidget     *lblGagLog;
  GtkTextBuffer *bufCommands;
  gboolean       disable_triggers;
  guchar        *line;
  guchar        *stripped;
  gint           n_matches;
  GSList        *trigger_response = NULL;
  gsize          output_len;
  gboolean       gag_output = FALSE;
  gboolean       gag_log = FALSE;
  gchar         *n_matches_str;
  GSList        *cmdptr;
  GtkTextIter    iter;

  gladexml    = glade_get_widget_tree(widget);
  txtLine     = glade_xml_get_widget(gladexml, "txtLine");
  lblMatches  = glade_xml_get_widget(gladexml, "lblMatches");
  txtOutput   = glade_xml_get_widget(gladexml, "txtOutput");
  txtCommands = glade_xml_get_widget(gladexml, "txtCommands");
  lblGagLog   = glade_xml_get_widget(gladexml, "lblGagLog");
  bufCommands = gtk_text_view_get_buffer(GTK_TEXT_VIEW(txtCommands));

  line     = (guchar *) g_strdup(gtk_entry_get_text(GTK_ENTRY(txtLine)));
  stripped = strip_ansi(line, strlen((gchar *) line));

  /* We need the triggers to be tried. */
  disable_triggers = world->disable_triggers;
  world->disable_triggers = FALSE;

  match_triggers(world, &stripped, &line,
                 &output_len, &gag_output, &gag_log,
                 &n_matches, &trigger_response);

  world->disable_triggers = disable_triggers;

  /* Show output */
  n_matches_str = g_strdup_printf("%d", n_matches);
  gtk_label_set_text(GTK_LABEL(lblMatches), n_matches_str);
  g_free(n_matches_str);

  gtk_text_buffer_set_text(bufCommands, "", 0);
  gtk_text_buffer_get_end_iter(bufCommands, &iter);
  cmdptr = trigger_response;
  while (cmdptr) {
    gtk_text_buffer_insert(bufCommands, &iter, (gchar *) cmdptr->data, -1);
    gtk_text_buffer_insert(bufCommands, &iter, "\n", 1);
    cmdptr = cmdptr->next;
  }
  g_slist_foreach(trigger_response, (GFunc) g_free, NULL);
  g_slist_free(trigger_response);

  if (!gag_output) {
    gtk_entry_set_text(GTK_ENTRY(txtOutput), (gchar *) line);
  } else {
    gtk_entry_set_text(GTK_ENTRY(txtOutput), "");
  }

  gtk_label_set_text(GTK_LABEL(lblGagLog),
                     gag_log ? _("No") : _("Yes"));

  g_free(line);
  g_free(stripped);
}
