#!/bin/sh
# -*- Tcl -*- \
exec tclsh "$0" -- ${1+"$@"}

#  Copyright Mission Critical Linux, 2000

#  Kimberlite is free software; you can redistribute it and/or modify it
#  under the terms of the GNU General Public License as published by the
#  Free Software Foundation; either version 2, or (at your option) any
#  later version.

#  Kimberlite is distributed in the hope that it will be useful, but
#  WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
#  General Public License for more details.

#  You should have received a copy of the GNU General Public License
#  along with Kimberlite; see the file COPYING.  If not, write to the
#  Free Software Foundation, Inc.,  675 Mass Ave, Cambridge,
#  MA 02139, USA.

# cluadmin.tcl
#
#   author: Ron Lawrence <lawrence@missioncriticallinux.com>
#   $Revision: 1.117 $
#

set version  {$Id: cluadmin.tcl,v 1.117 2000/11/27 20:14:31 lawrence Exp $}

if { ![info exists env(CLU_PATH)] } {
    set cluster_bin_path "/opt/cluster/bin"
} else {
    set cluster_bin_path "$env(CLU_PATH)"
}
if { ![info exists env(CLU_LIB_PATH)] } {
    set cluster_lib_path "/opt/cluster/bin"
} else {
    set cluster_lib_path "$env(CLU_LIB_PATH)"
}
set CLU_CONFIG_FILE "/etc/opt/cluster/cluster.conf"

set escape_to_tcl 0;                    # default: no escape
set interactive 1;                      # default: interactive
set one_shot_mode 0;                    # default: commandloop-age.
set stack_traces 0;                     # default: no stack traces.
set no_cluster_ok 0;                    # default: must use the cluster.
set no_getline 0;                       # default: history and completion.
set no_more 0;                            # default: break up long output.

# Cheezy debugging tools
set spew 0
proc pr { args } {
    global spew
    if { $spew } {
        puts stderr "$args"
    }
}
proc pv { prefix args } {
    global spew
    if { $spew } {
        set str "puts stderr \"$prefix:"
        foreach arg [lrange $args 0 [expr {[llength $args] - 2}]] {
            set str [format {%1$s%2$s = $%2$s,} $str $arg]
            if {[string length $str] - [string last \n $str ] > 80} {
                set str [format "%s\n" $str]
            } else {
                set str [format "%s " $str]
            }
        }
        set str [format {%1$s%2$s = $%2$s} $str [lindex $args end]]

        set str "$str\""
        uplevel $str
    }
}
# Additional list processing procs.
proc listinfo {kind pattern} {
    regsub -all { } [lsort [info $kind $pattern]] "\n" res
    puts $res
}
proc lreverse { ls } {
    set res {}
    foreach el $ls {
        set res [linsert $res 0 $el]
    }
    return $res
}
proc lrest { ls } {
    return [lrange $ls 1 end]
}
proc lrm { list n } {
    return [concat [lreplace $list $n end] [lreplace $list 0 $n]]
}
proc lremove { args } {
    set lx lindex
    if { [llength $args] == 3 } {
        %lremove [$lx $args 0] [$lx $args 1] [$lx $args 2]
    } elseif { [llength $args] == 2 } {
        %lremove -exact [$lx $args 0] [$lx $args 1]
    } else {
        error "wrong \# of args, should be lremove ?option? list pattern"
    }
}
proc %lremove { type list pattern } {
    set idx [lsearch $type $list $pattern]
    if { $idx != -1 } {
        return [lrm $list $idx]
    } else {
        return $list
    }
}
proc lmemberp { el list } {
    if { -1 == [lsearch -exact $list $el] } {
        return 0
    } else {
        return 1
    }
}
proc lassign { list args } {
    set variables $args
    foreach var $variables {
        set command [format "set %s \[list %s\]" $var [lindex $list 0]]
        uplevel $command
        set list [lrange $list 1 end]
    }
    if { [llength $list] > 0 } {
        uplevel lappend [lindex $variables end] $list
    }
}
proc lrmduplicates { lst } {
    # Remove duplicate items from the given list; preserve the order
    # of items that are retained.
    set res {} 
    foreach item $lst {
        if { ![lmemberp $item $res] } {
            lappend res $item
        }
    }
    return $res
}
proc ldivide { list predicate } {
    # Divide a list into the part before the predicate returns true,
    # and the part after the element that first causes the predicate
    # to return true
    set res {}
    set i 0
    foreach item $list {
        if { [eval $predicate $item] } {
            return [list $res [lrange $list $i end]]
        } else {
            lappend res $item
        }
        incr i
    }
    return [list $res {}]
}
proc is_minus_minus { item } {
    return [expr {![string compare "--" $item]}]
}
proc canonicalize_args { arglist } {
    # Grovel over the command line arguments looking for args of the
    # form: {^-[a-zA-Z]+}  When such an argument is found, split it up 
    # into its component arguments.
    set res {}
    set minus_minus_seen 0
    foreach arg $arglist {
        if { [regexp {^--$} $arg] } {
            set minus_minus_seen 1;     # Stop looking after --
        }
        if { [regexp {^-([a-zA-Z]+)$} $arg ignore letters] && !$minus_minus_seen } {
            set letters_list [split $letters {}]
            foreach letter $letters_list {
                lappend res "-$letter"
            }
        } else {
            lappend res $arg
        }
    }
    # Remove duplicates from the portion of the list before the first
    # occurance of "--".
    lassign [ldivide $res is_minus_minus] switches rest
    return [concat [lrmduplicates $switches] $rest]
}
proc pretty_print_array { array { width 42} } {
    upvar $array a
    foreach { k v } [array get a] {
        regsub -all "%" $k "." pk
        puts [format "%-${width}s = %s" $pk $v]
    }
}
proc all_procs {} {
    foreach ns [lsort [namespace children ::]] {
        listinfo commands ${ns}::*
    }
}
proc prompt { tag } {
    set res [prompter::getPrompt $tag]
    regsub -all {\[yesnoprompt\]} $res [yesnoprompt] res
    return $res
}
proc yesnoprompt {} {
    return "(yes/no/?)"
}

proc load_shared_cluster { } {
    # Load the shared library, if it hasn't already been loaded.  This
    # is only to be done if the cluadmin binary has not been
    # statically linked.  Since we are desperate to be independent of
    # any tcl library that might reside on the target machine at run
    # time, we never do that anymore.  Though it is maintained here
    # to permit easy debugging.
    global cluster_lib_path
    if { ! [info exists cluster_loaded] } {
        if { [catch {load "${cluster_lib_path}/cluster.so" cluster} result] } {
            puts stderr [prompt failedToLoad]
            #puts stderr "$result"
            exit 1
        }
        namespace eval cluster { namespace export CFG_* ptr* }
    }
}

# Cluster_Init has already been called as a part of starting up the
# tcl interpreter.

# All that remains to be done is to export the portions of the
# cluster namespace that we will be referring to below.
namespace eval cluster { namespace export CFG_* ptr* }

# Set up to ignore Ctrl-c, and ctrl-\.
cluster::ignore_signals

# Certain functions in certain namespaces are no longer called.  I
# think.  This proc will allow a proc to announce that it has been
# called when it shouldn't be.  If treat_deprecated_calls_as_errors is
# 1, then a call to a proc marked as deprecated causes an error to be
# thrown.  Note: Leave treat_deprecated_calls_as_errors set to 0 in
# the sources.  Change its value from the command line using the tcl
# escape hatch.  This will keep spurious error messages from appearing
# on the command line for years and years and years.
#
set treat_deprecated_calls_as_errors 0
proc deprecated { } {
    global treat_deprecated_calls_as_errors
    set fname [lindex [info level -1] 0]
    if { $treat_depcrecated_calls_as_errors } {
        error "called $fname, deprecated."
    } else {
        puts stderr "called $fname, deprecated."
    }
}

namespace eval cfg {
    namespace import -force ::cluster::*

    variable CFG_vals
    variable CFG_LOCK_vals

    # Create arrays from which to look up return codes
    if { [info exists CFG_vals] } {unset CFG_vals}
    if { [info exists CFG_LOCK_vals] } {unset CFG_LOCK_vals}

    set CFG_LOCK_vals(1)   CFG_LOCK_LOCKED_THIS_SESSION
    set CFG_LOCK_vals(2)   CFG_LOCK_LOCKED_OTHER_SESSION
    set CFG_LOCK_vals(4)   CFG_LOCK_NOT_LOCKED
    set CFG_LOCK_vals(8)   CFG_LOCK_LOST
    set CFG_LOCK_vals(16)  CFG_LOCK_EXPIRED
    set CFG_LOCK_vals(32)  CFG_LOCK_FAILURE
    set CFG_LOCK_vals(64)  CFG_LOCK_STATUS_UNKNOWN

    foreach v [info vars ::cluster::CFG_*] {
        if { ![string match "*LOCK*" $v] } {
            regsub {::cluster::} $v {} v1
            set CFG_vals([set [set v]]) $v1
        }
    }
    set CFG_vals(-99) CFG_ZERO_LENGTH_FILE
    proc Read  {} {
        variable CFG_vals
        return $CFG_vals([CFG_Read])
    }
    proc Write {} {
        variable CFG_vals
        return $CFG_vals([CFG_Write])
    }
    proc ReadFile {filename} {
        variable CFG_vals
        return $CFG_vals([CFG_ReadFile $filename])
    }
    proc WriteFile {filename} {
        variable CFG_vals
        return $CFG_vals([CFG_WriteFile $filename])
    }
    proc Initialized {} {
        variable CFG_vals
        return $CFG_vals([CFG_Initialized])
    }
    proc Loaded {} {
        variable CFG_vals
        return $CFG_vals([CFG_Loaded])
    }
    proc Destroy {} {
        variable CFG_vals
        return $CFG_vals([CFG_Destroy])
    }
    proc Backup {} {
        variable CFG_vals
        return $CFG_vals([CFG_Backup])
    }
    proc Restore {} {
        variable CFG_vals
        return $CFG_vals([CFG_Restore])
    }
    proc Get {key {defaulted {}}} {
        variable CFG_vals
        regsub -all {\.} $key {%} key
        set ff [ptrcreate "char *"]
        set r [CFG_Get $key $defaulted $ff]
        if { [string match "CFG_OK" $CFG_vals($r)] ||
             [string match "CFG_DEFAULT" $CFG_vals($r)] } {
            set res [ptrvalue $ff]
            return $res
        } else {
            error "in cfg::Get: $CFG_vals($r), key=$key"
        }
    }
    proc Put {key value} {
        variable CFG_vals
        regsub -all {\.} $key {%} key
        return $CFG_vals([CFG_Set $key $value])
    }
    proc Remove {key} {
        variable CFG_vals
        regsub -all {\.} $key {%} key
        return $CFG_vals([CFG_Remove $key])
    }
    proc GetMatch { pattern } {
        array set result_array {}
        Collect result_array $pattern
        return [array get result_array]
    }
    proc RemoveMatch {pattern} {
        variable CFG_vals
        regsub -all {\.} $pattern {%} key
        return $CFG_vals([CFG_RemoveMatch $pattern])
    }
    proc InsertEntryFromFile { key filename } {
        variable CFG_vals
        regsub -all {\.} $key {%} key
        return $CFG_vals([CFG_InsertEntryFromFile $key $filename])
    }
    proc ExtractEntryToFile { key filename } {
        variable CFG_vals
        regsub -all {\.} $key {%} key
        return $CFG_vals([CFG_ExtractEntryToFile $key $filename])
    }

    proc MakeLockResultList { n } {
        variable CFG_LOCK_vals
        variable CFG_vals
        set res {}
        for { set i 0 } { $i < 6 } { incr i } {
            set k [expr {1 << $i}]
            if { $k & $n } {
                lappend res $CFG_LOCK_vals($k)
            }
        }
        return $res
    }
    proc CheckLock {session} {
        variable CFG_LOCK_vals
        variable CFG_vals
        set res [CFG_CheckLock $session]
        return [MakeLockResultList $res]
    }
    proc Lock {session { forced 0 } } {
        variable CFG_LOCK_vals
        variable CFG_vals
        global CLU_CONFIG_FILE
        set res [CFG_Lock $session $forced]
        return [MakeLockResultList $res]
    }
    proc UnLock {session { forced 0 } } {
        variable CFG_LOCK_vals
        variable CFG_vals
        global CLU_CONFIG_FILE
        set res [CFG_Unlock $session $forced]
        return [MakeLockResultList $res]
    }
    proc LockString {} {
        return [CFG_LockString]
    }
    proc UserTimeString {timestring} {
        return [CFG_UserTimeString $timestring]
    }
    proc Collect {arrname pattern} {
        upvar $arrname a
        
        if { [info exists a] } {
            unset a
        }
        set it [MakeIterator -glob $pattern]
        set k {} ; set v {} ; set res {}
        while { [string match "CFG_OK" [IterMore $it]] } {
            set ok [IterNext $it k v]
            if { $ok && (! [string match "NULL" $k]) } {
                set res [concat $res [list $k $v]]
            }
        }
        array set a $res
    }
    proc UnCollect {arrname} {
        upvar $arrname a
        foreach { k v } [array get a] {
            Put $k $v
        }
    }
    proc MakeIterator {args} {
        set type none
        set newargs {}
        foreach arg $args {
            switch -glob -- $arg {
                -g* { set type glob }
                default { lappend newargs $arg }
            }
        }
        if { [string match $type "glob"] &&
             [llength $newargs] != 1 } {
            error "Wrong number of arguments to MakeIterator, $args"
        }
        if { [string match $type "glob"] } {
            set iter [ptrcast [ptrcreate "char"] "struct CFG_Iter **"]
            #puts stderr "newargs = $newargs"
            CFG_CreateGlobIterator $newargs $iter
            return [::cluster::cfg_deref_iter $iter]
        } else {
            set iter [ptrcast [ptrcreate "char"] "struct CFG_Iter **"]
            CFG_CreateIterator $iter
            return [::cluster::cfg_deref_iter $iter]
        }
    }
    proc IterNext {iter key val} {
        variable CFG_vals
        upvar $key k
        upvar $val v
        set kk [ptrcreate "char *"]
        set vv [ptrcreate "char *"]
        if { [string match "CFG_OK" $CFG_vals([CFG_IteratorNext $iter $kk $vv])] } {
            set k [ptrvalue $kk]
            set v [ptrvalue $vv]
            return 1
        } else {
            return 0
        }
    }
    proc IterMore {iter} {
        variable CFG_vals
        return $CFG_vals([CFG_IteratorMore $iter])
    }
}

namespace eval svc {
    namespace eval ::cluster {
        namespace export -clear *Svc* ptr* disableService enableService
    }
    namespace import -force ::cluster::*

    variable result_map_n2i
    array set result_map_n2i {
        FAIL                    -1
        SUCCESS                 0
        YES                     2
        NO                      3
        TIMEOUT                 4
        NOT_FOUND               5
    }
    variable result_map_i2n
    foreach {k v} [array get result_map_n2i] {
        set result_map_i2n($v) $k
    }

    proc dollarize_list { args } {
        set res {}
        foreach l $args {
            append res " \$$l"
        }
        return $res
    }
    proc defineGetter { name params {type {char *}}} {
        regsub {^get} $name {} name
        set cmd \
            [format \
                 {proc get%1$s \{ %2$s \} \{
                     set res \[ptrcreate \"%3$s\"\]
                     getSvc%1$s [dollarize_list %2$s] \$res
                     return \[ptrvalue \$res\]
                     \}\n} \
                 $name $params $type]
        #puts [concat [list [subst -novar $cmd]]]
        uplevel [subst -novar $cmd]
    }

    defineGetter getID svcName int
    defineGetter getName ID
    defineGetter getScript svcID
    defineGetter getRelocateOnPreferredNodeBoot svcID
    defineGetter getPreferredNodeName svcID
    defineGetter getPreferredNodeID svcID int
    defineGetter getDisabled svcID
    defineGetter getIPaddress { svcID N }
    defineGetter getDevice { svcID N }
    defineGetter getMountPoint { svcID N }
    defineGetter getMountFstype { svcID N }
    defineGetter getMountOptions { svcID N }
    defineGetter getForceUnmount { svcID N }
    defineGetter getDeviceOwner { svcID N }
    defineGetter getDeviceGroup { svcID N }
    defineGetter getDeviceMode { svcID N }
    defineGetter getMgrLogLevel { }
    defineGetter getNetmask { svcID N }
    defineGetter getBroadcast { svcID N }

    proc getDatabaseToken {  token  } {
        set res [ptrcreate "char *"]
        getSvcDatabaseToken  $token $res
        return [ptrvalue $res]
    }

    proc defineSetter { name params } {
        regsub {^set} $name {} name
        set cmd \
            [format \
                 {proc set%1$s \{ %2$s \} \{
                     set buffer \[binary format \"A1024\" \{\}\]
                     set res \[setSvc%1$s [dollarize_list %2$s] \$buffer \]
                     switch -- \$res \{
                     0 \{ return 1 \}
                     -1 \{
                     error \[trim_trailing_nulls \$buffer\]
                     \}
                     -2 \{
                     error \"Warning: \[trim_trailing_nulls \$buffer\]\"
                     \}
                     default \{
                     return 0
                     \}
                     \}
                     \}\n} \
                 $name $params]
        uplevel [subst -novar $cmd]
    }

    defineSetter setName { svcID svcName }
    defineSetter setScript { svcID svcScript }
    defineSetter setRelocateOnPreferredNodeBoot { svcID relocate }
    defineSetter setPreferredNodeName { svcID nodeName }
    defineSetter setPreferredNodeID { svcID nodeID }
    defineSetter setDisabled { svcID disabled }
    defineSetter setIPaddress { svcID N IPaddr }
    defineSetter setDevice { svcID N device }
    defineSetter setMountPoint { svcID N mountPoint }
    defineSetter setMountFstype { svcID N mountFstype }
    defineSetter setMountOptions { svcID N mountOptions }
    defineSetter setForceUnmount { svcID N forceUnmount }
    defineSetter setDeviceOwner { svcID N deviceOwner }
    defineSetter setDeviceGroup { svcID N deviceGroup }
    defineSetter setDeviceMode { svcID N deviceMode }
    defineSetter setNetmask { scvID N netMask }
    defineSetter setBroadcast { svcID N broadcast }

    proc defineChecker { name params } {
        regsub {^ck} $name {} name
        set cmd \
            [format \
                 {proc ck%1$s \{ %2$s \} \{
                     set buffer \[binary format \"A1024\" \{\}\]
                     set res \[ckSvc%1$s [dollarize_list %2$s] \$buffer \]
                     switch -- \$res \{
                     0 \{ return 1 \} 
                     -1 \{
                     error \[trim_trailing_nulls \$buffer\]
                     \}
                     -2 \{
                     error \"Warning: \[trim_trailing_nulls \$buffer\]\"
                     \}
                     default \{
                     return 0
                     \}
                     \}
                     \}\n} \
                 $name $params]
        uplevel [subst -novar $cmd]
    }

    defineChecker ckName svcName 
    defineChecker ckRelocateOnPreferredNodeBoot relocate 
    defineChecker ckPreferredNodeName nodeName 
    defineChecker ckPreferredNodeID nodeID 
    defineChecker ckDisabled disabled 
    defineChecker ckBroadcast Broadcast 
    defineChecker ckNetmask Netmask 
    defineChecker ckMountOptions mountOptions 
    defineChecker ckForceUnmount forceUnmount 
    defineChecker ckDeviceOwner deviceOwner 
    defineChecker ckDeviceGroup deviceGroup 
    defineChecker ckDeviceMode deviceMode 

    proc capitalize  { str } {
        return "[string toupper [string range $str 0 0]][string range $str 1 end]"
    }
    proc defineTokenRetriever { name n } {
        incr n ;# Need to include id parameter.
        set realName [format "%s%s%s" "getSvc" [capitalize $name] "TokenStr"]
        set params {}
        set name "${name}Token"
        for { set i 1 } { $i < $n } { incr i } {
            set params [format "%s n%d" $params $i]
        }
        set cmd \
            [format \
                 {proc %1$s \{  %2$s \} \{
                     set buffer \[binary format \"A1024\" \{\}\]
                     set res \[::cluster::%3$s [dollarize_list %2$s] \$buffer\]
                     switch -- \$res \{
                     0 \{ 
                     return \[trim_trailing_nulls \$buffer\]
                     \}
                     -1 \{
                     error \[trim_trailing_nulls \$buffer\]
                     \}
                     default \{ return 0 \}
                     \}
                     \}\n} \
                 $name $params $realName]
        uplevel [subst -novar $cmd]
    }

    defineTokenRetriever name 1
    defineTokenRetriever script 1
    defineTokenRetriever relocateOnPreferredNodeBoot 1
    defineTokenRetriever preferredNodeName 1
    defineTokenRetriever disabled 1
    defineTokenRetriever IPaddress 2
    defineTokenRetriever netmask 2
    defineTokenRetriever broadcast 2
    defineTokenRetriever device 2
    defineTokenRetriever mountPoint 2
    defineTokenRetriever mountFstype 2
    defineTokenRetriever mountOptions 2
    defineTokenRetriever forceUnmount 2
    defineTokenRetriever deviceOwner 2
    defineTokenRetriever deviceGroup 2
    defineTokenRetriever deviceMode 2

    # ckMountPoint is now hairy and complicated, so it doesn't fit in
    # with the rest of the checking routines.  This is because the
    # checkers are called in validation routines, where argument
    # passing is troublesome.
    proc ckMountPoint {  mountPoint } {
        variable result_map_i2n
        set buffer [binary format "A1024" {}]
        set res [::cluster::ckSvcMountPoint [set ::Service::V::serviceID] \
                     [set ::Service::V::deviceID] $mountPoint $buffer]
        switch -- $res {
            0 { return 1 }
            -1 {
                error [trim_trailing_nulls $buffer]
            }
            default { return 0 }
        }
    }

    # ckDevice is now hairy and complicated, so it doesn't fit in
    # with the rest of the checking routines.  This is because the
    # checkers are called in validation routines, where argument
    # passing is troublesome.
    proc ckDevice {  deviceName } {
        variable result_map_i2n
        set buffer [binary format "A1024" {}]
        set res [::cluster::ckSvcDevice [set ::Service::V::serviceID] \
                     [set ::Service::V::deviceID] $deviceName $buffer]
        switch -- $res {
            0 { return 1 }
            -1 {
                error [trim_trailing_nulls $buffer]
            }
            default { return 0 }
        }
    }
 
    proc ckMountFstype { mountFstype } {
        variable result_map_i2n
        
        if { [regexp -nocase "none" $mountFstype] } {
            return 1
        }
        set buffer [binary format "A1024" {}]
        set res [::cluster::ckSvcMountFstype $mountFstype $buffer]
        switch -- $res {
            0 { return 1 }
            -1 {
                error [trim_trailing_nulls $buffer]
            }
            default { return 0 }
        }        
    }
    proc ckScript { svcScript } {
        variable result_map_i2n
        
        if { [regexp -nocase "none" $svcScript] } {
            return 1
        }
        set buffer [binary format "A1024" {}]
        set res [::cluster::ckSvcScript $svcScript $buffer]
        switch -- $res {
            0 { return 1 }
            -1 {
                error [trim_trailing_nulls $buffer]
            }
            -2 {
                error "Warning: [trim_trailing_nulls $buffer]"
            }
            default { return 0 }
        }        
    }

    proc ckIPaddress {  ipaddress } {
        variable result_map_i2n
        set buffer [binary format "A1024" {}]
        set res [::cluster::ckSvcIPaddress [set ::Service::V::serviceID] \
                     [set ::Service::V::netID] $ipaddress $buffer]
        switch -- $res {
            0 { return 1 }
            -1 {
                error [trim_trailing_nulls $buffer]
            }
            default { return 0 }
        }
    }

    proc setDatabaseToken {  token value  } {
        return [setSvcDatabaseToken  $token $value]
    }
    proc enableService { ID } {
        variable result_map_i2n
#         if { [string match "SVC_ERROR" [svc::getServiceState $ID]] &&
#              [expr {[svc::getServiceOwner $ID] != [svc::getLocalNodeID]}] } {
#             error "Cannot enable service in error state unless owned by local node."
#         }
        set buffer [binary format "A1024" {}]
        set res [::cluster::reqEnableService $ID $buffer]
        switch -- $res {
            0 { return 1 }
            -1 {
                error [trim_trailing_nulls $buffer]
            }
            default { return 0 }
        }
    }
    proc disableService { ID } {
        variable result_map_i2n
#         if { [string match "SVC_ERROR" [getServiceState $ID]] } {
#             error "Cannot disable service in error state."
#         }
        set buffer [binary format "A1024" {}]
        set res [::cluster::reqDisableService $ID $buffer]
        switch -- $res {
            0 { return 1 }
            -1 { 
                error [trim_trailing_nulls $buffer]
            }
            default { return 0 }
        }
    }
    proc getNextFreeSvcID { } {
        return [::cluster::getNextFreeSvcID]
    }
    proc addService { ID } {
        variable result_map_i2n
        set buffer [binary format "A1024" {}]
        set res [::cluster::addService $ID $buffer ]
        if { [string match $res "-1"] } {
            puts stderr "[trim_trailing_nulls $buffer]"
            set res 0
        } else {
            set res 1
        }
        return $res
    }
    proc serviceExists { ID } {
        variable result_map_i2n
        set buffer [binary format "A1024" {}]
        set res [::cluster::serviceExists $ID]
        set res $result_map_i2n($res)
        pv "in serviceExists" res
        if { [string match $res "YES"] } {
            return "1"
        } elseif { [string match $res "NO"] } {
            return "0"
        } else {
            set res "Service exists returned error"
            error $res
        }
    }
    proc isServiceStateDisabled { ID } {
        variable result_map_i2n
        set buffer [binary format "A1024" {}]
        set res [::cluster::isServiceStateDisabled $ID $buffer ]
        set res $result_map_i2n($res)
        pv "in isServiceStateDisabled" res
        if { [string match $res "YES"] } {
            return "1"
        } elseif { [string match $res "NO"] } {
            return "0"
        } else {
            error [trim_trailing_nulls $buffer]
        }
    }
    proc deleteService { ID } {
        variable result_map_i2n
        set buffer [binary format "A1024" {}]
        set res [::cluster::deleteService $ID $buffer ]
        if { [string match $res "-1"] } {
            set res [trim_trailing_nulls $buffer]
            if { [string match "" $res] } {
                set res "0"
            }
        } else {
            set res 1
        }
        return $res
    }
    proc mapCodeToServiceState { s } {
        array set state_map {
            0 SVC_UNINITIALIZED
            1 SVC_STARTING
            2 SVC_RUNNING
            3 SVC_STOPPING
            4 SVC_STOPPED 
            5 SVC_DISABLING
            6 SVC_DISABLED
            7 SVC_ERROR
        }
        return $state_map($s)
    }
    proc getServiceState { ID } {
        set svc_state [::cluster::cluadmin_getServiceStatus $ID]
        if { $svc_state == -1 }  {
            error "getServiceState failed for service\# $ID."
        } else {
            return [mapCodeToServiceState $svc_state]
        }
    }
    proc getServiceOwner { ID } {
        set owner [::cluster::cluadmin_getServiceOwner $ID]
        if { -1 == $owner } {
            error "getServiceOwner failed for service\# $ID."
        } else {
            return $owner
        }
    }
    proc getLocalNodeID {} {
        return [::cluster::cluadmin_getLocalNodeID]
    }
    proc isServiceEnableable { ID } {
        set svc_state [::cluster::cluadmin_getServiceStatus $ID]
        if { $svc_state != -1 }  {
            if { 0 == [string compare [mapCodeToServiceState $svc_state] "SVC_RUNNING"] } {
                return 0
            } else {
                return 1
            }
        } else {
            return 0
        }
    }
    proc getNodeName { nodeID } {
        variable result_map_i2n
        set buffer [binary format "A1024" {}]
        set res [::cluster::cluadmin_getNodeName $nodeID $buffer]
        switch -- $res {
            0 { return [trim_trailing_nulls $buffer] }
            -1 {
                error [trim_trailing_nulls $buffer]
            }
            default { return 0 }
        }
    }
        
}

# Ensembles
#
# Create a collection of commands callable through a common base
# command.  Uses the interpreter to find the sub commands.
#
# This implementation is mostly satisfactory, though the error
# messages that are returned are not at all pretty.
#
namespace eval Ensemble { }

proc ensemble { name body } {
    set bodlines [split $body "\n"]
    set accum {}
    foreach line $bodlines {
        if { [regexp "^\[ \t\]*$" $line] ||
             [regexp "^\[ \t\]*\\#" $line] } {
            continue
        }
        if { [string match "" $accum] ||
             ! [info complete $accum] } {
            append accum "$line\n"
        }
        if { ![string match "" $accum] && [info complete $accum] } {
            set subname [format "Ensemble::%s.%s" $name [lindex $accum 1]]
            eval proc $subname [lrange $accum 2 end]
            set accum {}
        }
    }
    set cmd [format "%s" "proc \"::$name\" { subcommand args } {
            if { \[llength \[info commands ::Ensemble::$name.\$subcommand\]\] == 0} {
                error \"invalid command: $name \$subcommand\"
              }
              eval \"::Ensemble::$name.\$subcommand\" \$args
            }"]
    eval $cmd
}

namespace eval Service {
    variable names {}
    variable lineCounter 0
    proc checkServiceName { name } {
        return [regexp {^[A-Za-z_][A-Za-z0-9_]*$} $name]
    }

    proc chooseService { { prompt "Choose service: " }
                         { help "Choose a service by entering the number next to it." }
                         {filterCmd {identity 1}}} {
        set names [namelist $filterCmd]
        if { [llength $names] == 0 } {
          return -1
        }
        set numberednames {}
        set printedrep {}
        set i 0
        set names [split [string trim $names] "\n"]
        foreach name $names {
            if { ![string match {} $name] } {
                lappend numberednames $i 
                lappend numberednames [string trim $name]
                set printedrep [::format "%s  %d) %s\n" $printedrep $i [string trim $name]]
            }
            incr i
        }
        set printedrep [::format "%s  %s) %s\n" $printedrep "c" "cancel"]
        array set nmap $numberednames
        while { 1 } {
            puts $printedrep
            ::cluster::detach_completer
            set response [getline $prompt]
            ::cluster::attach_completer
            if { [regexp {^[0-9]+$} "$response"] } {
                if { [info exists nmap($response)] } {
                    return [string trim $nmap($response)]
                } else {
                    puts "Bad number, $response."
                }
            } elseif { [regexp {^\?$} $response] } {
                puts $help
            } elseif { [regexp {^c$} $response] } {
                error "NoError: canceled."
            } else {
                puts "Not a number, $response."
            }
        }
    }
    proc namelist {{filterCmd {identity 1}}} {
        # Return a vertically formatted list of names.
        set serviceNames [names $filterCmd]
        pv "in Service::namelist" serviceNames
        set serviceList "\n"
        foreach name $serviceNames {
            if { ![string match {} $name] } {
                set serviceList [::format "%s  %s\n" $serviceList $name]
            }
        }
        return $serviceList
    }
    proc names {{filterCmd {identity 1}}} {
        # list all of the defined services by name
        cfg::Destroy
        cfg::Read
        set res {}
        for {set i 0} {$i < $::cluster::MAX_SERVICES} {incr i} {
            regsub -all {%v} $filterCmd "$i" do_filterCmd
            # When looping through the services, don't call getName
            # unless the service exists. Wed Jul 12 17:26:41 2000
            if { [svc::serviceExists $i] && 
                 ![string match "$i" [set name [svc::getName $i]]] && 
                 [eval $do_filterCmd] } {
                lappend res $name
            }
        }
        return $res
    }
    proc delete { serviceID } {
        pv "in Service::delete" service serviceID
        cfg::RemoveMatch "services%service${serviceID}%*"
    }
    proc add { name args } {
        for {set i 0} {$i < $cluster::MAX_SERVICES} {incr i} {
            if { [svc::serviceExists $i] } {
                if { [string match [svc::getName $i] $name] } {
                    error "Service already exists"
                }
            }
        }
        cfg::Put services.service[svc::getNextFreeSvcID].name $name
    }
    proc disable { service args } {
        set id [svc::getID $service]
        if { ![svc::serviceExists $id] } {
            error "No such service: $service"
        }
        if { [svc::disableService $id] } {
            #puts "service $service disabled"
        } else {
            error "failed: service $service not disabled."
        }
    }
    proc enable { service args } {
        set id [svc::getID $service]
        if { ![svc::serviceExists $id] } {
            error "No such service: $service"
        }
        if { [svc::enableService $id] } {
            #puts "service $service enabled"
        } else {
            error "failed: service $service not enabled."
        }
    }
    proc relocate { service } {
        error "Service::relocate not implemented"
        puts "service $service relocated"
    }
    proc _lookup { field array } {
        pv "in _lookup, " field array
        upvar $array a
        if { [info exists a($field)] } {
            return $a($field)
        } else {
            return "NULL"
        }
    }
    proc _fetch { function args } {
        pv "in _fetch" function args
        return [uplevel svc::$function \$id $args]
    }
    proc _print_either_wnl { prefix lookup fetch } {
        if { 0 == [string compare $lookup "__cleared_out__"] } {
            # Don't print anything if the user cleared out the item.
            return {}
        }
        if { [notNULL $lookup] && ![string match {} $lookup]} {
            puts_with_more [::format "%s%s" ${prefix} [join ${lookup}]]
        } elseif { [notNULL $fetch] && ![string match {} $fetch]} {
            puts_with_more [::format "%s%s" ${prefix} [join ${fetch}]]
        } else {
            return {}
        }
    }
    proc _print_either_nonl { prefix lookup fetch } {
        if { 0 == [string compare $lookup "__cleared_out__"] } {
            # Don't print anything if the user cleared out the item.
            return {}
        }
        if { [notNULL $lookup] && ![string match {} $lookup]} {
            puts -nonewline [::format "%s%s" ${prefix} [join ${lookup}]]
        } elseif { [notNULL $fetch] && ![string match {} $fetch]} {
            puts -nonewline [::format "%s%s" ${prefix} [join ${fetch}]]
        } else {
            return {}
        }
    }
    proc _print_either { args } {
        if { [string match "-nonewline" [lindex $args 0]] } {
            eval _print_either_nonl [lrange $args 1 end]
        } else {
            eval _print_either_wnl $args
        }
    }
    proc _map_yes_or_no { yorn } {
        if { [regexp {^[Yy](es)?$} $yorn] } {
            return yes
        } elseif { [regexp {^[Nn](o)?$} $yorn] } {
            return no
        } else {
            return $yorn
        }
    }
    variable rows 25
    proc reset_line_counter {} {
        variable lineCounter
        variable rows
        set rows [lindex [get_term_dimensions] 0]
        if { $rows == -1 } {
            set rows 25
        }
        set lineCounter 0
    }
    proc incr_line_counter {{n 1}} {
        variable lineCounter
        incr lineCounter $n
    }
    # This has been changed to no longer use termcap, which is
    # "obsolete".  Instead we use the LINES and COLUMNS environment
    # variables.  These will work, provided that LINES and COLUMNS are
    # exported.  I'm not certain what will happen if a user is using a
    # lame shell, though.  At any rate, even if we can't figure out
    # what size the screen is, the output gets printed out.
    proc get_term_dimensions {} {
        global env
        if { [info exists env(LINES)] &&
             [info exists env(COLUMNS)] } {
            return [list $env(LINES) $env(COLUMNS)]
        } else {
            return {-1 -1}
        }
    }
    proc puts_with_more { args } {
        variable lineCounter
        variable rows
        global no_more
        if { ! $no_more } {
            incr lineCounter
            if { 0 == [expr {$lineCounter % $rows}] } {
                promptFor nothing {more: return, or q}
                if { [string match "q" $nothing] } {
                    error "quitted"
                }
            }
        }
        eval puts $args
    }

    proc _set_if { value id setter args } {
        if { [string compare "__cleared_out__" $value] &&
             ![string match {} $value] &&
             ![string match {NULL} $value] } {
            pr "uplevel svc::$setter $id $args $value"

            # Catch is needed here in case the setter throws a warning.
            if { [catch {uplevel svc::$setter $id $args $value} result] } {
                if { [string match "Warning*" $result] } {
                    puts $result
                }
            }
        }
    }
    proc _set_always { value id setter args } {
        # Force the value to be set into the database even if the
        # value is NULL, or, for example, an empty string.
        pr "uplevel svc::$setter $id $args [list $value]"
        if { [string compare "__cleared_out__" $value] } {
            # Unless the user has explicitely cleared the field out.
            # Catch is needed here in case the setter throws a warning.
            if { [catch {uplevel svc::$setter $id $args [list $value]} result] } {
                if { [string match "Warning:*" $result] } {
                    puts $result
                }
            }
        }
    }
    proc _set_if_not_none { value id setter args } {
        if { [string compare "__cleared_out__" $value] &&
             ![string match {} $value] &&
             ![string match {^ +$} $value] &&
             ![string match {NULL} $value] &&
             ![regexp {[Nn][Oo][Nn][Ee]} $value] &&
             ![string match "__not_set__" $value] } {
            pr "uplevel svc::$setter $id $args $value"
            # Catch is needed here in case the setter throws a warning.
            if { [catch {uplevel svc::$setter $id $args $value} result] } {
                if { [string match "Warning:*" $result] } {
                    puts $result
                }
            }
        }
    }

    proc _promptFor { id prompt_item res_item promptsArray resarray {default "__not_set__"}} {
        upvar $promptsArray prompts
        upvar $resarray results

        set el $prompts($prompt_item)
	regsub -all {\[yesnoprompt\]} $el [yesnoprompt] el
        set lx lindex
        if { ![info exists results($res_item)] } {
            set results($res_item) {}
        } else {
            set default $results($res_item)
        }
        if { [string match "__not_set__" $default] } {
            set default [compute_default [$lx $el 4]]
        }
        set response [promptWithNavigationFor results($res_item) \
                          [$lx $el 2] \
                          [$lx $el 3] \
                          [$lx $el 1] \
                          $default 9999]
        if { [string match "restart" $response] } {
            error "restart"
        }
    }
    # Remove all keys that match pattern from the given tcl array.
    proc _array_removematch { array pattern } {
        upvar $array a
        foreach {key} [array names a "$pattern"] {
            catch {unset a($key)}
        }
    }
    # How many of the characters at the start of str match the
    # corresponding characters of what.  If no characters match
    # returns zero.
    proc _starts_with { what str } {
        set pattern [string toupper $what]
        set string [string toupper $str]
        set i 0
        for { set i 0 } {$i < [string length $str]} {incr i} {
            if { [string index $pattern $i] == [string index $string $i] } {
                incr i
            } else {
                break
            }
        }
        return $i
    }
    proc _map_member_id_to_name { id_or_name } {
        if { [regexp {^[01]$} $id_or_name] } {
            array set idmap [Cluster::member_names_and_ids]
            return $idmap($id_or_name)
        } else {
            return $id_or_name
        }
    }
    proc _comma_separate_list { lst } {
        set res {}
        foreach item $lst {
            set res [::format "%s%s, " $res $item]
        }
        return $res
    }
    namespace eval V {
        variable serviceID -1
        variable deviceID  -1
        variable netID     -1
        variable exportID  -1
        variable clientID  -1
        variable mountPoint ""

        proc reset {} {
            variable serviceID
            variable deviceID
            variable netID
            variable exportID
            variable clientID
            variable mountPoint 
            set serviceID  -1
            set deviceID   -1
            set netID      -1
            set exportID   -1
            set clientID   -1
            set mountPoint ""
        }
    }
    ################################################################################

    # The purpose of the following procs is to manage the interaction
    # with the user when the user is being prompted for adding,
    # modifying, or deleting the various aspects of a service.

    # higher level prompting includes prompting for what, or whether a
    # given thing is to be done, and getting confirmation that the
    # whole service is done being edited.

    # The prompts are taken from a modlist variable in the prompter
    # namespace.  This list of prompts is a flat list, suitable for
    # reading into an array that has indices that correspond to the
    # indices in the cluadmin database when it is loaded into memory.

    # Each of the interactive procs that follow may throw "canceled"

    # Note that in many cases the strings which are used to look up
    # information in the resarray are absurdly long.  This is most
    # likely a bad sign, though of what I am not sure.  Use a large
    # monitor and a tiny font to edit, or view, what follows.

    # Nit.  In the following, lines like this:

    #        array set prompts [prompter::getModlist]

    # appear repeatedly.  These are a waste of time, though not yet
    # noticeably so from across the interface.  They should probably
    # be replaced with a single global like resarray, that gets passed
    # everywhere by name.

    proc _addservice { name resarrayin } {
        # The new service has already been created.  Just call modify
        # to get it modified.
        upvar $resarrayin resarray
        _modifyservice $name resarray
    }
    proc _modifyservice { name resarrayin } {
        global errorInfo
        upvar $resarrayin resarray
        array set prompts [prompter::getModlist]

        set id [svc::getID $name]
        set V::serviceID $id

        set done 0
        while { !$done } {
            if { [catch {
                # Hack together a help prompt that includes the member 
                # names, just in case the user asks for it.
                set prompts(preferredNode) [lreplace $prompts(preferredNode) 3 3 \
                                                [::format "%s\n%s %s%s" \
                                                     [lindex $prompts(preferredNode) 3] \
                                                     "Enter" \
                                                     [Cluster::member_names_and_ids_string] \
                                                     "or None."]]
                _promptFor $id "preferredNode" \
                    [svc::preferredNodeNameToken $id] prompts resarray
                # Don't prompt for relocate unless a preferred node
                # has been specified.
                if { (! [info exists resarray([svc::preferredNodeNameToken $id])]) || \
                     (! [_starts_with "none" $resarray([svc::preferredNodeNameToken $id])]) } {
                    _promptFor $id "relocateOnPreferredNodeBoot" \
                        [svc::relocateOnPreferredNodeBootToken $id] prompts resarray
                }
                _promptFor $id "userScript" \
                    [svc::scriptToken $id] prompts resarray

                _addmodifyordelete_network $id resarray
                _addmodifyordelete_device $id resarray

                _promptFor $id "disabled" \
                    [svc::disabledToken $id] prompts resarray
            } result] } {
                if { [string match $result "restart"] } {
                    continue
                } else {
                    Service::V::reset
                    set savedInfo $errorInfo
                    error "$result" $savedInfo
                }
            } else {
                set done 1
            }
        }
        Service::V::reset
        return {}
    }
    # Networks
    proc _addmodifyordelete_network { serviceID resarrayin } {
        upvar $resarrayin resarray
        set res ""
        while { 1 } {
            if { 0 == [_countnetworks $serviceID resarray] } {
                promptWithNavigationFor res \
                    "Do you want to add an IP address to the service [yesnoprompt]" \
                    "Enter yes to add an IP address." \
                    {expr [Service::_starts_with yes %v] || [Service::_starts_with no %v]} \
                    no
                if { [_starts_with {yes} $res] } {
                    puts "\n\tIP Address Information\n"
                    _addnetwork $serviceID resarray
                } else {
                    return finished
                }
            } else {
                #_shownetworks $serviceID resarray
                promptWithNavigationFor res "Do you want to (a)dd, (m)odify, (d)elete or (s)how an IP address, or are you (f)inished adding IP addresses" \
                    "Enter a to add an IP address, m to modify one, d to delete one, or f when finished"
                if { [_starts_with {add} $res] } {
                    _addnetwork $serviceID resarray
                } elseif { [_starts_with {modify} $res] } {
                    _modifynetwork $serviceID [_whichnetwork "modify" $serviceID resarray] resarray
                } elseif { [_starts_with {delete} $res] } {
                    _deletenetwork $serviceID [_whichnetwork "delete" $serviceID resarray] resarray
                } elseif { [_starts_with {show} $res] } {
                    _shownetworks $serviceID resarray
                } elseif { [_starts_with {finished} $res] } {
                    return finished
                }
            }
        }
    }
    proc _addnetwork { serviceID resarrayin } {
        upvar $resarrayin resarray
        array set prompts [prompter::getModlist]

        set netid [_nextnetworkID $serviceID resarray]
        _modifynetwork $serviceID $netid resarray
        return {}
    }
    proc _modifynetwork { serviceID networkID resarrayin } {
        upvar $resarrayin resarray
        array set prompts [prompter::getModlist]
        if { [string match "c" $networkID] } {
            return {}
        }
        set V::netID $networkID

        _promptFor $serviceID "network.ipAddress" \
            [svc::IPaddressToken $serviceID $networkID] prompts resarray
        _promptFor $serviceID "network.netmask" \
            [svc::netmaskToken $serviceID $networkID] prompts resarray
        _promptFor $serviceID "network.broadcast" \
            [svc::broadcastToken $serviceID $networkID] prompts resarray

        return {}
    }
    proc _whichnetwork { what serviceID resarrayin } {
        # Find out which network the user wants to modify or delete.
        upvar $resarrayin resarray
        set networks [lsort -integer [_listnetworkIDs $serviceID resarray]]
        set i 0
        set freshline 0
        foreach networkID $networks {
            incr i
            puts -nonewline [::format "%4d) %-20s" \
                                 $networkID \
                                 $resarray([svc::IPaddressToken $serviceID $networkID])]
            set freshline 0
            if { ! ($i % 3) } {
                puts ""
                set freshline 1
            }
        }
        incr i
        set freshline 0
        puts -nonewline "   c) cancel"
        if { ! ($i % 3) } {
            puts ""
            set freshline 1
        }
        if { ! $freshline } {
            puts ""
        }
        promptFor choice "Which of the above IP addresses do you want to ${what}? " \
            "Enter the number that precedes the IP address that you want to ${what}. Type c to cancel." \
            "lmemberp %v [list [lappend networks c]]" \
            "__not_set__" 9999
        return $choice
    }
    proc _shownetworks { serviceID resarrayin } {
        upvar $resarrayin resarray
        set networks [lsort -integer [_listnetworkIDs $serviceID resarray]]
        set i 0
        set freshline 0
        foreach networkID $networks {
            incr i
            puts -nonewline [::format "%4d) %-20s" \
                                 $networkID \
                                 $resarray([svc::IPaddressToken $serviceID $networkID])]
            set freshline 0
            if { ! ($i % 3) } {
                puts ""
                set freshline 1
            }
        }
        if { ! $freshline } {
            puts ""
        }
    }
    proc _deletenetwork { serviceID networkID resarrayin } {
        upvar $resarrayin resarray
        array set prompts [prompter::getModlist]

        if { [string match "c" $networkID] } {
            return {}
        }
        if { [confirm "Delete IP address $networkID [yesnoprompt]? " \
                 "Y to delete the IP address, N otherwise."] } {
            _array_removematch resarray "services%service${serviceID}%network${networkID}%*"
        }
    }

    # Devices
    proc _addmodifyordelete_device { serviceID resarrayin } {
        upvar $resarrayin resarray
        set res {}
        while { 1 } {
            if { 0 == [_countdevices $serviceID resarray] } {
                promptWithNavigationFor res \
                    "Do you want to add a disk device to the service [yesnoprompt]" \
                    "Enter yes to add a device." \
                    {expr [Service::_starts_with yes %v] || [Service::_starts_with no %v]} \
                    no
                if { [_starts_with {yes} $res] } {
                    puts "\n\tDisk Device Information\n"
                    _adddevice $serviceID resarray
                } else {
                    return finished
                }
            } else {
                #_showdevices $serviceID resarray
                promptWithNavigationFor res "Do you want to (a)dd, (m)odify, (d)elete or (s)how devices, or are you (f)inished adding devices" \
                    "Enter a to add a device, m to modify one, d to delete one, or f if finished."
                if { [_starts_with {add} $res] } {
                    _adddevice $serviceID resarray
                } elseif { [_starts_with {modify} $res] } {
                    _modifydevice $serviceID [_whichdevice "modify" $serviceID resarray] resarray
                } elseif { [_starts_with {delete} $res] } {
                    _deletedevice $serviceID [_whichdevice "delete" $serviceID resarray] resarray
                } elseif { [_starts_with {show} $res] } {
                    _showdevices $serviceID resarray
                } elseif { [_starts_with {finished} $res] } {
                    return finished
                }
            }
        }
    }
    proc _adddevice { serviceID resarrayin } {
        upvar $resarrayin resarray

        set deviceID [_nextdeviceID $serviceID resarray]
        _modifydevice $serviceID $deviceID resarray
    }
    proc _modifydevice { serviceID deviceID resarrayin } {
        upvar $resarrayin resarray
        array set prompts [prompter::getModlist]

        if { [string match "c" $deviceID] } {
            return {}
        }
        set V::deviceID $deviceID
        _promptFor $serviceID "device.name" \
            [svc::deviceToken $serviceID $deviceID] prompts resarray
        _promptFor $serviceID "device.mount_fstype" \
            [svc::mountFstypeToken $serviceID $deviceID] prompts resarray

        # The test here should be isNULL?
        if { ![regexp {^[Nn][Oo][Nn][Ee]$} $resarray([svc::mountFstypeToken $serviceID $deviceID])] } {
            # Prompt for the mount point.  Before prompting, save the
            # old mount point name, if any.  After the user specifies
            # the new mount point, compare with the old one.  If they
            # are different, do special magic.
            set oldMountpoint [compute_default [lindex $prompts(device.mount_point) 4]]
            _promptFor $serviceID "device.mount_point" \
                [svc::mountPointToken $serviceID $deviceID] prompts resarray

            set newMountpoint $resarray([svc::mountPointToken $serviceID $deviceID])
            if { 0 != [string compare $oldMountpoint $newMountpoint] } {
                if { ![string match "" $oldMountpoint] && \
                         ![regexp {[Nn][Oo][Nn][Ee]} $oldMountpoint] } {
                    _spewMountPointChangedMessage
                }
                set resarray($serviceID,$deviceID,oldMountpoint) $oldMountpoint
                set resarray($serviceID,$deviceID,newMountpoint) $newMountpoint
                set resarray([svc::mountPointToken $serviceID $deviceID]) $newMountpoint
            }
            set V::mountPoint $resarray([svc::mountPointToken $serviceID $deviceID])
            _promptFor $serviceID "device.mount_options" \
                [svc::mountOptionsToken $serviceID $deviceID] prompts resarray
            # Strip out all of the whitespace.
            regsub -all "\[ \t\]" $resarray([svc::mountOptionsToken $serviceID $deviceID]) \
                {} resarray([svc::mountOptionsToken $serviceID $deviceID])

            _promptFor $serviceID "device.force_unmount" \
                [svc::forceUnmountToken $serviceID $deviceID] prompts resarray
        }
        _promptFor $serviceID "device.device_owner" \
            [svc::deviceOwnerToken $serviceID $deviceID] prompts resarray
        _promptFor $serviceID "device.device_group"  \
            [svc::deviceGroupToken $serviceID $deviceID] prompts resarray
        _promptFor $serviceID "device.device_mode"   \
            [svc::deviceModeToken $serviceID $deviceID] prompts resarray
        _addmodifyordelete_export $serviceID $deviceID resarray
        
    }
    proc _whichdevice { what serviceID resarrayin } {
        # Find out which device the user wants to modify or delete.
        upvar $resarrayin resarray
        set devices [lsort -integer [_listdeviceIDs $serviceID resarray]]
        set i 0
        set freshline 0
        foreach deviceID $devices {
            incr i
            puts -nonewline [::format "%4d) %-20s" \
                                 $deviceID \
                                 $resarray([svc::deviceToken $serviceID $deviceID])]
            set freshline 0
            if { ! ($i % 3) } {
                puts ""
                set freshline 1
            }
        }
        set freshline 0
        puts -nonewline "   c) cancel"
        if { ! ($i % 3) } {
            puts ""
            set freshline 1
        }
        if { ! $freshline } {
            puts ""
        }        
        promptFor choice "Which of the above devices do you want to ${what}? " \
            [::format "%s%s" "Enter the number that precedes the pathname of the " \
                 "device you want to ${what}.  Type c to cancel."] \
            "lmemberp %v [list [lappend devices c]]" \
            "__not_set__" 9999
        return $choice
    }
    proc _showdevices { serviceID resarrayin } {
        upvar $resarrayin resarray
        set devices [lsort -integer [_listdeviceIDs $serviceID resarray]]
        set i 0
        set freshline 0
        foreach deviceID $devices {
            incr i
            puts -nonewline [::format "%4d) %-20s" \
                                 $deviceID \
                                 $resarray([svc::deviceToken $serviceID $deviceID])]
            set freshline 0
            if { ! ($i % 3) } {
                puts ""
                set freshline 1
            }
        }
        if { ! $freshline } {
            puts ""
        }        
    }        
    proc _deletedevice { serviceID deviceID resarrayin } {
        upvar $resarrayin resarray
        array set prompts [prompter::getModlist]

        if { [string match "c" $deviceID] } {
            return {}
        }
        set V::deviceID $deviceID
        if { [confirm "Delete device ${deviceID} [yesnoprompt]?" \
                 "Y to delete the device, N otherwise."] } {
            _array_removematch resarray "services%service${serviceID}%device${deviceID}%*"
        }
    }
    # Utility procs to support service add and modify.
    # Count various things.
    proc _countServices { resarrayin } {
        upvar $resarrayin resarray
        return [regsub -all  {services%service\[0-9\]+%name} \
                [array names resarray ] xx xx]
    }
    proc _countnetworks { serviceID resarrayin } {
        upvar $resarrayin resarray
        return [regsub -all  "services%service${serviceID}%network\[0-9]+%ipAddress" \
                [array names resarray] xx xx]
    }
    proc _countdevices { serviceID resarrayin } {
        upvar $resarrayin resarray
        return [regsub -all  \
                    "services%service${serviceID}%device\[0-9\]+%name" \
                    [array names resarray] xx xx]
    }
    # List various things.
    proc _listServiceIDs { resarrayin } {
        upvar $resarrayin resarray
        set res {}
        foreach name [array names resarray] {
            if { [regexp {services%service(\[0-9\]+)%name} $name m id] } {
                lappend res $id
            }
        }
        return $res
    }
    proc _listnetworkIDs { serviceID resarrayin } {
        upvar $resarrayin resarray
        set res {}
        foreach name [array names resarray] {
            pv names name
            if { [regexp "services%service${serviceID}%network(\[0-9]+)%ipAddress" $name m id] } {
                pv regexp m id
                lappend res $id
            }
        }
        return $res
    }
    proc _listdeviceIDs { serviceID resarrayin } {
        upvar $resarrayin resarray
        set res {}
        foreach name [array names resarray] {
            if { [regexp "services%service${serviceID}%device(\[0-9\]+)%name" $name m id] } {
                lappend res $id
            }
        }
        return $res
    }
    # Get the next id for various things.
    # Note that these may duplicate functions written in C.  If so,
    # the C functions should be called instead.
    # 200 is hardwired in to the following routines.
    proc _nextnetworkID { serviceID resarrayin } {
        upvar $resarrayin resarray
        set ids [lsort -integer [_listnetworkIDs $serviceID resarray]]
        for {set i 0} {$i < 200} {incr i} {
            if { -1 == [lsearch -exact $ids $i] } {
                pv returning i
                return $i
            }
        }
    }
    proc _nextdeviceID { serviceID resarrayin } {
        upvar $resarrayin resarray
        set ids [lsort -integer [_listdeviceIDs $serviceID resarray]]
        for {set i 0} {$i < 200} {incr i} {
            if { -1 == [lsearch -exact $ids $i] } {
                return $i
            }
        }
    }
    # Delete all of the currently defined services.
    proc deleteAllServices {} {
        for {set i 0} {$i < $cluster::MAX_SERVICES} {incr i} {
            if { [svc::serviceExists $i] } {
                if { ![svc::isServiceStateDisabled $i] } {
                    svc::disableService $i
                }
                svc::deleteService $i
            }
        }
    }
    # Create all of the services currently defined in the cluster
    # config database.  Use the id's that are defined in the database.
    # This proc is specifically for creating the services after a
    # cluster database has been restored, or loaded from a file.  It
    # should not be used otherwise.
    proc createAllServices {} {
        cfg::Collect svcs "services%service*%name"
        foreach name [array names svcs] {
            if { ![regexp "^services%service\[0-9\]+%name$" $name] } {
                unset svcs($name)
            }
        }
        foreach name [array names svcs] {
            regexp "^services%service(\[0-9\]+)%name$" $name ignore n
            svc::addService $n
        }
    }
    # Write the attributes of the service that are never quereid for
    # to the resarray so they won't be lost.

    # This has to be done because the information for the service is
    # erased before the newly collecte dinformation is inserted.  The
    # erasure is done to ensure that items that the user wanted
    # removed from the in-memory database actually go away.
    proc SaveSecretServiceAttributes { id resarrayin } {
        upvar $resarrayin resarray

        # This will work as long as the secret attributes of interest
        # are unique to the service.  Once there are secret attributes 
        # that are associated with specific sub-portions of the
        # service this will break down.
        set secretBits { 
	    services%%service%d%%NFS%%serverDir
	    services%%service%d%%NFS%%serverDev
	}

        foreach secretBit $secretBits {
            set attribute [cfg::Get [::format $secretBit $id] "__not__found__"]
            if { ![string match $attribute "__not__found__"] } {
                set resarray([::format $secretBit $id]) $attribute
            }
        }
        return {}
    }
    proc WriteSecretServiceAttributes { id resarrayin } {
        upvar $resarrayin resarray
        set secretBits { 
	    services%%service%d%%NFS%%serverDir 
	    services%%service%d%%NFS%%serverDev
	}

        foreach secretBit $secretBits {
            if { [info exists resarray([::format $secretBit $id])] } {
                set attribute $resarray([::format $secretBit $id])
                cfg::Put [::format $secretBit $id] $attribute
            }
        }
        return {}
    }
}
namespace eval Cluster {

    proc member_names {} {
        set result {}
        foreach {pat name} [cfg::GetMatch "members%member?%name"] {
            lappend result $name
        }
        return $result
    }
    proc member_names_and_ids {} {
        set result {}
        foreach {pat name} [cfg::GetMatch "members%member?%name"] {
            regexp {([01])[^01]*$} $pat ignore id
            lappend result $id $name
        }
        return $result
    }
    proc member_names_and_ids_string {} {
        set result {}
        foreach {pat name} [cfg::GetMatch "members%member?%name"] {
            regexp {([01])[^01]*$} $pat ignore id
            set result [format "%s$name (or $id), " $result]
        }
        return $result
    }

    proc reload {} {
        global CLU_CONFIG_FILE
        pr "in reload"
        if { [session::locked] } {
            # if we have the lock, write out the config database.
            cfg::Write
        } else {
            error "No session lock."
        }
        KillThemAll
    }
    proc KillThemAll {} {
        # Notify the quorum daemon that the database has changed.  Or, 
        # that the daemons need to be HUP'ed.

        ::cluster::cluConfigChangeNotification
    }
    variable name "cluster"
    variable num_nodes "2"
    variable nodes

    proc status { args } {
        global cluster_bin_path

        # Call clustat to get status information for the cluster.

        catch {exec $cluster_bin_path/clustat -v -i0} result
        puts $result
    }
    variable afterid
    variable monitoring 0
    variable clear 0
    variable interval 5

    proc monitor { args } {
        variable afterid
        variable monitoring
        variable clear
        variable interval
        if { -1 != [set clearoptn [lsearch -exact $args "-clear"]] } {
            if { [string match "yes" \
                      [lindex $args [expr {$clearoptn + 1}]]] } {
                set ::Cluster::clear 1
            } else {
                set ::Cluster::clear 0
            }
        }
        if { -1 != [set intervaloptionn \
                        [lsearch -exact $args "-interval"]] } {
            set interval [lindex $args [expr {$intervaloptionn + 1}]]
        }
        if { -1 != [lsearch -exact $args "stop"] } {
            set monitoring 0
            after cancel $afterid
            fconfigure stdin -blocking true
            fileevent stdin readable {}
            return {}
        } else {
            if { ! $monitoring } {
                fconfigure stdin -blocking false
            }
        }
        if { $Cluster::clear } {
            # This is only for vt100 and its relatives.
            puts stdout "\033c"
        }
        puts "\n"
        status
        puts "\n"
        fileevent stdin readable "::Cluster::monitor stop"
        set afterid [after [expr {$interval * 1000}] ::Cluster::monitor1]
        set monitoring 1
        vwait ::Cluster::monitoring
    }
    proc monitor1 { args } {
        variable interval
        variable afterid
        variable clear
        if { $Cluster::clear } {
            # This is only for vt100 and its relatives.
            puts stdout "\033c"
        }
        puts "\n"
        status
        puts "\n"
        set afterid [after [expr {$interval * 1000}] ::Cluster::monitor1]
    }
}
namespace eval session {
    variable have_lock 0
    variable session
    proc user_session_string {session_string} {
        set s [split $session_string "-"]
        set lx lindex
        set res [::format "\n  host: %s\n  application: %s\n  process id: %s\n  user: %s" \
                 [$lx $s 0] \
                 [$lx $s 1] \
                 [$lx $s 2] \
                 [$lx $s 3]]
        return $res
    }
    proc session_string {} {
        global env
        set res {}
        set host [exec hostname]
        set user $env(LOGNAME)
        set pid [pid]
        set appname cluadmin

        pv "in session_string" host appname pid user

        set res [format "%s-%s-%s-%s" $host $appname $pid $user]
        return $res
    }
    proc lock {{force 0}} {
        variable session
        variable have_lock
        # Attempt to get a session lock on the cluster database.  If
        # the database is already locked, allow the user to seize the
        # lock.  It the database is locked and the user does not want
        # to seize the lock, allow the user to proceed with out the
        # lock.  When the user proceeds without the lock, she must be
        # prompted to seize the lock the first time she tries to write
        # to the database.  Currently, though, if the user proceeds
        # without the lock, she is never afterward allowed to perform
        # write operations on the cluster database.  She is advised
        # that this will be the case in the help for the "proceed
        # without the lock prompt.

        set session [session_string]
        pv "in session::lock " session

        # Check the advisory lock status.
        if { [lmemberp "CFG_LOCK_NOT_LOCKED" \
                  [set res [cfg::CheckLock $session]]] } {
            pv "No session lock " res
            # There is no lock currently on the database.  Get one.
            if { [catch {cfg::Lock $session} result] } {
                pv "lock failed " result
                return 0
            }
            if { [lmemberp "CFG_LOCK_LOCKED_THIS_SESSION" $result] } {
                pv "locked" result
                set have_lock 1
                return 1
            } else {
                # Failed to get the lock for some reason.
                pv "lock result " result
                set have_lock 0
                return 0
            }
        } else {
            # What if this session already has the lock?
            if { $force } {
                if { [catch {cfg::Lock $session 1} result] } {
                    pv "failed to seize the lock " result
                    set have_lock 0
                    return 0
                } else {
                    pv "seized the lock" result
                    set have_lock 1
                    return 1
                }
            }
            pr "Lock locked. result = $res"
            set lock_data [cfg::LockString]
            puts "Session locked by:[user_session_string [lindex $lock_data 0]]"
            puts "Lock taken at: [cfg::UserTimeString [lindex $lock_data 1]]"
            if { [confirm "[prompt seizeTheLockHelp]\n[prompt seizeTheLock]" \
                      [prompt seizeTheLockHelp]]} {
                if { [catch {cfg::Lock $session 1} result] } {
                    pv "failed to seize the lock " result
                    set have_lock 0
                    return 0
                } else {
                    pv "seized the lock" result
                    set have_lock 1
                    return 1
                }
            } elseif { [confirm "[prompt proceedWithoutLockHelp]\n[prompt proceedWithoutLock]" \
                            [prompt proceedWithoutLockHelp]] } {
                set have_lock 0
                return 1
            } else {
                set have_lock 0
                return 0
            }
        }
    }
    # This is not called anywhere, but I am keeping it here for now.
    # Who knows how soon I'll need it.
    proc stealTheLock {} {
        variable session
        variable have_lock
        if { [confirm "[prompt seizeTheLockHelp]\n[prompt seizeTheLock]" \
                  [prompt seizeTheLockHelp]]} {
            if { [catch {cfg::Lock $session 1} result] } {
                pv "failed to seize the lock " result
                set have_lock 0
                return 0
            } else {
                pv "seized the lock" result
                set have_lock 1
                return 1
            }
        }
    }
    proc locked {} {
        global CLU_CONFIG_FILE
        variable session
        variable have_lock
        pr "called locked"
        if { [lmemberp "CFG_LOCK_LOCKED_THIS_SESSION" \
                  [set res [cfg::CheckLock $session]]] } {
            # Good, we have the lock...
            set have_lock 1
        } elseif {[lmemberp "CFG_LOCK_NOT_LOCKED" $res]} {
            # The lock is not there.  If no one has it, why can't I
            # take it?  Should I?
            puts "[prompt lostTheLock]"
            set have_lock 0
        } else {
            # Someone else has the lock.  We have to tell the user who
            # has it.
            set lock_data [cfg::LockString]
            puts "Session locked by:[user_session_string [lindex $lock_data 0]]"
            puts "Lock taken at: [cfg::UserTimeString [lindex $lock_data 1]]"
            set have_lock 0
        }
        return $have_lock
    }
    proc unlock {} {
        variable session
        variable have_lock
        set session [session_string]
        pv "in unlock " session
        if { [lmemberp "CFG_LOCK_LOCKED_THIS_SESSION" \
                  [set res [cfg::CheckLock $session]]] } {
            # Good, we have the lock...
            if { [catch {cfg::UnLock $session} result] } {
                pv "unlock failed " result
                return 0
            } else {
                pv "unlock succeeded " result
                set have_lock 0
                return 1
            }
        } elseif { [lmemberp "CFG_LOCK_NOT_LOCKED" $res] } {
            # The database file is not locked; pretend that that's
            # perfectly alright.
            pv "database not locked " res
            set have_lock 0
            return 1
        } else {
            if { $have_lock } {
                pr "lost the lock."
                return 0
            } else {
                pr "locked by another session"
                return 0
            }
        }
    }
}
# Prompt the user for a value, after displaying the prompt passed in
# as prompt.  Check the result with validateCommand, if one was passed
# in.  Keep trying until the validate accepted.

# validateCommand is called after replacing occurances of %V with the
# name of the variable being set, and with %v replaced with the value
# that was entered by the user.

proc promptWithNavigationFor { variable prompt
                 { helpText "No help for this prompt." }
                 { validateCommand "__not_set__" }
                 { default "__not_set__" }
                 { max_tries 9999} } {
    upvar $variable var
    set ok 0
    set tries 0
    set cleared 0
    pv "in promptWithNavigationFor" variable var prompt helpText validateCommand default max_tries
    while { ! $ok } {
        incr tries
        if { $tries > $max_tries } {
            error "$tries responses not accepted.  Try help command."
        }
        if { [string match {} $var] && ![string match "__not_set__" $default] } {
            set var $default
        }
        if { [string match "__not_set__" $default] } {
            set ppt "${prompt}: "
        } else {
            set ppt "$prompt \[${default}\]: "
        }
        ::cluster::detach_completer
        set v [getline $ppt]
        ::cluster::attach_completer
        # Accept a whitespace only string as input.  In this case, we
        # need to reprompt, but with the default string set to empty.
        if { [regexp "^\[ \t\]+$" $v] && ![string match "__not_set__" $default] } {
            set var {}
            set default ""
            set cleared 1
            continue
        }
        if { [string match {} $v] && ([notNULL $default] || $cleared)} {
            if { $cleared } {
                # The user has cleared the variable.
                set var "__cleared_out__"
                return {}
            } else {
                set var $default
                # Defaults aren't validated?  Can this be true?
                return $var
            }
        }
        if { [regexp {^\?} $v] } {
            puts [string trim $helpText]
            continue
        }
        if { [regexp {^(:[crp]?)$} $v match] } {
            # special operations
            if { 2 == [string length $match] } {
                if {[string match ":c" $match]} {
                    error "canceled"
                } elseif { [string match ":r" $match] } {
                    error "restart"
                } elseif { [string match ":p" $match] } {
                    continue
                } else {
                }
            }
            promptFor special [prompt navigationCommands] \
                [prompt navigationCommandsHelp]
            if { [regexp {^:?([crp])$} $special m char] } {
                if { [string match $char "c"] } {
                    error "canceled"
                } elseif { [string match $char "r"] } {
                    error "restart"
                } elseif { [string match $char "p"] } {
                    continue
                } else {
                }
            } else {
                continue
            }
        }
        if { ![string match "__not_set__" $validateCommand] } {
            regsub -all {%V} $validateCommand "[list $variable]" cmd
            regsub -all {%v} $cmd "[list $v]" cmd
            pv "in promptWithNavigationFor, validating, " cmd default
            if { [catch {
                set ok [eval $cmd]
            } result] } {
                if { [regexp {^Warning:} $result] } {
                    # Print the warning.
                    puts stderr "$result"
                    # Set the default to the last user input.
                    set default $v
                    # Prompt again.  N.B. defaults, when accepted, are 
                    # not validated. 
                    continue
                } else {
                    puts stderr "$result"
                    continue
                }
            }
            if { !$ok } {
                puts "Invalid input: $v"
            }
        } else {
            set ok 1
        }
        set var $v
    }
    return $var
}
proc promptFor { variable prompt
                 { helpText "No help for this prompt." }
                 { validateCommand "__not_set__" }
                 { default "__not_set__" }
                 { max_tries 9999} } {
    upvar $variable var
    set ok 0
    set tries 0
    while { ! $ok } {
        incr tries
        if { $tries > $max_tries } {
            error "$tries responses not accepted.  Try help command."
        }
        ::cluster::detach_completer
        set var [getline $prompt]
        ::cluster::attach_completer
        if { [string match {} $var] && ![string match "__not_set__" $default] } {
            set var $default
        }
        if { [regexp {^\?} $var] } {
            puts [string trim $helpText]
            continue
        }
        if { ![string match "__not_set__" $validateCommand] } {
            regsub -all {%V} $validateCommand "[list $variable]" cmd
            regsub -all {%v} $cmd "[list $var]" cmd
            pv "in promptFor, validating, " cmd default
            set ok [eval $cmd]
            if { ! $ok } {
                puts "Invalid input: $var"
            }
        } else {
            set ok 1
        }
    }
    return $var
}
proc confirm { { prompt "Are you sure? [yesnoprompt] " }
               { helptext {Type "Y" for yes, "N" for no, then return.}}
               { failtext "Too many tries, giving up.  Taken to mean no."}
               { max_tries 9999} } {
    set prompt [subst $prompt]
    
    ::cluster::detach_completer
    set response [getline $prompt]
    ::cluster::attach_completer

    set try 0
    while { $try < $max_tries } {
        if { [regexp {^[Yy]} $response] } {
            return 1
        } elseif { [regexp {^[Nn]} $response] } {
            return 0
        } elseif { [regexp {^\?} $response] } {
            puts $helptext
        }
        ::cluster::detach_completer
        set response [getline $prompt]
        ::cluster::attach_completer
        incr try
    }
    puts $failtext
    return 0
}
proc compute_default { default_command } {
    if { [string match "__not_set__" $default_command] } {
        return $default_command
    } else {
        set result [uplevel \#0 $default_command]
        if { ![string match NULL $result] &&
             ![string match {} $result] } {
            return $result
        } else {
            return "__not_set__"
        }
    }
}
proc identity { args } {
    return $args
}
proc any { args } {
    foreach arg $args {
        if { ![string match NULL $arg] && ![string match {} $arg] } {
            return $arg
        }
    }
}
namespace eval Help {
    proc help { args } {
        variable helpdata
        if { [llength $args] == 0 } {
            puts $helpdata(help)
            return {}
        }
        if { [llength $args] == 1 } {
            if { [catch {
                regsub -all {   *} $helpdata([lindex $args 0]) {  } out
                regsub -all "\n +" $out "\n" out
                puts $out
            } result ] } {
                puts stderr "Cannot find help for [lindex $args 0]"
            }
            return {}
        }
        if { [catch {
            regsub -all {   *} $helpdata([join $args "."]) {  } out
            regsub -all "\n +" $out "\n" out
            puts $out
        } result ] } {
            puts stderr "Cannot find help for: $args"
        }
    }
    proc apropos { args } {
        variable helpdata

        if { [llength $args] != 1 } {
           apropos *
           return {}
        }
        foreach name [array names helpdata] {
            set truename [split $name "."]
            foreach t $truename {
                if { [string match [lindex $args 0] $t] } {
                    lappend result $truename
                    break
                }
            }
        }
        foreach r [lsort $result] {
            puts $r
        }
    }
}
namespace eval prompter {
    variable modlist
    # Modlist is defined in cluadmin.help.tcl.
    variable verbose "0"
    variable depth "0"

    variable prompts
    proc getPrompt { id } {
        variable prompts
        regsub -all {\\} [string trim $prompts($id)] {} prompt
        return "$prompt "
    }
    proc getModlist {} {
        variable modlist
        return $modlist
    }
    proc indent { prompt } {
        set res {}
        set lines [split $prompt "\n"]
        foreach line $lines {
            set res [format "%s  %s\n" $res $line]
        }
        return $res
    }
}
proc puts_if { predicate string } {
    set command [format {if { [eval "%s"] } { \
                 puts [subst {%s}] }} $predicate $string]
    pv "in puts_if" command
    uplevel $command
}
proc notNULL { args } {
    if { [regexp "^\[ \t\]*$" [lindex $args 0]] } {
        return 0
    }
    return [expr {!([string match "NULL" $args] || [string match "__not_set__" $args])}]
}
######################################################################
# Ensembles to make dispatch easier.
#
ensemble service {
    proc show { args } {
        global cluster_bin_path CLU_CONFIG_FILE errorInfo
        set id "__not_set__"

        Service::reset_line_counter
        if { [string match "state" [lindex $args 0]] } {
            # We get state info from clustat, so we need to be on a
            # cluster system.
            if { 1 == [::cluster::cluster_member_check] } {
                catch {exec $cluster_bin_path/clustat -v -i0} result
                set lines [split $result "\n"]
                if { [llength $args] == 2 } {
                    set pattern [lindex $args 1]
                } else {
                    set pattern {.*}
                }
                set printing 0
                set count 0
                set service_info {}
                set preamble {}
                foreach line $lines {
                    # Print everything after and including the line that
                    # reads: Service status:
                    if { [regexp "^Service status:" $line] } {
                        set printing 1
                    }
                    if { $printing } {
                        if { $count < 4 } {
                            lappend preamble $line
                        } elseif { [regexp "^\[ \t]*${pattern}\[ \t]" $line] } {
                            lappend service_info  $line
                        }
                        incr count
                    }
                }
                if { [llength $service_info] == 0 } {
                    puts "No services available to show."
                    return {}
                } else {
                    foreach p $preamble {
                        puts $p
                    }
                    foreach s $service_info {
                        puts $s
                    }
                }
            } else {
                error "Not running on a cluster member"
            }
        } elseif { [string match "conf*" [lindex $args 0]] } {
            # Contained commands are "mored" and may throw "quitted"
            if { [catch {
                # Print configuration
                if { [string match "CFG_FALSE" [cfg::Initialized] ] } {
                    cfg::Read
                }
                # Need to check args. FIX ME
                if { [llength [lrest $args]] == 0 } {  
                    set service [Service::chooseService "Choose service: "]
                } elseif { [llength [lrest $args]] == 1 &&
                           [string match "all" [lindex $args 1]] } {
                    # Show all service configuration information.
                    set services [Service::names]
                    foreach s $services {
                        Service::incr_line_counter
                        puts {}
                        set id [svc::getID $s]
                        cfg::Collect xxx "services%service${id}%*"
                        Service::format $s xxx
                        unset xxx
                    }
                    return {}
                } else {
                    set service [lindex $args 1]
                }
                if { [string match "-1" $service] } {
                    puts "No services available to show"
                    return {}
                }
                set id [svc::getID $service]
                if { -1 == $id } {
                    error "Unknown service: $service"
                }
                # Print out the attributes of the service if they come
                # back non-null from the svc interface.
                #
                cfg::Collect xxx "services%service${id}%*"
                Service::format $service xxx
            } result] } {
                if { [string match {} $result] } {
                    return {}
                }
                # If quitted was thrown, suppress the message,
                # otherwise, rethrow the error.
                if { ![string match "quitted" $result] } {
                    set savedInfo $errorInfo
                    error $result $savedInfo
                }
            }
        } elseif { [regexp "services" [lindex $args 0]]} {
            cfg::Destroy
            cfg::Read
            puts ""
            set services {}
            for {set i 0} {$i < $::cluster::MAX_SERVICES} {incr i} {
                # When looping through the services, don't call
                # getName unless the service is known to exist.
                # Wed Jul 12 17:27:18 2000
                if { [svc::serviceExists $i] && \
                         ![string match "$i" [set name [svc::getName $i]]] } {
                    lappend services $name
                }
            }
            if { [llength $services] == 0 } {
                puts "No services available to show."
                return {}
            } else {
                foreach s $services {
                    puts $s
                }
            }
            puts ""
        } else {
            promptFor which [format "  %s\n  %s\n  %s\n%s" "1) state" \
                                 "2) config" "3) services" "service show what? "] \
                "Enter 1 for service show state\n\
                  2 for service show config,\n\
                  or 3 for shervice show services." \
                {regexp {^[231]$} %v} 
            switch "$which" {
                1 { service show state }
                2 { service show config }
                3 { service show services }
            }
        }
    }
    proc modify {  args } {
        global CLU_CONFIG_FILE errorInfo one_shot_mode
        if { $one_shot_mode } {
            puts stderr "No service modify in one-shot mode."
            exit 1
        }
        set serviceDisabledByModify 0
        Service::reset_line_counter
        if { 1 == [::cluster::cluster_member_check] } {
            if { ![session::locked] } {
                puts "Cannot modify services without session lock."
                return {}
            }
            puts "\n[prompter::indent [prompt serviceModifyPrompt1]]\n"

            if { 0 == [llength $args]} {
                set service [Service::chooseService "Choose service to modify: "]
            } else {
                set service [lindex $args 0]
            }
            if { [string match "-1" $service] } {
                puts "No services available to modify"
                return {}
            }
            set lx lindex
            puts "Modifying: $service"

            # Make sure the service exists before modifying it.
            if { -1 == [lsearch -exact [Service::names] $service] } {
                error "No such service: $service"
            }
            # Check that the service isn't in error state, or that, if
            # it is, that the user isn't trying to modify it on the
            # wrong node.
            if { [string match "SVC_ERROR" [svc::getServiceState [svc::getID $service]]] &&
                 [expr {[svc::getServiceOwner [svc::getID $service]] != [svc::getLocalNodeID]}] } {
                error [::format "%s\n%s\n%s%s, node %s" "Service $service is in error state." \
                           "A service in error state can only be modified on the node that owns it." \
                           "Service $service is owned by node\# " \
                           "[set nodeID [svc::getServiceOwner [svc::getID $service]]]" \
                           "[svc::getNodeName $nodeID]"]
            }
            # Make sure the service is disabled before proceeding.
            if { [catch {svc::isServiceStateDisabled [svc::getID $service]} isDisabled] } {
                puts "svc::isServiceStateDisabled failed, $isDisabled"
                return 0
            }
            if { ! $isDisabled && 
                 ![string match "SVC_ERROR" [svc::getServiceState [svc::getID $service]]] } {
                promptFor disableIt "Service is not disabled.  Disable it? [yesnoprompt] " \
                    "A service must be disabled before it can be modified."
                if { ![regexp {^[Yy]} $disableIt] } {
                    return "Service $service not modified."
                } else {
                    if { ![svc::disableService [svc::getID $service]] } {
                        error "Failed to disable service: $service"
                    } else {
                        set serviceDisabledByModify 1
                    }
                }
            }
            set modlist [prompter::getModlist]
            array set resarray [cfg::GetMatch "services%service[svc::getID $service]%*"]
            if { [catch {::Service::_modifyservice \
                             $service resarray} result] } {
                if { [string match "canceled" $result] } {
                    puts "service modify canceled."
                    if { $serviceDisabledByModify } {
                        if { [confirm [subst [prompt restartAfterModify]] \
                                  [prompt restartAfterModifyHelp]] } {
                            if { ![svc::enableService [svc::getID $service]] } {
                                error "Failed to enable service: $service"
                            }
                        }
                    }
                    return {}
                } else {
                    set savedInfo $errorInfo
                    error $result $savedInfo
                }
            }
            # XXX Call the aggregate validation command here, if there is
            # one.

            # The user may have moved mountpoints.  If so they need to
            # be remapped.
            
            foreach oldIndex [array names resarray "[svc::getID $service],*,oldMountpoint"] {
                regexp {([0-9]+),([0-9]+),oldMountpoint} $oldIndex ignore ss dd
                set oldMountpoint $resarray($oldIndex)
                if { [info exists resarray($ss,$dd,newMountpoint)] } {
                    set newMountpoint $resarray($ss,$dd,newMountpoint)
                    svc::renameExportDirs $ss $oldMountpoint $newMountpoint
                    foreach exportID [Service::_listexportIDs $ss $dd resarray] {
                        if { 0 != [string compare \
                                       [set dir \
                                            [cfg::Get \
                                                 [svc::exportDirToken \
                                                      $ss $dd $exportID] \
                                                 "__not_set__"]] \
                                       "__not_set__"] } {
                            set resarray([svc::exportDirToken $ss $dd $exportID]) \
                                $dir
                        }
                    }
                }
            }
            Service::format $service resarray
            if { [confirm [subst [prompt serviceModifyConfirm]] \
                      [prompt serviceModifyConfirmHelp ]]} {
                puts "Modified $service."
                if { [session::locked] } {
                    # Save the service id.
                    set id [svc::getID $service]
                    
                    # Save the service attributes that the user isn't
                    # aware of.
                    Service::SaveSecretServiceAttributes $id resarray

		    # Here we go again...
		    
		    # Sigh ... Save off a copy of the service, in case
		    # something--in particular, the assignNFSserverDir
		    # call--fails after this point.
		    
		    cfg::Collect savedService "services%service${id}%*"

                    # Remove any information about the service to be
                    # modified from the in memory database.

                    cfg::RemoveMatch "services%service${id}%*"

		    # This entry has to be in there, or else...
                    cfg::Put "services%service${id}%name" $service

                    # Now, write the service information into the
                    # in-memory database:

                    Service::write $service resarray 1 ;# 1 means modify.
                    
                    # Write saved secrets into the in-memory copy of
                    # the database
                    Service::WriteSecretServiceAttributes $id resarray

                    # The service may have been modified, and some of
                    # its elements may have been deleted.  If so, some
                    # of its contents may no longer have consecutive
                    # indices.  Compact the service in memory, before
                    # writing it to disk.
                    
                    cluster::compact_service $id

                    # Ensure that the NFS server directory is
                    # writeable, adjust if necessary.
		    if { [catch {svc::assignNFSserverDir $id}] } {
			# There was an error in the foregoing, so:
			# Remove the service entry just made:
			cfg::RemoveMatch "services%service${id}%*"

			# Then copy in the saved service from before
			# all of this crap was attempted:
			foreach name [array names savedService] {
			    cfg::Put $name $savedService($name)
			}
			# Whine piteously:
			error "Failed to assign an NFS server dir.\
                               Service modify aborted."
			# Even though the user will, most likely, have
			# no idea what that means.  The user's work in
			# modifying the service will, at this point,
			# have been thrown away.
		    } else {
			# otherwise, write the
			# database out:
			cfg::Write
		    }
                    if { ![regexp {^[Yy]} $resarray([svc::disabledToken [svc::getID $service]])] } {
                        # start up the service right away.
                        # Do I need to check if the service is in
                        # error state here?
                        if { ![svc::enableService [svc::getID $service]] } {
                            error "Failed to start service $service"
                        }
                    }
                } else {
                    error "No session lock."
                }
            } else {
                puts "$service not modified."
                if { $serviceDisabledByModify &&
                     ![string match "SVC_ERROR" [svc::getServiceState [svc::getID $service]]] } {
                    if { [confirm [subst [prompt restartAfterModify]] \
                              [prompt restartAfterModifyHelp]] } {
                        if { ![svc::enableService [svc::getID $service]] } {
                            error "Failed to enable service: $service"
                        }
                    }
                }
                if { [string match "SVC_ERROR" [svc::getServiceState [svc::getID $service]]] } {
                    puts [format "%s\n%s" "Service $service modified." \
                              "$service is in error state, use service enable to enable it."]
                }
            }
        } else {
            error "Not running on a cluster member"
        }
    }
    proc disable { args } {
        if { 1 == [::cluster::cluster_member_check] } {
            if { ![session::locked] } {
                puts "Cannot disable a service without session lock."
                return {}
            }
            if { 0 == [llength $args]} {
                set service [Service::chooseService "Choose service to disable: " \
                                 "Choose a service by entering the number next to it." \
                                 {expr ![svc::isServiceStateDisabled %v]}]
                if { [string match "-1" $service] } {
                    puts "No services available to disable."
                    return {}
                }
                if { [confirm] } {
                    puts -nonewline stderr "Disabling ${service}. "
                    Service::disable $service
                } else {
                    return {}
                }
            } elseif { 1 == [llength $args] } {
                set service [lindex $args 0]
                if { [confirm] } {
                    # We cd to / here because sometimes we might be
                    # running from a mounted filesystem, and, when the 
                    # service is disabled, the filesystem may be
                    # forcebly unmounted.  After that bad things of
                    # various kinds begin to happen.
                    cd /
                    puts -nonewline stderr "Disabling ${service}. "
                    Service::disable $service
                } else {
                    return {}
                }
            } else {
                error "Too many arguments to service disable"
            }
            set id [svc::getID $service]
            cfg::Put "[svc::disabledToken $id]" "yes"
            if { [session::locked] } {
                cfg::Write
                ::cluster::cluConfigChangeNotification
            } else {
                error "No session lock."
            }
            puts "Service [lindex $args 0] disabled."
        } else {
            error "Not running on a cluster member"
        }
    }
    proc enable { args } {
        if { 1 == [::cluster::cluster_member_check] } {
            if { ![session::locked] } {
                puts "Cannot enable a service without session lock."
                return {}
            }
            if { 0 == [llength $args]} {
                set service [Service::chooseService "Choose service to enable: " \
                                 "Choose a service by entering the number next to it." \
                                 {svc::isServiceEnableable %v}]
                if { [string match "-1" $service] } {
                    puts "No services available to enable."
                    return {}
                }
                if { [confirm] } {
                    Service::enable $service
                    puts -nonewline stderr "Enabling ${service}. "
                } else {
                    return {}
                }
            } elseif { 1 == [llength $args] } {
                set service [lindex $args 0]
                if { [confirm] } {
                    puts -nonewline stderr "Enabling ${service}. "
                    Service::enable $service
                } else {
                    return {}
                }
            } else {
                error "Too many arguments to service enable"
            }
            set id [svc::getID $service]
            cfg::Put "[svc::disabledToken $id]" "no"
            if { [session::locked] } {
                cfg::Write
                ::cluster::cluConfigChangeNotification
            } else {
                error "No session lock."
            }
            puts "Service [lindex $args 0] enabled."
        } else {
            error "Not running on a cluster member"
        }
    }
    proc add { args } {
        global CLU_CONFIG_FILE errorInfo one_shot_mode
        if { $one_shot_mode } {
            puts stderr "No service add in one-shot mode."
            exit 1
        }
        Service::reset_line_counter
        if { 1 == [::cluster::cluster_member_check] } {
            if { ![session::locked] } {
                puts "Cannot add a service without session lock."
                return {}
            }
            set modlist [prompter::getModlist]
            set modlist [lrange $modlist 1 end]

            puts "\n[prompter::indent [prompt serviceAddPrompt1]]\n"

            if { 0 == [llength $args]} {
                puts "Currently defined services: "
                puts "[Service::namelist]"
                set prompt "Service name: "
                ::cluster::detach_completer
                set service [getline $prompt]
                ::cluster::attach_completer
                if { [string match {} $service] } {
                    error "No service specified."
                }
                Service::add $service
            } elseif { 1 == [llength $args]} {
                set service [lindex $args 0]
                Service::add [lindex $args 0]
            } else {
                error "Too many arguments to service add"
            }
            # Check that the service name has the form of a c-style
            # identifier.
            if { ! [Service::checkServiceName $service] } {
                error "Invalid service name: $service"
            }
            # XXX Do a service modify $service, now?
            set lx lindex

            array set resarray {}
            if { [catch {::Service::_addservice \
                             $service resarray} result] } {
                if { [string match "canceled" $result] } {
                    cfg::RemoveMatch "services%service[svc::getID $service]%*"
                    puts "service add canceled."
                    return {}
                } else {
                    set savedInfo $errorInfo
                    error $result $savedInfo
                }
            }
            # XXX Call the aggregate validation command here, if there is
            # one.
            Service::format $service resarray
            if { [confirm [subst [prompt serviceAddConfirm]] \
                      [prompt serviceAddConfirmHelp]]} {
                if { [session::locked] } {
                    set id [svc::getID $service]

                    Service::write $service resarray 0 ;# 0 because
                    # this is not
                    # a modify.

                    # Ensure that the NFS server directory is
                    # writeable, adjust if necessary.
		    if { [catch {svc::assignNFSserverDir $id}] } {
			cfg::RemoveMatch "services%service$id%*"
			error "Failed to assign an NFS server dir.\
                               Service add aborted."
		    }
                    if { ![svc::addService [svc::getID $service]] } {
                        # Possibly prompt the user to modify the
                        # service?  We need to do something to avoid
                        # throwing out the user's input.

                        cfg::RemoveMatch "services%service[svc::getID $service]%*"
                        error "Failed to add service $service"
                    }
                    cfg::Write
                    if { ![regexp {^[Yy]} $resarray([svc::disabledToken [svc::getID $service]])] } {
                        # start up the service right away.
                        if { ![svc::enableService [svc::getID $service]] } {
                            error "Failed to start service $service"
                        }
                    }
                } else {
                    cfg::RemoveMatch "services%service[svc::getID $service]%*"
                    error "No session lock."
                }
                puts "Added $service."
            } else {
                cfg::RemoveMatch "services%service[svc::getID $service]%*"
                puts "$service not added."
            }
        } else {
            error "Not running on a cluster member"
        }
    }
    proc delete { args } {
        global CLU_CONFIG_FILE
        if { 1 == [::cluster::cluster_member_check] } {
            if { ![session::locked] } {
                puts "Cannot delete a service without session lock."
                return {}
            }
            if { 0 == [llength $args]} {
                set service [Service::chooseService "Choose service to delete: "]
                if { [string match "-1" $service] } {
                    puts "No services available to delete."
                    return {}
                }
            } elseif { [llength $args] > 1 } {
                error "Delete one service at a time."
            } else {
                set service [lindex $args 0]
            }
            # Make sure the service exists before modifying it.
            if { ![svc::serviceExists [svc::getID $service]] } {
                error "No such service: $service"
            }
            set i [svc::getID $service]
            if { [confirm "Deleting $service, are you sure? [yesnoprompt]: "]} {
                if { [catch {svc::isServiceStateDisabled \
                                 [svc::getID $service]} isDisabled] } {
                    puts "svc::isServiceStateDisabled failed, $isDisabled"
                    return 0
                }
                if { ! $isDisabled } {
                    if { ![svc::disableService $i] } {
                        puts stderr "Cannot disable service: $service"
                        puts stderr "Service not deleted."
                        return 0
                    } else {
                        puts "Service $service disabled"
                    }
                }
                if { ![svc::deleteService $i] } {
                    puts stderr "svc::deleteService failed for $service"
                    return 0
                }
                Service::delete $i
                if { [session::locked] } {
                    cfg::Write
                } else {
                    error "No session lock."
                }
                puts "$service deleted."
            }
            return {}

        } else {
            error "Not running on a cluster member"
        }
    }
}
array set loglevels2n {
    EMERG    0
    ALERT    1
    CRIT     2
    ERR      3
    WARNING  4
    NOTICE   5
    INFO     6
    DEBUG    7
}
foreach {k v} [array get loglevels2n] {
    set n2loglevels($v) $k
}
ensemble cluster {

    proc reload {} {
        # Arrange to get HUPS sent to all of the daemons on both
        # cluster members.
        Cluster::reload

        # How do we make the hup go out to the daemons on the other
        # node?
    }
    proc loglevel { {daemon "__not_set__" } { severity "__not_set__" }} {
        global CLU_CONFIG_FILE
        global loglevels2n
        global n2loglevels
        set old_severity "defaulted"
        # Read the database every time.  It may have been changed by
        # another process.
        ::cfg::Read
        if { [string match "__not_set__" $daemon] } {
            set daemons {quorumd heartbeat svcmgr powerd} 
        } else {
            set daemons $daemon
        }
        foreach d $daemons {
            if { [string compare $severity "__not_set__"] != 0 } {
                if { [regexp -nocase {^[A-Za-z]+$} $severity] } {
                    if { [info exists loglevels2n([string toupper $severity])] } {
                        set severity $loglevels2n([string toupper $severity])
                    } else {
                        error "bad severity level: \"$severity\", \n\
                           must be one of [array names loglevels2n]"
                    }
                } elseif { ![regexp {^[0-9]+$} $severity] } {
                    error "Severity $severity must be a log level string or number"
                } elseif { $severity < 0 || $severity > 7 } {
                    error "Severity $severity must be between 0 and 7 inclusive."
                }
                switch -exact -- "$d" {
                    powerd {
                        set old_severity [::cfg::Get powerd.logLevel defaulted]
                        ::cfg::Put powerd.logLevel $severity
                    }
                    quorumd {
                        set old_severity [::cfg::Get quorumd.logLevel defaulted]
                        ::cfg::Put quorum.logLevel $severity
                    }
                    svcmgr {
                        set old_severity [::cfg::Get svcmgr.logLevel defaulted]
                        ::cfg::Put svcmgr.logLevel $severity
                        # Any time the service manager's loglevel
                        # changes, change cluadmin's loglevel.
                        cluster::clu_set_loglevel $severity
                    }
                    heartbeat {
                        set old_severity [::cfg::Get powerd.logLevel defaulted]
                        ::cfg::Put heartbeat.logLevel $severity
                    }
                    default {
                        error "Bad argument: \"$d\", must be heartbeat,\
                               powerd, quorumd, or svcmgr"
                    }
                }
            } else {
                switch -exact -- "$d" {
                    powerd {
                        puts "powerd: [::cfg::Get powerd.logLevel]"
                    }
                    quorumd {
                        puts "quorumd: [::cfg::Get quorum.logLevel]"
                    }
                    svcmgr {
                        puts "svcmgr: [::cfg::Get svcmgr.logLevel]"
                    }
                    heartbeat {
                        puts "heartbeat: [::cfg::Get heartbeat.logLevel]"
                    }
                    default {
                        error "Bad argument: \"$d\", must be heartbeat,\
                               powerd, quorumd, or svcmgr"
                    }
                }
            }
        }
        if { [string compare "__not_set__" $daemon] != 0 ||
             ([string compare "__not_set__" $severity] != 0 && 
              [string compare $old_severity $severity] != 0) } {
            # Only kick the daemons if arguments were passed to this
            # function, and the severity argument is different than
            # the one already stored in the database.
            if { [catch {Cluster::reload} result] } {
                puts $result
            }
        }
    }
    proc name { args } {
        # Return the name of the cluster, modify the name if new is
        # not equal to __not_set__
        cfg::Destroy
        cfg::Read
        set name [set oldname [cfg::Get "cluster.name"]]
        if { [llength $args] != 0 } {
            set name [eval concat $args]
            if { [string compare "$oldname" "$name"] != 0 } {
                cfg::Put "cluster.name" [eval concat $args]
                if { [catch {Cluster::reload} result] } {
                    puts $result
                    return
                }
            }
        }
        puts $name
        return $oldname
    }
    proc status { } {
        ::Cluster::status
    }
    proc monitor { args } {
        # Call through to the cluster monitor command.
        eval ::Cluster::monitor $args
    }
    # Procs to save, restore, and edit the cluster database:
    proc backup { } {
        global CLU_CONFIG_FILE
        if { [string match "CFG_FALSE" [cfg::Loaded]] } {
            cfg::Read
        }
        cfg::Backup
    }
    proc restore { } {
        global CLU_CONFIG_FILE
        if { 1 != [::cluster::cluster_member_check] } {
            error "Not running on a cluster member"
        }
        if { ! [file exists "${CLU_CONFIG_FILE}.bak"] ||
             ! [file readable "${CLU_CONFIG_FILE}.bak"] } {
            error "Cannot open ${CLU_CONFIG_FILE}.bak"
        }
        if { [session::unlock] } {
            if { [cluster::noNonDisabledServicesOtherNode] ||
                 [cluster::allOtherNodesDown] } {
                Service::deleteAllServices

                if { [string match "CFG_OK" [cfg::Loaded]] } {
                    cfg::Destroy
                }
                cfg::Restore
                # Hold yer nose...
                cfg::Write
                session::unlock
                session::lock
                pr "about to call cfg::ReadFile"
                cfg::Read
                pr "about to call Cluster::reload"
                Service::createAllServices
                Cluster::reload
            } else {
                error "Cannot restore while other nodes are not down,\n\
                       or any services are not disabled on another node."
            }
        } else {
            # being conservative...
            error "failed to release the session lock"
        }
    }
    proc saveas { filename } {
        global CLU_CONFIG_FILE
        cfg::Destroy
        if { ![string match "CFG_OK" [cfg::Loaded]] } {
            cfg::Read
        }
        cfg::WriteFile $filename
    }
    proc restorefrom { filename { force 0 } } {
        global CLU_CONFIG_FILE
        pv "in restorefrom" filename
        if { 1 != [::cluster::cluster_member_check] } {
            error "Not running on a cluster member"
        }
        if { [session::unlock] } {
            if { [cluster::noNonDisabledServicesOtherNode] ||
                 [cluster::allOtherNodesDown] ||
                 $force } {
                Service::deleteAllServices

                if { [string match CFG_OK [cfg::Loaded]] } {
                    cfg::Destroy
                }
                if { [file exists $filename] &&
                     [file readable $filename] } {
                    pv "2nd" filename
                    pr "cfg::ReadFile $filename"
                    cfg::ReadFile $filename
                } else {
                    error "Can't read $filename"
                }
                # Hold yer nose...
                cfg::Write
                session::unlock
                session::lock
                pr "about to call cfg::ReadFile"
                cfg::Read
                pr "about to call Cluster::reload"
                Service::createAllServices
                Cluster::reload
            } else {
                error "Cannot restorefrom $filename\n while other nodes are not down,\n\
                       or any services are not disabled on another node."
            }
        } else {
            # being conservative...
            error "failed to release the session lock"
        }
    }
    proc edit { { filename "__not_set__" } } {
        global CLU_CONFIG_FILE
        global env one_shot_mode

        if { $one_shot_mode } {
            puts stderr "No cluster edit in one-shot mode."
            exit 1
        }

        # Originally, we grabbed the EDITOR environment variable from
        # the global env variable.  This variable, however, contains
        # the environment as it appeared at the time the program was
        # started.  Instead, we now make a call to libc's getenv,
        # which should get the value of the environment as it exists
        # at the time of the call.  I think.  The call to getenv is
        # wrapped in make-i-file, and the wrapper provided through
        # swig.

        if { [info exists env(EDITOR)] } {
            set editor $env(EDITOR)
        } else {
            set editor "vi"
        }
        if { [string match "__not_set__" $filename] ||
             [string match "$CLU_CONFIG_FILE" $filename] } {
            if { !([cluster::noNonDisabledServicesOtherNode]) && \
                     !([cluster::allOtherNodesDown]) } {
                error "Cannot edit cluster database while other nodes are up,\n\
                       or any services are not disabled on another node."
            }
            if { ![session::locked] } {
                puts "Cannot edit cluster database without session lock."
                return {}
            }
            if { [confirm [prompt editDiscouragement]] } {
                set filename "/tmp/cluster.[pid]"
                cluster saveas $filename
                set mustsave 1
            } else {
                puts "Edit cancelled."
                return {}
            }
        } else {
            set mustsave 0
        }
        # Whether this works will depend on whether X is running, and
        # whether editor gets set to some sensible value for this
        # environment.  Also whether the script is running from an
        # emacs process buffer.  Also, undoubtedly, there are
        # dependencies on a myriad of other variables.  This is almost
        # certainly a bad idea, but if it works once, it will be cool.
        if { [string match "vi*" $editor]  } {
            exec $editor $filename < /dev/tty > /dev/tty 2> /dev/tty
        } elseif { [string match "gnuclient" $editor] } {
            catch {exec $editor $filename}
        } elseif { [string match $editor {}] } {
            # The default behavior used to be that an xterm would try
            # to run the program specified in the EDITOR environment
            # variable. This won't work in general because a user may
            # have her DISPLAY variable unset, or, even if it is set,
            # may have xhosts - in effect.  So, instead of trying to
            # launch xterm with no editor, we will carp and give up.
            # puts "EDITOR environment variable not set."  return {}
            catch {exec vi $filename < /dev/tty > /dev/tty 2> /dev/tty}
        } else {
            # wing it...
            catch {exec xterm -e $editor $filename}
        }
        pv "mustsave? " mustsave filename
        if { [session::locked] } {
            if { $mustsave } {
                if { [cluster::noNonDisabledServicesOtherNode] ||
                     [cluster::allOtherNodesDown] } {
                    cfg::Destroy
                    cluster restorefrom $filename 1 ;# 1 means force.
                    if { [file writable $CLU_CONFIG_FILE] } {
                        cfg::Write
                    } else {
                        file delete /tmp/cluster.[pid]
                        error "can't write $CLU_CONFIG_FILE"
                    }
                }
            }
        } else {
            error "No session lock."
        }
    }
}
# For externalized help functions and data.
source "cluadmin.help.tcl"
source "proprietary.tcl"

proc help { args } {
    eval Help::help $args
}

proc apropos { args } {
    eval Help::apropos $args
}

proc clear { args } {
    global env
    if { ![string match "*emacs*" $env(TERM)] } {
        puts stdout "\033c"
    }
}

set commands {
    cluster     service        clear
    help        apropos        exit
}

# Spew a pretty error message if the luser types the wrong crap.  The
# ensemble command doesn't print a sufficiently respectible error
# message.  These messages are much, much better.
proc checkCommand { command arguments } {
    if { [string match "cluster" $command] } {
        if { 0 == [llength $arguments] } {
            error "missing subcommand, should be
                   cluster status, monitor, loglevel, reload, name, edit, 
                   backup, restore, saveas, or restorefrom."
        }
        if { ![lmemberp [lindex $arguments 0] \
                   {status monitor loglevel reload name edit backup restore saveas restorefrom}] } {
            error "bad subcommand: [lindex $arguments 0], should be
                   cluster status, monitor, loglevel, reload, name, edit, 
                   backup, restore, saveas, or restorefrom."
         }
    }
    if { [string match "service" $command] } {
        if { 0 == [llength $arguments] } {
            error "missing subcommand, should be
                   service add, show, modify, disable, enable, or delete"
        }
        if { ![lmemberp [lindex $arguments 0] \
                   {add show modify disable enable delete}] } {
            error "bad subcommand: [lindex $arguments 0], should be
                   service add, show, modify, disable, enable, or delete"
        }        
    }
    return 1
}
proc trim_trailing_nulls { string } {
    global tcl_version

    if { [string match "8.0" $tcl_version] } {
        # tcl8.0 thinks \000 is the string terminating null.  So, in
        # this case, use a piece of c code to chop off trailing
        # garbage.
        return [cluster::minus_trailing $string 1024]
    } else {
        regsub "( ?)\[\000 \t\]+$" $string {\1} m1
        return $m1
    }        
}
proc getline { prompt } {
    set buffer [binary format "A1024" {}]
    cluster::getaline $prompt $buffer 1024
    return [trim_trailing_nulls $buffer]
}
proc commandloop {} {
    set quit 0
    global errorInfo stack_traces
    global command_map
    global commands
    global escape_to_tcl
    puts [string trim [subst [prompt initial_prompt]]]
    while { ! $quit } {
        update
        if { [catch {
            set line [string trim [getline [prompt base_prompt]]]
            if { ! [regexp "^\[ \t\]*$" $line] } {
                regsub -all { +} $line { } line
                set elements [split $line " \t"]
                set command [lindex $elements 0]
                if { ([lsearch -exact $commands $command] != -1) ||
                     [string match $command "tcl"] } {
                    # do the command
                    if {[string match $command "exit"]} {
                        eval pv [list "in commandloop"] [info locals]
                        session::unlock
                        cluster::save_history
                        exit
                    } elseif { [string match $command "tcl"] &&
                               $escape_to_tcl } {
                        puts [uplevel eval [lrange $elements 1 end]]
                    } else {
                        if { [checkCommand $command [lrange $elements 1 end]] } {
                            uplevel eval $command [lrange $elements 1 end]
                        }
                    }
                } else {
                    error "unrecognised command: $command"
                }
            }
        } result] } {
            if { [string match "NoError*" $result] } {
                regsub {^NoError:} $result {} result
                puts stderr "$result"
            } else {
                puts stderr "Error: $result"
                if { $stack_traces } {
                    puts stderr "$errorInfo"
                }
            }
        }
    }
}
proc usage {} {
    global argv0
    set ustr [format "
Usage: %s \[OPTIONS\] \[COMMAND\]
cluadmin - Administer cluster.

  -h,-?, --help              Give short usage message.
  -n,    --nointeractive     Don't interact.
  -d,    --debug             Spew copious diagnostic info.
  -s,    --stack-trace       Show tcl stack traces.
  -t,    --tcl               Allow an escape to tcl.
  -V,    --version           Print version information.
         --no-cluster        Run off-cluster for development.
         --no-getline        No completion and no history.
         --no-more           Don't break up long output.
  --,                        No more options, everything else is a command\n

If arguments are left over on the command line, interpret the arguments
as a command, then exit.
If no arguments are left after OPTIONS, start an interactive interpreter.
" [file tail $argv0]]
puts [string trim $ustr]
}

# Main

# Parse command line arguments:
set argv [canonicalize_args $argv]
set nonopts {}
if { -1 != [lsearch -exact $argv {--help}] ||
     -1 != [lsearch -exact $argv {-h}] ||
     -1 != [lsearch -exact $argv {-?}] } {
    usage
    exit 0
}
if { -1 != [set where [lsearch -exact $argv {-V}]]} {
    puts "[file tail $argv0] $version"
    exit 0
}
if { -1 != [set where [lsearch -exact $argv {--version}]]} {
    puts "[file tail $argv0] $version"
    exit 0
}
if { -1 != [set where [lsearch -exact $argv {--}]] } {
    set nonopts [lrange $argv [expr {$where + 1}] end]
    set argv [lrange $argv 0 [expr {$where - 1}]]
}
if { -1 != [set where [lsearch -exact $argv {-t}]]} {
    set escape_to_tcl 1
    set argv [lrm $argv $where]
}
if { -1 != [set where [lsearch -exact $argv {--tcl}]]} {
    set escape_to_tcl 1
    set argv [lrm $argv $where]
}
if { -1 != [set where [lsearch -glob $argv {-d}]] } {
    set spew 1
    set argv [lrm $argv $where]
}
if { -1 != [set where [lsearch -glob $argv {--debug}]] } {
    set spew 1
    set argv [lrm $argv $where]
}
if { -1 != [set where [lsearch -glob $argv {-s}]] } {
    set stack_traces 1
    set argv [lrm $argv $where]
}
if { -1 != [set where [lsearch -glob $argv {--stack-trace}]] } {
    set stack_traces 1
    set argv [lrm $argv $where]
}
if { -1 != [lsearch -exact $argv {-n}] } {
    set interactive 0
    proc confirm {args} {
        return 1
    }
}
if { -1 != [lsearch -exact $argv {--nointeractive}] } {
    set interactive 0
}
if { -1 != [set where [lsearch -exact $argv {--no-cluster}]] } {
    set no_cluster_ok 1
    set argv [lrm $argv $where]
}
if { -1 != [set where [lsearch -exact $argv {--no-getline}]] } {
    set no_getline 1
    set argv [lrm $argv $where]
}
if { -1 != [set where [lsearch -exact $argv {--no-more}]] } {
    set no_more 1
    set argv [lrm $argv $where]
}

eval lappend argv $nonopts 

# Here is a hack to let me do some things on a non-cluster machine.
# This will let cluadmin start up on a non-cluster machine, but it
# won't run well, or do much of anything correctly.  It is nice to be
# able to start cluadmin, and call tcl commands, or call through to
# commands implemented in C, though.  Anyway, this switch isn't really
# intended for general use.
#
# Note that, to use the switch, cluster_locking should be set to 0 in
# parseconf.c.
#
if { $no_cluster_ok } {
    set CLU_CONFIG_FILE "/extra/src/cluster.conf"; # Or to wherever
                                                   # you want to put it.
    proc cluster::cluster_member_check {} {
        return 1;                       # pretend we are in a cluster.
    }
    proc cluster::ensure_running_as_root {} {
        # pretend we are.
    }
    proc cfg::Read  {} {
        global CLU_CONFIG_FILE
        variable CFG_vals
        return $CFG_vals([CFG_ReadFile $CLU_CONFIG_FILE])
    }
    proc cfg::Write {} {
        global CLU_CONFIG_FILE
        variable CFG_vals
        return $CFG_vals([CFG_WriteFile $CLU_CONFIG_FILE])
    }
    proc session::lock {{force 0}} {
        return 1
    }
    proc session::stealTheLock {} {
        return 1
    }
    proc session::locked {} {
        return 1
    }
    proc session::unlock {} {
        return 1
    }
}
# Sometimes we might want completion and history turned off.  This
# might be the case when we are running cluadmin under the control of
# some other program.  Esp. one that isn't based on expect.
#
if { $no_getline } {
    proc getline { prompt } {
        puts -nonewline stdout $prompt
        flush stdout
        set res {}
        gets stdin res
        return $res
    }
}

cluster::ensure_running_as_root
# Set log level to svcmgr log level. Remember to set the cluadmin
# logging level when the svcmgr log level has been changed.
set loglevel [cfg::Get svcmgr%loglevel]
if { [string match {} $loglevel] } {
    cluster::clu_set_loglevel 4
} else {
    cluster::clu_set_loglevel $loglevel
}
if { $interactive } {
    cfg::Read
    if { [llength $argv] == 0 } {
        if { [session::lock] } {
            commandloop
        } else {
            puts stderr "Unable to obtain session lock, exiting."
            exit 1
        }
    } else {
        set one_shot_mode 1
        if { [session::lock 1] } {
            proc confirm {args} {
                return 1
            }
            set elements [lrange $argv 1 end]
            set command [lindex $argv 0]
            if { [lsearch -exact $commands $command] != -1 } {
                # do the command
                if { [string match $command "exit"]} {
                    session::unlock
                    exit 0
                }
                if { [catch {
                    eval $command $elements
                } result] } {
                    if { [string match "NoError*" $result] } {
                        regsub {^NoError:} $result {} result
                        puts stderr "$result"
                    } else {
                        puts stderr "Error: $result"
                        if { $stack_traces } {
                            puts stderr "$errorInfo"
                        }
                    }
                    session::unlock
                    exit 1 
                } else {
                    session::unlock
                    exit 0
                }
            } else {
                session::unlock
                error "unrecognised command: $command"
            }
        } else {
            puts stderr "Unable to obtain session lock, exiting."
            exit 1
        }
    }
}
