/*
 * Copyright Mission Critical Linux LLC, 2000
 *
 * All Rights Reserved. Unpublished rights reserved under the
 * copyright laws of the United States. This software and all
 * associated intellectual property embodies the confidential
 * technology of Mission Critical Linux LLC. Possession, use,
 * duplication or dissemination of the software is authorized
 * only pursuant to a valid written license from Mission
 * Critical Linux LLC.
 */
/*
 *  Copyright (C) 2000 Mission Critical Linux, LLC
 *
 *  author: Tim Burke <burke@missioncriticallinux.com>
 *  description: Interface to net authentication token.
 *
 * disknetblock.c
 *
 * This file implements the routines used to read and write the network
 * information block.
 */
#include <stdio.h>
#include <unistd.h>
#include <string.h>
#include <fcntl.h>
#include <sys/types.h>
#include <sys/param.h>
#include <sys/stat.h>
#include <sys/wait.h>
#include <sys/time.h>
#include <signal.h>
#include <errno.h>
#include <clu_lock.h>

#include <logger.h>
#include <sys/syslog.h>
#include <diskstate.h>
#include <disk_proto.h>
#include <diskapis.h>

/*
 * Forward routine declarations.
 */
static void initNetBlock(DiskNetBlock *netblk);
static int writeNetBlock(DiskNetBlock *netblk);
static int initNetBlockSubsys(void);

static const char *version __attribute__ ((unused)) = "$Revision: 1.3 $";
static int subsysInitialized = 0;
/*
 * .............Configurable Parameters...................
 *
 * The following tuning knobs are intended to allow customization.
 */
/*
 * We tolerate a few IO errors before reacting.
 * This parameter defines how many consecutive errors are needed
 * to declare a true IO failure condition.  It is intended to avoid
 * over-reacting to an intermittent error.
 */
static int max_consecutive_io_errors = MAX_CONSECUTIVE_IO_ERRORS;

/*
 * Called to initialize subsystem state variables.
 * Also opens the file descriptor representing the shared state partition.
 * 
 * Returns: 0 on success.
 */
static int initNetBlockSubsys() {
    // Get diagnostic print level from cluster config file.
    // XXX TIM, FIXME - getVerboseLevel();  
    if (subsysInitialized) {
	clulog(LOG_DEBUG, "initNetBlockSubsys: already initialized.\n");
	return(0);
    }

    if (initAlignedBufStuff() < 0) {
        clulog(LOG_EMERG, "initNetBlockSubsys: unable to init rawio support.\n");
        return(-1);
    }
    subsysInitialized = 1;
    return(0);
}

#ifdef NOTYET /* XXX - We don't yet call this function */
/*
 * Called to release resources obtained in initNetBlockSubsys.
 * 
 * Returns: 0 on success.
 */
static int closeNetBlockSubsys(void) {

    if (subsysInitialized == 0) {
	clulog(LOG_DEBUG, "closeNetBlockSubsys: Subsystem not open.\n");
	return(0);
    }
    deinitAlignedBufStuff();
    subsysInitialized = 0;
    return 0;
}
#endif

/*
 * Called to initialize a net block.  
 * During the course of normal operation, various fields within this block
 * will be updated accordingly.
 */
static void initNetBlock(DiskNetBlock *netblk) {
    bzero((void *)netblk, sizeof(DiskNetBlock));
    netblk->magic_number = NET_BLOCK_MAGIC_NUMBER;
    netblk->version = NET_BLOCK_LATEST_VERSION;
    netblk->check_sum = 0;
    netblk->state = DISK_NET_BLOCK_INVALID;
}

/*
 * Write the net block out to disk.
 * Returns: -1 on IO error, -2 on parameter error, 0 on success.
 */
static int writeNetBlock(DiskNetBlock *netblk) {

        if (subsysInitialized == 0) {
	    if (initNetBlockSubsys() != 0) {
	        clulog(LOG_ERR, "writeNetBlock: Subsystem init failure.\n");
	        return(-2);
	    }
        }
	if (netblk->magic_number != NET_BLOCK_MAGIC_NUMBER) {
		clulog(LOG_ERR, "writeNetBlock: invalid magic# 0x%lx\n",
			netblk->magic_number);
		return(-2);
	}
	/*
	 * XXX -- This was a check for != DISK_NET_BLOCK_VALID.  It is
	 *        most certainly legal to write VALID or INVALID to the
	 *        disk block.  VALID is written when we put the data
	 *        into the block, INVALID is put in there when we go to
	 *        initialize it.
	 */
#if TIM_SAYS_ITS_OKAY
	if (netblk->state != DISK_NET_BLOCK_VALID) {
		clulog(LOG_ERR, "writeNetBlock: invalid state %d\n",
		       netblk->state);
		return(-2);
	}
#endif
	assert_clu_lock_held("writeNetBlock");
	return (diskRawWriteShadow(OFFSET_NET_BLOCK, (char *)netblk, 
				  sizeof(DiskNetBlock),
				  (ulong)&((DiskNetBlock *)0)->check_sum));

}

/*
 * Reads in the net block from the shared partition.
 * Stuffing the results into the passed data struct.
 * Returns: -1 on IO error, -2 on parameter error, 0 on success.
 */
static int readNetBlock(DiskNetBlock *netblk) {
	int ret;

#ifdef DISKSTATE_DEBUG
        bzero((void *)netblk, sizeof(DiskNetBlock)); // Debug
#endif // DISKSTATE_DEBUG
        if (subsysInitialized == 0) {
	    if (initNetBlockSubsys() != 0) {
	        clulog(LOG_ERR, "readNetBlock: Subsystem init failure.\n");
	        return(-2);
	    }
        }


	ret = diskRawReadShadow(OFFSET_NET_BLOCK, (char *)netblk, 
				  sizeof(DiskNetBlock),
				  (ulong)&((DiskNetBlock *)0)->check_sum, 1);

	if(ret) {
		clulog(LOG_ERR, "readNetBlock: bad ret %d from diskRawReadShadow\n", ret);
		return(ret);
	}		
	/*
         * Do at least a primitive level of validation to see if it looks like
	 * a viable net block.
	 */
	if (netblk->magic_number != NET_BLOCK_MAGIC_NUMBER) {
		clulog(LOG_ERR, "readNetBlock: Invalid magic # 0x%lx.\n",
			netblk->magic_number);
		return(-2);
	} 
	return(0);
}

/*
 * Initialize the on-disk data structures representing net block.
 * Returns: 0 on success.
 */
int initializePartitionNetBlock(void) {
    DiskNetBlock netblk;
    int retval;

    /*
     * Just wiping out any prior settings.
     */
    initNetBlock(&netblk);
    retval = writeNetBlock(&netblk);
    if (retval != 0) {
        clulog(LOG_CRIT, "initializePartitionNetBlock: unable to initialize block.\n");
        return(retval);
    }
    clulog(LOG_DEBUG, "initializePartitionNetBlock: successfully initialized net block.\n");
    return(0);
}

/*
 * Read the on-disk data structures representing the network block.
 * This is called periodically as part of the read/repair service.
 * Returns: 0 on success.
 */
int readRepairNetBlock(void) {
    DiskNetBlock netblk;
    int retval;

    if(clu_try_lock() == LOCK_SUCCESS) {
	    retval = readNetBlock(&netblk);
	    if (retval != 0) {
		    clulog(LOG_ERR, "readRepairNetBlock: unable to read block.\n");
		    return(retval);
	    }
	    clu_un_lock();
    }
    return(0);
}


#ifdef NOTYET /* Not used at the moment */
/*
 * Before we can read or write to the net block, we must first determine
 * cluster membership for all cluster members.  Fail the call if we 
 * aren't a cluster member.
 * Returns: 1 - Success, we are a cluster member, and so is our partner.
 *          0 - Success, we are a cluster member and the state of the other 
 *              node has been determined as down.  Therefore we can trust the 
 *              integrity of the contents of the net block to be INVALID.
 *	   -2 - We are not a cluster member.
 *	   -3 - We are a member, but the state of the partner is uncertain.
 */
static int checkMembership() {
    int retval;
    SharedDiskNodeStates nodeStates;
    int myNodeNumber, partnerNodeNumber;
    int myState, partnerState;

    myNodeNumber = cluGetLocalNodeId();
    if (myNodeNumber == -1) {
	clulog(LOG_ERR, "checkMembership: can't determine my node number.\n");
	return(-2);
    }
    partnerNodeNumber = myNodeNumber ^ 1; /* kludge */
    /*
     * Send a message to quorumd requesting cluster membership status.
     */
    retval = cluGetDiskNodeStates(&nodeStates);
    if (retval < 0) {
	/*
	 * Unable to contact quorumd for node status info.  Consider 
	 * ourselves out of the cluster.
	 */
	clulog(LOG_DEBUG, "checkMembership: failing, unable to contact quorumd.\n");
	return(-2);
	
    }
    myState = nodeStates.states[myNodeNumber];
    partnerState = nodeStates.states[partnerNodeNumber];
    if (myState != NODE_UP) {
	clulog(LOG_DEBUG, "checkMembership: sorry, I'm not a member.\n");
	return(-2);
    }
    if ((partnerState != NODE_UP) && (partnerState != NODE_DOWN)) {
	clulog(LOG_DEBUG, "checkMembership: partner status indeterminate %d.\n",
			partnerState);
	return(-3);
    }
    /*
     * If we make it this far, we know the state of all cluster members.
     * Return success.
     */
    return(partnerState == NODE_UP ? 1 : 0);
}
#endif /* NOTYET */


/*
 * Externally accessible API used to retrieve the contents of the net block.
 * Basically this is a wrapper over the routine which does the actual reading
 * and abstracts the caller from seeing the metadata header info associated
 * with an on-disk net block.
 *
 * Parameter: *netBlockData user buffer into which the contents of the 
 *	    net block will be copied.  This better be at least
 *	    SPACE_NET_BLOCK_DATA in size!
 * Returns: greater than zero - indicating how many bytes of data constitute
 *	    the net block.
 *	   -1 - Unable to read from partition, IO error.
 *	   -2 - Invalid (inactive) net block.
 *	   -3 - Not currently a cluster member.
 *	   -4 - Uninitialized net block
 */
int getNetBlockData(char *netBlockData) {
    DiskNetBlock netblk;
    int retval;
    int retries = 0;

    /*
     * Fail the call if we are not a cluster member.
     * Note: this is actually somewhat recursive in that the call uses
     * the messaging library; consequently this will have to be a call
     * which is operational in a non-authenticated case.
     */
#if 0 /* BROKEN */
    partnerState = checkMembership();
    if (partnerState < 0) {
	return(partnerState);
    }
#endif
    if (subsysInitialized == 0) {
	    /* XXX - Why init the Service subsys? */
	    /*if (initServiceSubsys() != 0) { */
	    if (initNetBlockSubsys() != 0) {
	        clulog(LOG_ERR, "getNetBlockData: Subsystem init failure.\n");
	        return(-1);
	    }
    }
    while (retries++ < max_consecutive_io_errors) {
        retval = readNetBlock(&netblk);
	if (retval == 0) { // success
	    if (netblk.state != DISK_NET_BLOCK_VALID) {
		return(-4);
	    }
    	    bcopy(&netblk.data, netBlockData, SPACE_NET_BLOCK_DATA);
	    return(SPACE_NET_BLOCK_DATA);
	}
    }
    clulog(LOG_ERR, "getNetBlockData: IO error reading quorum partition.\n");
    return(-1);
}

/*
 * Externally accessible API used to write out the contents of the net block.
 * Basically this is a wrapper over the routine which does the actual writing
 * and abstracts the caller from seeing the metadata header info associated
 * with an on-disk net block.
 *
 * Parameter: *netBlockData user buffer containing the source data to be
 *	    written out to disk.  This better be at least
 *	    SPACE_NET_BLOCK_DATA in size!
 * Returns: greater than zero - indicating how many bytes of data written.
 *	   -1 - Unable to initialize/write to partition, or no memory.
 */
int setNetBlockData(char *netBlockData) {
    DiskNetBlock netblk;
    int retval;
    int retries = 0;

    if (subsysInitialized == 0) {
	    if (initNetBlockSubsys() != 0) {
	        clulog(LOG_ERR, "setNetBlockData: Subsystem init failure.\n");
	        return(-1);
	    }
    }
    initNetBlock(&netblk);
    bcopy(netBlockData, &netblk.data, SPACE_NET_BLOCK_DATA);
    netblk.state = DISK_NET_BLOCK_VALID;
    while (retries++ < max_consecutive_io_errors) {
        retval = writeNetBlock(&netblk);
	if (retval >= 0) { // success
	    return(SPACE_NET_BLOCK_DATA);
	}
    }
    clulog(LOG_ERR, "setNetBlockData: IO error writing quorum partition.\n");
    return(-1);
}
