/*
  Copyright Mission Critical Linux, 2000

  Kimberlite is free software; you can redistribute it and/or modify it
  under the terms of the GNU General Public License as published by the
  Free Software Foundation; either version 2, or (at your option) any
  later version.

  Kimberlite is distributed in the hope that it will be useful, but
  WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with Kimberlite; see the file COPYING.  If not, write to the
  Free Software Foundation, Inc.,  675 Mass Ave, Cambridge, 
  MA 02139, USA.
*/

/* APC prototype power switch library.

   author: Ron Lawrence <lawrence@missioncriticallinux.com>
           Mike Ledoux  <mwl@missioncriticallinux.com>   
*/
#include <power.h>
#include <parseconf.h>
#include <clusterdefs.h>
#include "plock.h"

#include <unistd.h>
#include <stdlib.h>

#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <termios.h>
#include <stdio.h>
#include <sys/time.h>
#include <string.h>
#include <errno.h>
#include <regex.h>
#include <signal.h>

#include <diskapis.h>
#include <clucfg.h>

static const char *version __attribute__ ((unused)) = "$Id: power_apc.c,v 1.8 2000/11/10 19:29:48 burke Exp $";

/*#define LOG_STDERR*/
/*#define USE_LOGGING*/

#if defined(LOG_STDERR)
#define PR(format, args...) fprintf(stderr, "PWR_APC: " format, ## args)
#elif defined(USE_LOGGING)
#include <sys/syslog.h>
#include <msgsvc.h>
#include <logger.h>
#define PR(format, args...) clulog(LOG_DEBUG, format, ## args)
#else
#define PR(format, args...)
#endif 

enum PWR_boolean PWR_APC_configure(char *config_file_name);
enum PWR_boolean PWR_APC_set(char* option, char* value);
enum PWR_boolean PWR_APC_get(char* option, char *dflt, char** value);
enum PWR_boolean PWR_APC_init(PWR_result *status);
enum PWR_boolean PWR_APC_release(void);
PWR_result  PWR_APC_status(void);
PWR_result  PWR_APC_reboot(void);
PWR_result  PWR_APC_off(void);
PWR_result  PWR_APC_on(void);


/* Implementation Details */

#define ONESECOND 1000
#define BUFFER_LEN 1025         /* must be > ONESECOND */
#define DFLT_TMP_BUFFER_SZ 32

#define STREQ(s1,s2) (strcmp((s1),(s2)) == 0)

#define PWR_OFF_SW      9000
#define PWR_ON_SW       9001
#define PWR_REBOOT_SW   9002

static int initialized = 0;
static int locked = 0;

/* the name of the serial device file. */ 
// XXX - have no default device; thats a configuration error.
#define DFLT_NO_DEVICE "none"
static char serial_device_file[BUFFER_LEN] = DFLT_NO_DEVICE;

/* Serial device modes. Things like baud, parity, stop bits, etc. */

#define TTY_LOCK_D "/var/lock"

static int serial_fd = 0;
static struct termios oldtio, newtio;
static char buffer[BUFFER_LEN];

static long timeout = 2;
static long init_timeout = 10;

static int perform_reset = 1;

#define BAUDRATE B9600

/* Implementation Variables */

static PWR_result status;
static char dflt_config_filename[BUFFER_LEN] = CLU_CONFIG_FILE;

#define SET(where, what)        (where |= (what))
#define RESET(where, what)      (where &= ~(what))
#define ISSET(where, what)      (where & (what))
#define SETIFF(condition, where, what) {	\
  if((condition)) {				\
    SET((where),(what));} 			\
  else {					\
    RESET((where),(what));			\
  }}

/* Private: send a command to the switch. */
static int send_command(char *s, long ttimeout) {
  int length, result;
  fd_set wfds, xfds;
  struct timeval tv;
  int retval;

  FD_ZERO(&wfds);
  FD_ZERO(&xfds);

  FD_SET(serial_fd, &wfds);
  FD_SET(serial_fd, &xfds);

  tv.tv_sec = ttimeout;
  tv.tv_usec = 0;

  retval = select(serial_fd+1, NULL, &wfds, &xfds, &tv);
  switch(retval) {
  case -1:                      /* Error */
    return 0;
  case 1:                       /* something happened on our fd */
    if (FD_ISSET(serial_fd, &xfds))
      return 0;                 /* an exception occurred */
    break;
  case 0:                       /* timeout */
    return 0;
  }
  snprintf(buffer, BUFFER_LEN, "%s", s);
  length = strlen(buffer);
  result = write(serial_fd, buffer, length);
  if (result == length)
    return 1;
  else
    return 0;
}

/* Private: Read a response from the switch. */
static int read_response(char s[], long ttimeout) {
  int result;
  fd_set rfds, xfds;
  struct timeval tv;
  int retval;

  FD_ZERO(&rfds);
  FD_ZERO(&xfds);

  FD_SET(serial_fd, &rfds);
  FD_SET(serial_fd, &xfds);

  if(-1L == ttimeout) {
  /* Set the timeout to 1/2 sec.  This minimizes the delay when we
     stop getting chars back after the switch has opened. */
    tv.tv_sec = 0;
    tv.tv_usec = 500;
  }
  else {
    tv.tv_sec = ttimeout;
    tv.tv_usec = 0;
  }    

  retval = select(serial_fd+1, &rfds, NULL, &xfds, &tv);
  switch(retval) {
  case -1:                      /* Error */
    return 0;
  case 1:                       /* something happend on our fd */
    if (FD_ISSET(serial_fd, &xfds)) /* an exception occurred */
      return 0;
    break;
  case 0:                       /* timeout */
    return 2;
  }
  result = read(serial_fd, buffer, BUFFER_LEN);
  if (result == -1)
    return 0;
  if (strcpy(s, buffer) != NULL)
    return 1;
  return 0;
}

/* Private: Get the status from the switch, constrained by the given
   timeout. 
*/
static PWR_result internal_status(long ttimeout) {
  int rc;
  PWR_result result = 0;
  char s[BUFFER_LEN];
  char r[BUFFER_LEN];

  strcpy(s, "Z\n");

  rc = send_command(s, ttimeout);
  if (!rc) {
    PR("send returned bad rc returning result: %d\n", result);
    SET(result, PWR_ERROR);
    return result;
  }
  RESET(result, PWR_SWITCH);
  rc = read_response(r, ttimeout);
  if (2 == rc) {
    PR("timed out on read\n");
    SET(result, PWR_ERROR);
    SET(result, PWR_TIMEOUT);
    return result;
  }
  if (!rc) {
    PR("read returned bad rc returning result: %d\n", result);
    SET(result, PWR_ERROR);
    return result;
  }
  /* The switch is always closed, unless we have opened it, so, report
     the switch closed. */
  SET(result, PWR_SWITCH);
  SETIFF(initialized, result, PWR_INIT);
  return result;
}

/* Private: send a command to the switch.  Return the result. 
*/
static PWR_result internal_command(int cmd) {
  PWR_result result = 0;
  char s[BUFFER_LEN];
  char r[BUFFER_LEN]; 
  int i;
  int opened = 0;

  if(initialized) {
    SET(result, PWR_INIT);

    switch (cmd) {
    case PWR_OFF_SW:
      /* send a constant stream of 'U' chars across the serial line
         until the PWR_ON_SW command is issued */
      fprintf(stderr, "Power Off not implemented.\n");
      break;
    case PWR_ON_SW:
      /* stop sending 'U' chars across the serial line send 'Z' chars
         one per second until one is read back successfully */
      fprintf(stderr, "Power On not implemented.\n");
      break;
    case PWR_REBOOT_SW:
      /* check the switch */

      PR("PWR_REBOOT_SW, result = %d\n", result);

      result = internal_status(timeout);

      PR("PWR_REBOOT_SW, after internal_status, result = %d\n", result);

      if (!(result & PWR_ERROR) &&
          !(result & PWR_TIMEOUT)) {
        /* send enough 'U' chars to power down the switch for
           approximately 10 seconds. */
        memset(s,0,BUFFER_LEN);
        memset(s,'U', ONESECOND);
        strcat(s,"\n");

        PR("Starting to send U's\n");
        for (i = 0; i < 10; i++) {
          if(send_command(s, timeout) == 0) {
            PR("error on send command, i=%d, result = %d\n", i, result);
            SET(result, PWR_ERROR);
            break;
          }
          /* it doesn't matter if the following read is successful,
             but the attempt has to be made for the switch to open.

             The read will return immediately with result 1 until the
             switch opens, when it will timeout.  After the first
             timeout, stop trying to read responses from the serial
             port. */
          PR("calling read_response\n");
          if(!opened && 2 == read_response(r, -1L)) {
            PR("switch opened\n");
            opened = 1;
          }
        }
      }
      break;
    }
  }
  else {
    PR(__FUNCTION__ " not initialised");
    SET(result, PWR_ERROR);
  }
  SETIFF(initialized, result, PWR_INIT);
  SET(result, PWR_SWITCH);
  return result;
}

/* Private: Reset the serial device. */
static void serial_reset(void) {
  struct termios tty, old;
  int sec = 2;

  tcgetattr(serial_fd, &tty);
  tcgetattr(serial_fd, &old);
  cfsetospeed(&tty, B0);
  cfsetispeed(&tty, B0);
  tcsetattr(serial_fd, TCSANOW, &tty);
  if (sec>0) {
    sleep(sec);
    tcsetattr(serial_fd, TCSANOW, &old);
  }
}

/* Private: Open the serial device.  Set the parameters on the serial
   line.  If the device cannot be opened, we return some negative
   error code.  
*/
static int open_serial(char * device_file) {
  int rc;

  serial_fd = open(device_file, O_RDWR | O_NOCTTY | O_NDELAY );
  if (serial_fd <0) {
#ifdef DEBUG
    perror(device_file);
#endif
    return -1;
  }
  rc = tcgetattr(serial_fd, &oldtio);
  if(rc < 0) {
    PR("tcgetattr(serial_fd, &oldtio) failed\n");
    return rc;
  }
  memset(&newtio, 0, sizeof(newtio));

  /* Don't use CRTSCTS for the APC switch. */
  newtio.c_cflag = BAUDRATE | CS8 | CLOCAL | CREAD;
  newtio.c_iflag = 0;
  newtio.c_oflag = 0;
  newtio.c_lflag = ICANON;

  newtio.c_cc[VINTR]    = 0;
  newtio.c_cc[VQUIT]    = 0;
  newtio.c_cc[VERASE]   = 0;
  newtio.c_cc[VKILL]    = 0;
  newtio.c_cc[VEOF]     = 4;
  newtio.c_cc[VTIME]    = 0;
  newtio.c_cc[VMIN]     = 1;
  newtio.c_cc[VSWTC]    = 0;
  newtio.c_cc[VSTART]   = 0;
  newtio.c_cc[VSTOP]    = 0;
  newtio.c_cc[VSUSP]    = 0;
  newtio.c_cc[VEOL]     = 0;
  newtio.c_cc[VREPRINT] = 0;
  newtio.c_cc[VDISCARD] = 0;
  newtio.c_cc[VWERASE]  = 0;
  newtio.c_cc[VLNEXT]   = 0;
  newtio.c_cc[VEOL2]    = 0;

  rc = tcsetattr(serial_fd, TCSANOW, &newtio);
  if(rc < 0) {
    PR("tcsetattr(serial_fd, TCSANOW, &newtio) failed\n");
    return rc;
  }
  rc = tcflush(serial_fd, TCIFLUSH);
  if(rc < 0) {
    PR("tcflush(serial_fd, TCIFLUSH) failed\n");
    return rc;
  }
  if(perform_reset) {
    serial_reset();
  }
  fcntl(serial_fd, F_SETFL, O_NONBLOCK);
  return 0;
}

/****************************************************************************/

/* Public Interface Functions */

/* Read a configuration file for settings and parameters. 
*/
enum PWR_boolean PWR_APC_configure(char *config_file_name) {
  char *pstr;
  /* when a configuration filename hasn't been passed in...
   */
  if( NULL == config_file_name ||
      ! strcmp(config_file_name, "")) {
    /* If a config file has already been read, it may have an entry
       that tells us where to read a config file from.
    */
    CFG_Get("power%configfile", dflt_config_filename, &pstr);
    /* Read the file specified, or the default configu file.
     */
    return CFG_OK == CFG_ReadFile(pstr);
  }
  else {
    return CFG_OK == CFG_ReadFile(config_file_name);
  }
}

/* Set or query the configuration parameters of the switch interface. */
enum PWR_boolean PWR_APC_set(char* option, char* value) {
  CFG_status rc;
  rc = CFG_Set(option, value);
  if(CFG_OK == rc) {
    return PWR_TRUE;
  }
  else {
    return PWR_FALSE;
  }
}

enum PWR_boolean PWR_APC_get(char* option, char* dflt, char** value) {
  CFG_status rc;
  rc = CFG_Get(option, dflt, value);
  if(CFG_OK == rc || CFG_DEFAULT == rc) {
    return PWR_TRUE;
  }
  else {
    return PWR_FALSE;
  }
}

/* Initialize the power switch communication paths. */
enum PWR_boolean PWR_APC_init(PWR_result *s) {
  char sdflt[DFLT_TMP_BUFFER_SZ];
  char *svalue;
  long lvalue;
  int lock_result;
  CluCfg              *cfg;
  char                *file = NULL;

  if (!initialized) {
    /* read configuration variables */
    snprintf(sdflt, DFLT_TMP_BUFFER_SZ, "%s", DFLT_NO_DEVICE);
    PWR_get("power%device", sdflt, &svalue);
    if( STREQ(svalue,DFLT_NO_DEVICE)) {
      /* Read configuration variables. */
      cfg = get_clu_cfg(file);
      if (cfg != NULL) {
        strcpy(serial_device_file, 
               cfg->nodes[cfg->lid].powerSerialPort);
        free(cfg);
      }
      else {
        strcpy(serial_device_file, DFLT_NO_DEVICE);
      }
    } else {
      strcpy(serial_device_file, svalue);
    }
    snprintf(sdflt, DFLT_TMP_BUFFER_SZ, "%ld", init_timeout);
    PWR_get("power%init_timeout", sdflt, &svalue);
    sscanf(svalue, "%ld", &lvalue);
    init_timeout = lvalue;

    snprintf(sdflt, DFLT_TMP_BUFFER_SZ, "%ld", timeout);
    PWR_get("power%timeout", sdflt, &svalue);
    sscanf(svalue, "%ld", &lvalue);
    timeout = lvalue;
  
    if (!locked) {
      /* Create a lock file for the serial device. */
      if (strcmp(serial_device_file, DFLT_NO_DEVICE) == 0) {
        PR("serialPort not specified\n");
        SET(status, PWR_ERROR);
      }
      else if((lock_result = pwr_lock_serial_device(serial_device_file)) < 0) {
        PR("can't lock %s, lock result: %d\n", 
           serial_device_file,
           lock_result);
        SET(status, PWR_ERROR);
      }
      else {
        PR("locked %s\n", serial_device_file);
        locked = 1;
        /* Open and query the serial device. */
        if(open_serial(serial_device_file) < 0) {
          SET(status,PWR_ERROR);
          *s = status;
          return PWR_FALSE;
        }
      }
    } else { /* locked */
      /*
       * This situation is basically the following:  we already called
       * this initialization routine, but we were unable to contact the
       * power switch.  Thus, we have it locked, we need only cycle dtr
       * and hope that we can talk to it.
       */
      if (serial_fd > 0)
        serial_reset();
      else
        open_serial(serial_device_file);
    }
    status = internal_status(init_timeout);
    if(!(status & PWR_ERROR) && 
       !(status & PWR_TIMEOUT) && 
       !(status & PWR_INIT)) {
      /* The switch isn't fully initialized, yet it is not in error,
         nor has it timed out.  This can only mean that we have just
         received the RPS-10 Ready string.  Send the status command
         again; the switch should now be ready to act on
         commands. 
      */
      status = internal_status(init_timeout);
    }
  }
  else {
    status = internal_status(timeout);
  }
  if (ISSET(status, PWR_TIMEOUT) || ISSET(status, PWR_ERROR)) {
    initialized = 0;
    RESET(status, PWR_INIT);
    *s = status;
    /* 'close()' will cause the program to hang if the fd hasn't been
       tcflush'ed beforehand. */
    tcflush(serial_fd, TCIOFLUSH);

    close(serial_fd);
    return PWR_FALSE;
  }
  else {
    initialized = 1;
    SET(status, PWR_INIT);
    *s = status;
    return PWR_TRUE;
  }
}

/* Give back resources to the system.  Flush the serial port, and
   close the file.  This must be done prior to calling exit.  
*/
enum PWR_boolean PWR_APC_release(void) {
  if(initialized) {
    if(pwr_unlock_serial_device(serial_device_file) < 0 ) {
      PR("failed to unlock serial device\n");
    }
    /* 'close()' will cause the program to hang if the fd hasn't been
       tcflush'ed beforehand. */
    tcflush(serial_fd, TCIOFLUSH);
    close(serial_fd);
  }
  initialized = 0;
  return 1;
}

/* Query the switch to determine its current status, and its readiness
   to carry out additional commands.  Returns a PWR_result that
   contains flags that tell whether the switch is open or closed.
*/
PWR_result PWR_APC_status(void) {
  return internal_status(timeout);
}

/* Send a command to the switch that will cause the system powered by
   the switch to reboot.  Normally, this means that the switch will be
   opened for some period of time, after which it will be closed
   again.
*/
PWR_result PWR_APC_reboot(void) {
  return internal_command(PWR_REBOOT_SW);
}

/* Send a command to the switch that will cause the switch to open,
   securing power to the effected system.

   Note: not implemented for APC switches.
*/
PWR_result PWR_APC_off(void) {
  return internal_command(PWR_OFF_SW);
}

/* Send a command to the switch that will cause the switch to close,
   supplying power to eht effected system.

   Note: not implemented for APC switches.
*/
PWR_result PWR_APC_on(void) {
  return internal_command(PWR_ON_SW);
}
