/*
  Copyright Mission Critical Linux, 2000

  Kimberlite is free software; you can redistribute it and/or modify it
  under the terms of the GNU General Public License as published by the
  Free Software Foundation; either version 2, or (at your option) any
  later version.

  Kimberlite is distributed in the hope that it will be useful, but
  WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with Kimberlite; see the file COPYING.  If not, write to the
  Free Software Foundation, Inc.,  675 Mass Ave, Cambridge, 
  MA 02139, USA.
*/
/*
 * Program used to initiate a clean cluster shutdown.
 *
 * Author: Tim Burke (burke@missioncriticallinux.com)
 * $Revision: 1.10 $
 *
 * Currently this consists of sending the service manager a message
 * indicating that this node has had a state transition to DOWN.
 * In response, SM will stop all services.  Upon successful service
 * stopping, SM then sends a message to the quorum daemon notifying it
 * that its safe to stop pinging and marking our state as down.
 */
#include <stdio.h>
#include <stdlib.h>
#include <sys/param.h>
#include <clucfg.h>
#include <msgsvc.h>
#include <clusterdefs.h>

static void fill_generic_hdr(generic_msg_hdr *msghdr, int cmd, int len);

int
main(
    int			argc,
    char		**argv)
{
    CluCfg		*cfg = (CluCfg *)NULL;    
    char		*file = NULL;
    msg_handle_t	con;
    DiskMessageSt sendMsgBuf;
    int 		retval;
    char		opt;
    int			try_count=0;

    int	send_stop_to_SM = 1;
    int	send_stop_to_quorumd = 0;

    if (geteuid() != (uid_t)0) {
          fprintf(stderr, "%s must be run as the root user.\n",
                argv[0]);
        exit(1);    
    }

    if (argc > 1)
        file = argv[1];

    while ((opt = getopt(argc, argv, "p")) != EOF) {
        switch (opt) {
        case 'p': // Send stop message to quorum daemon
	    /* Note: generally speaking this is not a safe thing to do as it
	     * could cause other cluster members to concurrently failover
	     * services which could still be running if service manager has
	     * not been cleanly shutdown.
	     */
	    send_stop_to_quorumd = 1;
	    send_stop_to_SM = 0;
            break;
        default:
	    fprintf(stderr, "Unknown option %c.\n",opt);
            break;
        }
    }                                      

    cfg = get_clu_cfg(file);
    if (cfg == NULL) {
        fprintf(stderr, "error in get_clu_cfg\n");
	perror("get_clu_cfg");
	exit(1);
    }

  try_count=0;
  if (send_stop_to_SM) {
    while (1) {
      con = msg_open(PROCID_SVCMGR, cfg->lid);
      if (con < 0) {
        fprintf(stderr, 
          "Unable to establish connection to SM, error=%d, retrying.\n",
          con);
        try_count=try_count+1;
	if (try_count >= 10)
          {
	    exit(1);
          }
        sleep(5);		// give the svcmgr some time to respond
	continue;
      } 
      else {
	fill_generic_hdr(&sendMsgBuf.hdr, DISK_NODE_STATECHANGE, DISK_MESSAGE_SIZE);
   	sendMsgBuf.data.statusMsg.nodeNumber = cfg->lid;
   	sendMsgBuf.data.statusMsg.nodeStatus = NODE_DOWN;
   	retval = msg_send(con, &sendMsgBuf, DISK_MESSAGE_SIZE);
   	if (retval != DISK_MESSAGE_SIZE){
       	  fprintf(stderr, 
            "Unable to send node down to SM, error=%d retrying.\n", retval);
          sleep(5);
	  continue;
   	}
   	else {
       	  fprintf(stderr, "Succesfully sent node down to SM.\n");
   	}
   	msg_close(con);
        break;
      }
    }
  }

  try_count=0;
  if (send_stop_to_quorumd) {
    while (1) {
      con = msg_open(PROCID_QUORUMD, cfg->lid);
      if (con < 0) {
        fprintf(stderr, 
          "Unable to establish connection to quorumd, error=%d, retrying.\n",
           con);
        try_count=try_count+1;
        if (try_count >= 10) {
	    exit(1);
        }
        sleep(5);
        continue;
      }
      else {
	fill_generic_hdr(&sendMsgBuf.hdr, DISK_SM_EXITING, DISK_MESSAGE_SIZE);
   	retval = msg_send(con, &sendMsgBuf, DISK_MESSAGE_SIZE);
   	if (retval != DISK_MESSAGE_SIZE) {
       	  fprintf(stderr, 
            "Unable to send stop message to quorumd, error=%d, retrying.\n", 
            retval);
          sleep(5);
	  continue;
   	}
   	else {
       	  fprintf(stderr, "Succesfully sent stop message to quorumd.\n");
   	}
   	msg_close(con);
        break;
      }
    }
  }
  if (cfg != NULL) {
	free(cfg);
  }
  exit(0);
}


static void
fill_generic_hdr(generic_msg_hdr *msghdr, int cmd, int len)
{
    msghdr->magic = GENERIC_HDR_MAGIC;
    msghdr->command = cmd;
    msghdr->length = len;
}

