
#include <config.h>

#include <map>
using std::map;

#define WITH_LIBDV

#include <kino_extra.h>
#include <image_create.h>
#include <image_transitions.h>
#include <image_filters.h>
#include <audio_filters.h>
#include "widgets.h"
#include "time_map.h"
#include "affine.h"
#include <sys/types.h>
#include <sys/wait.h>
#include <libdv/dv.h>

#include <stdint.h>
#include <algorithm>
#include <cmath>
#include <functional>
#include <string>
using std::string;

#include <stdint.h>
#include "kino_plugin_utility.h"
#include "PixbufUtils.h"


extern "C" {
#include <gtk/gtkruler.h>
#include "interface.h"
#include "support.h"

#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include <dirent.h>

extern void sigpipe_clear( );
extern int sigpipe_get( );

}

class Pixelate : public GDKImageFilter
{
	private:
		GtkWidget *window;
		int sw, sh;
		int ew, eh;

	public:
		Pixelate( )
		{
			window = create_window_pixelate( );
		}

		virtual ~Pixelate( ) 
		{
			gtk_widget_destroy( window );
		}

		char *GetDescription( ) const 
		{ 
			return "Pixelate";
		}

		void Average( uint8_t *start, int width, int height, int offset )
		{
			double r = (double)*( start );
			double g = (double)*( start + 1 );
			double b = (double)*( start + 2 );

			for ( int y = 0; y < height; y ++ )
			{
				uint8_t *p = start + y * offset;
				for ( int x = 0; x < width; x ++ )
				{
					r = ( r + (double)*( p ++ ) ) / 2;
					g = ( g + (double)*( p ++ ) ) / 2;
					b = ( b + (double)*( p ++ ) ) / 2;
				}
			}

			for ( int y = 0; y < height; y ++ )
			{
				uint8_t *p = start + y * offset;
				for ( int x = 0; x < width; x ++ )
				{
					*p ++ = (uint8_t)r;
					*p ++ = (uint8_t)g;
					*p ++ = (uint8_t)b;
				}
			}
		}

		void FilterFrame( uint8_t *io, int width, int height, double position, double frame_delta ) 
		{
			int rw = (int)( sw + ( ew - sw ) * position );
			int rh = (int)( sh + ( eh - sh ) * position );
			for ( int w = 0; w < width; w += rw )
			{
				for ( int h = 0; h < height; h += rh )
				{
					int aw = w + rw > width ? rw - ( w + rw - width ) : rw;
					int ah = h + rh > height ? rh - ( h + rh - height ) : rh;
					Average( io + h * width * 3 + w * 3, aw, ah, width * 3 );
				}
			}
		}
		
		void AttachWidgets( GtkBin *bin ) 
		{
			gtk_widget_reparent( ( GTK_BIN( window ) )->child, GTK_WIDGET( bin ) );
		}

		void DetachWidgets( GtkBin *bin ) 
		{ 
			gtk_widget_reparent( ( GTK_BIN( bin ) )->child, GTK_WIDGET( window ) );
		}

		void InterpretWidgets( GtkBin *bin ) 
		{
			GtkEntry *spin = GTK_ENTRY( my_lookup_widget( window, "spinbutton_start_width" ) );
			ew = sw = atoi( gtk_entry_get_text( spin ) );
			spin = GTK_ENTRY( my_lookup_widget( window, "spinbutton_start_height" ) );
			eh = sh = atoi( gtk_entry_get_text( spin ) );
			if ( gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON( my_lookup_widget( window, "checkbutton_change_size" ) ) ) )
			{
				spin = GTK_ENTRY( my_lookup_widget( window, "spinbutton_end_width" ) );
				ew = atoi( gtk_entry_get_text( spin ) );
				spin = GTK_ENTRY( my_lookup_widget( window, "spinbutton_end_height" ) );
				eh = atoi( gtk_entry_get_text( spin ) );
			}
		}
};

class Jerker : public GDKImageFilter
{
	private:
		uint8_t tempframe[ 720 * 576 * 3 ];
		GtkWidget *window;
		int show_every;
		int current;

	public:
		Jerker( )
		{
			window = create_window_slow_mo( );
		}

		virtual ~Jerker( ) 
		{
			gtk_widget_destroy( window );
		}

		char *GetDescription( ) const 
		{ 
			return "Jerk";
		}

		void FilterFrame( uint8_t *io, int width, int height, double position, double frame_delta ) 
		{
			if ( current ++ % show_every == 0 )
				memcpy( tempframe, io, width * height * 3 );
			else
				memcpy( io, tempframe, width * height * 3 );
		}
		
		void AttachWidgets( GtkBin *bin ) 
		{
			gtk_widget_reparent( ( GTK_BIN( window ) )->child, GTK_WIDGET( bin ) );
		}

		void DetachWidgets( GtkBin *bin ) 
		{ 
			gtk_widget_reparent( ( GTK_BIN( bin ) )->child, GTK_WIDGET( window ) );
		}

		void InterpretWidgets( GtkBin *bin ) 
		{
			GtkRange *range = GTK_RANGE( my_lookup_widget( window, "hscale_slow_mo" ) );
			GtkAdjustment *adjust = gtk_range_get_adjustment( range );
			show_every = (int)adjust->value; 
			current = 0;
		}
};

static void on_entry_file_changed( GtkEditable *editable, gpointer user_data );

class MultipleImport : public GDKImageCreate, protected PixbufUtils
{
	private:
		GtkWidget *window;
		char directory[ 1024 ];
		int files;
		int index;
		int repeat_times;
		int repeater;
		bool maintain_aspect_ratio;

	public:
		MultipleImport( ) : files(0)
		{
			window = create_window_multiple_import( );
			strcpy( directory, "" );
			gtk_signal_connect( GTK_OBJECT( my_lookup_widget( window, "entry_directory" ) ), "changed", 
							    GTK_SIGNAL_FUNC( on_entry_file_changed ), this );
		}

		virtual ~MultipleImport( ) 
		{
			gtk_widget_destroy( window );
		}

		char *GetDescription( ) const 
		{ 
			return "Multiple Image Import";
		}

		void CreateFrame( uint8_t *io, int width, int height, double position, double frame_delta  ) 
		{
			if ( repeater == 0 )
			{
				GtkTreeView *list = GTK_TREE_VIEW( my_lookup_widget( window, "clist_files" ) );
				GtkTreeSelection *selection = gtk_tree_view_get_selection( list );

				GtkTreeModel *model = NULL;
				GList *selected = gtk_tree_selection_get_selected_rows( selection, &model );

				GtkTreePath *path = ( GtkTreePath * )g_list_nth_data( selected, index ++ );
				GtkTreeRowReference *row =  gtk_tree_row_reference_new( model, path );
				GtkTreeIter iter;
				GValue value;
				memset( &value, 0, sizeof( GValue ) );
				gtk_tree_model_get_iter( model, &iter, path );
				gtk_tree_model_get_value( model, &iter, 0, &value );

				char *filename = g_strdup_printf( "%s/%s", directory, (char *)g_value_peek_pointer( &value ) );

				if ( maintain_aspect_ratio )
					SetScale( SCALE_ASPECT_RATIO );
				else
					SetScale( SCALE_FULL );

				ReadImageFile( filename, io, width, height );
				free( filename );

				g_value_unset( &value );

				//g_list_foreach( selected, ( void (*)( void *, void *) )gtk_tree_path_free, NULL);
				g_list_free( selected );
			}

			repeater ++;
			if ( repeater >= repeat_times )
				repeater = 0;
		}

		int GetNumberOfFrames( )
		{
			return files * repeat_times;
		}
		
		void AttachWidgets( GtkBin *bin ) 
		{
			gtk_widget_reparent( ( GTK_BIN( window ) )->child, GTK_WIDGET( bin ) );
		}

		void DetachWidgets( GtkBin *bin ) 
		{ 
			gtk_widget_reparent( ( GTK_BIN( bin ) )->child, GTK_WIDGET( window ) );
		}

		void InterpretWidgets( GtkBin *bin ) 
		{
			GtkTreeView *list = GTK_TREE_VIEW( my_lookup_widget( window, "clist_files" ) );
			GtkTreeSelection *selection = gtk_tree_view_get_selection( list );
			files = gtk_tree_selection_count_selected_rows( selection );

			GtkEntry *spin = GTK_ENTRY( my_lookup_widget( window, "spinbutton_repeat" ) );
			repeat_times = atoi( gtk_entry_get_text( spin ) );

			maintain_aspect_ratio = gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON( my_lookup_widget( window, "checkbutton_aspect" ) ) );

			// Initialise to start...
			index = 0;
			repeater = 0;
		}

		void DirectorySelected( )
		{
			char *new_dir = (char*)gtk_entry_get_text( GTK_ENTRY( my_lookup_widget( window, "entry_directory" ) ) );

			if ( strcmp( new_dir, directory ) )
			{
				GtkTreeView *list = GTK_TREE_VIEW( my_lookup_widget( window, "clist_files" ) );

				GtkListStore *store = GTK_LIST_STORE( gtk_tree_view_get_model( list ) );

				if ( gtk_tree_view_get_model( list ) == NULL )
				{
					GtkCellRenderer *renderer;
					GtkTreeViewColumn *column;

					store = gtk_list_store_new( 1, G_TYPE_STRING );
					gtk_tree_view_set_model( list, GTK_TREE_MODEL( store ) );
			
					renderer = gtk_cell_renderer_text_new( );
					column = gtk_tree_view_column_new_with_attributes ( "Description", renderer, "text", 0, NULL);
					gtk_tree_view_column_set_sort_column_id( column, 0 );
					gtk_tree_view_append_column( list, column );
				}
				else
				{
					gtk_list_store_clear( store );
				}

				strcpy( directory, new_dir );
				files = 0;

				char *filename; 
				char *title;
				DIR *dir;
				struct dirent *entry;

				dir = opendir( directory );

				if ( dir )
				{
					GtkTreeIter iter;

					while ( ( entry = readdir( dir ) ) != NULL )
					{
						filename = g_strdup_printf( "%s/%s", directory, entry->d_name );
						GError *err = NULL;
						GdkPixbuf *splash = gdk_pixbuf_new_from_file( filename, &err );

						if ( splash != NULL )
						{
							gtk_list_store_append( store, &iter );
							gtk_list_store_set( store, &iter, 0, entry->d_name, -1 );
						}

						gdk_pixbuf_unref( splash );
						g_free(filename);
					}
					closedir(dir);
				}

				GtkTreeSelection *selection = gtk_tree_view_get_selection( list );
				gtk_tree_selection_set_mode( selection, GTK_SELECTION_MULTIPLE );
				gtk_tree_selection_select_all( selection );
			}
		}
};


/** GTK callback on directory selection change.
*/

static void on_entry_file_changed( GtkEditable *editable, gpointer user_data )
{
	( ( MultipleImport * ) user_data )->DirectorySelected( );
}

/** Imagemagick support class.
*/

class Convert 
{
	private:
		char namedpipe[ PATH_MAX ];
		char args[ 10240 ];
	public:
		Convert( );
		~Convert();
		void Clear();
		void AddTransform( char *transform );
		void Transform( uint8_t *image, int width, int height );
		void Transform( char *file, uint8_t *image, int width, int height );
};

/** Convert class implementation - will probably be removed.
*/

Convert::Convert( ) 
{
	// Decided against using a fifo in the end.. too messy
	strcpy( namedpipe, "kinoXXXXXX" );
	close( mkstemp( namedpipe ) );
	unlink( namedpipe );
	strcat( this->namedpipe, ".ppm" );
	strcpy( args, "" );
}

Convert::~Convert() 
{
}

void Convert::Clear() 
{
	strcpy( args, "" );
}

void Convert::AddTransform( char *t ) 
{
	strcat( args, t );
	strcat( args, " " );
}

void Convert::Transform( uint8_t *image, int width, int height ) 
{
	char command[ 10240 ];
	sprintf( command, "convert %s ppm:- ppm:- > %s", args, namedpipe );
	printf( "Running %s\n", command );

	sigpipe_clear();
	FILE *output = popen( command, "w" );
	if ( output != NULL ) 
	{
		fprintf( output, "P6\n%d %d\n255\n", width, height );
		fwrite( image, width * height * 3, 1, output );
		pclose( output );
	}
	else 
	{
		throw "convert utility not found - get it from www.imagemagick.org";
	}

	if ( !sigpipe_get() ) 
	{
		char temp[ 132 ];
		FILE *input = fopen( namedpipe, "r" );
		fgets( temp, 132, input );
		fgets( temp, 132, input );
		fgets( temp, 132, input );
		fread( image, width * height * 3, 1, input );
		fclose( input );
		unlink( namedpipe );
	}
	else 
	{
		fprintf( stderr, "Failed %s\n", command );
		throw "Error in generated command for convert";
	}
}

void Convert::Transform( char *file, uint8_t *image, int width, int height ) 
{
	char command[ 10240 ];
	sprintf( command, "convert %s %s ppm:-", args, file );
	printf( "Running %s\n", command );

	sigpipe_clear();
	FILE *input = popen( command, "r" );
	if ( input != NULL ) 
	{
		char temp[ 132 ];
		fgets( temp, 132, input );
		fgets( temp, 132, input );
		fgets( temp, 132, input );
		fread( image, width * height * 3, 1, input );
		pclose( input );
		unlink( namedpipe );
	}
	else 
	{
		fprintf( stderr, "Rejected %s\n", command );
		throw "convert utility not found - get it from www.imagemagick.org";
	}
}

class MagickImageLookup 
{
	protected:

		// Hack to avoid common code duplication for option menu shortcomings
		char *GetGravity( int value ) 
		{
			char *gravity = NULL;
			if ( value == 0 )
				gravity = "-gravity center";
			else if ( value == 1 )
				gravity = "-gravity northwest";
			else if ( value == 2 )
				gravity = "-gravity north";
			else if ( value == 3 )
				gravity = "-gravity northeast";
			else if ( value == 4 )
				gravity = "-gravity west";
			else if ( value == 5 )
				gravity = "-gravity east";
			else if ( value == 6 )
				gravity = "-gravity southwest";
			else if ( value == 7 )
				gravity = "-gravity south";
			else if ( value == 8 )
				gravity = "-gravity southeast";
			return gravity;
		}

		char *GetCompose( int value ) 
		{
			char *compose = NULL;
			if ( value == 0 )
				compose = "Over";
			else if ( value == 1 )
				compose = "In";
			else if ( value == 2 )
				compose = "Out";
			else if ( value == 3 )
				compose = "Atop";
			else if ( value == 4 )
				compose = "Xor";
			else if ( value == 5 )
				compose = "Plus";
			else if ( value == 6 )
				compose = "Minus";
			else if ( value == 7 )
				compose = "Add";
			else if ( value == 8 )
				compose = "Subtract";
			else if ( value == 9 )
				compose = "Difference";
			else if ( value == 10 )
				compose = "Multiply";
			else if ( value == 11 )
				compose = "Bumpmap";
			else if ( value == 12 )
				compose = "Copy";
			else if ( value == 13 )
				compose = "CopyRed";
			else if ( value == 14 )
				compose = "CopyGreen";
			else if ( value == 15 )
				compose = "CopyBlue";
			else if ( value == 16 )
				compose = "CopyOpacity";
			return compose;
		}
};

/** ImageMagick titling class.
*/

class ImageTitler : public GDKImageFilter, MagickImageLookup
{
	private:
		Convert convert;
		GtkWidget *window;
		char *gravity;
		int startx;
		int starty;
		int endx;
		int endy;
		char *text;
		char *font;
		char colour[ 30 ];
		int fontsize;

	public:
		ImageTitler( ) : window(NULL), gravity(NULL), startx(0), starty(0),
								   endx(0), endy(0), text(NULL), font(NULL)
		{
			strcpy( colour, "" );
			window = create_image_filter_title( );
		}

		virtual ~ImageTitler( ) 
		{
			g_free( text );
		}

		char *GetDescription( ) const 
		{ 
			return "ImageMagick Titler";
		}

		void FilterFrame( uint8_t *pixels, int width, int height, double position, double frame_delta ) 
		{ 
			char command[ 1024 ];
			int x = (int)( startx + position * ( endx - startx ) );
			int y = (int)( starty + position * ( endy - starty ) );

			convert.Clear();
			sprintf( command, "%s -fill \"%s\" -font \"%s\" -pointsize %d -draw \"text %d,%d '%s'\"", 
					  gravity, colour, font, fontsize, x, y, text );
			convert.AddTransform( command );
			convert.Transform( pixels, width, height );
		}

		void AttachWidgets( GtkBin *bin ) 
		{
			gtk_widget_reparent( ( GTK_BIN( window ) )->child, GTK_WIDGET( bin ) );
		}

		void DetachWidgets( GtkBin *bin ) 
		{ 
			gtk_widget_reparent( ( GTK_BIN( bin ) )->child, GTK_WIDGET( window ) );
		}

		void InterpretWidgets( GtkBin *bin ) 
		{ 
			GtkMenu *gravityMenu = GTK_MENU( gtk_option_menu_get_menu( GTK_OPTION_MENU( my_lookup_widget( GTK_WIDGET( window ), "optionmenu_magick_title_gravity" ) ) ) );
			GtkEntry *positionEntry = GTK_ENTRY( my_lookup_widget( GTK_WIDGET( window ), "entry_magick_title_position" ) );	
			GtkEntry *endPositionEntry = GTK_ENTRY( my_lookup_widget( GTK_WIDGET( window ), "entry_magick_end_position" ) );	
			GtkTextView *titleView = GTK_TEXT_VIEW( my_lookup_widget( GTK_WIDGET( window ), "text_magick_title" ) );	
			GtkTextBuffer *titleEntry = gtk_text_view_get_buffer( titleView );

			GnomeFontPicker *fontPicker = GNOME_FONT_PICKER( my_lookup_widget( GTK_WIDGET( window ), "fontpicker_magick_title" ) );	
			GnomeColorPicker *colourPicker = GNOME_COLOR_PICKER( my_lookup_widget( GTK_WIDGET( window ), "colorpicker_magick_title" ) );

			// Obtain the gravity to apply to the convert command
			GtkWidget *active_item = gtk_menu_get_active( gravityMenu );
			gravity = GetGravity( g_list_index (GTK_MENU_SHELL ( gravityMenu )->children, active_item) ) ;

			// Obtain the position
			char *position = (char *)gtk_entry_get_text( positionEntry );
			sscanf( position, "%d,%d", &startx, &starty );
			position = (char *)gtk_entry_get_text( endPositionEntry );
			sscanf( position, "%d,%d", &endx, &endy );

			// Obtain the text
			g_free( text );
			GtkTextIter start;
			GtkTextIter end;

			gtk_text_buffer_get_start_iter( titleEntry, &start );
			gtk_text_buffer_get_end_iter( titleEntry, &end );
			text = (char *) gtk_text_buffer_get_text( titleEntry, &start, &end, TRUE );

			// Obtain the font
			font = (char *)gnome_font_picker_get_font_name( fontPicker );
			fontsize = atoi( strrchr( font, ' ' ) + 1 );

			// Obtain the colour
			guint8 r, g, b, a;
			gnome_color_picker_get_i8( colourPicker, &r, &g, &b, &a);
			sprintf( colour, "#%02X%02X%02X", r, g, b );
		}
};

class ImageOverlay : public GDKImageFilter, MagickImageLookup
{
	private:
		Convert convert;
		GtkWidget *window;
		char *image;
		char *gravity;
		char *compose;
		int startx;
		int starty;
		int endx;
		int endy;
		int startwidth;
		int startheight;
		int endwidth;
		int endheight;

	public:
		ImageOverlay( ) : window(NULL), image(NULL), gravity(NULL), compose(NULL), startx(0), starty(0),
								   endx(0), endy(0), startwidth(0), startheight(0), endwidth(0), endheight(0)
		{
			window = create_image_filter_overlay( );
		}

		virtual ~ImageOverlay( ) 
		{
		}

		char *GetDescription( ) const 
		{ 
			return "ImageMagick Overlay";
		}

		void FilterFrame( uint8_t *pixels, int width, int height, double position, double frame_delta ) 
		{ 
			char command[ 1024 ];
			int x = (int)( startx + ( endx - startx ) * position );
			int y = (int)( starty + ( endy - starty ) * position );
			int w = (int)( startwidth + ( endwidth - startwidth ) * position );
			int h = (int)( startheight + ( endheight - startwidth ) * position );

			convert.Clear();
			sprintf( command, "%s -draw \"image %s %d,%d %d,%d '%s'\"", 
					  		gravity, compose, x, y, w, h, image );
			convert.AddTransform( command );
			convert.Transform( pixels, width, height );
		}
		
		void AttachWidgets( GtkBin *bin ) 
		{
			gtk_widget_reparent( ( GTK_BIN( window ) )->child, GTK_WIDGET( bin ) );
		}

		void DetachWidgets( GtkBin *bin ) 
		{ 
			gtk_widget_reparent( ( GTK_BIN( bin ) )->child, GTK_WIDGET( window ) );
		}

		void InterpretWidgets( GtkBin *bin ) 
		{ 
			GtkEntry *fileEntry = GTK_ENTRY( gnome_file_entry_gtk_entry( GNOME_FILE_ENTRY( my_lookup_widget( window, "fileentry_magick_overlay_image" ) ) ) );
			GtkMenu *gravityMenu = GTK_MENU( gtk_option_menu_get_menu( GTK_OPTION_MENU( my_lookup_widget( window, "optionmenu_magick_overlay_gravity" ) ) ) );
			GtkMenu *composeMenu = GTK_MENU( gtk_option_menu_get_menu( GTK_OPTION_MENU( my_lookup_widget( window, "optionmenu_magick_overlay_compose" ) ) ) );
			GtkEntry *positionEntry = GTK_ENTRY( my_lookup_widget( window, "entry_magick_overlay_start" ) );	
			GtkEntry *endPositionEntry = GTK_ENTRY( my_lookup_widget( window, "entry_magick_overlay_end" ) );	
			GtkEntry *startSizeEntry = GTK_ENTRY( my_lookup_widget( window, "entry_magick_overlay_start_size" ) );	
			GtkEntry *endSizeEntry = GTK_ENTRY( my_lookup_widget( window, "entry_magick_overlay_end_size" ) );	

			// Obtain the gravity to apply to the convert command
			GtkWidget *active_item = gtk_menu_get_active( gravityMenu );
			gravity = GetGravity( g_list_index (GTK_MENU_SHELL ( gravityMenu )->children, active_item) );

			// Obtain the compose to apply to the convert command
			active_item = gtk_menu_get_active( composeMenu );
			compose = GetCompose( g_list_index (GTK_MENU_SHELL ( composeMenu )->children, active_item) );

			// Obtain the position
			char *position = (char *)gtk_entry_get_text( positionEntry );
			sscanf( position, "%d,%d", &startx, &starty );
			position = (char *)gtk_entry_get_text( endPositionEntry );
			sscanf( position, "%d,%d", &endx, &endy );

			// Obtain the size
			char *size = (char *)gtk_entry_get_text( startSizeEntry );
			sscanf( size, "%d,%d", &startwidth, &startheight );
			size = (char *)gtk_entry_get_text( endSizeEntry );
			sscanf( size, "%d,%d", &endwidth, &endheight );

			// Obtain the file name
			image = (char*)gtk_entry_get_text( fileEntry );
		}
};

/** Chroma key on blue....
    It is assumed that the blue stuff is in mesh, io contains the background,
    mesh should contain the foreground
*/

class ImageTransitionChromaKeyBlue : public ImageTransition
{
	public:
		char *GetDescription( ) const
  		{
    		return "Chroma Key"; 
  		}

  		void GetFrame( uint8_t *io, uint8_t *mesh, int width, int height, double position, double frame_delta, bool reverse )
  		{
    		uint8_t *k = mesh;
    		uint8_t *p = io;
    		uint8_t kr, kg, kb;
			int max = width * height * 3;
    		while ( p < ( io + max ) ) 
      		{
        		kr = *p;
        		kg = *( p + 1 );
        		kb = *( p + 2 );
        		if (kb >= 240 && kr <= 5 && kg <= 5) 
				{
          			*p ++ = *k ++; 
          			*p ++ = *k ++; 
          			*p ++ = *k ++; 
        		} 
				else 
				{
          			p += 3;
          			k += 3;
        		}
      		}
  		}
};

static gboolean on_scale_release_event( GtkWidget *widget, GdkEventButton *event, gpointer user_data);

class LineDraw : public GDKImageFilter, public GtkKinoContainer, private KeyFrameControllerClient, private PreviewAreaClient,
				 public SelectionNotification
{
	private:
		GtkWidget *window;
		InterlaceOptions *interlace;
		KeyFrameController *controller;
		PreviewArea *area;
		uint8_t *temp;
		double scale;
		bool interlace_on;
		bool interlace_first_field;
		int y_scatter;
		int x_scatter;
		double mix;

	public:
		LineDraw( ) : y_scatter( 2 ), x_scatter( 2 ), mix( 0 )
		{
			window = create_window_line_draw( );
			interlace = CreateWidgetInterlaceOptions( );
			AttachGtkWidget( "frame_interlace", interlace );
			scale = 2;

			GtkWidget *widget = my_lookup_widget( window, "hscale" );
			gtk_signal_connect( GTK_OBJECT( widget ), "button_release_event", GTK_SIGNAL_FUNC( on_scale_release_event ), this );
			widget = my_lookup_widget( window, "hscale_x_scatter" );
			gtk_signal_connect( GTK_OBJECT( widget ), "button_release_event", GTK_SIGNAL_FUNC( on_scale_release_event ), this );
			widget = my_lookup_widget( window, "hscale_y_scatter" );
			gtk_signal_connect( GTK_OBJECT( widget ), "button_release_event", GTK_SIGNAL_FUNC( on_scale_release_event ), this );
			widget = my_lookup_widget( window, "vscale_mix" );
			gtk_signal_connect( GTK_OBJECT( widget ), "button_release_event", GTK_SIGNAL_FUNC( on_scale_release_event ), this );
		}

		virtual ~LineDraw( ) 
		{
			gtk_widget_destroy( window );
		}

		char *GetDescription( ) const 
		{ 
			return "Charcoal";
		}

		GtkWidget *GetGtkWidget()
		{
			return window;
		}

		int GetPix( uint8_t *pixels, int width, int height, int x, int y )
		{
			if ( x < 0 || x >= width || y < 0 || y >= height )
			{
				return 0;
			}
			else
			{
				uint8_t *pixel = pixels + y * width * 3 + x * 3;
				//kino::basic_rgb<uint8_t>* pixel = reinterpret_cast<kino::basic_rgb<uint8_t>*>(pixels + y * width * 3 + x * 3 );
				//return pixel->red;
				return *pixel;
			}
		}

		void FilterFrame( uint8_t *pixels, int width, int height, double position, double frame_delta ) 
		{
			uint8_t *copy = NULL;
			if ( mix != 0 )
			{
				copy = new uint8_t[ width * height * 3 ];
				memcpy( copy, pixels, width * height * 3 );
			}

			for ( int y = ( interlace_on && interlace_first_field ) ? 0 : 1; y < height; y += ( interlace_on ? 2 : 1 ) ) 
			{
				uint8_t *p = pixels + y * width * 3;

				for ( int x = 0; x < width; x ++ )
				{
					uint8_t r = *( p );
					uint8_t g = *( p + 1 );
					uint8_t b = *( p + 2 );
					r = (uint8_t)( 0.299 * r + 0.587 * g + 0.114 * b );
					*p ++ = r;
					*p ++ = r;
					*p ++ = r;
				}

				if ( interlace_on && interlace_first_field )
					memcpy( p, p - width * 3, width * 3 );
				else if ( interlace_on && !interlace_first_field )
					memcpy( p - width * 6, p - width * 3, width * 3 );
			}

			temp = new uint8_t[ width * height * 3 ];
			uint8_t *dest = temp;
			uint8_t *hack = copy;

			for ( int y = 0; y < height; y ++ )
			{
				for ( int x = 0; x < width; x ++ )
				{
					int pix[ 3 ][ 3 ];

					pix[ 0 ][ 0 ] = GetPix( pixels, width, height, x - x_scatter, y - y_scatter );
					pix[ 0 ][ 1 ] = GetPix( pixels, width, height, x            , y - y_scatter );
					pix[ 0 ][ 2 ] = GetPix( pixels, width, height, x + x_scatter, y - y_scatter );
					pix[ 1 ][ 0 ] = GetPix( pixels, width, height, x - x_scatter, y             );
					pix[ 1 ][ 2 ] = GetPix( pixels, width, height, x + x_scatter, y             );
					pix[ 2 ][ 0 ] = GetPix( pixels, width, height, x - x_scatter, y + y_scatter );
					pix[ 2 ][ 1 ] = GetPix( pixels, width, height, x            , y + y_scatter );
					pix[ 2 ][ 2 ] = GetPix( pixels, width, height, x + x_scatter, y + y_scatter );

		      		double sum1 = (pix[2][0] - pix[0][0]) + 2*(pix[2][1] - pix[0][1]) + (pix[2][2] - pix[2][0]);
		      		double sum2 = (pix[0][2] - pix[0][0]) + 2*(pix[1][2] - pix[1][0]) + (pix[2][2] - pix[2][0]);
		      		double sum = sqrt( (long) sum1 * sum1 + (long) sum2 * sum2 );

					sum = (int)( sum * scale );
					if ( sum > 255 )
						sum = 255;

					if ( copy == NULL )
					{
						*dest ++ = (uint8_t)( 255 - sum );
						*dest ++ = (uint8_t)( 255 - sum );
						*dest ++ = (uint8_t)( 255 - sum );
					}
					else
					{
						*dest ++ = (uint8_t)( ( 255 - sum ) * ( 1 - mix ) + ( *hack ++ ) * mix );
						*dest ++ = (uint8_t)( ( 255 - sum ) * ( 1 - mix ) + ( *hack ++ ) * mix );
						*dest ++ = (uint8_t)( ( 255 - sum ) * ( 1 - mix ) + ( *hack ++ ) * mix );
					}
				}
			}

			memcpy( pixels, temp, width * height * 3 );
			delete temp;
			delete copy;
		}
		
		void AttachWidgets( GtkBin *bin ) 
		{
			// Put the key frame controller widget in place
			controller = CreateWidgetKeyFrameController( this );
			AttachGtkWidget( "frame_key_frame_control", controller );

			// Put the preview widget in place
			area = CreateWidgetPreviewArea( this );
			AttachGtkWidget( "frame_preview", area );

			gtk_widget_reparent( ( GTK_BIN( window ) )->child, GTK_WIDGET( bin ) );
		}

		void DetachWidgets( GtkBin *bin ) 
		{ 
			gtk_widget_reparent( ( GTK_BIN( bin ) )->child, GTK_WIDGET( window ) );
			DetachGtkWidget( "frame_key_frame_control", controller );
			delete controller;
			DetachGtkWidget( "frame_preview", area );
			delete area;
		}

		void InterpretWidgets( GtkBin *bin ) 
		{
			GtkRange *range = GTK_RANGE( my_lookup_widget( window, "hscale" ) );
			GtkAdjustment *adjust = gtk_range_get_adjustment( range );
			scale = adjust->value; 
			range = GTK_RANGE( my_lookup_widget( window, "hscale_x_scatter" ) );
			adjust = gtk_range_get_adjustment( range );
			x_scatter = (int)adjust->value; 
			range = GTK_RANGE( my_lookup_widget( window, "hscale_y_scatter" ) );
			adjust = gtk_range_get_adjustment( range );
			y_scatter = (int)adjust->value; 
			range = GTK_RANGE( my_lookup_widget( window, "vscale_mix" ) );
			adjust = gtk_range_get_adjustment( range );
			mix = adjust->value; 
			interlace_on = interlace->IsInterlaceOn();
			interlace_first_field = interlace->IsFirstFieldOn();
		}

		void Refresh( double position )
		{
			InterpretWidgets( GTK_BIN( window ) );
			controller->ShowCurrentStatus( position, LOCKED_KEY, false, false );

			int preview_width = 360;
			int preview_height = 288;
			uint8_t *preview = new uint8_t[ preview_width * preview_height * 3 ];
			SelectedFrames &selected = GetSelectedFramesForFX( );
			if ( selected.GetNumInputFrames() > 0 )
				selected.GetImageA( controller->GetCurrentPosition( ), preview, preview_width, preview_height );
			else
				memset( preview, 255, preview_width * preview_height * 3 );

			FilterFrame( preview, preview_width, preview_height, controller->GetCurrentPosition( ), 0.01 );
			area->ShowImage( 50, 50, preview, preview_width, preview_height );
			delete preview;
		}

		void Refresh( )
		{
			Refresh( controller->GetCurrentPosition( ) );
		}

		void OnControllerPositionChanged( double position )
		{
			Refresh( position );
		}

		void OnPreviewAreaRefresh( PreviewArea *area )
		{
			Refresh( );
		}

		void OnSelectionChange( )
		{
			Refresh( );
		}
};

static gboolean on_scale_release_event( GtkWidget *widget, GdkEventButton *event, gpointer user_data)
{
	( *( LineDraw * )user_data ).Refresh( );
	return FALSE;
}

class TweenieEntry : public TransitionTimeEntry < TweenieEntry >, protected PixbufUtils
{
	public:
		double x;
		double y;
		double width;
		double height;
		double angle;
		double fade;
		double shear;

		bool scaled;
		bool interlace_on;
		bool interlace_first_field;
		uint8_t *luma;
		int luma_width;
		int luma_height;
		double luma_softness;
		double frame_delta;

		TweenieEntry() : x(50), y(50), width(50), height(50), angle(0), fade(0), shear(0), luma(NULL)
		{
			SetEditable( false );
		}

		TweenieEntry( double position ) : x(50), y(50), width(50), height(50), angle(0), fade(0), shear(0), luma(NULL)
		{
			SetEditable( false );
			SetPosition( position );
		}

		TweenieEntry( double position, TweenieEntry entry )
		{
			SetPosition( position );
			SetEditable( false );
			x = entry.x;
			y = entry.y;
			width = entry.width;
			height = entry.height;
			angle = entry.angle;
			fade = entry.fade;
			shear = entry.shear;
		}

		TweenieEntry *Get( double position, TweenieEntry *ante )
		{
			TweenieEntry *current = new TweenieEntry( );
			double r = ( position - GetPosition( ) ) / ( ante->GetPosition() - GetPosition( ) );
			current->SetEditable( false );
			current->SetPosition( position );
			current->x = x + ( ante->x - x ) * r;
			current->y = y + ( ante->y - y ) * r;
			current->width = width + ( ( ante->width - width ) * r );
			current->height = height + ( ( ante->height - height ) * r );
			current->angle = angle + ( ( ante->angle - angle ) * r );
			current->fade = fade + ( ( ante->fade - fade ) * r );
			current->shear = shear + ( ( ante->shear - shear ) * r );
			return current;
		}

		void Composite( uint8_t *dest, int width, int height, uint8_t *src, double x, double y, int src_width, int src_height, double angle, bool scaled, double merge )
		{
			// Set up the affine transform (rescaling could be done here, but quality sucks)
			AffineTransform affine;
			affine.Shear( shear / 100 );
			affine.Rotate( angle );

			int origin_x = (int)( width * x / 100 );
			int origin_y = (int)( height * y / 100 );

			// Rescale the luma if it exists
			uint8_t *scaled_luma = NULL;

			if ( luma != NULL )
			{
				SetScale( SCALE_FULL );
				GdkPixbuf *input = gdk_pixbuf_new_from_data( luma, GDK_COLORSPACE_RGB, FALSE, 8, luma_width, luma_height, ( luma_width * 3 ), NULL, NULL );
				scaled_luma = new uint8_t[ src_width * src_height * 3 ];
				ScalePixbuf( input, scaled_luma, src_width, src_height );
				gdk_pixbuf_unref( input );
			}
			else
			{
				scaled_luma = new uint8_t[ src_width * src_height * 3 ];
				memset( scaled_luma, 0, src_width * src_height * 3 );
			}

			// could probably calculate these a bit more accurately...
			int max_height = (int)sqrt( 2 * ( src_height > src_width ? src_height * src_height : src_width * src_width ) ) / 2;
			int max_width = max_height;

			//if ( shear != 0 )
			{
				max_height = height / 2;
				max_width = width / 2;
			}

			for ( int iy = 0 - max_height; iy < max_height; iy ++ )
			{
				int real_y = iy + origin_y;

				if ( real_y >= 0 && real_y < height )
				{
					uint8_t *dest_row = dest + real_y * width * 3 + origin_x * 3 - max_width * 3;

					for ( int ix = 0 - max_width; ix < max_width; ix ++ )
					{
						int map_x = (int)( src_width / 2 + affine.MapX( ix, iy ) );
						int map_y = (int)( src_height / 2 + affine.MapY( ix, iy ) );

						int real_x = origin_x + ix;

						if ( real_x >= 0 && real_x < width && map_x >= 0 && map_y >= 0 && map_x < src_width && map_y < src_height )
						{
							uint8_t *pixel;

							if ( scaled )
								pixel = src + map_y * src_width * 3 + map_x * 3;
							else
								pixel = src + real_y * width * 3 + real_x * 3;

							const double adjusted_position = kino::lerp(0.0, 1.0 + luma_softness, merge );
							uint8_t *luma_pixel = scaled_luma + map_y * src_width * 3 + map_x * 3;
							double sample = 1 - (double)*luma_pixel / 255;
							const double mix = kino::smoothstep( sample, sample + luma_softness, adjusted_position );

							*dest_row = (uint8_t)( *dest_row * mix + ( *pixel ++ * ( 1 - mix ) ) );
							dest_row ++;
							*dest_row = (uint8_t)( *dest_row * mix + ( *pixel ++ * ( 1 - mix ) ) );
							dest_row ++;
							*dest_row = (uint8_t)( *dest_row * mix + ( *pixel ++ * ( 1 - mix ) ) );
							dest_row ++;
						}
						else
						{
							dest_row += 3;
						}
					}
				}
			}

			delete scaled_luma;
		}

		void RenderPreview( uint8_t *background, uint8_t *foreground, int width, int height )
		{
			RenderFinal( background, foreground, width, height );
		}

		void RenderFinal( uint8_t *background, uint8_t *foreground, int width, int height )
		{
			GdkPixbuf *i1 = gdk_pixbuf_new_from_data( foreground, GDK_COLORSPACE_RGB, FALSE, 8, width, height, ( 3 * width ), NULL, NULL );
			int new_width = (int)( width * this->width / 100 );
			int new_height = (int)( height * this->height / 100 );

			if ( new_width > 1 && new_height > 1 )
			{
				SetScale( SCALE_NONE );
				if ( scaled )
				{
					GdkPixbuf *im = gdk_pixbuf_scale_simple( i1, new_width, new_height, GDK_INTERP_HYPER );
					uint8_t *temp_image = new uint8_t[ new_width * new_height * 3 ];
					ScalePixbuf( im, temp_image, new_width, new_height );
					Composite( background, width, height, temp_image, this->x, this->y, new_width, new_height, this->angle, true, this->fade / 100 );
					delete temp_image;
					gdk_pixbuf_unref( im );
				}
				else
				{
					uint8_t *temp_image = new uint8_t[ new_width * new_height * 3 ];
					ScalePixbuf( i1, temp_image, new_width, new_height );
					Composite( background, width, height, temp_image, this->x, this->y, new_width, new_height, this->angle, true, this->fade / 100 );
					delete temp_image;
				}
			}

			gdk_pixbuf_unref( i1 );
		}
};

static void on_key_frame_changed( GtkWidget *some_widget, gpointer user_data );
static void on_rescale_changed( GtkWidget *some_widget, gpointer user_data );
static gboolean on_predefine_changed( GtkWidget *some_widget, void *event, gpointer user_data );

class Tweenies : public GDKImageTransition, public GtkKinoContainer, private KeyFrameControllerClient, 
				 private PairPickerClient, private LumaPickerClient, private PreviewAreaClient, 
				 public SelectionNotification, protected PixbufUtils
{
	private:
		GtkWidget *window;
		KeyFrameController *controller;
		PreviewArea *area;
		bool gui_active;
		string luma;
		uint8_t *luma_image;
		double luma_softness;
		uint8_t *preview;
		uint8_t *overlay;
		static const int preview_width = 180;
		static const int preview_height = 144;
		int last_predefine;
		bool scaled;
		bool reversed;

		int luma_width;
		int luma_height;

		PairPicker *coord_picker;
		PairPicker *size_picker;
		LumaPicker *luma_picker;

		InterlaceOptions *interlace;
		bool interlace_on;
		bool interlace_first_field;

		TimeMap < TweenieEntry > time_map;

	public:
		Tweenies( ) : gui_active( true ), luma( "" ), luma_image( NULL ), luma_softness( 0.20 ), last_predefine( 0 ), scaled( true ), reversed( false )
		{
			// Define the widget
			window = create_window_tweenies( );

			// Define the default animation
			TweenieEntry *entry = time_map.SetEditable( 0, true );
			entry->x = 50;
			entry->y = 50;
			entry->width = 1;
			entry->height = 1;
			entry->fade = 100;
			time_map.FinishedWith( entry );

			entry = time_map.SetEditable( 0.99, true );
			entry->x = 50;
			entry->y = 50;
			entry->width = 100;
			entry->height = 100;
			entry->fade = 0;
			time_map.FinishedWith( entry );

			// Create the 
			preview = new uint8_t[ preview_width * preview_height * 3 ];
			overlay = new uint8_t[ preview_width * preview_height * 3 ];
			memset( overlay, 0, preview_width * preview_height * 3 );
		}

		virtual ~Tweenies( ) 
		{
			delete luma_image;
			delete preview;
			delete overlay;
			gtk_widget_destroy( window );
		}

		char *GetDescription( ) const 
		{ 
			return "Tweenies";
		}

		GtkWidget *GetGtkWidget()
		{
			return window;
		}

		void GetFrame( uint8_t *io, uint8_t *mesh, int width, int height, double position, double frame_delta, bool reverse ) 
		{
			uint8_t *in = mesh;
			uint8_t *out = io;

			// Reverse for fly-out
			if ( reverse )
			{
				in = io;
				out = mesh;
				if ( !reversed )
					position = 0.99 - position;
			}

			TweenieEntry *entry = time_map.Get( position );
			entry->luma = luma_image;
			entry->luma_width = luma_width;
			entry->luma_height = luma_height;
			entry->luma_softness = luma_softness;
			entry->frame_delta = frame_delta;
			entry->scaled = scaled;
			entry->RenderFinal( out, in, width, height );
			time_map.FinishedWith( entry );

			// Correct for fly out
			if ( reverse )
				memcpy( io, mesh, width * height * 3 );
		}
		
		void AttachWidgets( GtkBin *bin ) 
		{
			// Put the key frame controller widget in place
			controller = CreateWidgetKeyFrameController( this );
			AttachGtkWidget( "frame_key_frame_control", controller );

			// Put the preview widget in place
			area = CreateWidgetPreviewArea( this );
			AttachGtkWidget( "frame_preview", area );

			// Attach the coordinate picker
			coord_picker = CreateWidgetPairPicker( this );
			AttachGtkWidget( "frame_coords", coord_picker );

			// Attach the size picker
			size_picker = CreateWidgetPairPicker( this );
			size_picker->SetFirstMinMax( 0, 500 );
			size_picker->SetSecondMinMax( 500, 0 );
			AttachGtkWidget( "frame_size", size_picker );

			// Attach the size picker
			luma_picker = CreateWidgetLumaPicker( this );
			AttachGtkWidget( "frame_luma", luma_picker );

			// Attach the interlace options
			interlace = CreateWidgetInterlaceOptions( );
			AttachGtkWidget( "frame_interlace", interlace );

			// define the callbacks...
			GtkWidget *widget = my_lookup_widget( window, "spinbutton_angle" );
			gtk_signal_connect( GTK_OBJECT( widget ), "changed", GTK_SIGNAL_FUNC( on_key_frame_changed ), this );
			widget = my_lookup_widget( window, "spinbutton_fade" );
			gtk_signal_connect( GTK_OBJECT( widget ), "changed", GTK_SIGNAL_FUNC( on_key_frame_changed ), this );
			widget = my_lookup_widget( window, "spinbutton_shear" );
			gtk_signal_connect( GTK_OBJECT( widget ), "changed", GTK_SIGNAL_FUNC( on_key_frame_changed ), this );
			widget = my_lookup_widget( window, "checkbutton_rescale" );
			gtk_signal_connect( GTK_OBJECT( widget ), "toggled", GTK_SIGNAL_FUNC( on_rescale_changed ), this );
			widget = my_lookup_widget( window, "optionmenu_predefines" );
			gtk_signal_connect( GTK_OBJECT( widget ), "event", GTK_SIGNAL_FUNC( on_predefine_changed ), this );
	
			gtk_widget_reparent( ( GTK_BIN( window ) )->child, GTK_WIDGET( bin ) );
			Refresh( );
		}

		void DetachWidgets( GtkBin *bin ) 
		{ 
			gtk_widget_reparent( ( GTK_BIN( bin ) )->child, GTK_WIDGET( window ) );
			DetachGtkWidget( "frame_key_frame_control", controller );

			DetachGtkWidget( "frame_preview", area );
			delete area;
			DetachGtkWidget( "frame_coords", coord_picker );
			delete coord_picker;
			DetachGtkWidget( "frame_size", size_picker );
			delete size_picker;
			DetachGtkWidget( "frame_luma", luma_picker );
			delete luma_picker;
			DetachGtkWidget( "frame_interlace", interlace );
			delete interlace;
		}

		void InterpretWidgets( GtkBin *bin )
		{
		}

		void Refresh( TweenieEntry *entry )
		{
			SelectedFrames &selected = GetSelectedFramesForFX( );
			double frame_position = entry->GetPosition( );

			entry->luma = luma_image;
			entry->luma_width = luma_width;
			entry->luma_height = luma_height;
			entry->luma_softness = luma_softness;
			entry->scaled = scaled;

			if ( selected.GetNumInputFrames() > 0 )
			{
				entry->frame_delta = selected.GetFrameDelta( );
				if ( !selected.IsEffectReversed() )
				{
					selected.GetImageA( frame_position, preview, preview_width, preview_height );
					selected.GetImageB( frame_position, overlay, preview_width, preview_height );
				}
				else
				{
					selected.GetImageB( frame_position, preview, preview_width, preview_height );
					selected.GetImageA( frame_position, overlay, preview_width, preview_height );
				}
				entry->RenderFinal( preview, overlay, preview_width, preview_height );
			}
			else
			{
				entry->frame_delta = 1 / 100;
				memset( preview, 255, preview_width * preview_height * 3 );
				memset( overlay, 0, preview_width * preview_height * 3 );
				entry->RenderPreview( preview, overlay, preview_width, preview_height );
			}

			area->ShowImage( (int)entry->x, (int)entry->y, preview, preview_width, preview_height );
		}

		void Refresh( bool with_control_sync = false )
		{
			double position = controller->GetCurrentPosition( );
			SelectedFrames &selected = GetSelectedFramesForFX( );
			if ( selected.IsEffectReversed( ) != reversed )
			{
				time_map.Invert( );
				reversed = selected.IsEffectReversed( );
			}
			TweenieEntry *entry = time_map.Get( position );
			Refresh( entry );
			if ( with_control_sync )
				ChangeController( entry );
			time_map.FinishedWith( entry );
		}

		void RescaleChanged( )
		{
			scaled = !scaled;
			Refresh( );
		}

		void OnSelectionChange( )
		{
			Refresh( );
		}

		void OnPreviewAreaRefresh( PreviewArea *area )
		{
			Refresh( true );
		}

		void ChangeController( TweenieEntry *entry )
		{
			frame_type type = FRAME;
			if ( entry->GetPosition() == 0 )
				type = LOCKED_KEY;
			else if ( entry->IsEditable() )
				type = KEY;

			controller->ShowCurrentStatus( entry->GetPosition(), type, entry->GetPosition() > time_map.GetFirst(), entry->GetPosition() < time_map.GetLast() );

			gui_active = false;
			GtkWidget *widget = my_lookup_widget( window, "spinbutton_angle" );
			gtk_spin_button_set_value( GTK_SPIN_BUTTON( widget ), entry->angle );
			widget = my_lookup_widget( window, "spinbutton_fade" );
			gtk_spin_button_set_value( GTK_SPIN_BUTTON( widget ), entry->fade );
			widget = my_lookup_widget( window, "spinbutton_shear" );
			gtk_spin_button_set_value( GTK_SPIN_BUTTON( widget ), entry->shear );
			widget = my_lookup_widget( window, "frame_key_input" );
			gtk_widget_set_sensitive( widget, entry->IsEditable() );
			gui_active = true;

			coord_picker->SetValues( entry->x, entry->y );
			size_picker->SetValues( entry->width, entry->height );
		}

		void OnControllerKeyChanged( double position, bool value )
		{
			TweenieEntry *entry = time_map.SetEditable( position, value );
			time_map.FinishedWith( entry );
			Refresh( true );
		}

		void OnControllerPrevKey( double position )
		{
			TweenieEntry *entry = time_map.GotoPreviousKey( position );
			ChangeController( entry );
			time_map.FinishedWith( entry );
			Refresh( );
		}

		void OnControllerNextKey( double position )
		{
			TweenieEntry *entry = time_map.GotoNextKey( position );
			ChangeController( entry );
			time_map.FinishedWith( entry );
			Refresh( );
		}

		void OnControllerPositionChanged( double position )
		{
			TweenieEntry *entry = time_map.Get( position );
			ChangeController( entry );
			time_map.FinishedWith( entry );
			Refresh( );
		}

		// Definition changed - this method is responsible for updating the current frame in relation to modifications made
		void KeyFrameChanged( )
		{
			if ( gui_active )
			{
				double position = controller->GetCurrentPosition( );
				TweenieEntry *entry = time_map.Get( position );

				if ( entry->IsEditable() )
				{
					GtkWidget *widget = my_lookup_widget( window, "spinbutton_angle" );
					entry->angle = atoi( gtk_entry_get_text( GTK_ENTRY( widget ) ) );
					widget = my_lookup_widget( window, "spinbutton_fade" );
					entry->fade = atoi( gtk_entry_get_text( GTK_ENTRY( widget ) ) );
					widget = my_lookup_widget( window, "spinbutton_shear" );
					entry->shear = atoi( gtk_entry_get_text( GTK_ENTRY( widget ) ) );
				}

				time_map.FinishedWith( entry );
				Refresh( );
			}
		}

		void OnPairPickerChangeValue( PairPicker *picker, double first, double second )
		{
			double position = controller->GetCurrentPosition( );
			TweenieEntry *entry = time_map.Get( position );

			if ( picker == coord_picker )
			{
				entry->x = first;
				entry->y = second;
			}
			else if ( picker == size_picker )
			{
				entry->width = first;
				entry->height = second;
			}

			time_map.FinishedWith( entry );
			Refresh( );
		}

		void OnLumaPickerChange( LumaPicker *picker, char *file, double softness )
		{
			luma = file == NULL ? "" : file;
			luma_softness = softness;
			
			delete luma_image;
			luma_image = NULL;

			if ( file != NULL )
			{
				GError *err = NULL;
				GdkPixbuf *pix = gdk_pixbuf_new_from_file( file, &err );
				if ( pix != NULL )
				{
					luma_width = gdk_pixbuf_get_width( pix );
					luma_height = gdk_pixbuf_get_height( pix );
					luma_image = new uint8_t[ luma_width * luma_height * 3 ];
					ScalePixbuf( pix, luma_image, luma_width, luma_height );
					gdk_pixbuf_unref( pix );
				}
			}

			Refresh();
		}

		void OnPredefineChange( )
		{
	       	GtkMenu *menu = GTK_MENU( gtk_option_menu_get_menu( GTK_OPTION_MENU( my_lookup_widget( window, "optionmenu_predefines" ) ) ) );
        	GtkWidget *item = gtk_menu_get_active( menu );
        	int direction = g_list_index( GTK_MENU_SHELL ( menu )->children, item );

			// Hack - can't find a better way to determine changes in the option menu (grrr...)
			if ( direction == last_predefine )
				return;
			last_predefine = direction;

			// Wipe any previous definition
			time_map.Clear( );

			// Obtain first and last entries as keys
			TweenieEntry *first_entry = time_map.SetEditable( 0, true );
			TweenieEntry *last_entry = time_map.SetEditable( 0.99, true );

			// Define the default last entry
			last_entry->x = 50;
			last_entry->y = 50;
			last_entry->width = 100;
			last_entry->height = 100;
			last_entry->fade = 0;

			// Define the first according to the selected direction
			if ( direction == 0 )
			{
				first_entry->x = 50;
				first_entry->y = 50;
				first_entry->width = 1;
				first_entry->height = 1;
				first_entry->fade = 100;
			}
			else if ( direction == 1 )
			{
				first_entry->x = 0;
				first_entry->y = 50;
				first_entry->width = 1;
				first_entry->height = 100;
				first_entry->fade = 100;
			}
			else if ( direction == 2 )
			{
				first_entry->x = 100;
				first_entry->y = 50;
				first_entry->width = 1;
				first_entry->height = 100;
				first_entry->fade = 100;
			}
			else if ( direction == 3 )
			{
				first_entry->x = 50;
				first_entry->y = 50;
				first_entry->width = 1;
				first_entry->height = 100;
				first_entry->fade = 100;
			}
			else if ( direction == 4 )
			{
				first_entry->x = 50;
				first_entry->y = 0;
				first_entry->width = 100;
				first_entry->height = 1;
				first_entry->fade = 100;
			}
			else if ( direction == 5 )
			{
				first_entry->x = 50;
				first_entry->y = 100;
				first_entry->width = 100;
				first_entry->height = 1;
				first_entry->fade = 100;
			}
			else if ( direction == 6 )
			{
				first_entry->x = 50;
				first_entry->y = 50;
				first_entry->width = 100;
				first_entry->height = 1;
				first_entry->fade = 100;
			}
			else if ( direction == 7 )
			{
				first_entry->x = 0;
				first_entry->y = 0;
				first_entry->width = 1;
				first_entry->height = 1;
				first_entry->fade = 100;
			}
			else if ( direction == 8 )
			{
				first_entry->x = 100;
				first_entry->y = 0;
				first_entry->width = 1;
				first_entry->height = 1;
				first_entry->fade = 100;
			}
			else if ( direction == 9 )
			{
				first_entry->x = 0;
				first_entry->y = 100;
				first_entry->width = 1;
				first_entry->height = 1;
				first_entry->fade = 100;
			}
			else if ( direction == 10 )
			{
				first_entry->x = 100;
				first_entry->y = 100;
				first_entry->width = 1;
				first_entry->height = 1;
				first_entry->fade = 100;
			}

			time_map.FinishedWith( first_entry );
			time_map.FinishedWith( last_entry );

			reversed = false;

			Refresh( true );
		}
};

static gboolean on_predefine_changed( GtkWidget *some_widget, void *event, gpointer user_data )
{
	( (Tweenies *)user_data )->OnPredefineChange( );
	return FALSE;
}

static void on_key_frame_changed( GtkWidget *some_widget, gpointer user_data )
{
	( (Tweenies *)user_data )->KeyFrameChanged( );
}

static void on_rescale_changed( GtkWidget *some_widget, gpointer user_data )
{
	( (Tweenies *)user_data )->RescaleChanged( );
}

class PanZoomEntry : public FilterTimeEntry < PanZoomEntry >, public PixbufUtils
{
	public:
		double x;
		double y;
		double width;
		double height;
		bool interlace_on;
		bool interlace_first_field;

		PanZoomEntry() : x(50), y(50), width(50), height(50)
		{
			SetEditable( false );
		}

		PanZoomEntry( double position ) : x(50), y(50), width(50), height(50)
		{
			SetEditable( false );
			SetPosition( position );
		}

		PanZoomEntry( double position, PanZoomEntry &entry )
		{
			SetPosition( position );
			SetEditable( false );
			x = entry.x;
			y = entry.y;
			width = entry.width;
			height = entry.height;
		}

		PanZoomEntry *Get( double position, PanZoomEntry *ante )
		{
			PanZoomEntry *current = new PanZoomEntry( );
			double r = ( position - GetPosition( ) ) / ( ante->GetPosition() - GetPosition( ) );
			current->SetEditable( false );
			current->SetPosition( position );
			current->x = x + ( ante->x - x ) * r;
			current->y = y + ( ante->y - y ) * r;
			current->width = width + ( ( ante->width - width ) * r );
			current->height = height + ( ( ante->height - height ) * r );
			return current;
		}

		void RenderPreview( uint8_t *image, int width, int height )
		{
			memset( image, 255, width * height * 3 );

			int x = (int)( ( this->x * width ) / 100 );
			int y = (int)( ( this->y * height ) / 100 );
			int cut_width = (int)( ( this->width * width ) / 100 );
			int cut_height = (int)( ( this->height * height ) / 100 );
			int lower_x = x - cut_width / 2;
			int lower_y = y - cut_height / 2;
			int upper_x = x + cut_width / 2;
			int upper_y = y + cut_height / 2;

			if ( lower_x < 0 )
			{
				cut_width = cut_width - ( 0 - lower_x );
				lower_x = 0;
			}
			if ( lower_y < 0 )
			{
				cut_height = cut_height - ( 0 - lower_y );
				lower_y = 0;
			}
			if ( upper_x > width )
			{
				cut_width = cut_width - ( upper_x - width );
				upper_x = width;
			}
			if ( upper_y > height )
			{
				cut_height = cut_height - ( upper_y - height );
				upper_y = height;
			}

			for ( y = lower_y; y < upper_y; y ++ )
				memset( image + y * width * 3 + lower_x * 3, 0, ( upper_x - lower_x ) * 3 );
		}

		void RenderFinal( uint8_t *image, int width, int height )
		{
			int x = (int)( ( this->x * width ) / 100 );
			int y = (int)( ( this->y * height ) / 100 );
			int cut_width = (int)( ( this->width * width ) / 100 );
			int cut_height = (int)( ( this->height * height ) / 100 );
			int lower_x = x - cut_width / 2;
			int lower_y = y - cut_height / 2;
			int upper_x = x + cut_width / 2;
			int upper_y = y + cut_height / 2;

			if ( lower_x < 0 )
			{
				cut_width = cut_width - ( 0 - lower_x );
				lower_x = 0;
			}
			if ( lower_y < 0 )
			{
				cut_height = cut_height - ( 0 - lower_y );
				lower_y = 0;
			}
			if ( upper_x > width )
			{
				cut_width = cut_width - ( upper_x - width );
				upper_x = width;
			}
			if ( upper_y > height )
			{
				cut_height = cut_height - ( upper_y - height );
				upper_y = height;
			}

			if ( interlace_on )
			{
				for ( int row = interlace_first_field ? 0 : 1; row < height; row += 2 )
				{
					if ( interlace_first_field )
						memcpy( image + ( row + 1 ) * width * 3, image + row * width * 3, width * 3 );
					else
						memcpy( image + ( row - 1 ) * width * 3, image + row * width * 3, width * 3 );
				}
			}

			SetScale( SCALE_FULL );
			ZoomAndScaleRGB( image, width, height, upper_x, upper_y, lower_x, lower_y );
		}
};

static void on_reverse_toggled( GtkWidget *some_widget, gpointer user_data );

class PanZoom : public GDKImageFilter, public GtkKinoContainer, private KeyFrameControllerClient, private PairPickerClient, 
				private MultiMapClient, private PreviewAreaClient, public SelectionNotification
{
	private:
		GtkWidget *window;
		KeyFrameController *controller;
		PreviewArea *area;
		PairPicker *coord_picker;
		PairPicker *size_picker;
		MultiMap *multi;
		uint8_t *preview;
		static const int preview_width = 180;
		static const int preview_height = 144;
		bool gui_active;
		bool reverse;
		InterlaceOptions *interlace;
		bool interlace_on;
		bool interlace_first_field;

		TimeMap < PanZoomEntry > time_map;

	public:
		PanZoom( ) : gui_active( true ), reverse( false )
		{
			// Create the window
			window = create_window_pan_zoom( );

			// Define the default animation
			PanZoomEntry *entry = time_map.SetEditable( 0, true );
			entry->x = 50;
			entry->y = 50;
			entry->width = 50;
			entry->height = 50;
			time_map.FinishedWith( entry );

			entry = time_map.SetEditable( 0.99, true );
			entry->x = 50;
			entry->y = 50;
			entry->width = 100;
			entry->height = 100;
			time_map.FinishedWith( entry );

			// Preview and overlay images
			preview = new uint8_t[ preview_width * preview_height * 3 ];

			//multi = CreateWidgetMultiMap( this );
			//AttachGtkWidget( "frame_timemap", multi );
		}

		virtual ~PanZoom( ) 
		{
			gtk_widget_destroy( window );
		}

		GtkWidget *GetGtkWidget()
		{
			return window;
		}

		char *GetDescription( ) const 
		{ 
			return "Pan and Zoom";
		}

		void FilterFrame( uint8_t *io, int width, int height, double position, double frame_delta ) 
		{
			PanZoomEntry *entry = time_map.Get( position );
			entry->interlace_on = interlace_on;
			entry->interlace_first_field = interlace_first_field;
			entry->RenderFinal( io, width, height );
			time_map.FinishedWith( entry );
		}
		
		void AttachWidgets( GtkBin *bin ) 
		{
			// Put the key frame controller widget in place
			controller = CreateWidgetKeyFrameController( this );
			AttachGtkWidget( "frame_key_frame_control", controller );

			// Put the preview widget in place
			area = CreateWidgetPreviewArea( this );
			AttachGtkWidget( "frame_preview", area );

			coord_picker = CreateWidgetPairPicker( this );
			AttachGtkWidget( "frame_coords", coord_picker );

			size_picker = CreateWidgetPairPicker( this );
			size_picker->SetSecondMinMax( 100, 0 );
			AttachGtkWidget( "frame_size", size_picker );

			interlace = CreateWidgetInterlaceOptions( );
			AttachGtkWidget( "frame_interlace", interlace );

			GtkWidget *widget = my_lookup_widget( window, "checkbutton_reverse" );
			gtk_signal_connect( GTK_OBJECT( widget ), "toggled", GTK_SIGNAL_FUNC( on_reverse_toggled ), this );

			gtk_widget_reparent( ( GTK_BIN( window ) )->child, GTK_WIDGET( bin ) );
		}

		void DetachWidgets( GtkBin *bin ) 
		{ 
			gtk_widget_reparent( ( GTK_BIN( bin ) )->child, GTK_WIDGET( window ) );
			DetachGtkWidget( "frame_coords", coord_picker );
			DetachGtkWidget( "frame_size", size_picker );
			DetachGtkWidget( "frame_key_frame_control", controller );
			DetachGtkWidget( "frame_preview", area );
			delete controller;
			delete area;
			delete coord_picker;
			delete size_picker;
		}

		void InterpretWidgets( GtkBin *bin )
		{
			interlace_on = interlace->IsInterlaceOn();
			interlace_first_field = interlace->IsFirstFieldOn();
		}

		void ReverseToggled( )
		{
			reverse = !reverse;
			time_map.Invert( );
			Refresh( );
		}

		void Refresh( PanZoomEntry *entry )
		{
			SelectedFrames &selected = GetSelectedFramesForFX( );
			double frame_position = entry->GetPosition( );

			if ( selected.GetNumInputFrames() > 0 )
			{
				selected.GetImageA( frame_position, preview, preview_width, preview_height );
				entry->RenderFinal( preview, preview_width, preview_height );
			}
			else
			{
				memset( preview, 0, preview_width * preview_height * 3 );
				entry->RenderPreview( preview, preview_width, preview_height );
			}

			area->ShowImage( (int)entry->x, (int)entry->y, preview, preview_width, preview_height );
		}

		void Refresh( bool with_control_sync = false )
		{
			double position = controller->GetCurrentPosition( );
			PanZoomEntry *entry = time_map.Get( position );
			Refresh( entry );
			if ( with_control_sync )
				ChangeController( entry );
			time_map.FinishedWith( entry );
		}

		void OnSelectionChange( )
		{
			Refresh( );
		}

		void OnPreviewAreaRefresh( PreviewArea *area )
		{
			Refresh( true );
		}

		void ChangeController( PanZoomEntry *entry )
		{
			frame_type type = FRAME;
			if ( entry->GetPosition() == 0 )
				type = LOCKED_KEY;
			else if ( entry->IsEditable() )
				type = KEY;

			controller->ShowCurrentStatus( entry->GetPosition(), type, entry->GetPosition() > time_map.GetFirst(), entry->GetPosition() < time_map.GetLast() );

			GtkWidget *widget = my_lookup_widget( window, "frame_key_input" );
			gtk_widget_set_sensitive( widget, entry->IsEditable() );

			coord_picker->SetValues( entry->x, entry->y );
			size_picker->SetValues( entry->width, entry->height );
		}

		void OnControllerKeyChanged( double position, bool value )
		{
			PanZoomEntry *entry = time_map.SetEditable( position, value );
			time_map.FinishedWith( entry );
			Refresh( true );
		}

		void OnControllerPrevKey( double position )
		{
			PanZoomEntry *entry = time_map.GotoPreviousKey( position );
			ChangeController( entry );
			time_map.FinishedWith( entry );
			Refresh( );
		}

		void OnControllerNextKey( double position )
		{
			PanZoomEntry *entry = time_map.GotoNextKey( position );
			ChangeController( entry );
			time_map.FinishedWith( entry );
			Refresh( );
		}

		void OnControllerPositionChanged( double position )
		{
			PanZoomEntry *entry = time_map.Get( position );
			ChangeController( entry );
			time_map.FinishedWith( entry );
			Refresh( );
		}

		void OnPairPickerChangeValue( PairPicker *picker, double first, double second )
		{
			double position = controller->GetCurrentPosition( );
			PanZoomEntry *entry = time_map.Get( position );

			if ( picker == coord_picker )
			{
				entry->x = first;
				entry->y = second;
			}
			else if ( picker == size_picker )
			{
				entry->width = first;
				entry->height = second;
			}

			ChangeController( entry );
			time_map.FinishedWith( entry );
			Refresh( );
		}

		void OnPropertySelected( MultiMap *multi, int value )
		{
		}
};

static void on_reverse_toggled( GtkWidget *some_widget, gpointer user_data )
{
	( (PanZoom *)user_data )->ReverseToggled( );
}

static gboolean on_average_scale_release_event( GtkWidget *widget, GdkEventButton *event, gpointer user_data);

class ColourAverage : public GDKImageFilter, public GtkKinoContainer, private KeyFrameControllerClient, private PreviewAreaClient,
				 public SelectionNotification
{
	private:
		GtkWidget *window;
		KeyFrameController *controller;
		PreviewArea *area;
		uint8_t *temp;
		int scale;

	public:
		ColourAverage( )
		{
			window = create_window_colour_average( );
			scale = 2;

			GtkWidget *widget = my_lookup_widget( window, "scale" );
			gtk_signal_connect( GTK_OBJECT( widget ), "button_release_event", GTK_SIGNAL_FUNC( on_average_scale_release_event ), this );
		}

		virtual ~ColourAverage( ) 
		{
			gtk_widget_destroy( window );
		}

		char *GetDescription( ) const 
		{ 
			return "Colour Average";
		}

		GtkWidget *GetGtkWidget()
		{
			return window;
		}

		void FilterFrame( uint8_t *pixels, int width, int height, double position, double frame_delta ) 
		{
			for ( int h = 0; h < height; h += 1 )
			{
				for ( int w = 0; w < width; w += 1 )
				{
					uint8_t *p = pixels + h * width * 3 + w * 3;
					*p ++ = ( *p / scale ) * scale + scale / 2;
					*p ++ = ( *p / scale ) * scale + scale / 2;
					*p ++ = ( *p / scale ) * scale + scale / 2;
				}
			}
		}
		
		void AttachWidgets( GtkBin *bin ) 
		{
			// Put the key frame controller widget in place
			controller = CreateWidgetKeyFrameController( this );
			AttachGtkWidget( "frame_key_frame_control", controller );

			// Put the preview widget in place
			area = CreateWidgetPreviewArea( this );
			AttachGtkWidget( "frame_preview", area );

			gtk_widget_reparent( ( GTK_BIN( window ) )->child, GTK_WIDGET( bin ) );
		}

		void DetachWidgets( GtkBin *bin ) 
		{ 
			gtk_widget_reparent( ( GTK_BIN( bin ) )->child, GTK_WIDGET( window ) );
			DetachGtkWidget( "frame_key_frame_control", controller );
			delete controller;
			DetachGtkWidget( "frame_preview", area );
			delete area;
		}

		void InterpretWidgets( GtkBin *bin ) 
		{
			GtkRange *range = GTK_RANGE( my_lookup_widget( window, "scale" ) );
			GtkAdjustment *adjust = gtk_range_get_adjustment( range );
			scale = (int)adjust->value; 
		}

		void Refresh( double position )
		{
			InterpretWidgets( GTK_BIN( window ) );
			controller->ShowCurrentStatus( position, LOCKED_KEY, false, false );

			int preview_width = 360;
			int preview_height = 288;
			uint8_t *preview = new uint8_t[ preview_width * preview_height * 3 ];
			SelectedFrames &selected = GetSelectedFramesForFX( );
			if ( selected.GetNumInputFrames() > 0 )
				selected.GetImageA( controller->GetCurrentPosition( ), preview, preview_width, preview_height );
			else
				memset( preview, 255, preview_width * preview_height * 3 );

			FilterFrame( preview, preview_width, preview_height, controller->GetCurrentPosition( ), 0.01 );
			area->ShowImage( 50, 50, preview, preview_width, preview_height );
			delete preview;
		}

		void Refresh( )
		{
			Refresh( controller->GetCurrentPosition( ) );
		}

		void OnControllerPositionChanged( double position )
		{
			Refresh( position );
		}

		void OnPreviewAreaRefresh( PreviewArea *area )
		{
			Refresh( );
		}

		void OnSelectionChange( )
		{
			Refresh( );
		}
};

static gboolean on_average_scale_release_event( GtkWidget *widget, GdkEventButton *event, gpointer user_data)
{
	( *( ColourAverage * )user_data ).Refresh( );
	return FALSE;
}

static gboolean on_gamma_release_event( GtkWidget *widget, GdkEventButton *event, gpointer user_data);

class Gamma : public GDKImageFilter, public GtkKinoContainer, private KeyFrameControllerClient, 
	      private PreviewAreaClient, public SelectionNotification
{
 private:
   double gamma;
   GtkWidget *window;
   KeyFrameController *controller;
   PreviewArea *area;
   
 public:
   Gamma()
   {
     window = create_window_gamma();
     gamma = 1.0;
     
     GtkWidget *widget = my_lookup_widget( window, "hscale_gamma" );
     gtk_signal_connect( GTK_OBJECT( widget ), "button_release_event", GTK_SIGNAL_FUNC( on_gamma_release_event ), this );
   }
  
   virtual ~Gamma()
   {
     gtk_widget_destroy(window);
   }
   
   char *GetDescription( ) const
   {
     return "Gamma";
   }

   GtkWidget *GetGtkWidget()
   {
     return window;
   }
   
   void FilterFrame(uint8_t *io, int width, int height, double position, double frame_delta)
   {
     int r,g,b;
     int index;
     uint8_t *p = io;
     unsigned char lookup[256];
     double exp = 1/gamma;
     
     for(index = 0; index < 256; index++)
       lookup[index] = (unsigned char) (pow ((double) (index) / 255.0, (exp)) * 255);
     
     for(int h = 0; h < height; h++)
     {
       for(int w = 0; w < width; w++)
       {
	 r = *p;
	 g = *(p+1);
	 b = *(p+2);
	 
	 *p++ = lookup[r];
	 *p++ = lookup[g];
	 *p++ = lookup[b];
       }
     }	
   }
   
   void AttachWidgets(GtkBin *bin)
   {
     // Create controller
     controller = CreateWidgetKeyFrameController(this);
     AttachGtkWidget("frame_key_frame_control", controller);
     
     // Create preview area
     area = CreateWidgetPreviewArea(this);
     AttachGtkWidget("frame_preview", area);
     
     // Attach the window
     gtk_widget_reparent( ( GTK_BIN( window ) )->child, GTK_WIDGET( bin ) );
   }
   
   void DetachWidgets( GtkBin *bin ) 
   { 
     gtk_widget_reparent( ( GTK_BIN( bin ) )->child, GTK_WIDGET( window ) );
     DetachGtkWidget("frame_key_frame_control", controller);
     delete controller;
     DetachGtkWidget("frame_preview", area);
     delete area;
   }
   
   void InterpretWidgets( GtkBin *bin ) 
   {
     GtkRange *range = GTK_RANGE( my_lookup_widget( window, "hscale_gamma" ) );
     GtkAdjustment *adjust = gtk_range_get_adjustment( range );
     gamma = (double) adjust->value; 
   }
   
   void Refresh(double position)
   {     
     /* Create controller */
     InterpretWidgets(GTK_BIN(window));
     controller->ShowCurrentStatus(position, LOCKED_KEY, false, false);
     
     /* Set size of the preview window */
     int preview_width = 360;
     int preview_height = 288;
     uint8_t *preview = new uint8_t[preview_width * preview_height *3];
     SelectedFrames &selected = GetSelectedFramesForFX();
     
     if(selected.GetNumInputFrames() > 0)
     {
       selected.GetImageA(controller->GetCurrentPosition(), preview, preview_width, preview_height);
     }
     else
     {
       memset(preview, 255, preview_width * preview_height * 3);       
     }
     
     FilterFrame(preview, preview_width, preview_height, controller->GetCurrentPosition(), 0.01);
     area->ShowImage(50, 50, preview, preview_width, preview_height);
     delete preview;       
   }
   
   void Refresh()
   {
     Refresh(controller->GetCurrentPosition());
   }
   
   void OnControllerPositionChanged(double position)
   {
     Refresh(position);
   }
   
   void OnPreviewAreaRefresh(PreviewArea *area)
   {
     Refresh();
   }
   
   void OnSelectionChange()
   {
     Refresh();
   }     
};

static gboolean on_gamma_release_event( GtkWidget *widget, GdkEventButton *event, gpointer user_data)
{
	( *( Gamma * )user_data ).Refresh( );
	return FALSE;
}

class RWPipe
{
	private:
		int pid;

	protected:
		int reader;
		int writer;
		GError *error;
		string m_command;

	public:
		RWPipe( ) : pid( -1 ), error( NULL ), m_command( "" )
		{
		}

		~RWPipe( )
		{
			Close( );
		}

		bool Open( string command )
		{
			char *fuckwits[ 4 ];
			fuckwits[ 0 ] = "/bin/sh";
			fuckwits[ 1 ] = "-c";
			fuckwits[ 2 ] = ( char * )command.c_str( );
			fuckwits[ 3 ] = NULL;

			m_command = command;

			return g_spawn_async_with_pipes( ".", fuckwits, NULL, G_SPAWN_LEAVE_DESCRIPTORS_OPEN, NULL, NULL, &pid, &writer, &reader, NULL, &error );
		}
	
		bool IsRunning( )
		{
			return pid != -1;
		}

		int Read( void *data, int size )
		{
			if ( pid != -1 )
			{
				int bytes = 0;
				int len;
				uint8_t *p = ( uint8_t * )data;

				while ( size > 0 )
				{
					len = read( reader, p, size );
					if ( len <= 0 )
						break;
					p += len;
					size -= len;
					bytes += len;
				}

				return bytes;
			}
			else
				return -1;
		}

		int ReadLine( char *text, int max )
		{
			int len = -1;
			strcpy( text, "" );
			if ( pid != -1 )
			{
				while ( len < max - 1 && Read( &text[ ++ len ], 1 ) )
				{
					if ( text[ len ] == '\n' )
						break;
				}
				if ( len >= 0 )
					text[ len ] = '\0';
			}
			return len;
		}

		int Write( void *data, int size )
		{
			if ( pid != -1 )
				return write( writer, (uint8_t *)data, size );
			else
				return -1;
		}

		void Close( )
		{
			if ( pid != -1 )
			{
				close( reader );
				close( writer );
				waitpid( pid, NULL, 0 );
				pid = -1;
			}
		}

		string GetCurrentCommand( )
		{
			return m_command;
		}
};

class EffectTV : public GDKImageFilter
{
	private:
		GtkWidget *window;
		RWPipe filter;

	public:
		EffectTV( )
		{
			window = create_window_effectv( );
		}

		virtual ~EffectTV( ) 
		{
			gtk_widget_destroy( window );
		}

		char *GetDescription( ) const 
		{ 
			return "Effect TV";
		}

		void FilterFrame( uint8_t *io, int width, int height, double position, double frame_delta ) 
		{
			char temp[ 132 ];
			sprintf( temp, "P6\n%d %d\n255\n", width, height );
			filter.Write( temp, strlen( temp ) );
			filter.Write( io, width * height * 3 );
			filter.ReadLine( temp, 132 );
			filter.ReadLine( temp, 132 );
			filter.ReadLine( temp, 132 );
			filter.Read( io, width * height * 3 );
		}
		
		void PopulateFX( )
		{
			// Get the option menu
			GtkOptionMenu *menu = GTK_OPTION_MENU( my_lookup_widget( window, "optionmenu" ) );

			// Add the filters to the menu
			GtkMenu *menu_new = GTK_MENU( gtk_menu_new( ) );
			char temp[ 1024 ];
 			RWPipe pip;

			pip.Open( "exec ppmeffectv --list" );
			while ( pip.ReadLine( temp, 1024 ) > 0 )
			{
				if ( strstr( temp, ":" ) )
				{
					GtkWidget *item = gtk_menu_item_new_with_label( temp );
					gtk_widget_show( item );
					gtk_menu_append( menu_new, item );
				}
			}
			gtk_menu_set_active( menu_new, 0 );
			gtk_option_menu_set_menu( menu, GTK_WIDGET( menu_new ) );
			pip.Close( );
		}

		void AttachWidgets( GtkBin *bin ) 
		{
			gtk_widget_reparent( ( GTK_BIN( window ) )->child, GTK_WIDGET( bin ) );
			PopulateFX( );
		}

		void DetachWidgets( GtkBin *bin ) 
		{ 
			gtk_widget_reparent( ( GTK_BIN( bin ) )->child, GTK_WIDGET( window ) );
			filter.Close( );
		}

		void InterpretWidgets( GtkBin *bin ) 
		{
			filter.Close( );
			GtkOptionMenu *menu = GTK_OPTION_MENU( my_lookup_widget( window, "optionmenu" ) );
			GtkMenu *filterMenu = GTK_MENU( gtk_option_menu_get_menu( menu ) );
			GtkWidget *active_item = gtk_menu_get_active( filterMenu );
			char temp[ 132 ];
			sprintf( temp, "exec ppmeffectv -e %d", g_list_index( GTK_MENU_SHELL( filterMenu )->children, active_item ) );
			filter.Open( temp );
		}
};

class PipeFilter : public GDKImageFilter, public GtkKinoContainer, private KeyFrameControllerClient, 
	      private PreviewAreaClient, public SelectionNotification
{
	private:
		GtkWidget *window;
		KeyFrameController *controller;
		PreviewArea *area;
 		RWPipe filter;

	public:
		PipeFilter( )
		{
			window = create_window_pipe_filter( );
		}

		virtual ~PipeFilter( ) 
		{
			gtk_widget_destroy( window );
		}

		char *GetDescription( ) const 
		{ 
			return "Pipe Filter";
		}

		void FilterFrame( uint8_t *io, int width, int height, double position, double frame_delta ) 
		{
			char temp[ 132 ];
			sprintf( temp, "P6\n%d %d\n255\n", width, height );
			filter.Write( temp, strlen( temp ) );
			filter.Write( io, width * height * 3 );
			filter.ReadLine( temp, 132 );
			filter.ReadLine( temp, 132 );
			filter.ReadLine( temp, 132 );
			filter.Read( io, width * height * 3 );
		}
		
		void AttachWidgets( GtkBin *bin ) 
		{
			// Create controller
			controller = CreateWidgetKeyFrameController(this);
			AttachGtkWidget("frame_key_frame_control", controller);
     
			// Create preview area
			area = CreateWidgetPreviewArea(this);
			AttachGtkWidget("frame_preview", area);
 			gtk_widget_reparent( ( GTK_BIN( window ) )->child, GTK_WIDGET( bin ) );
		}
  
		void DetachWidgets( GtkBin *bin ) 
		{ 
			gtk_widget_reparent( ( GTK_BIN( bin ) )->child, GTK_WIDGET( window ) );
			DetachGtkWidget("frame_key_frame_control", controller);
			delete controller;
			DetachGtkWidget("frame_preview", area);
			delete area;
			filter.Close( );
		}

		void InterpretWidgets( GtkBin *bin ) 
		{
			RefreshCommand( true );
		}

		void RefreshCommand( bool restart = false )
		{
			GtkCombo *combo = GTK_COMBO( my_lookup_widget( window, "combo1" ) );
			GtkEntry *entry = GTK_ENTRY( combo->entry );
			char temp[ 1024 ];
			sprintf( temp, "exec %s", gtk_entry_get_text( entry ) );

			if ( restart || string( temp ) != filter.GetCurrentCommand( ) )
			{
				filter.Close( );
				filter.Open( temp );
			}
		}

		void Refresh(double position)
		{     
			/* Create controller */
			RefreshCommand( );
			controller->ShowCurrentStatus(position, LOCKED_KEY, false, false);
     
			/* Set size of the preview window */
			int preview_width = 360;
			int preview_height = 288;
			uint8_t *preview = new uint8_t[preview_width * preview_height *3];
			SelectedFrames &selected = GetSelectedFramesForFX();
     
			if(selected.GetNumInputFrames() > 0)
				selected.GetImageA(controller->GetCurrentPosition(), preview, preview_width, preview_height);
			else
				memset(preview, 255, preview_width * preview_height * 3);       
     
			FilterFrame(preview, preview_width, preview_height, controller->GetCurrentPosition(), 0.01);
			area->ShowImage(50, 50, preview, preview_width, preview_height);
			delete preview;       
		}
 
		void Refresh()
		{
			Refresh(controller->GetCurrentPosition());
		}
   
		void OnControllerPositionChanged(double position)
		{
			Refresh(position);
		}
   
		void OnPreviewAreaRefresh(PreviewArea *area)
		{
			Refresh();
		}
   
		void OnSelectionChange()
		{
			Refresh();
		}

		GtkWidget *GetGtkWidget()
		{
			return window;
		}
};


static void on_entry_ffmpeg_changed( GtkEditable *editable, gpointer user_data );

class FfmpegImport : public GDKAudioImport, protected PixbufUtils
{
	private:
		GtkWidget *window;
		bool pal;
		bool maintain_aspect_ratio;
		char selected[ 1024 ];
		RWPipe convert;
		int frames;
		int16_t audio_buffer[ 4 * DV_AUDIO_MAX_SAMPLES ];
		uint8_t *buffer;
		bool silent;

	public:
		FfmpegImport( ) : frames( 0 )
		{
			window = create_window_ffmpeg_import( );
			strcpy( selected, "" );
			gtk_signal_connect( GTK_OBJECT( my_lookup_widget( window, "entry_file" ) ), "changed", 
							    GTK_SIGNAL_FUNC( on_entry_ffmpeg_changed ), this );
			Refresh( "" );

			buffer = new uint8_t[ 1024 * 768 * 3 ];
		}

		virtual ~FfmpegImport( ) 
		{
			convert.Close( );
			gtk_widget_destroy( window );
			delete buffer;
		}

		char *GetDescription( ) const 
		{ 
			return "FFMPEG Import";
		}

		void CreatePAL( bool is_pal )
		{
			pal = is_pal;
		}

		void CreateFrame( uint8_t *io, int width, int height, double position, double frame_delta  ) 
		{
			if ( !convert.IsRunning( ) )
			{
				double in;
				double out;
				double rate = 0;
				int video_track;
				int audio_track;
				char command[ 1024 ];

				// Determine and set scaling option
				GtkMenu *scaling = GTK_MENU( gtk_option_menu_get_menu( GTK_OPTION_MENU( my_lookup_widget( GTK_WIDGET( window ), "optionmenu_scale" ) ) ) );
				GtkWidget *active_item = gtk_menu_get_active( scaling );
				switch ( g_list_index( GTK_MENU_SHELL( scaling )->children, active_item ) )
				{
					case 0:
						SetScale( SCALE_ASPECT_RATIO );
						break;
					case 1:
						SetScale( SCALE_FULL );
						break;
					case 2:
					default:
						SetScale( SCALE_NONE );
						break;
				}

				// Determine start and end time
				GtkSpinButton *spin = GTK_SPIN_BUTTON( my_lookup_widget( window, "spinbutton1" ) );
				double start_h = gtk_spin_button_get_value( spin );
				spin = GTK_SPIN_BUTTON( my_lookup_widget( window, "spinbutton2" ) );				
				double start_m = gtk_spin_button_get_value( spin );
				spin = GTK_SPIN_BUTTON( my_lookup_widget( window, "spinbutton3" ) );				
				double start_s = gtk_spin_button_get_value( spin );
				spin = GTK_SPIN_BUTTON( my_lookup_widget( window, "spinbutton4" ) );				
				double final_h = gtk_spin_button_get_value( spin );
				spin = GTK_SPIN_BUTTON( my_lookup_widget( window, "spinbutton5" ) );				
				double final_m = gtk_spin_button_get_value( spin );
				spin = GTK_SPIN_BUTTON( my_lookup_widget( window, "spinbutton6" ) );				
				double final_s = gtk_spin_button_get_value( spin );

				in = start_h * 60 * 60 + start_m * 60 + start_s;
				out = final_h * 60 * 60 + final_m * 60 + final_s;

				// Determine fps
				GtkEntry *entry_fps = GTK_ENTRY( my_lookup_widget( GTK_WIDGET( window ), "entry_fps" ) );
				if ( strcmp( (char *)gtk_entry_get_text( entry_fps ), "default" ) )
					rate = atof( (char *)gtk_entry_get_text( entry_fps ) );
				
				// Determine video track
				GtkMenu *vts = GTK_MENU( gtk_option_menu_get_menu( GTK_OPTION_MENU( my_lookup_widget( GTK_WIDGET( window ), "optionmenu_video" ) ) ) );
				active_item = gtk_menu_get_active( vts );
				video_track = (int)g_object_get_data( G_OBJECT( active_item ), "kinoplus" );

				// Determine audio track
				GtkMenu *ats = GTK_MENU( gtk_option_menu_get_menu( GTK_OPTION_MENU( my_lookup_widget( GTK_WIDGET( window ), "optionmenu_audio" ) ) ) );
				active_item = gtk_menu_get_active( ats );
				audio_track = (int)g_object_get_data( G_OBJECT( active_item ), "kinoplus" );
				silent = audio_track == -1;

				// Generate the command
				sprintf( command, "kinoplusimport \"%s\" %f %f %f %d %d %s", selected, in, out, rate, video_track, audio_track, pal ? "" : "-n" );
				convert.Open( command );
			}

			char temp[ 132 ];
			convert.ReadLine( temp, 132 );

			if ( !strncmp( temp, "P6", 2 ) )
			{
				int in_width;
				int in_height;

				convert.ReadLine( temp, 132 );
				sscanf( temp, "%d %d", &in_width, &in_height );
				convert.ReadLine( temp, 132 );
				convert.Read( buffer, in_width * in_height * 3 );
				GdkPixbuf *pix = ConvertRGBToPixbuf( buffer, in_width, in_height );
				ScalePixbuf( pix, io, width, height );
				gdk_pixbuf_unref( pix );
			}
			else
			{
				throw "Unrecognised image data received in raw import.";
			}
		}

		void CreateAudio( int16_t **buffer, short int *channels, int *frequency, int *samples )
		{
			if ( !silent )
			{
				char temp[ 132 ];
				convert.ReadLine( temp, 132 );
				if ( !strncmp( temp, "A6", 2 ) )
				{
					sscanf( temp + 3, "%d %d %d", frequency, channels, samples );
					convert.Read( audio_buffer, *samples * *channels * sizeof( int16_t ) );
					for ( int i = 0; i < *samples; i ++ )
					{
						buffer[ 0 ][ i ] = audio_buffer[ i ];
						buffer[ 1 ][ i ] = audio_buffer[ *samples + i ];
					}
				}
				else
				{
					throw "Unrecognised audio data received in raw import.";
				}
			}
			else
			{
				for ( int i = 0; i < *samples; i ++ )
					buffer[ 0 ][ i ] = buffer[ 1 ][ i ] = 0;
			}
		}

		int GetNumberOfFrames( )
		{
			GtkSpinButton *spin = GTK_SPIN_BUTTON( my_lookup_widget( window, "spinbutton1" ) );
			double start_h = gtk_spin_button_get_value( spin );
			spin = GTK_SPIN_BUTTON( my_lookup_widget( window, "spinbutton2" ) );				
			double start_m = gtk_spin_button_get_value( spin );
			spin = GTK_SPIN_BUTTON( my_lookup_widget( window, "spinbutton3" ) );				
			double start_s = gtk_spin_button_get_value( spin );
			spin = GTK_SPIN_BUTTON( my_lookup_widget( window, "spinbutton4" ) );				
			double final_h = gtk_spin_button_get_value( spin );
			spin = GTK_SPIN_BUTTON( my_lookup_widget( window, "spinbutton5" ) );				
			double final_m = gtk_spin_button_get_value( spin );
			spin = GTK_SPIN_BUTTON( my_lookup_widget( window, "spinbutton6" ) );				
			double final_s = gtk_spin_button_get_value( spin );

			double start_seconds = start_h * 60 * 60 + start_m * 60 + start_s;
			double final_seconds = final_h * 60 * 60 + final_m * 60 + final_s;

			fprintf( stderr, "%f %f %d\n", start_seconds, final_seconds, ( int )( ( final_seconds - start_seconds ) * ( pal ? 25 : 29.97 ) ) );

			return ( int )( ( final_seconds - start_seconds ) * ( pal ? 25 : 29.97 ) );
		}
		
		void AttachWidgets( GtkBin *bin ) 
		{
			gtk_widget_reparent( ( GTK_BIN( window ) )->child, GTK_WIDGET( bin ) );
		}

		void DetachWidgets( GtkBin *bin ) 
		{ 
			gtk_widget_reparent( ( GTK_BIN( bin ) )->child, GTK_WIDGET( window ) );
		}

		void InterpretWidgets( GtkBin *bin ) 
		{
			convert.Close();
			if ( !strcmp( selected, "" ) )
				throw "No file selected for import";
		}

		int Refresh( char *filename )
		{
			if ( strcmp( filename, "" ) )
			{
				float hours = 0, minutes = 0, seconds = 0;
				int size = 0;
				char *command = g_strdup_printf( "ffmpeg2raw --stats %s \"%s\"", pal ? "" : "-n", filename );
				char description[ 1024 ];
				char temp[ 1024 ];
				RWPipe pip;

				// Get the option menus
				GtkOptionMenu *video = GTK_OPTION_MENU( my_lookup_widget( window, "optionmenu_video" ) );
				GtkOptionMenu *audio = GTK_OPTION_MENU( my_lookup_widget( window, "optionmenu_audio" ) );
				GtkMenu *menu_video = NULL;
				GtkMenu *menu_audio = NULL;

				strcpy( description, filename );

				pip.Open( command );
				while( pip.ReadLine( temp, 1024 ) )
				{
					fprintf( stderr, "%s\n", temp );
					if ( temp[ 0 ] == 'T' )
					{
						strcat( description, " [" );
						strcat( description, temp + 3 );
						strcat( description, "]" );
						sscanf( temp + 3, "%f:%f:%f", &hours, &minutes, &seconds );
					}
					else if ( temp[ 0 ] == 'F' )
					{
						size = atoi( temp + 3 );
					}
					else if ( size > 0 && temp[ 0 ] == 'V' && strcmp( selected, filename ) )
					{
						if ( menu_video == NULL )
							menu_video = GTK_MENU( gtk_menu_new( ) );
						GtkWidget *item = gtk_menu_item_new_with_label( temp + 1 );
						g_object_set_data( G_OBJECT( item ), "kinoplus", (gpointer)( atoi( temp + 1 ) ) );
						gtk_widget_show( item );
						gtk_menu_append( menu_video, item );
					}
					else if ( size > 0 && temp[ 0 ] == 'A' && strcmp( selected, filename ) )
					{
						if ( menu_audio == NULL )
							menu_audio = GTK_MENU( gtk_menu_new( ) );
						GtkWidget *item = gtk_menu_item_new_with_label( temp + 1 );
						g_object_set_data( G_OBJECT( item ), "kinoplus", (gpointer)( atoi( temp + 1 ) ) );
						gtk_widget_show( item );
						gtk_menu_append( menu_audio, item );
					}
				}
				pip.Close( );
				free( command );
				if ( size != 0 )
				{
					gtk_label_set_text( GTK_LABEL( my_lookup_widget( window, "label" ) ), description );
					if ( menu_video != NULL )
					{
						gtk_menu_set_active( menu_video, 0 );
						gtk_option_menu_set_menu( video, GTK_WIDGET( menu_video ) );
						if ( menu_audio != NULL )
						{
							GtkWidget *item = gtk_menu_item_new_with_label( "[Silent]" );
							g_object_set_data( G_OBJECT( item ), "kinoplus", (gpointer)( -1 ) );
							gtk_widget_show( item );
							gtk_menu_append( menu_audio, item );
							gtk_menu_set_active( menu_audio, 0 );
							gtk_option_menu_set_menu( audio, GTK_WIDGET( menu_audio ) );
						}
						else
						{
							menu_audio = GTK_MENU( gtk_menu_new( ) );
							GtkWidget *item = gtk_menu_item_new_with_label( "[Silent]" );
							g_object_set_data( G_OBJECT( item ), "kinoplus", (gpointer)( -1 ) );
							gtk_widget_show( item );
							gtk_menu_append( menu_audio, item );
							gtk_menu_set_active( menu_audio, 0 );
							gtk_option_menu_set_menu( audio, GTK_WIDGET( menu_audio ) );
						}
					}

					GtkEntry *entry_fps = GTK_ENTRY( my_lookup_widget( GTK_WIDGET( window ), "entry_fps" ) );
					gtk_entry_set_text( entry_fps, "default" );

					GtkSpinButton *spin = GTK_SPIN_BUTTON( my_lookup_widget( window, "spinbutton1" ) );
					gtk_spin_button_set_value( spin, 0 );
					spin = GTK_SPIN_BUTTON( my_lookup_widget( window, "spinbutton2" ) );				
					gtk_spin_button_set_value( spin, 0 );
					spin = GTK_SPIN_BUTTON( my_lookup_widget( window, "spinbutton3" ) );				
					gtk_spin_button_set_value( spin, 0 );
					spin = GTK_SPIN_BUTTON( my_lookup_widget( window, "spinbutton4" ) );				
					gtk_spin_button_set_value( spin, hours );
					spin = GTK_SPIN_BUTTON( my_lookup_widget( window, "spinbutton5" ) );				
					gtk_spin_button_set_value( spin, minutes );
					spin = GTK_SPIN_BUTTON( my_lookup_widget( window, "spinbutton6" ) );				
					gtk_spin_button_set_value( spin, seconds );
				}
				return size;
			}
			else
			{
				gtk_label_set_text( GTK_LABEL( my_lookup_widget( window, "label" ) ), "No file selected." );
				strcpy( selected, "" );
				return 0;
			}
		}

		void FileSelected( )
		{
			char *filename = (char *)gtk_entry_get_text( GTK_ENTRY( my_lookup_widget( window, "entry_file" ) ) );
			int size = Refresh( filename );
			if ( size > 0 )
				strncpy( selected, filename, 1024 );
		}
};

class FfmpegDub : public GDKAudioFilter
{
	private:
		GtkWidget *window;
		string m_file;
		RWPipe m_pipe;
		int16_t m_buffer[ 4 * DV_AUDIO_MAX_SAMPLES ];
		char *m_command;

	public:
		FfmpegDub( ) :
			m_command( NULL )
		{
			window = create_window_ffmpeg_dub( );
		}

		~FfmpegDub( )
		{
			gtk_widget_destroy( window );
			free( m_command );
		}

		char *GetDescription( ) const
		{
			return "FFMPEG Dub"; 
		}

		void AttachWidgets( GtkBin *bin ) 
		{ 
			gtk_widget_reparent( ( GTK_BIN( window ) )->child, GTK_WIDGET( bin ) );
		}

		void InterpretWidgets( GtkBin *bin ) 
		{ 
			m_pipe.Close( );
		}

		void DetachWidgets( GtkBin *bin ) 
		{ 
			gtk_widget_reparent( ( GTK_BIN( bin ) )->child, GTK_WIDGET( window ) );
		}

		void GetFrame( int16_t **buffer, int frequency, int channels, int samples, double position, double frame_delta )
		{
			if ( !m_pipe.IsRunning( ) )
			{
				char *filename = (char *)gtk_entry_get_text( GTK_ENTRY( my_lookup_widget( window, "entry_file" ) ) );
				free( m_command );
				m_command = g_strdup_printf( "kinoplusdub \"%s\" %d", filename, frequency );
				m_pipe.Open( m_command );
			}

			if ( m_pipe.Read( m_buffer, samples * channels * sizeof( int16_t ) ) == 0 )
			{
				if ( gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON( my_lookup_widget( window, "checkbutton_loop" ) ) ) )
				{
					m_pipe.Close( );
					m_pipe.Open( m_command );
				}
			}

			for ( int i = 0; i < samples; i ++ )
			{
				buffer[ 0 ][ i ] = m_buffer[ 2 * i ];
				buffer[ 1 ][ i ] = m_buffer[ 2 * i + 1 ];
			}
		}
};


/** GTK callback on file selection change.
*/

static void on_entry_ffmpeg_changed( GtkEditable *editable, gpointer user_data )
{
	( ( FfmpegImport * ) user_data )->FileSelected( );
}

extern "C" {

GDKImageCreate *GetImageCreate( int index )
{
	switch( index )
	{
		case 0:
			return new FfmpegImport();
		case 1:
			return new MultipleImport();
	}
	return NULL;
}

GDKImageFilter *GetImageFilter( int index )
{
	switch( index )
	{
		case 0:
			return new Jerker();
		case 1:
			return new ImageTitler();
		case 2:
			return new ImageOverlay();
		case 3:
			return new Pixelate();
		case 4:
			return new LineDraw();
		case 5:
			return new PanZoom();
		case 6:
			return new ColourAverage();
		case 7:
			return new Gamma();
		case 8:
			return new EffectTV();
		case 9:
			return new PipeFilter();
	}
	return NULL;
}

GDKImageTransition *GetImageTransition( int index )
{
	switch( index )
	{
		case 0:
			return new GDKImageTransitionAdapter( new ImageTransitionChromaKeyBlue() );
		case 1:
			return new Tweenies();
	}
	return NULL;
}

GDKAudioFilter *GetAudioFilter( int index )
{
	switch( index )
	{
		case 0:
			return new FfmpegDub();
	}
	return NULL;
}
}
