#include <string.h>
#include <time.h>

#include <gtk/gtk.h>
#include <glade/glade.h>

#include "kpnewsplitworkoutdialog.h"
#include "kpsportentry.h"
#include "kpmainwindow.h"
#include "kpsplitsview.h"
#include "kpguiutils.h"
#include "kpdateentry.h"

#include "../kpsplitworkout.h"
#include "../kptraininglog.h"
#include "../kppresetdata.h"
#include "../kipina-i18n.h"
#include "../kpworkout.h"
#include "../kputil.h"


#define SPORT_MENU_TEXT    "Choose a Sport"
#define SPORT_MENU_OTHER   "Other"

static void     kp_new_split_workout_dialog_class_init  (KPNewSplitWorkoutDialogClass *klass);
static void     kp_new_split_workout_dialog_init        (KPNewSplitWorkoutDialog *dialog);
static void     kp_new_split_workout_dialog_finalize    (GObject *object);
static void     kp_new_split_workout_dialog_response_cb (GtkDialog *gtkdialog,
                                                         int response_id,
                                                         KPNewSplitWorkoutDialog *dialog);
static void     build_detail_menu                       (KPNewSplitWorkoutDialog *dialog);
static void     set_detail_tree_view                    (KPNewSplitWorkoutDialog *dialog);
static gboolean export_list_row                         (GtkTreeModel *model,
                                                         GtkTreePath *path,
                                                         GtkTreeIter *iter,
                                                         KPNewSplitWorkoutDialog *dialog);
static void     detail_list_row_selected                (GtkTreeSelection * selection,
                                                         KPNewSplitWorkoutDialog *dialog);
static void     on_detail_add_button_clicked            (GtkButton *button,
                                                         KPNewSplitWorkoutDialog *dialog);
static void     on_detail_delete_button_clicked         (GtkButton *button,
                                                         KPNewSplitWorkoutDialog *dialog);
/* SPLITS */
static void     split_view_row_selected_cb              (GtkTreeSelection *selection,
                                                         KPNewSplitWorkoutDialog *dialog);
static void     on_split_add_button_clicked             (GtkButton *button,
                                                         KPNewSplitWorkoutDialog *dialog);
static void     on_split_delete_button_clicked          (GtkButton *button,
                                                         KPNewSplitWorkoutDialog *dialog);
static void     set_comment                             (KPNewSplitWorkoutDialog *dialog);
static void     date_selected_cb                        (KPDateEntry *entry,
                                                         KPDate *date,
                                                         KPNewSplitWorkoutDialog *dialog);
static void     time_selected_cb                        (KPDateEntry *entry,
                                                         KPTime *time,
                                                         KPNewSplitWorkoutDialog *dialog);


static GuiModuleSignalsData signals_data[] = {
{"detail_add_button",     "clicked",CB (on_detail_add_button_clicked),    NULL},
{"detail_delete_button",  "clicked",CB (on_detail_delete_button_clicked), NULL},
{"split_add_button",      "clicked",CB (on_split_add_button_clicked),     NULL},
{"split_delete_button" ,  "clicked",CB (on_split_delete_button_clicked),  NULL},
{ NULL,                    NULL,    NULL,                                 NULL},
};


typedef struct KPNewSplitWorkoutDialogPrivateData_
{
  KPTrainingLog  *log;
  KPSplitWorkout *wo;
 
  GtkWidget    *details_view;
  GtkWidget    *detail_add_button;
  GtkWidget    *detail_delete_button;
  GtkWidget    *detail_preset_menu;
  GtkWidget    *detail_value;

  GtkWidget    *split_spin;
  GtkWidget    *split_type_menu;
  GtkWidget    *split_distance_spin;
  GtkWidget    *split_duration_entry;
  GtkWidget    *split_comment_entry;
  GtkWidget    *split_view;
  GtkWidget    *split_delete_button;
  GtkWidget    *split_add_button;

  GtkWidget    *date_entry;
  GtkWidget    *okbutton;
  GtkWidget    *cancelbutton;
  GtkWidget    *comment;
  GtkWidget    *sportbox;
  
  gboolean      is_new_workout;
} KPNewSplitWorkoutDialogPrivateData;

#define KP_NEW_SPLIT_WORKOUT_DIALOG_PRIVATE_DATA(widget) (((KPNewSplitWorkoutDialogPrivateData*) \
        (KP_NEW_SPLIT_WORKOUT_DIALOG (widget)->private_data)))

static GObjectClass *parent_class = NULL;

GType
kp_new_split_workout_dialog_get_type (void)
{
  static GType kp_new_split_workout_dialog_type = 0;

  if (kp_new_split_workout_dialog_type == 0) {
    static const GTypeInfo our_info = {
      sizeof (KPNewSplitWorkoutDialogClass),
      NULL,
      NULL,
      (GClassInitFunc) kp_new_split_workout_dialog_class_init,
      NULL,
      NULL,
      sizeof (KPNewSplitWorkoutDialog),
      0,
      (GInstanceInitFunc) kp_new_split_workout_dialog_init,
      NULL,
    };

    kp_new_split_workout_dialog_type = g_type_register_static (GTK_TYPE_DIALOG,
                                                              "KPNewSplitWorkoutDialog",
                                                              &our_info, 0);
  }

  return kp_new_split_workout_dialog_type;
}


static void
kp_new_split_workout_dialog_class_init (KPNewSplitWorkoutDialogClass * klass)
{
  GObjectClass *object_class = G_OBJECT_CLASS (klass);

  parent_class = g_type_class_peek_parent (klass);
  object_class->finalize = kp_new_split_workout_dialog_finalize;
}


static void
split_selected_cb (KPSplitsView *sv, KPSplit *split, 
                   KPNewSplitWorkoutDialog *dialog)
{
  g_print ("Split: %u\n", split->n);
}


static void
kp_new_split_workout_dialog_init (KPNewSplitWorkoutDialog * dialog)
{
  KPNewSplitWorkoutDialogPrivateData *p_data;
  GtkTreeSelection *selection;
  GladeXML *xml;

  xml = kp_gui_load ("new_split_workout_dialog", "split_workout");
  
  dialog->private_data = g_new0 (KPNewSplitWorkoutDialogPrivateData, 1);
  p_data = KP_NEW_SPLIT_WORKOUT_DIALOG_PRIVATE_DATA (dialog);

  g_signal_connect (G_OBJECT (dialog),
                   "response",
                    G_CALLBACK (kp_new_split_workout_dialog_response_cb), dialog);

  g_signal_connect (G_OBJECT (dialog), "response",
                    G_CALLBACK (gtk_widget_destroy), NULL);
  
  gtk_dialog_set_has_separator (GTK_DIALOG (dialog), FALSE);
  gtk_container_set_border_width (GTK_CONTAINER (dialog), 5);
  gtk_box_set_spacing (GTK_BOX (GTK_DIALOG (dialog)->vbox), 2);

  gtk_dialog_set_default_response (GTK_DIALOG (dialog), GTK_RESPONSE_NONE);
  gtk_window_set_title (GTK_WINDOW (dialog), _("New Split Workout"));

  p_data->cancelbutton = gtk_dialog_add_button (GTK_DIALOG (dialog),
                                                GTK_STOCK_CANCEL,
                                                GTK_RESPONSE_CANCEL);

  p_data->okbutton = gtk_dialog_add_button (GTK_DIALOG (dialog),
                                            GTK_STOCK_ADD,
                                            GTK_RESPONSE_OK);
  gtk_container_add (GTK_CONTAINER (GTK_DIALOG (dialog)->vbox),
                     KP_W (xml, "split_workout"));

  p_data->wo = kp_split_workout_new ();
  
  /* Put widgets to private data */
  p_data->details_view = KP_W (xml, "details_view");
  p_data->detail_add_button = KP_W (xml, "detail_add_button");  
  p_data->detail_delete_button = KP_W (xml, "detail_delete_button");  
  p_data->detail_value = KP_W (xml, "detail_value_entry");
  p_data->detail_preset_menu = KP_W (xml, "detail_preset_menu");

  p_data->split_spin = KP_W (xml, "split_spin");
  p_data->split_type_menu = KP_W (xml, "split_type_menu");
  p_data->split_distance_spin = KP_W (xml, "distance_spin");
  p_data->split_duration_entry = KP_W (xml, "duration_entry");
  p_data->split_comment_entry = KP_W (xml, "comment_entry");
  p_data->split_delete_button = KP_W (xml, "split_delete_button");
  p_data->split_add_button = KP_W (xml, "split_add_button");

  gtk_paned_set_position (GTK_PANED (KP_W (xml, "splitpaned")), 300);
  
  p_data->split_view = kp_splits_view_new (NULL);
  gtk_container_add (GTK_CONTAINER (KP_W (xml, "splitview")), 
                     p_data->split_view);
  g_signal_connect (G_OBJECT (p_data->split_view), "split-selected",
                    G_CALLBACK (split_selected_cb), dialog);
  gtk_widget_show (p_data->split_view);
 
  selection = gtk_tree_view_get_selection (GTK_TREE_VIEW (p_data->split_view));
  g_signal_connect (G_OBJECT (selection), "changed",
                    G_CALLBACK (split_view_row_selected_cb), dialog);
  
  p_data->comment = KP_W (xml, "comment_textview");

  /* Date Entry */
  p_data->date_entry = kp_date_entry_new ();
  gtk_box_pack_start (GTK_BOX (KP_W (xml, "datebox")),
                      GTK_WIDGET (p_data->date_entry), TRUE, TRUE, 0);
  gtk_widget_show (GTK_WIDGET (p_data->date_entry));

  g_signal_connect (G_OBJECT (p_data->date_entry), "date-selected",
                    G_CALLBACK (date_selected_cb), dialog);
  g_signal_connect (G_OBJECT (p_data->date_entry), "time-selected",
                    G_CALLBACK (time_selected_cb), dialog);
  

  /* Sport entry */
  p_data->sportbox = kp_sport_entry_new ();
  gtk_box_pack_start (GTK_BOX (KP_W (xml, "sportbox")),
                      p_data->sportbox, TRUE, TRUE, 0);
  gtk_widget_show (p_data->sportbox);

  gtk_widget_set_sensitive (p_data->split_delete_button, FALSE);
  
  build_detail_menu (dialog);

  kp_gui_module_signals_connect_data (xml, signals_data, dialog);
  g_object_unref (G_OBJECT (xml));
}

static void
kp_new_split_workout_dialog_finalize (GObject *object)
{
  KPNewSplitWorkoutDialog *dialog;

  g_return_if_fail (object != NULL);
  g_return_if_fail (KP_IS_NEW_SPLIT_WORKOUT_DIALOG (object));

  dialog = KP_NEW_SPLIT_WORKOUT_DIALOG (object);

  g_return_if_fail (dialog->private_data != NULL);
  g_free (dialog->private_data);

  G_OBJECT_CLASS (parent_class)->finalize (object);
}

GtkWidget *
kp_new_split_workout_dialog_new (KPDate *date, KPCalendarEntry *entry)
{
  KPNewSplitWorkoutDialogPrivateData *p_data;
  KPNewSplitWorkoutDialog *dialog;
  GtkTextBuffer *buffer;
  const gchar *comment;
  KPTime time;

  dialog = g_object_new (KP_TYPE_NEW_SPLIT_WORKOUT_DIALOG, NULL);
  
  p_data = KP_NEW_SPLIT_WORKOUT_DIALOG_PRIVATE_DATA (dialog);
  buffer = gtk_text_view_get_buffer (GTK_TEXT_VIEW (p_data->comment));
  
  if (KP_IS_SPLIT_WORKOUT (entry)) {
    p_data->is_new_workout = FALSE;
    p_data->wo = KP_SPLIT_WORKOUT (entry);

    kp_calendar_entry_get_time (KP_CALENDAR_ENTRY (p_data->wo), &time);
    kp_date_entry_set_time (KP_DATE_ENTRY (p_data->date_entry), &time);
   
    comment = kp_param_list_get_string (KP_WORKOUT (p_data->wo)->param_list,
                                       "comment");
    if (comment != NULL)
      gtk_text_buffer_set_text (GTK_TEXT_BUFFER (buffer), comment, -1);
    kp_splits_view_set_splits (KP_SPLITS_VIEW (p_data->split_view), 
                               p_data->wo->splits);
  }
  else {
    p_data->is_new_workout = TRUE;
    p_data->wo = kp_split_workout_new ();
  
    kp_calendar_time_set_dmy (KP_CALENDAR_ENTRY (p_data->wo)->datetime,
                              date->d, date->m, date->y);
  }

  /* Don't allow adding empty workouts without any splits. That would
   * be OK but every workout must have "sport" and kp_workout_get_sport()
   * doesn't work without any splits.. */
  if (kp_split_workout_get_splits (p_data->wo) == NULL)
    gtk_widget_set_sensitive (GTK_WIDGET (p_data->okbutton), FALSE);
  
  kp_date_entry_set_date (KP_DATE_ENTRY (p_data->date_entry), date);
  
  set_detail_tree_view (dialog);

  return GTK_WIDGET (dialog);
}


static void
kp_new_split_workout_dialog_response_cb (GtkDialog *gtk_dialog, int response_id,
                                         KPNewSplitWorkoutDialog *dialog)
{
  KPNewSplitWorkoutDialogPrivateData *p_data;

  p_data = KP_NEW_SPLIT_WORKOUT_DIALOG_PRIVATE_DATA (dialog);

  if (response_id != GTK_RESPONSE_OK) {
    return;
  }

  /* Add details to the workout */
  gtk_tree_model_foreach (gtk_tree_view_get_model (GTK_TREE_VIEW (p_data->details_view)),
                         (GtkTreeModelForeachFunc) export_list_row,
                         (gpointer) dialog);

  /* Add text from the textview to the workout */
  set_comment (dialog);

  p_data->log = kp_main_window_get_log ();
 
  g_return_if_fail (KP_IS_TRAINING_LOG (p_data->log));
  
  /* If workout is valid, add it to the log. */
  if (p_data->is_new_workout) 
    kp_training_log_add (p_data->log, KP_CALENDAR_ENTRY (p_data->wo));
}


static void
set_comment (KPNewSplitWorkoutDialog *dialog)
{
  KPNewSplitWorkoutDialogPrivateData *p_data;
  GtkTextBuffer *buffer;
  GtkTextView *textview;
  GtkTextIter start;
  GtkTextIter end;
  gchar *text;

  p_data = KP_NEW_SPLIT_WORKOUT_DIALOG_PRIVATE_DATA (dialog);
  
  textview = GTK_TEXT_VIEW (p_data->comment);
  buffer = gtk_text_view_get_buffer (textview);

  gtk_text_buffer_get_start_iter (buffer, &start);
  gtk_text_buffer_get_end_iter (buffer, &end);
  text = gtk_text_buffer_get_text (buffer, &start, &end, TRUE);

  if (strlen (text) == 0)
    return;
  
  g_assert (text != NULL);
 
  kp_workout_set_comment (KP_WORKOUT (p_data->wo), text);
}


static gboolean
export_list_row (GtkTreeModel *model, GtkTreePath *path, GtkTreeIter *iter,
                 KPNewSplitWorkoutDialog *dialog)
{
  KPNewSplitWorkoutDialogPrivateData *p_data;
  KPParam *param;
  gchar *key;
  gchar *val;

  p_data = KP_NEW_SPLIT_WORKOUT_DIALOG_PRIVATE_DATA (dialog);
  g_return_val_if_fail (p_data != NULL, TRUE); 
  
  gtk_tree_model_get (model, iter, 0, &key, 1, &val, -1);

  param = kp_param_new (key);
  kp_param_set_automatic_as_string (param, val);
  kp_param_list_insert (KP_WORKOUT (p_data->wo)->param_list, "detail", param);

  g_free (key);
  g_free (val);

  return FALSE;
}

/*
 * notebook's note 'DETAILS'
 * 
 */

static void
set_detail_tree_view (KPNewSplitWorkoutDialog *dialog)
{
  KPNewSplitWorkoutDialogPrivateData *p_data;
  GtkTreeSelection *selection;
  GtkTreeViewColumn *column1;
  GtkTreeViewColumn *column2;
  GtkCellRenderer *renderer;
  GtkListStore *store;
  GtkTreeView *treeview;

  p_data = KP_NEW_SPLIT_WORKOUT_DIALOG_PRIVATE_DATA (dialog);
  treeview = GTK_TREE_VIEW (p_data->details_view);
  
  store = gtk_list_store_new (2, G_TYPE_STRING, G_TYPE_STRING);
  gtk_tree_view_set_model (treeview, GTK_TREE_MODEL (store));

  renderer = gtk_cell_renderer_text_new ();

  column1 = gtk_tree_view_column_new_with_attributes (_("Field"), renderer,
                                                     "text", 0, NULL);
  column2 = gtk_tree_view_column_new_with_attributes (("Value"), renderer,
                                                     "text", 1, NULL);
  gtk_tree_view_append_column (treeview, column1);
  gtk_tree_view_append_column (treeview, column2);

  selection = gtk_tree_view_get_selection (treeview);

  gtk_tree_selection_set_mode (selection, GTK_SELECTION_SINGLE);
  g_signal_connect (G_OBJECT (selection), "changed",
                    G_CALLBACK (detail_list_row_selected), dialog);
}

static void
detail_list_row_selected (GtkTreeSelection *selection,
                          KPNewSplitWorkoutDialog *dialog)
{
  KPNewSplitWorkoutDialogPrivateData *p_data;
  p_data = KP_NEW_SPLIT_WORKOUT_DIALOG_PRIVATE_DATA (dialog);
  gtk_widget_set_sensitive (p_data->detail_delete_button, TRUE);
}

static void
on_detail_delete_button_clicked (GtkButton *button,
                                 KPNewSplitWorkoutDialog *dialog)
{
  KPNewSplitWorkoutDialogPrivateData *p_data;
  GtkTreeSelection *selection;
  GtkTreeModel *model;
  GtkTreeIter iter;

  p_data = KP_NEW_SPLIT_WORKOUT_DIALOG_PRIVATE_DATA (dialog);
  
  selection = gtk_tree_view_get_selection (GTK_TREE_VIEW (p_data->details_view));
  model = gtk_tree_view_get_model (GTK_TREE_VIEW (p_data->details_view));

  if (gtk_tree_selection_get_selected (selection, &model, &iter)) {
    (void) gtk_list_store_remove (GTK_LIST_STORE (model), &iter);

    if (gtk_tree_model_iter_n_children (model, NULL) == 0)
      gtk_widget_set_sensitive (GTK_WIDGET (button), FALSE);
  }
}

static void
on_detail_add_button_clicked (GtkButton *button,
                              KPNewSplitWorkoutDialog *dialog)
{
  KPNewSplitWorkoutDialogPrivateData *p_data;
  GtkTreeModel *model;
  GtkTreeIter iter;

  gchar *value_str;
  gchar *name;

  p_data = KP_NEW_SPLIT_WORKOUT_DIALOG_PRIVATE_DATA (dialog);
  
  value_str = g_strdup (gtk_entry_get_text (GTK_ENTRY (p_data->detail_value)));
  name = kp_gui_get_option_menu_active (
      GTK_OPTION_MENU (p_data->detail_preset_menu));
  g_return_if_fail (name != NULL);
  
  model = gtk_tree_view_get_model (GTK_TREE_VIEW (p_data->details_view));
  gtk_list_store_append (GTK_LIST_STORE (model), &iter);
  gtk_list_store_set (GTK_LIST_STORE (model), &iter, 0, name, 1, value_str, -1);

  g_free (value_str);
  g_free (name);

  gtk_entry_set_text (GTK_ENTRY (p_data->detail_value), "");
}


static void
build_detail_menu (KPNewSplitWorkoutDialog *dialog)
{
  KPNewSplitWorkoutDialogPrivateData *p_data;
  KPPresetDataItem *detail;
  GtkWidget *item;
  GtkWidget *menu;
  GSList *list;

  g_return_if_fail (dialog != NULL);

  p_data = KP_NEW_SPLIT_WORKOUT_DIALOG_PRIVATE_DATA (dialog);
  
  menu = gtk_menu_new ();
  item = gtk_menu_item_new_with_label (_("Choose preset detail"));
  gtk_menu_shell_append (GTK_MENU_SHELL (menu), item);
 
  list = kp_preset_data_get_items (KP_PRESET_DATA_DETAIL); 
  for (;list; list = list->next) {
    detail = KP_PRESET_DATA_ITEM (list->data);
    
    item = gtk_menu_item_new_with_label (detail->name);
    gtk_menu_shell_append (GTK_MENU_SHELL (menu), item);
  }

  gtk_option_menu_set_menu (GTK_OPTION_MENU (p_data->detail_preset_menu), menu);
  gtk_widget_show_all (menu);
}


static void
split_view_row_selected_cb (GtkTreeSelection *selection,
                            KPNewSplitWorkoutDialog *dialog)
{
  KPNewSplitWorkoutDialogPrivateData *p_data;
  p_data = KP_NEW_SPLIT_WORKOUT_DIALOG_PRIVATE_DATA (dialog);

  gtk_widget_set_sensitive (p_data->split_delete_button, TRUE);
}


void
on_split_delete_button_clicked (GtkButton *button,
                                KPNewSplitWorkoutDialog *dialog)
{
  KPNewSplitWorkoutDialogPrivateData *p_data;
  guint split_n; 
    
  p_data = KP_NEW_SPLIT_WORKOUT_DIALOG_PRIVATE_DATA (dialog);

  split_n = kp_splits_view_get_selected_split_n (KP_SPLITS_VIEW (p_data->split_view));
  kp_split_workout_remove_split (KP_SPLIT_WORKOUT (p_data->wo), split_n);
 
  /* Must update the splits view */
  kp_splits_view_set_splits (KP_SPLITS_VIEW (p_data->split_view), 
                             p_data->wo->splits);

  
}

void
on_split_add_button_clicked (GtkButton *button, KPNewSplitWorkoutDialog *dialog)
{
  KPNewSplitWorkoutDialogPrivateData *p_data;
  KPSplit *split;
  GtkTextIter s, e;
  GtkTextBuffer *buffer;
  gchar *type;
  const gchar *sport;
  gdouble distance;
  guint n;
  gchar *comment;
  const gchar *time;
  gint msec;
  
  p_data = KP_NEW_SPLIT_WORKOUT_DIALOG_PRIVATE_DATA (dialog);
 
  sport = kp_sport_entry_get_sport (KP_SPORT_ENTRY (p_data->sportbox));
  if (sport == NULL) {
    kp_gui_err_message (GTK_WINDOW (dialog), 
                     _("You must enter a sport for the split!"));
    return;
  }
    
  /* Get split number */
  n = gtk_spin_button_get_value_as_int (GTK_SPIN_BUTTON (p_data->split_spin)),
  distance = gtk_spin_button_get_value (GTK_SPIN_BUTTON (p_data->split_distance_spin)),

  buffer = gtk_text_view_get_buffer (GTK_TEXT_VIEW (p_data->split_comment_entry));
  gtk_text_buffer_get_start_iter (buffer, &s);
  gtk_text_buffer_get_end_iter (buffer, &e);
  comment = gtk_text_buffer_get_text (buffer, &s, &e, TRUE);
  
  /* Get duration */
  time = gtk_entry_get_text (GTK_ENTRY (p_data->split_duration_entry));
  msec = kp_duration_str_to_ms (time);
  if (msec < 0) {
    kp_gui_report_error (GTK_WINDOW (dialog), NULL,
                      _("Duration entry is invalid!"));
    return;
  }

  /* Get split type */
  type = kp_gui_get_option_menu_active (
      GTK_OPTION_MENU (p_data->split_type_menu));
  
  split = kp_split_new_full (type, n, distance, msec, sport, comment);
  kp_split_workout_add_split_struct (p_data->wo, split);

  g_free (comment);
  
  kp_splits_view_set_splits (KP_SPLITS_VIEW (p_data->split_view), 
                             p_data->wo->splits);
  gtk_spin_button_spin (GTK_SPIN_BUTTON (p_data->split_spin), 
                        GTK_SPIN_STEP_FORWARD, 1.0);

  gtk_text_buffer_set_text (buffer, "", -1);
  gtk_entry_set_text (GTK_ENTRY (p_data->split_duration_entry), "");

  g_free (type);

  gtk_widget_set_sensitive (GTK_WIDGET (p_data->okbutton), TRUE);
}


static void
time_selected_cb (KPDateEntry *entry, KPTime *time, 
                  KPNewSplitWorkoutDialog *dialog)
{
  KPNewSplitWorkoutDialogPrivateData *p_data;
  p_data = KP_NEW_SPLIT_WORKOUT_DIALOG_PRIVATE_DATA (dialog);

  kp_calendar_time_set_hmst (KP_CALENDAR_ENTRY (p_data->wo)->datetime,
                             time->h, time->m, time->s, 0);
}


static void
date_selected_cb (KPDateEntry *entry, KPDate *date, 
                  KPNewSplitWorkoutDialog *dialog)
{
  KPNewSplitWorkoutDialogPrivateData *p_data;
  p_data = KP_NEW_SPLIT_WORKOUT_DIALOG_PRIVATE_DATA (dialog);

  kp_calendar_time_set_dmy (KP_CALENDAR_ENTRY (p_data->wo)->datetime,
                            date->d, date->m, date->y);
}

