#include <string.h>
#include <time.h>

#include <gconf/gconf-client.h>
#include <glade/glade.h>
#include <gtk/gtk.h>

#include "kppreferencesdialog.h"
#include "kpcalendarview.h"
#include "kpmainwindow.h"
#include "kpstatusbar.h"
#include "kpguiutils.h"

#include "../kptraininglog.h"
#include "../kppresetdata.h"
#include "../kipina-i18n.h"
#include "../kpsettings.h"
#include "../kpworkout.h"
#include "../kpplugin.h"
#include "../kputil.h"

static void     kp_preferences_dialog_class_init  (KPPreferencesDialogClass *klass);
static void     kp_preferences_dialog_init        (KPPreferencesDialog *dialog);
static void     kp_preferences_dialog_finalize    (GObject *object);
static void     kp_preferences_dialog_response_cb (GtkDialog *gtkdialog,
                                                   int response_id,
                                                   KPPreferencesDialog *dialog);
static void     detail_list_row_selected          (GtkTreeSelection *selection,
                                                   gpointer data);
static void     detail_entry_changed              (GtkEditable *editable,
                                                   KPPreferencesDialog *dialog);
static void     on_detail_add_button_clicked      (GtkButton *button,
                                                   gpointer data);
static void     on_detail_remove_button_clicked   (GtkButton *button,
                                                   gpointer data);
static void     set_detail_tree_view              (KPPreferencesDialog *dialog);
static gboolean set_detail_list_row               (GtkTreeModel *model,
                                                   GtkTreePath *path,
                                                   GtkTreeIter *iter,
                                                   gpointer data);
static void     sport_list_row_selected           (GtkTreeSelection *selection,
                                                   gpointer data);
static void     on_sport_add_button_clicked       (GtkButton *button,
                                                   gpointer data);
static void     on_sport_color_button_clicked     (GtkButton *button,
                                                   KPPreferencesDialog *dialog);
static void     on_sport_remove_button_clicked    (GtkButton *button,
                                                   gpointer data);
static void     sport_color_set                   (GtkColorButton *button,
                                                   KPPreferencesDialog *dialog);
static void     sport_entry_changed               (GtkEditable *editable,
                                                   KPPreferencesDialog *dialog);
static void     set_sport_tree_view               (KPPreferencesDialog *dialog);
static gboolean set_sport_list_row                (GtkTreeModel *model,
                                                   GtkTreePath *path,
                                                   GtkTreeIter *iter,
                                                   gpointer data);
static void     set_plugin_tree_view              (KPPreferencesDialog *dialog);
static void     plugin_load_state_toggled         (GtkCellRendererToggle *renderer,
                                                   gchar *path,
                                                   KPPreferencesDialog *dialog);
static void     populate_tree_sports              (GtkTreeView *treeview);
static void     populate_tree_details             (GtkTreeView *treeview);
static void     init_values                       (KPPreferencesDialog *dialog);
static gboolean set_values                        (KPPreferencesDialog *dialog);
static void     on_statusbar_show_sth_button_clicked
                                                  (GtkCheckButton *button,
                                                   KPPreferencesDialog *dialog);
static void     on_browse_button_clicked          (GtkButton *button,
                                                   KPPreferencesDialog *dialog);


static GuiModuleSignalsData signals_data[] = {
{"detail_add_button",   "clicked",   CB (on_detail_add_button_clicked),    NULL},
{"detail_remove_button","clicked",   CB (on_detail_remove_button_clicked), NULL},
{"sport_add_button",    "clicked",   CB (on_sport_add_button_clicked),     NULL},
{"sport_color_button",  "clicked",   CB (on_sport_color_button_clicked),   NULL},
{"sport_remove_button", "clicked",   CB (on_sport_remove_button_clicked),  NULL},
{"html_dir_browse",     "clicked",   CB (on_browse_button_clicked),        NULL},  
{"default_dir_browse",  "clicked",   CB (on_browse_button_clicked),        NULL},
{"sport_entry",         "changed",   CB (sport_entry_changed),             NULL},
{"sport_long_entry",    "changed",   CB (sport_entry_changed),             NULL},
{"sport_abbr_entry",    "changed",   CB (sport_entry_changed),             NULL},
{"sport_color_button",  "color-set", CB (sport_color_set),                 NULL},
{"detail_entry",        "changed",   CB (detail_entry_changed),            NULL},
{"detail_long_entry",   "changed",   CB (detail_entry_changed),            NULL},
{"detail_abbr_entry",   "changed",   CB (detail_entry_changed),            NULL},
{ NULL,                  NULL,       NULL,                                 NULL},
};

typedef struct KPPreferencesDialogPrivateData_
{
  KPTrainingLog *log;
  KPCalendarView *cv;
  KPWorkout *wo;

  GtkWidget *detail_add_button;
  GtkWidget *detail_remove_button;
  GtkWidget *detail_treeview;
  GtkWidget *detail_entry;
  GtkWidget *detail_long_entry;
  GtkWidget *detail_abbr_entry;
  
  GtkWidget *sport_add_button;
  GtkWidget *sport_remove_button;
  GtkWidget *sport_treeview;
  GtkWidget *sport_entry;
  GtkWidget *sport_color_button;
  GtkWidget *sport_long_entry;
  GtkWidget *sport_abbr_entry;

  GtkWidget *plugin_treeview;
  GtkWidget *plugin_title_label;
  GtkWidget *plugin_author_label;
  GtkWidget *plugin_homepage_label;
  GtkWidget *plugin_summary_label;
  GtkWidget *plugin_filename_label;
  GtkWidget *plugin_description_textview;
  
  GtkWidget *sb[KP_STATUSBAR_F_N];
  
  GtkWidget *window_geometry_button;
  GtkWidget *default_dir_entry;
  GtkWidget *html_dir_entry;
  GtkWidget *exit_button;

  GConfClient *client;
} KPPreferencesDialogPrivateData;

#define KP_PREFERENCES_DIALOG_PRIVATE_DATA(widget) \
  (((KPPreferencesDialogPrivateData*) \
  (KP_PREFERENCES_DIALOG (widget)->private_data)))

#define KEY_HTML_OUTPUT_DIR   "/apps/kipina/preferences/html_output_dir"
#define KEY_DEFAULT_DIR       "/apps/kipina/preferences/default_dir"
#define KEY_STATUSBAR_ITEMS   "/apps/kipina/preferences/statusbar_items"
#define KEY_WINDOW_GEOMETRY   "/apps/kipina/preferences/save_window_geometry"

enum {
  COLUMN_SPORT,
  COLUMN_SPORT_COLOR,
  COLUMN_SPORT_ABBR,
  COLUMN_SPORT_DESCRIPTION
};
    
enum {
  COLUMN_DETAIL,
  COLUMN_DETAIL_ABBR,
  COLUMN_DETAIL_DESCRIPTION
};

static GObjectClass *parent_class = NULL;


GType
kp_preferences_dialog_get_type (void)
{
  static GType kp_preferences_dialog_type = 0;

  if (kp_preferences_dialog_type == 0) {
    static const GTypeInfo our_info = {
      sizeof (KPPreferencesDialogClass),
      NULL,
      NULL,
      (GClassInitFunc) kp_preferences_dialog_class_init,
      NULL,
      NULL,
      sizeof (KPPreferencesDialog),
      0,
      (GInstanceInitFunc) kp_preferences_dialog_init,
      NULL,
    };

    kp_preferences_dialog_type = g_type_register_static (GTK_TYPE_DIALOG,
                                                         "KPPreferencesDialog",
                                                         &our_info, 0);
  }

  return kp_preferences_dialog_type;
}

static void
kp_preferences_dialog_class_init (KPPreferencesDialogClass *klass)
{
  GObjectClass *object_class = G_OBJECT_CLASS (klass);

  parent_class = g_type_class_peek_parent (klass);
  object_class->finalize = kp_preferences_dialog_finalize;
}

static gboolean
is_statusbar_item_active (GSList *items, const gchar *item)
{
  GSList *node;

  for (node = items; node; node = node->next) 
    if (strcmp (node->data, item) == 0)
      return TRUE;

  return FALSE;
}


static void
kp_preferences_dialog_init (KPPreferencesDialog *dialog)
{
  KPPreferencesDialogPrivateData *p_data;
  KPStatusbarFieldData *field;
  GtkWidget *notebook;
  GladeXML *xml;
  GSList *items;
  guint i;

  xml = kp_gui_load ("preferences", "preferences");
  
  dialog->private_data = g_new0 (KPPreferencesDialogPrivateData, 1);
  p_data = KP_PREFERENCES_DIALOG_PRIVATE_DATA (dialog);
  p_data->client = gconf_client_get_default ();

  g_signal_connect (G_OBJECT (dialog),
                   "response",
                    G_CALLBACK (kp_preferences_dialog_response_cb), dialog);

  gtk_dialog_set_has_separator (GTK_DIALOG (dialog), FALSE);
  gtk_container_set_border_width (GTK_CONTAINER (dialog), 7);
  
  gtk_box_set_spacing (GTK_BOX (GTK_DIALOG (dialog)->vbox), 5);
  gtk_dialog_set_default_response (GTK_DIALOG (dialog), GTK_RESPONSE_NONE);
  gtk_window_set_title (GTK_WINDOW (dialog),
                     _("Kipin\303\244 Preferences"));

  p_data->exit_button = gtk_dialog_add_button (GTK_DIALOG (dialog),
                                               GTK_STOCK_CLOSE,
                                               GTK_RESPONSE_CLOSE);

  notebook = KP_W (xml, "preferences");
  gtk_container_add (GTK_CONTAINER (GTK_DIALOG (dialog)->vbox), notebook);
  gtk_container_set_border_width (GTK_CONTAINER (notebook), 5);

  p_data->wo = kp_workout_new ();
  
  /* Put widgets to private data */
  p_data->window_geometry_button = KP_W (xml, "window_geometry_button");
  p_data->detail_remove_button = KP_W (xml, "detail_remove_button");  
  p_data->detail_add_button = KP_W (xml, "detail_add_button");  
  p_data->detail_treeview = KP_W (xml, "detail_treeview");
  p_data->detail_entry = KP_W (xml, "detail_entry");
  p_data->detail_long_entry = KP_W (xml, "detail_long_entry");
  p_data->detail_abbr_entry = KP_W (xml, "detail_abbr_entry");

  p_data->sport_remove_button = KP_W (xml, "sport_remove_button");  
  p_data->sport_add_button = KP_W (xml, "sport_add_button");  
  p_data->sport_treeview = KP_W (xml, "sport_treeview");
  p_data->sport_entry = KP_W (xml, "sport_entry");
  p_data->sport_color_button = KP_W (xml, "sport_color_button");
  p_data->sport_long_entry = KP_W (xml, "sport_long_entry");
  p_data->sport_abbr_entry = KP_W (xml, "sport_abbr_entry");

  p_data->plugin_treeview = KP_W (xml, "plugin_treeview");
  p_data->plugin_title_label = KP_W (xml, "plugin_title_label");
  p_data->plugin_author_label = KP_W (xml, "plugin_author_label");
  p_data->plugin_homepage_label = KP_W (xml, "plugin_homepage_label");
  p_data->plugin_summary_label = KP_W (xml, "plugin_summary_label");
  p_data->plugin_filename_label = KP_W (xml, "plugin_filename_label");
  p_data->plugin_description_textview = KP_W (xml, "plugin_description_textview");

  p_data->default_dir_entry = KP_W (xml, "default_dir_entry");
  p_data->html_dir_entry = KP_W (xml, "html_dir_entry");

  init_values (dialog);
 
  items = gconf_client_get_list (p_data->client, KEY_STATUSBAR_ITEMS,
                                 GCONF_VALUE_STRING, NULL);
    
  for (i=0; i < KP_STATUSBAR_F_N; i++) {
  
    field = kp_statusbar_get_field (i);
    
    
    p_data->sb[i] = KP_W (xml, field->name);
    gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (p_data->sb[i]),
                                  is_statusbar_item_active (items, field->setting));
    
    g_object_set_data (G_OBJECT (p_data->sb[i]), "setting_name",
                       field->setting);
    
    g_signal_connect (G_OBJECT (p_data->sb[i]), "clicked",
                      G_CALLBACK (on_statusbar_show_sth_button_clicked),
                      dialog);
    
    g_free (field->name);
    g_free (field);
  }
  
  kp_gui_module_signals_connect_data (xml, signals_data, dialog);
  
  g_object_unref (G_OBJECT (xml));
}


static void
kp_preferences_dialog_finalize (GObject *object)
{
  KPPreferencesDialog *dialog;

  g_return_if_fail (object != NULL);
  g_return_if_fail (KP_IS_PREFERENCES_DIALOG (object));

  dialog = KP_PREFERENCES_DIALOG (object);

  g_return_if_fail (dialog->private_data != NULL);
  g_free (dialog->private_data);

  G_OBJECT_CLASS (parent_class)->finalize (object);
}


GtkWidget *
kp_preferences_dialog_new (void)
{
  KPPreferencesDialog *dialog;

  dialog = g_object_new (KP_TYPE_PREFERENCES_DIALOG, NULL);

  set_detail_tree_view (dialog);
  set_sport_tree_view (dialog);
  set_plugin_tree_view (dialog);

  return GTK_WIDGET (dialog);
}


static void
kp_preferences_dialog_response_cb (GtkDialog *gtk_dialog, int response_id,
                                   KPPreferencesDialog *dialog)
{
  KPStatusbar *sbar;

  if (response_id != GTK_RESPONSE_CLOSE)
    return;
  if (set_values (dialog) == FALSE)
    return;
    
  kp_preset_data_write (NULL);
      
  kp_settings_save ();

  sbar = kp_main_window_get_statusbar ();
  kp_statusbar_set_message (sbar, _("Settings saved."));
            
  kp_debug ("Settings saved.\n");
}


static void
init_values (KPPreferencesDialog *dialog)
{
  KPPreferencesDialogPrivateData *p_data;
  GtkToggleButton *w_button;
  gboolean w_geometry;
  gchar *default_dir;
  gchar *out_dir;

  p_data = KP_PREFERENCES_DIALOG_PRIVATE_DATA (dialog);

  out_dir = gconf_client_get_string (p_data->client, KEY_HTML_OUTPUT_DIR, NULL);
  default_dir = gconf_client_get_string (p_data->client, KEY_DEFAULT_DIR, NULL);

  w_button = GTK_TOGGLE_BUTTON (p_data->window_geometry_button);
  w_geometry = gconf_client_get_bool (p_data->client, KEY_WINDOW_GEOMETRY, NULL);
  gtk_toggle_button_set_active (w_button, w_geometry);
  
  if (default_dir) {
    gtk_entry_set_text (GTK_ENTRY (p_data->default_dir_entry), default_dir);
    g_free (default_dir);
  }
  if (out_dir) {
    gtk_entry_set_text (GTK_ENTRY (p_data->html_dir_entry), out_dir);
    g_free (out_dir);
  }
}


static gboolean
set_values (KPPreferencesDialog *dialog)
{
  KPPreferencesDialogPrivateData *p_data;
  GtkToggleButton *wbutton;
  GtkTreeModel *model;
  const gchar *dir;
  const gchar *out_dir;
  
  p_data = KP_PREFERENCES_DIALOG_PRIVATE_DATA (dialog);

  dir = gtk_entry_get_text (GTK_ENTRY (p_data->default_dir_entry));
  if (dir)
    gconf_client_set_string (p_data->client, KEY_DEFAULT_DIR, dir, NULL);

  out_dir = gtk_entry_get_text (GTK_ENTRY (p_data->html_dir_entry));
  if (out_dir)
    gconf_client_set_string (p_data->client, KEY_HTML_OUTPUT_DIR, out_dir, NULL);
  
  wbutton = GTK_TOGGLE_BUTTON (p_data->window_geometry_button);
  
  gconf_client_set_bool (p_data->client,
                         KEY_WINDOW_GEOMETRY,
                         gtk_toggle_button_get_active (wbutton),
                         NULL);

  /* Set sport list items */
  kp_preset_data_clean (KP_PRESET_DATA_SPORT);
  model = gtk_tree_view_get_model (GTK_TREE_VIEW (p_data->sport_treeview));
  gtk_tree_model_foreach (model, set_sport_list_row, NULL);

  /* Set detail list items */
  kp_preset_data_clean (KP_PRESET_DATA_DETAIL);
  model = gtk_tree_view_get_model (GTK_TREE_VIEW (p_data->detail_treeview));
  gtk_tree_model_foreach (model, set_detail_list_row, dialog);

  return TRUE;
}


static void
statusbar_items_to_gconf (KPPreferencesDialog *dialog, GSList *strings)
{
  KPPreferencesDialogPrivateData *p_data;
  GSList *items_list;
  GSList *node;
  GConfValue *items;
  GConfValue *val;

  p_data = KP_PREFERENCES_DIALOG_PRIVATE_DATA (dialog);

  items = gconf_value_new (GCONF_VALUE_LIST);
  gconf_value_set_list_type (items, GCONF_VALUE_STRING);

  items_list = NULL;

  for (node = strings; node; node = node->next) {
    val = gconf_value_new_from_string (GCONF_VALUE_STRING, node->data, NULL);
    items_list = g_slist_prepend (items_list, val);
  }
  gconf_value_set_list_nocopy (items, items_list);
  items_list = NULL;

  gconf_client_set (p_data->client, KEY_STATUSBAR_ITEMS, items, NULL);
}


static void
update_gconf_list (KPPreferencesDialog *dialog, const gchar *item,
                   gboolean state)
{
  KPPreferencesDialogPrivateData *p_data;
  GConfValue *listvalue;
  GSList *strings = NULL;
  GSList *values;
  GSList *list;
  GSList *tmp;
  const gchar *str;
  
  p_data = KP_PREFERENCES_DIALOG_PRIVATE_DATA (dialog);

  listvalue = gconf_client_get (p_data->client, KEY_STATUSBAR_ITEMS, NULL);

  if (listvalue == NULL) {
    strings = g_slist_prepend (strings, g_strdup (item));
    statusbar_items_to_gconf (dialog, strings);

    for (list = strings; list; list = list->next)
      g_free (list->data);
    g_slist_free (strings);
    
    return;
  }
  
  values = gconf_value_get_list (listvalue);

  /* Copy the strings to list */
  for (list = values; list; list = list->next) {
    str = gconf_value_get_string ((GConfValue *) list->data);

    if (str) {
      tmp = g_slist_find_custom (strings, str, (GCompareFunc) strcmp);
     
      if (!tmp)
        /* str isn't already in the list */
        strings = g_slist_prepend (strings, g_strdup (str));
    }
  }

  if (state) {
    /* Add an item to list */
    strings = g_slist_prepend (strings, g_strdup (item));
  } else {
    /* Remove an item from list */    
    for (list = strings; list; list = list->next) {
      if (strcmp (list->data, item) == 0) {
        tmp = list;
        strings = g_slist_remove_link (strings, tmp);
        g_free (tmp->data);
        g_slist_free_1 (tmp);
        break;
      }
    }
  }
  statusbar_items_to_gconf (dialog, strings);
 
  /* Free the temporary list */
  for (list = strings; list; list = list->next)
    if (list->data)
      g_free (list->data);
  g_slist_free (strings);
 
  if (listvalue)
    gconf_value_free (listvalue);
}


static void
on_statusbar_show_sth_button_clicked (GtkCheckButton *button,
                                      KPPreferencesDialog *dialog)
{
  KPPreferencesDialogPrivateData *p_data;
  gboolean state;
  gchar *setting;
  
  p_data = KP_PREFERENCES_DIALOG_PRIVATE_DATA (dialog);
 
  setting = g_object_get_data (G_OBJECT (button), "setting_name");
  g_return_if_fail (setting != NULL);
  
  state = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (button));

  update_gconf_list (dialog, setting, state);
}


static void
on_browse_button_clicked (GtkButton *button, KPPreferencesDialog *dialog)
{
  KPPreferencesDialogPrivateData *p_data;
  const gchar *wname;
  gchar *dirname;
  
  p_data = KP_PREFERENCES_DIALOG_PRIVATE_DATA (dialog);
  
  dirname = kp_gui_get_dir (GTK_WINDOW (dialog));
  wname = gtk_widget_get_name (GTK_WIDGET (button));
  
  if (dirname == NULL)
    return;
  
  if (strcmp (wname, "html_dir_browse") == 0)
    gtk_entry_set_text (GTK_ENTRY (p_data->html_dir_entry), dirname);
  else if (strcmp (wname, "default_dir_browse") == 0)
    gtk_entry_set_text (GTK_ENTRY (p_data->default_dir_entry), dirname);

  g_free (dirname);
}


static gboolean
set_sport_list_row (GtkTreeModel *model, GtkTreePath *path, GtkTreeIter *iter,
                    gpointer data)
{
  KPPresetDataItem *sport;
  gchar *markup;
  gchar *descr;
  gchar *name;
  gchar *abbr;
  gchar *color;
 
  gtk_tree_model_get (model, iter,
                      COLUMN_SPORT, &name,
                      COLUMN_SPORT_DESCRIPTION, &descr,
                      COLUMN_SPORT_ABBR, &abbr,
                      COLUMN_SPORT_COLOR, &markup,
                     -1);

  sport = kp_preset_data_item_new ();
  sport->name = name;
  sport->description = descr;
  sport->abbreviation = abbr;

  if (!pango_parse_markup (markup, -1, 0, NULL, &color, NULL, NULL))
    color = g_strdup ("black");

  sport->data = color;
  kp_preset_data_add_item (KP_PRESET_DATA_SPORT, sport);
 
  return FALSE;
}

/*
 * notebook's note 'DETAILS'
 * 
 */
static void
set_detail_tree_view (KPPreferencesDialog *dialog)
{
  KPPreferencesDialogPrivateData *p_data;
  GtkTreeSelection *selection;
  GtkTreeViewColumn *col;
  GtkCellRenderer *renderer;
  GtkListStore *store;
  GtkTreeView *treeview;

  p_data = KP_PREFERENCES_DIALOG_PRIVATE_DATA (dialog);
  treeview = GTK_TREE_VIEW (p_data->detail_treeview);
  
  store = gtk_list_store_new (3, G_TYPE_STRING, G_TYPE_STRING, G_TYPE_STRING);
  gtk_tree_view_set_model (treeview, GTK_TREE_MODEL (store));

  renderer = gtk_cell_renderer_text_new ();

  col = gtk_tree_view_column_new_with_attributes (_("Detail"), renderer,
                                                 "text",
                                                  COLUMN_DETAIL,
                                                  NULL);
  gtk_tree_view_append_column (treeview, col);

  col = gtk_tree_view_column_new_with_attributes (_("Abbreviation"), renderer,
                                                 "text",
                                                  COLUMN_DETAIL_ABBR,
                                                  NULL);
  gtk_tree_view_append_column (treeview, col);

  col = gtk_tree_view_column_new_with_attributes (_("Long description"), renderer,
                                                 "text",
                                                  COLUMN_DETAIL_DESCRIPTION,
                                                  NULL);
  gtk_tree_view_append_column (treeview, col);

  selection = gtk_tree_view_get_selection (treeview);

  gtk_tree_selection_set_mode (selection, GTK_SELECTION_SINGLE);
  g_signal_connect (G_OBJECT (selection), "changed",
                    G_CALLBACK (detail_list_row_selected), dialog);

  populate_tree_details (treeview);
}


static void
populate_tree_sports (GtkTreeView *treeview)
{
  KPPresetDataItem *sport;
  GtkTreeModel *model;
  GtkTreeIter iter;
  gchar *markup;
  GSList *list;
  
  model = gtk_tree_view_get_model (GTK_TREE_VIEW (treeview));
  
  list = kp_preset_data_get_items (KP_PRESET_DATA_SPORT);
  
  kp_debug ("Found %u sports!", g_slist_length (list));
  
  for (; list; list = list->next) {
    sport = KP_PRESET_DATA_ITEM (list->data);

    if (sport->data)
      markup = kp_color_str_to_markup (sport->data);
    else
      markup = "black";
    
    gtk_list_store_append (GTK_LIST_STORE (model), &iter);
    gtk_list_store_set (GTK_LIST_STORE (model), &iter,
                        COLUMN_SPORT, sport->name,
                        COLUMN_SPORT_DESCRIPTION, sport->description,
                        COLUMN_SPORT_ABBR, sport->abbreviation,
                        COLUMN_SPORT_COLOR, markup,
                       -1);
  }
}


static void
populate_tree_details (GtkTreeView *treeview)
{
  KPPresetDataItem *detail;
  GtkTreeModel *model;
  GtkTreeIter iter;
  GSList *list;
  
  model = gtk_tree_view_get_model (GTK_TREE_VIEW (treeview));
  
  list = kp_preset_data_get_items (KP_PRESET_DATA_DETAIL); 
  for (;list; list = list->next) {
    detail = KP_PRESET_DATA_ITEM (list->data);
    
    gtk_list_store_append (GTK_LIST_STORE (model), &iter);
    gtk_list_store_set (GTK_LIST_STORE (model), &iter,
                        COLUMN_DETAIL, detail->name,
                        COLUMN_DETAIL_ABBR, detail->abbreviation,
                        COLUMN_DETAIL_DESCRIPTION, detail->description,
                       -1);
  }
}


static void
detail_entry_changed (GtkEditable *editable, KPPreferencesDialog *dialog)
{
  KPPreferencesDialogPrivateData *p_data;
  GtkTreeSelection *selection;
  GtkTreeModel *model;
  GtkTreeIter iter;
  const gchar *wname;
  gchar *str;
  gint field;

  wname = gtk_widget_get_name (GTK_WIDGET (editable));

  if (strcmp (wname, "detail_entry") == 0)
    field = COLUMN_DETAIL;
  else if (strcmp (wname, "detail_long_entry") == 0)
    field = COLUMN_DETAIL_DESCRIPTION;
  else if (strcmp (wname, "detail_abbr_entry") == 0)
    field = COLUMN_DETAIL_ABBR;
  else
    g_return_if_reached ();
  
  p_data = KP_PREFERENCES_DIALOG_PRIVATE_DATA (dialog);

  selection = gtk_tree_view_get_selection (GTK_TREE_VIEW (p_data->detail_treeview));
  
  if (!gtk_tree_selection_get_selected (selection, &model, &iter))
    return;

  str = gtk_editable_get_chars (editable, 0, -1);

  gtk_list_store_set (GTK_LIST_STORE (model), &iter, field, str, -1);
}



static void
detail_list_row_selected (GtkTreeSelection *selection, gpointer data)
{
  KPPreferencesDialogPrivateData *p_data;
  GtkTreeModel *model;
  GtkTreeIter iter;
  gchar *detail;
  gchar *descr;
  gchar *abbr;
  
  p_data = KP_PREFERENCES_DIALOG_PRIVATE_DATA (KP_PREFERENCES_DIALOG (data));
  gtk_widget_set_sensitive (p_data->detail_remove_button, TRUE);

  if (gtk_tree_selection_get_selected (selection, &model, &iter)) {
    gtk_tree_model_get (model, &iter,
                        COLUMN_DETAIL, &detail,
                        COLUMN_DETAIL_ABBR, &abbr,
                        COLUMN_DETAIL_DESCRIPTION, &descr,
                       -1);

    if (abbr)
      gtk_entry_set_text (GTK_ENTRY (p_data->detail_abbr_entry), abbr);
    if (descr)
      gtk_entry_set_text (GTK_ENTRY (p_data->detail_long_entry), descr);
    if (detail) {
      gtk_entry_set_text (GTK_ENTRY (p_data->detail_entry), detail);
      g_free (detail);
    }
  }
}


static void
on_detail_remove_button_clicked (GtkButton *button, gpointer data)
{
  KPPreferencesDialogPrivateData *p_data;
  GtkTreeSelection *selection;
  GtkTreeModel *model;
  GtkTreeIter iter;

  p_data = KP_PREFERENCES_DIALOG_PRIVATE_DATA (KP_PREFERENCES_DIALOG (data));
  
  selection = gtk_tree_view_get_selection (GTK_TREE_VIEW (p_data->detail_treeview));

  if (gtk_tree_selection_get_selected (selection, &model, &iter)) {
    (void) gtk_list_store_remove (GTK_LIST_STORE (model), &iter);

    if (gtk_tree_model_iter_n_children (model, NULL) == 0)
      gtk_widget_set_sensitive (GTK_WIDGET (button), FALSE);
  }
  gtk_entry_set_text (GTK_ENTRY (p_data->detail_long_entry), "");
  gtk_entry_set_text (GTK_ENTRY (p_data->detail_abbr_entry), "");
  gtk_entry_set_text (GTK_ENTRY (p_data->detail_entry), "");
}


static void
on_detail_add_button_clicked (GtkButton *button, gpointer data)
{
  KPPreferencesDialogPrivateData *p_data;
  GtkTreeModel *model;
  GtkTreeIter iter;
  const gchar *entry;
  const gchar *abbr;
  const gchar *descr;
  
  p_data = KP_PREFERENCES_DIALOG_PRIVATE_DATA (KP_PREFERENCES_DIALOG (data));
  
  descr = gtk_entry_get_text (GTK_ENTRY (p_data->detail_long_entry));
  entry = gtk_entry_get_text (GTK_ENTRY (p_data->detail_entry));
  abbr = gtk_entry_get_text (GTK_ENTRY (p_data->detail_abbr_entry));
  
  if (strlen (entry) == 0 || strlen (abbr) == 0)
    return;
  
  model = gtk_tree_view_get_model (GTK_TREE_VIEW (p_data->detail_treeview));
  gtk_list_store_append (GTK_LIST_STORE (model), &iter);
  gtk_list_store_set (GTK_LIST_STORE (model), &iter,
                      COLUMN_DETAIL, entry,
                      COLUMN_DETAIL_ABBR, abbr,
                      COLUMN_DETAIL_DESCRIPTION, descr,
                     -1);
  
  gtk_entry_set_text (GTK_ENTRY (p_data->detail_long_entry), "");
  gtk_entry_set_text (GTK_ENTRY (p_data->detail_abbr_entry), "");
  gtk_entry_set_text (GTK_ENTRY (p_data->detail_entry), "");
}


static gboolean
set_detail_list_row (GtkTreeModel *model, GtkTreePath *path, GtkTreeIter *iter,
                     gpointer data)
{
  KPPresetDataItem *detail;
  gchar *descr;
  gchar *abbr;
  gchar *val;

  gtk_tree_model_get (model, iter,
                      COLUMN_DETAIL, &val,
                      COLUMN_DETAIL_ABBR, &abbr,
                      COLUMN_DETAIL_DESCRIPTION, &descr,
                     -1);

  detail = kp_preset_data_item_new ();
  detail->name = val;
  detail->description = descr;
  detail->abbreviation = abbr;

  kp_preset_data_add_item (KP_PRESET_DATA_DETAIL, detail);
  
  return FALSE;
}


/*
 * notebook's note 'SPORTS'
 * 
 */
static void
set_sport_tree_view (KPPreferencesDialog *dialog)
{
  KPPreferencesDialogPrivateData *p_data;
  GtkTreeSelection *selection;
  GtkTreeViewColumn *col;
  GtkCellRenderer *renderer;
  GtkListStore *store;
  GtkTreeView *treeview;

  p_data = KP_PREFERENCES_DIALOG_PRIVATE_DATA (dialog);
  treeview = GTK_TREE_VIEW (p_data->sport_treeview);
  
  store = gtk_list_store_new (4, G_TYPE_STRING, G_TYPE_STRING, G_TYPE_STRING,
                              G_TYPE_STRING);
  gtk_tree_view_set_model (treeview, GTK_TREE_MODEL (store));

  renderer = gtk_cell_renderer_text_new ();
 
  col = gtk_tree_view_column_new_with_attributes (_("Sport"), renderer, "text",
                                                  COLUMN_SPORT,
                                                  NULL);
  gtk_tree_view_append_column (treeview, col);

  col = gtk_tree_view_column_new_with_attributes (_("Color"), renderer,
                                                 "markup",
                                                  COLUMN_SPORT_COLOR,
                                                  NULL);
  gtk_tree_view_append_column (treeview, col);

  col = gtk_tree_view_column_new_with_attributes (_("Abbreviation"), renderer,
                                                  "text",
                                                  COLUMN_SPORT_ABBR,
                                                  NULL);
  gtk_tree_view_append_column (treeview, col);

  col = gtk_tree_view_column_new_with_attributes (_("Long description"),
                                                  renderer, "text", 
                                                  COLUMN_SPORT_DESCRIPTION,
                                                  NULL);
  gtk_tree_view_append_column (treeview, col);

  selection = gtk_tree_view_get_selection (treeview);

  gtk_tree_selection_set_mode (selection, GTK_SELECTION_SINGLE);
  g_signal_connect (G_OBJECT (selection), "changed",
                    G_CALLBACK (sport_list_row_selected), dialog);
  
  populate_tree_sports (treeview);
}


static void
sport_list_row_selected (GtkTreeSelection *selection, gpointer data)
{
  KPPreferencesDialogPrivateData *p_data;
  GtkTreeModel *model;
  GtkTreeIter iter;
  GdkColor gdk_c;
  gchar *markup;
  gchar *descr;
  gchar *sport;
  gchar *abbr;
  gchar *color;
  
  p_data = KP_PREFERENCES_DIALOG_PRIVATE_DATA (KP_PREFERENCES_DIALOG (data));
  gtk_widget_set_sensitive (p_data->sport_remove_button, TRUE);
  
  if (gtk_tree_selection_get_selected (selection, &model, &iter)) {
    gtk_tree_model_get (model, &iter,
                        COLUMN_SPORT, &sport,
                        COLUMN_SPORT_COLOR, &markup,
                        COLUMN_SPORT_ABBR, &abbr,
                        COLUMN_SPORT_DESCRIPTION, &descr, 
                       -1);
    
    if (!pango_parse_markup (markup, -1, 0, NULL, &color, NULL, NULL))
      color = "black";

    gdk_color_parse (color, &gdk_c);
    gtk_color_button_set_color (GTK_COLOR_BUTTON (p_data->sport_color_button),
                               &gdk_c);

    if (descr)
      gtk_entry_set_text (GTK_ENTRY (p_data->sport_long_entry), descr);
    if (abbr)
      gtk_entry_set_text (GTK_ENTRY (p_data->sport_abbr_entry), abbr);
    if (sport) {
      gtk_entry_set_text (GTK_ENTRY (p_data->sport_entry), sport);
      g_free (sport);
    }
  }
  gtk_widget_set_sensitive (GTK_WIDGET (p_data->sport_add_button), TRUE);
}


static void
sport_entry_changed (GtkEditable *editable, KPPreferencesDialog *dialog)
{
  KPPreferencesDialogPrivateData *p_data;
  GtkTreeSelection *selection;
  GtkTreeModel *model;
  GtkTreeIter iter;
  const gchar *wname;
  gchar *str;
  gint field;

  wname = gtk_widget_get_name (GTK_WIDGET (editable));

  if (strcmp (wname, "sport_entry") == 0)
    field = COLUMN_SPORT;
  else if (strcmp (wname, "sport_long_entry") == 0)
    field = COLUMN_SPORT_DESCRIPTION;
  else if (strcmp (wname, "sport_abbr_entry") == 0)
    field = COLUMN_SPORT_ABBR;
  else
    g_return_if_reached ();
  
  p_data = KP_PREFERENCES_DIALOG_PRIVATE_DATA (dialog);

  selection = gtk_tree_view_get_selection (GTK_TREE_VIEW (p_data->sport_treeview));
  
  if (!gtk_tree_selection_get_selected (selection, &model, &iter))
    return;

  str = gtk_editable_get_chars (editable, 0, -1);

  gtk_list_store_set (GTK_LIST_STORE (model), &iter, field, str, -1);
}


static void
sport_color_set (GtkColorButton *button, KPPreferencesDialog *dialog)
{
  KPPreferencesDialogPrivateData *p_data;
  GtkTreeSelection *selection;
  GtkTreeModel *model;
  GtkTreeIter iter;
  GdkColor color;
  gchar *str;

  p_data = KP_PREFERENCES_DIALOG_PRIVATE_DATA (dialog);

  selection = gtk_tree_view_get_selection (GTK_TREE_VIEW (p_data->sport_treeview));
  if (!gtk_tree_selection_get_selected (selection, &model, &iter))
    return;

  gtk_color_button_get_color (button, &color);

  str = kp_color_to_markup (&color);

  gtk_list_store_set (GTK_LIST_STORE (model), &iter, COLUMN_SPORT_COLOR, str, -1);
  g_free (str);
}



static void
on_sport_remove_button_clicked (GtkButton *button, gpointer data)
{
  KPPreferencesDialogPrivateData *p_data;
  GtkTreeSelection *selection;
  GtkTreeModel *model;
  GtkTreeIter iter;

  p_data = KP_PREFERENCES_DIALOG_PRIVATE_DATA (KP_PREFERENCES_DIALOG (data));
  
  selection = gtk_tree_view_get_selection (GTK_TREE_VIEW (p_data->sport_treeview));

  if (gtk_tree_selection_get_selected (selection, &model, &iter)) {
    (void) gtk_list_store_remove (GTK_LIST_STORE (model), &iter);

    if (gtk_tree_model_iter_n_children (model, NULL) == 0)
      gtk_widget_set_sensitive (GTK_WIDGET (button), FALSE);
  }
  gtk_entry_set_text (GTK_ENTRY (p_data->sport_entry), "");
}


static void
on_sport_color_button_clicked (GtkButton *button, KPPreferencesDialog *dialog)
{
  KPPreferencesDialogPrivateData *p_data;
  p_data = KP_PREFERENCES_DIALOG_PRIVATE_DATA (KP_PREFERENCES_DIALOG (dialog));
}
  

static void
on_sport_add_button_clicked (GtkButton *button, gpointer data)
{
  KPPreferencesDialogPrivateData *p_data;
  GtkTreeModel *model;
  GtkTreeIter iter;
  const gchar *entry;
  GdkColor color;
  gchar *rgb_str;
  gchar *color_str;
  const gchar *descr;
  const gchar *abbr;
  
  p_data = KP_PREFERENCES_DIALOG_PRIVATE_DATA (KP_PREFERENCES_DIALOG (data));
  
  gtk_color_button_get_color (GTK_COLOR_BUTTON (p_data->sport_color_button),
                              &color);

  rgb_str = g_strdup_printf ("#%02x%02x%02x",
                             color.red / 256,
                             color.green / 256,
                             color.blue / 256);
  
  color_str = g_strdup_printf ("<span weight=\"bold\" color=\"%s\">%s</span>",
                               rgb_str, rgb_str);
  
  descr = gtk_entry_get_text (GTK_ENTRY (p_data->sport_long_entry));
  abbr = gtk_entry_get_text (GTK_ENTRY (p_data->sport_abbr_entry));
  
  entry = gtk_entry_get_text (GTK_ENTRY (p_data->sport_entry));
  if (strlen (entry) == 0)
    return;
  
  model = gtk_tree_view_get_model (GTK_TREE_VIEW (p_data->sport_treeview));
  gtk_list_store_append (GTK_LIST_STORE (model), &iter);
  gtk_list_store_set (GTK_LIST_STORE (model), &iter,
                      COLUMN_SPORT, entry,
                      COLUMN_SPORT_COLOR, color_str,
                      COLUMN_SPORT_ABBR, abbr,
                      COLUMN_SPORT_DESCRIPTION, descr,
                      -1);

  gdk_color_parse ("black", &color);
  gtk_color_button_set_color (GTK_COLOR_BUTTON (p_data->sport_color_button),
                              &color);
  gtk_entry_set_text (GTK_ENTRY (p_data->sport_entry), "");
  gtk_entry_set_text (GTK_ENTRY (p_data->sport_long_entry), "");
  gtk_entry_set_text (GTK_ENTRY (p_data->sport_abbr_entry), "");
  
  g_free (color_str);
  g_free (rgb_str);
}


static void
plugin_selection_changed (GtkTreeSelection *selection,
                          KPPreferencesDialog *dialog)
{
  KPPreferencesDialogPrivateData *p_data;
  GtkTextBuffer *buffer;
  GtkTreeModel *model;
  GtkTreeIter iter;
  KPPlugin *plugin;
  gchar *tmp;
  gchar *str;

  p_data = KP_PREFERENCES_DIALOG_PRIVATE_DATA (dialog);

  if (!gtk_tree_selection_get_selected (selection, &model, &iter))
    return;

  gtk_tree_model_get (model, &iter, 1, &str, -1);

  plugin = kp_plugin_get_plugin_by_name (str);

  if (!plugin)
    return;

  buffer = gtk_text_buffer_new (NULL);
  gtk_text_buffer_set_text (buffer, plugin->info->description, -1);
  gtk_text_view_set_buffer (GTK_TEXT_VIEW (p_data->plugin_description_textview),
                            buffer);
  
  tmp = g_strconcat ("<b>", str, "</b>", NULL);
  gtk_label_set_markup (GTK_LABEL (p_data->plugin_title_label), tmp);
  gtk_label_set_text (GTK_LABEL (p_data->plugin_summary_label), plugin->info->summary);
  gtk_label_set_text (GTK_LABEL (p_data->plugin_homepage_label), plugin->info->homepage);
  gtk_label_set_text (GTK_LABEL (p_data->plugin_author_label), plugin->info->author);
  gtk_label_set_text (GTK_LABEL (p_data->plugin_filename_label), plugin->fn);

  g_free (tmp);
  g_free (str);
}

static void
set_plugin_tree_view (KPPreferencesDialog *dialog)
{
  KPPreferencesDialogPrivateData *p_data;
  GtkTreeSelection *selection;
  GtkTreeViewColumn *col1;
  GtkTreeViewColumn *col2;
  GtkCellRenderer *renderer;
  GtkListStore *store;
  GtkTreeView *treeview;
  GtkTreeIter iter;
  GList *list;

  p_data = KP_PREFERENCES_DIALOG_PRIVATE_DATA (dialog);
  treeview = GTK_TREE_VIEW (p_data->plugin_treeview);
  
  store = gtk_list_store_new (2, G_TYPE_BOOLEAN, G_TYPE_STRING);
  gtk_tree_view_set_model (treeview, GTK_TREE_MODEL (store));

  renderer = gtk_cell_renderer_toggle_new ();
  
  g_signal_connect (G_OBJECT (renderer), "toggled",
                    G_CALLBACK (plugin_load_state_toggled), dialog);
  
  col1 = gtk_tree_view_column_new_with_attributes (_("Loaded"), renderer,
                                                     "active", 0, NULL);
  renderer = gtk_cell_renderer_text_new ();

  col2 = gtk_tree_view_column_new_with_attributes (_("Plugin"), renderer,
                                                     "text", 1, NULL);

  gtk_tree_view_column_set_clickable (col1, TRUE);
  
  gtk_tree_view_append_column (treeview, col1);
  gtk_tree_view_append_column (treeview, col2);

  selection = gtk_tree_view_get_selection (treeview);

  g_signal_connect (G_OBJECT (selection), "changed",
                    G_CALLBACK (plugin_selection_changed), dialog);

  gtk_tree_selection_set_mode (selection, GTK_SELECTION_SINGLE);

  list = kp_plugin_get_plugins ();
  
  while (list) {
    gtk_list_store_append (store, &iter);
    gtk_list_store_set (store, &iter, 0, KP_PLUGIN (list->data)->loaded, -1);
    gtk_list_store_set (store, &iter, 1, KP_PLUGIN (list->data)->info->name, -1);
    list = list->next;
  }
}


static void
plugin_load_state_toggled (GtkCellRendererToggle *renderer, gchar *path,
                           KPPreferencesDialog *dialog)
{
  KPPreferencesDialogPrivateData *p_data;
  GtkTreeModel *model;
  GtkTreeIter iter;
  GtkTreePath *p;
  gboolean ind;
  gchar *module;
  
  p_data = KP_PREFERENCES_DIALOG_PRIVATE_DATA (dialog);
  p = gtk_tree_path_new_from_string (path);

  model = gtk_tree_view_get_model (GTK_TREE_VIEW (p_data->plugin_treeview));
 
  gtk_tree_model_get_iter (model, &iter, p);
  gtk_tree_model_get (model, &iter, 0, &ind, 1, &module, -1);

  ind ^= 1;

  if (ind) {
    kp_plugin_load_by_name (module);
    kp_settings_add_to_list ("load_plugin", module);
  } else {
    kp_plugin_unload_by_name (module);
    kp_settings_remove_from_list ("load_plugin", module);
  }

  gtk_list_store_set (GTK_LIST_STORE (model), &iter, 0, ind, -1);
  gtk_tree_path_free (p);
}


