#include <string.h>

#include <libxml/tree.h>
#include <glib-object.h>
#include <glib.h>

#include "kpparamlist.h"
#include "kpparam.h"
#include "kputil.h"

enum {
  CHANGED_SIGNAL,
  LAST_SIGNAL
};

static guint      kp_param_list_signals[LAST_SIGNAL] = { 0 };
  
/* GObject stuff */
static void       kp_param_list_class_init           (GObjectClass *klass,
                                                      gpointer data);
static void       kp_param_list_instance_init        (GObject *object,
                                                      gpointer data);
static void       kp_param_list_instance_finalize    (GObject *object);


GType
kp_param_list_get_type ()
{
  static GType kp_param_list_type = 0;

  if (!kp_param_list_type) {
    static const GTypeInfo kp_param_list_info = {
      sizeof (KPParamListClass),
      (GBaseInitFunc) NULL,
      (GBaseFinalizeFunc) NULL,
      (GClassInitFunc) kp_param_list_class_init,
      (GClassFinalizeFunc) NULL,
      NULL,
      sizeof (KPParamList),
      0,
      (GInstanceInitFunc) kp_param_list_instance_init,
      NULL
    };

    kp_param_list_type = g_type_register_static (G_TYPE_OBJECT,
                                                "KPParamList",
                                                &kp_param_list_info,
                                                 0);
  }
  return kp_param_list_type;
}


static void
kp_param_list_class_init (GObjectClass *klass, gpointer data)
{
  GObjectClass *object_class;

  object_class = G_OBJECT_CLASS (klass);
  object_class->finalize = kp_param_list_instance_finalize;

  kp_param_list_signals[CHANGED_SIGNAL] = 
    g_signal_new ("changed",
                  G_OBJECT_CLASS_TYPE (object_class),
                  G_SIGNAL_RUN_FIRST | G_SIGNAL_ACTION,
                  G_STRUCT_OFFSET (KPParamListClass, changed),
                  NULL,
                  NULL,
                  g_cclosure_marshal_VOID__VOID,
                  G_TYPE_NONE,
                  0);
}

  
static void
kp_param_list_instance_init (GObject *object, gpointer data)
{
  KP_PARAM_LIST (object)->list = NULL;
}

static void
kp_param_list_instance_finalize (GObject *object)
{
  GObjectClass *parent_class;
  /*KPParamList *list = KP_PARAM_LIST (object);*/

  parent_class = g_type_class_peek_parent (G_OBJECT_GET_CLASS (object));
  parent_class->finalize (object);
}

/**
 * kp_param_list_new:
 * 
 * Create a new unset instance of #KPParamList.
 * 
 * Returns: A new #KPParamList.
 */
KPParamList *
kp_param_list_new (void)
{
  return g_object_new (kp_param_list_get_type (), NULL);
}


static KPParamListCategory *
kp_param_list_add_category (KPParamList *list, const gchar *name)
{ 
  KPParamListCategory *cat;
  g_return_val_if_fail (KP_IS_PARAM_LIST (list), NULL);
  
  cat = g_new0 (KPParamListCategory, 1);
  cat->name = g_strdup (name);
  cat->list = NULL;

  list->list = g_list_prepend (list->list, cat);

  return cat;
}


/* If a param is changed that means the list has changed too */  
static void
param_changed_cb (KPParam *param, KPParamList *list)
{
  g_signal_emit (G_OBJECT (list), kp_param_list_signals[CHANGED_SIGNAL], 0);
}



/**
 * kp_param_list_insert:
 * @list: A #KPParamList
 * @category: A category
 * @param: A #KPParam
 *
 * Insert the @param to the list of params. If there already is
 * param called kp_param_get_name (@param), it will be destroyed
 * and its resources will be freed.
 */  
void
kp_param_list_insert (KPParamList *list, const gchar *category,
                      KPParam *param)
{
  KPParamListCategory *cat;
  KPParam *old_param;

  g_return_if_fail (KP_IS_PARAM_LIST (list));
  g_return_if_fail (KP_IS_PARAM (param));
  
  if ((old_param = kp_param_list_get_param (list, kp_param_get_name (param))))
    kp_param_list_remove (list, kp_param_get_name (param));
  
  cat = kp_param_list_get_category (list, category);
  cat->list = g_list_append (cat->list, param);

  g_signal_connect (G_OBJECT (param), "changed",
                    G_CALLBACK (param_changed_cb), list);
  
  g_signal_emit (G_OBJECT (list), kp_param_list_signals[CHANGED_SIGNAL], 0);
}


KPParamListCategory *
kp_param_list_get_category (KPParamList *list, const gchar *name)
{
  GList *node;
  
  for (node = list->list; node; node = node->next) {
    if (strcmp (((KPParamListCategory *)node->data)->name, name) == 0)
      return (KPParamListCategory *) node->data;
  }

  return kp_param_list_add_category (list, name);
}


void
kp_param_list_print (KPParamList *list)
{
  KPParam *param;
  GList *node, *cat;
  gchar *str;

  g_print ("--------------\n PARAM LIST\n ------------------\n");
  
  for (cat = list->list; cat; cat = cat->next) {
    g_print ("CAT \"%s\"\n", ((KPParamListCategory *)cat->data)->name); 
    
    for (node = ((KPParamListCategory *) cat->data)->list; node;
         node = node->next) {

      param = KP_PARAM (node->data);
      str = kp_param_get_as_string (param);
      
      g_print ("  PARAM \"%s\" = \"%s\"\n", kp_param_get_name (param), str);
      g_free (str);
    }
  }
}


KPParam *
kp_param_list_get_param (KPParamList *list, const gchar *name)
{
  KPParam *param;
  GList *node, *cat;

  for (cat = list->list; cat; cat = cat->next)
    for (node = ((KPParamListCategory *) cat->data)->list; node;
         node = node->next) {

      param = KP_PARAM (node->data);
      if (strcmp (kp_param_get_name (KP_PARAM (param)), name) == 0)
        return param;
    }
  return NULL; 
}



void
kp_param_list_remove (KPParamList *list, const gchar *name)
{
  KPParam *param;
  GList *cn, *pn, *tmp;
  GList *plist;

  g_return_if_fail (KP_IS_PARAM_LIST (list));
  g_return_if_fail (name != NULL);

  param = kp_param_list_get_param (list, name);

  for (cn = list->list; cn; cn = cn->next) {
    plist = ((KPParamListCategory *) cn->data)->list;
    for (pn = plist; pn; pn = pn->next)
      if (KP_IS_PARAM (pn->data) && 
          strcmp (kp_param_get_name (KP_PARAM (pn->data)), name) == 0) {

        tmp = pn;
        ((KPParamListCategory *) cn->data)->list = g_list_remove_link (plist, tmp);
        g_object_unref (G_OBJECT (tmp->data));
        g_list_free_1 (tmp);

        g_signal_emit (G_OBJECT (list), kp_param_list_signals[CHANGED_SIGNAL],
                       0);
        
        return;
      }
  }
}


KPParamList *
kp_param_list_copy (KPParamList *list)
{
  KPParamListCategory *ccopy, *cat;
  KPParamList *copy;
  GList *cnode, *node;

  copy = kp_param_list_new ();
  
  for (cnode = list->list; cnode; cnode = cnode->next) {
    cat = (KPParamListCategory *) cnode->data;
   
    ccopy = g_new0 (KPParamListCategory, 1);
    ccopy->name = g_strdup (cat->name);
    ccopy->list = NULL;
   
    copy->list = g_list_prepend (copy->list, ccopy);
    
    for (node = cat->list; node; node = node->next) {
      ccopy->list = g_list_prepend (ccopy->list,
                                    kp_param_copy (KP_PARAM (node->data)));
    }
  }

  return copy;
}

gint
kp_param_list_get_int (KPParamList *list, const gchar *param_name)
{
  KPParam *param;
  
  g_return_val_if_fail (KP_IS_PARAM_LIST (list), 0);
  g_return_val_if_fail (param_name != NULL, 0);

  param = kp_param_list_get_param (list, param_name);

  if (param != NULL)
    return kp_param_get_int (param);

  return 0;
}


guint
kp_param_list_get_uint (KPParamList *list, const gchar *param_name)
{
  KPParam *param;
  
  g_return_val_if_fail (KP_IS_PARAM_LIST (list), 0);
  g_return_val_if_fail (param_name != NULL, 0);

  param = kp_param_list_get_param (list, param_name);
 
  if (param != NULL)
    return kp_param_get_uint (param);

  return 0;
}


gdouble
kp_param_list_get_double (KPParamList *list, const gchar *param_name)
{
  KPParam *param;
  
  g_return_val_if_fail (KP_IS_PARAM_LIST (list), 0.0);
  g_return_val_if_fail (param_name != NULL, 0.0);

  param = kp_param_list_get_param (list, param_name);
 
  if (param != NULL)
    return kp_param_get_double (param); 

  return 0.0;
}


gboolean
kp_param_list_get_boolean (KPParamList *list, const gchar *param_name)
{
  KPParam *param;
  
  g_return_val_if_fail (KP_IS_PARAM_LIST (list), FALSE);
  g_return_val_if_fail (param_name != NULL, FALSE);

  param = kp_param_list_get_param (list, param_name);
  
  return (param) ? kp_param_get_boolean (param) : FALSE;
}


G_CONST_RETURN gchar *
kp_param_list_get_string (KPParamList *list, const gchar *param_name)
{
  KPParam *param;
  
  g_return_val_if_fail (KP_IS_PARAM_LIST (list), NULL);
  g_return_val_if_fail (param_name != NULL, NULL);

  param = kp_param_list_get_param (list, param_name);
  
  return (param) ? kp_param_get_string (param) : NULL;
}


gpointer
kp_param_list_get_pointer (KPParamList *list, const gchar *param_name)
{
  KPParam *param;
  
  g_return_val_if_fail (KP_IS_PARAM_LIST (list), NULL);
  g_return_val_if_fail (param_name != NULL, NULL);

  param = kp_param_list_get_param (list, param_name);
  
  return (param) ? kp_param_get_pointer (param) : NULL;
}

  
GObject *
kp_param_list_get_object (KPParamList *list, const gchar *param_name)
{
  KPParam *param;
  
  g_return_val_if_fail (KP_IS_PARAM_LIST (list), NULL);
  g_return_val_if_fail (param_name != NULL, NULL);

  param = kp_param_list_get_param (list, param_name);
  
  return (param) ? kp_param_get_object (param) : NULL;
}


guint
kp_param_list_get_time (KPParamList *list, const gchar *param_name)
{
  KPParam *param;
  
  g_return_val_if_fail (KP_IS_PARAM_LIST (list), 0);
  g_return_val_if_fail (param_name != NULL, 0);

  param = kp_param_list_get_param (list, param_name);
  
  return (param) ? kp_param_get_time (param) : 0;
}


gchar *
kp_param_list_get_as_string (KPParamList *list, const gchar *param_name)
{
  KPParam *param;
  
  g_return_val_if_fail (KP_IS_PARAM_LIST (list), NULL);
  g_return_val_if_fail (param_name != NULL, NULL);

  param = kp_param_list_get_param (list, param_name);
  
  return (param) ? kp_param_get_as_string (param) : NULL;
}


void
kp_param_list_set_string (KPParamList *list, const gchar *category,
                          const gchar *name, const gchar *value)
{
  KPParam *param;
  
  g_return_if_fail (KP_IS_PARAM_LIST (list));
  g_return_if_fail (category != NULL);
  g_return_if_fail (name != NULL);
  g_return_if_fail (value != NULL);

  if ((param = kp_param_list_get_param (list, name))) {
    kp_param_set_automatic_as_string (param, value);  
    return;
  }

  param = kp_param_new (name);
  kp_param_set_automatic_as_string (param, value);
  kp_param_list_insert (list, category, param);
}


void
kp_param_list_set_time (KPParamList *list, const gchar *category,
                        const gchar *param_name, guint value)
{
  KPParam *param;

  g_return_if_fail (KP_IS_PARAM_LIST (list));
  g_return_if_fail (category != NULL);
  g_return_if_fail (param_name != NULL);
  
  param = kp_param_list_get_param (list, param_name);

  if (param) {
    kp_param_set_time (param, value);
    return;
  }

  param = kp_param_new (param_name);
  kp_param_set_time (param, value);
  kp_param_list_insert (list, category, param);
}

  
void
kp_param_list_set_uint (KPParamList *list, const gchar *category, 
                        const gchar *param_name, guint value)
{
  KPParam *param;

  g_return_if_fail (KP_IS_PARAM_LIST (list));
  g_return_if_fail (category != NULL);
  g_return_if_fail (param_name != NULL);
  
  param = kp_param_list_get_param (list, param_name);

  if (param) {
    kp_param_set_uint (param, value);
    return;
  }

  param = kp_param_new (param_name);
  kp_param_set_uint (param, value);
  kp_param_list_insert (list, category, param);
}


void
kp_param_list_set_double (KPParamList *list, const gchar *category,
                          const gchar *param_name, gdouble value)
{
  KPParam *param;

  g_return_if_fail (KP_IS_PARAM_LIST (list));
  g_return_if_fail (category != NULL);
  g_return_if_fail (param_name != NULL);
  
  param = kp_param_list_get_param (list, param_name);

  if (param) {
    kp_param_set_double (param, value);
    return;
  }

  param = kp_param_new (param_name);
  kp_param_set_double (param, value);
  kp_param_list_insert (list, category, param);
}


void
kp_param_list_export_as_xml (KPParamList *list, xmlNodePtr parent)
{
  KPParamListCategory *cat;
  GList *cn, *pn;

  for (cn = list->list; cn; cn = cn->next) {
    cat = ((KPParamListCategory *) cn->data); 
    for (pn = cat->list; pn; pn = pn->next)
      kp_param_export_as_xml (KP_PARAM (pn->data), parent);
  }
}


