#include <libxml/tree.h>
#include <glib.h>

#include "kppresetdataitem.h"
#include "kputil.h"

static KPPresetDataItem  *kp_preset_data_item_parse  (xmlNodePtr node);


/**
 * kp_preset_data_item_new_from_xml:
 * @node: A #xmlNodePtr
 *
 * Tries to get an item from XML node.
 *
 * Returns: A newly-allocated #KPPresetDataItem or NULL
 * if XML node is not OK.
 */
KPPresetDataItem *
kp_preset_data_item_new_from_xml (xmlNodePtr node)
{
  g_return_val_if_fail (node != NULL, NULL);
 
  kp_debug ("Somebody found an preset data item!");
  
  return kp_preset_data_item_parse (node);
}


/**
 * kp_preset_data_item_new:
 *
 *
 * Returns: A new #KPPresetDataItem
 */
KPPresetDataItem *
kp_preset_data_item_new (void)
{
  static guint id = 1;
  KPPresetDataItem *item;
  
  item = g_new0 (KPPresetDataItem, 1);
  item->abbreviation = NULL;
  item->name = NULL;
  item->description = NULL;
  item->data = NULL;
  item->id = id;
  
  id++;
  
  return item;
}



/**
 * kp_preset_data_item_to_xml:
 * @item: A #KPPresetDataItem
 *
 * Try to make XML node of @item.
 *
 * Returns: A #xmlNodePtr if @item is OK or NULL otherwise.
 */
xmlNodePtr
kp_preset_data_item_to_xml (KPPresetDataItem *item)
{
  xmlNodePtr parent;

  g_return_val_if_fail (item != NULL, NULL);

  kp_debug ("Item goes to disk: %s\n", item->name);
  
  parent = xmlNewNode (NULL, BAD_CAST ("item"));
  (void) xmlNewChild (parent, NULL, BAD_CAST ("name"), BAD_CAST (item->name));
  if (item->abbreviation)
    (void) xmlNewChild (parent, NULL, BAD_CAST ("abbr"), BAD_CAST (item->abbreviation));
  if (item->description)
    (void) xmlNewChild (parent, NULL, BAD_CAST ("descr"), BAD_CAST (item->description));
  if (item->data)
    (void) xmlNewChild (parent, NULL, BAD_CAST ("data"), BAD_CAST (item->data));

  return parent;
}



/* Parse preset data item */
static KPPresetDataItem *
kp_preset_data_item_parse (xmlNodePtr node)
{
  KPPresetDataItem *item;
  xmlNodePtr ptr;
  gchar *tmp;
  
  g_return_val_if_fail (node != NULL, FALSE);
 
  item = kp_preset_data_item_new ();
  
  for (ptr = node->children; ptr; ptr = ptr->next) {
    tmp = (gchar *) xmlNodeGetContent (ptr);
    
    if (tmp) {
      if (KP_TAG_MATCH (ptr, "name")) 
        item->name = tmp;
      else if (KP_TAG_MATCH (ptr, "abbr"))
        item->abbreviation = tmp;
      else if (KP_TAG_MATCH (ptr, "descr"))
        item->description = tmp;
      else if (KP_TAG_MATCH (ptr, "data"))
        item->data = tmp;
      else
        g_free (tmp);
    }
  }
  if (item->name) 
    return item;
  
  kp_preset_data_item_free (item);

  return NULL;
}


void
kp_preset_data_item_free (KPPresetDataItem *item)
{
  if (!item)
    return;

  if (item->name)
    g_free (item->name);
  if (item->abbreviation) 
    g_free (item->abbreviation);
  if (item->description) 
    g_free (item->description);
  if (item->data)
    g_free (item->data);
  g_free (item);
}


guint
kp_preset_data_item_get_id (KPPresetDataItem *item)
{
  return item->id;
}
