#include "config.h"

#include <string.h>

#ifdef KISSME_LINUX_USER
#include <stdio.h>
#endif

#include "vm/jni.h"
#include "vm/jni_data.h"
#include "vm/newobject.h"
#include "vm/interp_methods.h"
#include "vm/classfile_methods.h"
#include "vm/classloader_tuple.h"
#include "vm/loading_errors.h"

#include "lib/indigenous/java.lang/VMClass.h"
#include "lib/indigenous/java.lang/Class.h"
#include "lib/indigenous/java.lang/Class_Reflection.h"
#include "lib/indigenous/java.lang/VMClassLoader.h"
#include "lib/indigenous/java.lang.reflect/Method.h"

extern tOBREF OOMExceptionObject;
extern tClassLoaderTuple* pstClassType; 
extern tClassLoaderTuple* pstVMClassType; 

static jclass VMClass_forNameHelperFromStruct(JNIEnv* env, 
					      tClassLoaderTuple* ourClass);
static jclass VMClass_forNameHelperFromName(JNIEnv* env, char* name, 
					    tClassLoaderTuple* loadingClass);

/* 
 * Takes an existing exception and turns it into an
 * InvocationTargetException that wraps the original
 */

tOBREF CLASS_CreateInvocationTargetException(JNIEnv* env)
{
  /* We have to construct this puppy carefully, it doesn't have the
     normal String constructor*/
  jclass exClass;
  jmethodID consMethodID;
  jclass origException;
  jobject message;
  int failure;

  origException = (*env)->ExceptionOccurred(env);
  (*env)->ExceptionClear(env);
  
  exClass = (*env)->FindClass(env, 
			      "java/lang/reflect/InvocationTargetException");
  if (exClass == NULL) { 
    (*env)->Throw(env, INTERP_NewObjectFromName(env, 
						"java/lang/InternalError",
						&failure)); 
    return NULL; 
  }
  consMethodID = 
    (*env)->GetMethodID(env, exClass, "<init>", 
			"(Ljava/lang/Throwable;Ljava/lang/String;)V");
  if (consMethodID == NULL) { 
    (*env)->Throw(env, INTERP_NewObjectFromName(env, 
						"java/lang/InternalError",
						&failure)); 
    return NULL; 
  }
  message = INTERP_NewStringFromAsciz(env, 
				      "An exception was thrown while calling "
				      "java.lang.reflect.Method.invoke");
  (*env)->Throw(env, (*env)->NewObject(env, exClass, consMethodID,
				       origException, message));

  return NULL;	
}


/* 
 * Checks that the array of objects matches the types required by the method 
 */
int CLASS_CheckNumReflectionMethodArguments(JNIEnv* env, tMethod* method,
					    jobjectArray args)
{
  int expectedArguments = CLASSFILE_CalcNumArguments(method->uidSignature);
  if (expectedArguments != 0) {
    if (args == NULL) {
      (*env)->Throw(env, INTERP_ExceptionObjectFromNameAndMessage(env, "java/lang/NullPointerException", "Class[] passed to invoke/newInstance is NULL"));
      return -1;
    }
    assert(ADEREF(args)->i32Number >= 0);
    /* Check the number of arguments is correct */
    if ((*env)->GetArrayLength(env, args) != expectedArguments) {
      char* errMessage = (char*) sys_malloc(1024);
      if (errMessage == NULL) { 
	(*env)->Throw(env, OOMExceptionObject);
	return -1; 
      }
      snprintf(errMessage, sizeof(errMessage), 
	       "%i arguments passed to invoke, expecting %i",
	      (int) (*env)->GetArrayLength(env, args), expectedArguments);
      (*env)->Throw(env, INTERP_ExceptionObjectFromNameAndMessage(env, "java/lang/IllegalArgumentException", errMessage));
      sys_free(errMessage);
      return -1;
    }
  }
  else {
    if (args != NULL) {
      //we must check they didn't give us arguments by mistake
      if ((*env)->GetArrayLength(env, args) != 0) {
	(*env)->Throw(env, INTERP_ExceptionObjectFromNameAndMessage(env, "java/lang/IllegalArgumentException", "array of arguments passed to invoke had more than 0 elements (we were expecting 0)"));
	return -1;
      }
    }
  }
  return 0;
}


int CLASS_CreateReflectedMethodArguments(JNIEnv* env, tMethod* method, 
					 jobjectArray args, jvalue** ret_array)
{
  jvalue* jvalue_array;
  int expectedArguments = CLASSFILE_CalcNumArguments(method->uidSignature);
  int i = 0;
  
  //We have to convert the array of object into an array of jvalues
  if (expectedArguments != 0) {
    jvalue_array = sys_malloc((*env)->GetArrayLength(env, args) * 
			      sizeof(jvalue));
    if (jvalue_array == NULL) {
      (*env)->Throw(env, OOMExceptionObject);
      return -1;
    }
    for (i = 0; i < expectedArguments; i++) {
      //we must unwrap primitive arguments
      tOBREF an_arg = (*env)->GetObjectArrayElement(env, args, i);
      jclass formalClass = NULL;
      jclass formalVMClass = NULL;
      jclass actualClass = NULL;
      jclass actualVMClass = NULL;

      formalClass = CLASS_GetClassFromSig(env, method->uidSignature,
					  method->pstClass, i);
      if (formalClass == NULL) {
	// XXX - should throw an exception I think ...
	panic("CLASS_GetClassFromSig for signature %s, i %i returned NULL",
	      method->uidSignature, i);
      }
      formalVMClass = CLASS_GetVMClass(env, formalClass);

      if (Java_java_lang_VMClass_isPrimitive(env, formalVMClass)) {
	formalClass = CLASS_GetWrapperClass(env, formalClass);
	if (an_arg == NULL) {
	  sys_free(jvalue_array);
	  (*env)->Throw(env, INTERP_ExceptionObjectFromNameAndMessage(env, "java/lang/IllegalArgumentException", "(Wrapped primitive) was NULL"));
	  return -1;
	}
	actualClass = CLASS_GetClass(env, an_arg);
	if (actualClass != formalClass) {
	  if (actualClass == java_lang_VMClassLoader_getPrimitiveWrapperClassFromCode(env, T_VOID)) {
	    sys_free(jvalue_array);
	    (*env)->Throw(env, INTERP_ExceptionObjectFromNameAndMessage(env, "java/lang/IllegalArgumentException", "(Wrapped primitive) Void passed as an argument to invoke"));
	    return -1;
	  }
	  actualVMClass = CLASS_GetVMClass(env, actualClass);
	  if (!Java_java_lang_VMClass_isPrimitiveWrapper(env, actualVMClass)) {
	    sys_free(jvalue_array);
	    (*env)->Throw(env, INTERP_ExceptionObjectFromNameAndMessage(env, "java/lang/IllegalArgumentException", "(Wrapped primitive) Object passed as an argument to invoke is not of primitive type!"));
	    return -1;
	  }
	  if (!CLASS_CheckWrapperWidening(env, formalClass, actualClass)) {
	    sys_free(jvalue_array);
	    (*env)->Throw(env, INTERP_ExceptionObjectFromNameAndMessage(env, "java/lang/IllegalArgumentException", "(Wrapped primitive) Object passed as an argument to invoke is not of the expected type"));
	    return -1;
	  }
	}
	if (CLASS_UnwrapObject(env, an_arg, jvalue_array + i) != 0) {
	  sys_free(jvalue_array);
	  (*env)->Throw(env, INTERP_ExceptionObjectFromNameAndMessage(env, "java/lang/InternalError", "Failed to unwrap a primitive object"));
	  return -1;
	}
	if (actualClass != formalClass) {
	  //now we must do the widening of the result
	  if (CLASS_WidenResult(env, formalClass, actualClass, 
				jvalue_array + i) != 0) {
	    sys_free(jvalue_array);
	    (*env)->Throw(env, INTERP_ExceptionObjectFromNameAndMessage(env, "java/lang/InternalError", "Failed to widen a primitive object during invoke (even though the widening check succeeded)"));
	    return -1;
	  }
	}	
      }
      else {
	if (an_arg != NULL) {
	  if (!Java_java_lang_VMClass_isInstance(env, formalVMClass, an_arg)) {
	    int failure;
	    char* errMessage;
	    tClassLoaderTuple* pstFormalClass;
	    tClassLoaderTuple* pstActualClass;
	    
	    sys_free(jvalue_array);
	    pstFormalClass = CLASS_GetVMClassStruct(env, formalVMClass);
	    pstActualClass =
	      CLASS_GetClassStruct(env, CLASS_GetClass(env, an_arg));
	    
	    errMessage = sys_malloc(1024 + strlen(pstFormalClass->uidName) +
				    strlen(pstActualClass->uidName));
	    if (errMessage == NULL) {
	      (*env)->Throw(env, INTERP_NewObjectFromName(env, "java/lang/InternalError", &failure));
	      return -1; 
	    }
	    
	    sprintf(errMessage, 
		    "Object passed to invoke as an argument was type %s, "
		    "expected type %s\n", 
		    pstActualClass->uidName, pstFormalClass->uidName);
	    (*env)->Throw(env, INTERP_ExceptionObjectFromNameAndMessage(env, "java/lang/IllegalArgumentException", errMessage));
	    sys_free(errMessage);
	    return -1;
	  }
	}
	jvalue_array[i].l = an_arg;
      }
    }
  }
  else {
    jvalue_array = NULL;
  }
  
  //success
  *ret_array = jvalue_array;
  return 0;
}


int CLASS_CompareClassArrays(JNIEnv* env, jobjectArray a, jobjectArray b)
{
  int lengtha = (*env)->GetArrayLength(env, a);
  if (lengtha == (*env)->GetArrayLength(env, b)) {
    int i;
    for (i = 0; i < lengtha; i++) {
      if ((*env)->GetObjectArrayElement(env, a, i) != 
	  (*env)->GetObjectArrayElement(env, b, i)) {
	return 2;
      }
    }
    return 0; //match
  }
  else {
    return 1;
  }
}


int CLASS_getDeclaringClass(JNIEnv* env,tClassLoaderTuple* startingClass,
			    tMethod* method, jclass* origClazz, int* slotIndex)
{
  tClassLoaderTuple* searchClass = startingClass;
  while (searchClass != NULL) {
    int i = 0;
    for (i = 0; i < searchClass->pstClass->u16MethodsCount; i++) {
      if (searchClass->pstClass->pstMethods + i == method) {
	*slotIndex = i;
	*origClazz = CLASSLOADER_TUPLE_GetClassObject(env, searchClass);
	return 0;
      }
    }
    searchClass = searchClass->pstClass->pstSuperClass;
  }
  *slotIndex = -1;
  *origClazz = NULL;
  return -1;
}


static char sigbuffer[4096]; 
 

int ConvertClassToSig(JNIEnv* env, tClassLoaderTuple* clazz, int primitive) 
{ 
  if (primitive) { 
    if (uidcmp(clazz->uidName, UID_GetUid("int")) == 0) 
      strcat(sigbuffer, "I"); 
    else if (uidcmp(clazz->uidName, UID_GetUid("byte")) == 0) 
      strcat(sigbuffer, "B"); 
    else if (uidcmp(clazz->uidName, UID_GetUid("short")) == 0) 
      strcat(sigbuffer, "S"); 
    else if (uidcmp(clazz->uidName, UID_GetUid("char")) == 0) 
      strcat(sigbuffer, "C"); 
    else if (uidcmp(clazz->uidName, UID_GetUid("float")) == 0) 
      strcat(sigbuffer, "F"); 
    else if (uidcmp(clazz->uidName, UID_GetUid("double")) == 0) 
      strcat(sigbuffer, "D"); 
    else if (uidcmp(clazz->uidName, UID_GetUid("boolean")) == 0) 
      strcat(sigbuffer, "Z"); 
    else if (uidcmp(clazz->uidName, UID_GetUid("long")) == 0) 
      strcat(sigbuffer, "J"); 
  } 
  else { 
    if (clazz->uidName[0] == '[') { 
      int looper = 0; 
      int isClass = 0; 
      while (looper < strlen(clazz->uidName)) { 
	if (clazz->uidName[looper] != '[') { 
	  break; 
	}
	else {
	  strcat(sigbuffer, "["); 
	  looper++; 
	} 
      }
      
      if (clazz->uidName[looper] == 'L') { 
	isClass = 1; 
      } 
      while (looper < strlen(clazz->uidName)) { 
	char buf[2]; 
	buf[0] = clazz->uidName[looper]; 
	buf[1] = 0; 
	strcat(sigbuffer, buf); 
	looper++; 
      } 
    } 
    else { 
      //It's an object 
      strcat(sigbuffer, "L"); 
      strcat(sigbuffer, clazz->uidName); 
      strcat(sigbuffer, ";"); 
    }   
  } 
  return 0; 
} 
 


/* Takes an array Class[] and returns a java-style signature string */ 
 
Uid CLASS_GetSigFromClassArray(JNIEnv* env, tArray* args) 
{ 
  int numargs = (args == NULL) ? 0 : (args->i32Number); 
  int i; 
  
  // eprintf("Got an array with %i els, type %s\n", 
  //         (int) args->i32Number, args->pstType->uidName); 
  strcpy(sigbuffer, "("); 
  for (i = 0; i < numargs; i++) { 
    jclass clazzObj = ((jclass*) args->pvElements)[i]; 
    jobject vmClassObj = CLASS_GetVMClass(env, clazzObj);
    tClassLoaderTuple* tuple = CLASS_GetClassStruct(env, vmClassObj);
    
    if (Java_java_lang_VMClass_isPrimitive(env, vmClassObj)) { 
      ConvertClassToSig(env, tuple, 1); 
    } 
    else { 
      ConvertClassToSig(env, tuple, 0); 
    } 
  } 
  strcat(sigbuffer, ")"); 
  //Now the return type 
  
  return UID_GetUid(sigbuffer); 
} 


/* Since we're using these primitive class objects so often, I think
   we should create a method to access them directly out of their,
   with an enumerated type as index */


/*
 * Checks whether this is a wrapper class for a primitive value (like
 * java/lang/Integer java/lang/Byte etc)
 *
 * Note this returns true for java/lang/Void too
 */
jboolean Java_java_lang_VMClass_isPrimitiveWrapper(JNIEnv* env, 
						   jobject vmClass) 
{ 
  jclass clazz = CLASS_GetVMClassStruct(env, vmClass)->classObject;

  if ((clazz == java_lang_VMClassLoader_getPrimitiveWrapperClassFromCode(env, T_BYTE)) ||
      (clazz == java_lang_VMClassLoader_getPrimitiveWrapperClassFromCode(env, T_CHAR)) ||
      (clazz == java_lang_VMClassLoader_getPrimitiveWrapperClassFromCode(env, T_SHORT)) ||
      (clazz == java_lang_VMClassLoader_getPrimitiveWrapperClassFromCode(env, T_INT)) ||
      (clazz == java_lang_VMClassLoader_getPrimitiveWrapperClassFromCode(env, T_LONG)) ||
      (clazz == java_lang_VMClassLoader_getPrimitiveWrapperClassFromCode(env, T_DOUBLE)) ||
      (clazz == java_lang_VMClassLoader_getPrimitiveWrapperClassFromCode(env, T_FLOAT)) ||
      (clazz == java_lang_VMClassLoader_getPrimitiveWrapperClassFromCode(env, T_BOOLEAN)) ||
      (clazz == java_lang_VMClassLoader_getPrimitiveWrapperClassFromCode(env, T_VOID))) {
    return JNI_TRUE;
  }
  else {
    return JNI_FALSE;
  }
}


/* 
 * Checks whether the supplied wrapper type can be widened to the
 * target type. We assume the types aren't the same
 *
 * eg java/lang/Short can be widened to java/lang/Integer java/lang/Long
 *  returns JNI_TRUE or JNI_FALSE 
 *
 * XXX - if the widening rules are supposed to mirror the rules for
 * the primitive types, there are a number of bugs in the this code ...
 */
jboolean CLASS_CheckWrapperWidening(JNIEnv* env, jclass targetClass,
				    jclass suppliedClass)
{
  if (suppliedClass == java_lang_VMClassLoader_getPrimitiveWrapperClassFromCode(env, T_LONG)) {
    return JNI_FALSE; //no widening
  }
  else if (suppliedClass == java_lang_VMClassLoader_getPrimitiveWrapperClassFromCode(env, T_DOUBLE)) {
    return JNI_FALSE; //no widening
  } 
  else if (suppliedClass == java_lang_VMClassLoader_getPrimitiveWrapperClassFromCode(env, T_INT)) {
    //int can be widened to long 
    if ((targetClass == java_lang_VMClassLoader_getPrimitiveWrapperClassFromCode(env, T_LONG))) {
      return JNI_TRUE;
    }
    return JNI_FALSE;
  } 
  else if (suppliedClass == java_lang_VMClassLoader_getPrimitiveWrapperClassFromCode(env, T_SHORT)) {
    //short can be widened to int and long
    if (targetClass == java_lang_VMClassLoader_getPrimitiveWrapperClassFromCode(env, T_LONG)) {
      return JNI_TRUE;
    }
    if (targetClass == java_lang_VMClassLoader_getPrimitiveWrapperClassFromCode(env, T_INT)) {
      return JNI_TRUE;
    }
    return JNI_FALSE;
  } 
  else if (suppliedClass == java_lang_VMClassLoader_getPrimitiveWrapperClassFromCode(env, T_BYTE)) {
    //byte can be widened to short, int and long
    if (targetClass == java_lang_VMClassLoader_getPrimitiveWrapperClassFromCode(env, T_LONG)) {
      return JNI_TRUE;
    }
    if (targetClass == java_lang_VMClassLoader_getPrimitiveWrapperClassFromCode(env, T_INT)) {
      return JNI_TRUE;
    }
    if (targetClass == java_lang_VMClassLoader_getPrimitiveWrapperClassFromCode(env, T_SHORT)) {
      return JNI_TRUE;
    }
    return JNI_FALSE;
  } 
  else if (suppliedClass == java_lang_VMClassLoader_getPrimitiveWrapperClassFromCode(env, T_CHAR)) {
    return JNI_FALSE; //no widening
  } 
  else if (suppliedClass == java_lang_VMClassLoader_getPrimitiveWrapperClassFromCode(env, T_BOOLEAN)) {
    return JNI_FALSE; //no widening
  } 
  else if (suppliedClass == java_lang_VMClassLoader_getPrimitiveWrapperClassFromCode(env, T_FLOAT)) {
    if (targetClass == java_lang_VMClassLoader_getPrimitiveWrapperClassFromCode(env, T_DOUBLE)) {
      return JNI_TRUE;
    }
    return JNI_FALSE; 
  } 
  else {
    panic0("Unknown Wrapper class in CLASS_CheckWrapperWidening");
    (*env)->Throw(env, INTERP_ExceptionObjectFromNameAndMessage(env, "java/lang/InternalError", "Unknown Wrapper class in CLASS_CheckWrapperWidening"));
    return JNI_FALSE;
  }
}


jclass CLASS_GetWrapperClass(JNIEnv* env, jclass primClass)
{
  tClassLoaderTuple* pstType = CLASS_GetClassStruct(env, primClass);

  if (strcmp(pstType->uidName, "int") == 0) {
    return (*env)->FindClass(env, "java/lang/Integer");
  }
  else if (strcmp(pstType->uidName, "byte") == 0) {
    return (*env)->FindClass(env, "java/lang/Byte");
  }
  else if (strcmp(pstType->uidName, "char") == 0) {
    return (*env)->FindClass(env, "java/lang/Character");
  }
  else if (strcmp(pstType->uidName, "short") == 0) {
    return (*env)->FindClass(env, "java/lang/Short");
  }
  else if (strcmp(pstType->uidName, "long") == 0) {
    return (*env)->FindClass(env, "java/lang/Long");
  }
  else if (strcmp(pstType->uidName, "double") == 0) {
    return (*env)->FindClass(env, "java/lang/Double");
  }
  else if (strcmp(pstType->uidName, "float") == 0) {
    return (*env)->FindClass(env, "java/lang/Float");
  }
  else if (strcmp(pstType->uidName, "boolean") == 0) {
    return (*env)->FindClass(env, "java/lang/Boolean");
  }
  else {
    (*env)->Throw(env, INTERP_ExceptionObjectFromNameAndMessage(env, "java/lang/InternalError", "Couldn't get wrapper class for primitive class in CLASS_GetWrapperClass"));
    return NULL;
  }
}


/* 
 * Takes as a parameter a wrapped primitive object. The value
 * represented by this object is return in the retval structure
 *
 * returns 0 on success
 * returns -1 on failure
 */
int CLASS_UnwrapObject(JNIEnv* env, jobject object, jvalue* retval)
{
  tClassLoaderTuple* pstType = ODEREF(object)->pstType;
  jclass clazz = CLASS_GetClass(env, object);
  jfieldID fieldID;

  fieldID = (*env)->GetFieldID(env, clazz, "value", NULL);
  if (fieldID == NULL) {
    retval->j = 0;
    return -1;
  }
  if (strcmp(pstType->uidName, "java/lang/Integer") == 0) {
    retval->i = (*env)->GetIntField(env, object, fieldID);
    return 0;
  }
  else if (strcmp(pstType->uidName, "java/lang/Byte") == 0) {
    retval->b = (*env)->GetByteField(env, object, fieldID);
    return 0;
  }
  else if (strcmp(pstType->uidName, "java/lang/Short") == 0)  {
    retval->s = (*env)->GetShortField(env, object, fieldID);
    return 0;
  }
  else if (strcmp(pstType->uidName, "java/lang/Character") == 0)  {
    retval->c = (*env)->GetCharField(env, object, fieldID);
    return 0;
  }
  else if (strcmp(pstType->uidName, "java/lang/Long") == 0) {
    retval->j = (*env)->GetLongField(env, object, fieldID);
    return 0;
  }
  else if (strcmp(pstType->uidName, "java/lang/Double") == 0) {
    retval->d = (*env)->GetDoubleField(env, object, fieldID);
    return 0;
  }
  else if (strcmp(pstType->uidName, "java/lang/Float") == 0) {
    retval->f = (*env)->GetFloatField(env, object, fieldID);
    return 0;
  }
  else if (strcmp(pstType->uidName, "java/lang/Boolean") == 0) {
    retval->z = (*env)->GetBooleanField(env, object, fieldID);
    return 0;
  }
  else {
    return -1;
  }
}


/* 
 * The target class is something like java/lang/Integer and the 
 * suppliedClass is something like java/lang/Short
 *
 * Retval must contain the result to be widened
 * It is changed to reflect the widened result
 * The check to see if widening is possible must already have been performed
 *
 * Some of this behaviour may be unnecessary
 *
 * returns 0 for success
 * returns -1 for failure
 */
int CLASS_WidenResult(JNIEnv* env, jclass targetClass, jclass suppliedClass,
		      jvalue* retval)
{
  if (suppliedClass == java_lang_VMClassLoader_getPrimitiveWrapperClassFromCode(env, T_INT)) {
    //int can be widened to long 
    if ((targetClass == java_lang_VMClassLoader_getPrimitiveWrapperClassFromCode(env, T_LONG))) {
      retval->j = retval->i;
      return 0;
    }
    return -1;
  } 
  else if(suppliedClass == java_lang_VMClassLoader_getPrimitiveWrapperClassFromCode(env, T_SHORT)) {
    //short can be widened to int and long
    if (targetClass == java_lang_VMClassLoader_getPrimitiveWrapperClassFromCode(env, T_LONG)) {
      retval->j = retval->s;
      return 0;
    }
    if (targetClass == java_lang_VMClassLoader_getPrimitiveWrapperClassFromCode(env, T_INT)) {
      retval->i = retval->s;
      return 0;
    }
    return -1;
  } 
  else if(suppliedClass == java_lang_VMClassLoader_getPrimitiveWrapperClassFromCode(env, T_BYTE)) {
    //byte can be widened to short, int and long
    if (targetClass == java_lang_VMClassLoader_getPrimitiveWrapperClassFromCode(env, T_LONG)) {
      retval->j = retval->b;
      return 0;
    }
    if (targetClass == java_lang_VMClassLoader_getPrimitiveWrapperClassFromCode(env, T_INT)) {
      retval->i = retval->b;
      return 0;
    }
    if (targetClass == java_lang_VMClassLoader_getPrimitiveWrapperClassFromCode(env, T_SHORT)) {
      retval->s = retval->b;
      return 0;
    }
    return -1;
  } 
  else if (suppliedClass == java_lang_VMClassLoader_getPrimitiveWrapperClassFromCode(env, T_FLOAT)) {
    if (targetClass == java_lang_VMClassLoader_getPrimitiveWrapperClassFromCode(env, T_DOUBLE)) {
      retval->d = retval->f;
      return 0;
    }
    return -1;
  } 
  else {
    panic0("CLASS_WidenResult: unknown widening operation for Wrapper class");
    (*env)->Throw(env, INTERP_ExceptionObjectFromNameAndMessage(env, "java/lang/InternalError", "Unknown widening operation for Wrapper class in CLASS_WidenResult"));
    return -1;
  }
}


jclass CLASS_GetClassFromSig(JNIEnv* env, char* pszSignature,
			     tClassLoaderTuple* loadingClass, int index) 
{ 
  int pos,len; 
  int itemcount = 0; 
  uint16 jint_count=0; 
  byte *Bytes; 
  int looking_at_return_type = 0; //Just so we can examine the last item 
  
#ifdef DEBUG 
  assert(pszSignature); 
#endif 
  
#ifdef CLASSFILE_TRACE 
  eprintf("Parsing: %s\n", pszSignature); 
#endif 
  
  len = strlen(pszSignature); 
  Bytes = pszSignature; 
  for (pos = 0; pos < len; pos++) { 
    if (itemcount == index) { 
      //This is the item we want 
      switch (Bytes[pos]) { 
      case '(': 
	break; 

      case ')': 
	looking_at_return_type = 1; 
	break; 

      case 'V':
      case 'B': 		
      case 'C': 
      case 'F': 
      case 'S': 
      case 'Z': 
      case 'I': 
      case 'D': 
      case 'J': 
	return CLASS_GetPrimitiveClass(env, Bytes[pos]);
	break;
      
      case 'L': 
	{ 
	  char* orig = sys_malloc(4096); 
	  char* start = Bytes + pos; 
	  jclass retval; 
	  int origpos = pos; 
	  
	  assert(orig != NULL); 
	  assert(len < 4096); 
	  
	  for ( ; Bytes[pos] != ';' && pos < len; pos++); 
	  
	  assert(((pos - origpos) - 1) > 0);
	  strncpy(orig, start + 1, (pos - origpos) -1); 
	  orig[pos - origpos - 1] = 0; 
	  
	  retval = VMClass_forNameHelperFromName(env, orig, loadingClass);
          sys_free(orig);
	  return retval; 
	  break; 
	} 
      case '[': 
	{ 
	  int start = pos; 
	  tClassLoaderTuple* result;
	  
	  char buf[2048]; 
	  while ((Bytes[pos] == '[') && (pos < len)) { 
	    pos++; 
	  } 
	  
	  if (Bytes[pos] == 'L') { /* object signature */ 
	    while (Bytes[pos] != ';') { 
	      pos++; 
	    } 
	  } 
	  pos++; 
	  
	  memset(buf,0, 2048); 
	  strncpy(buf, pszSignature + start, pos - start); 
	  
	  result = CLASSFILE_FindOrLoad(env, buf, loadingClass);
	  if (result) {
	    return CLASSLOADER_TUPLE_GetClassObject(env, result);
	  }
	  else {
	    return NULL;
	  }
	  break; 
	} 
      default: 
	panic("illegal signature %s, index %i, itemcount %i", 
	      Bytes + pos, index, itemcount); 
      } //end switch 
			
    } 
    else { 
      //Just count 
      switch (Bytes[pos]) { 
      case '(': 
	break; 
	
      case ')': 
	looking_at_return_type = 1; 
	// drop through ...
	
      case 'B': 
      case 'C': 
      case 'F': 
      case 'S': 
      case 'Z': 
      case 'I': 
	itemcount++; 
	jint_count++; 
	break; 
	
      case 'D': 
      case 'J': 
	itemcount++; 
	jint_count += 2; 
	break; 
	
      case 'L': 
	itemcount++; 
	jint_count++; 
	for ( ; Bytes[pos] != ';' && pos < len; pos++); 
	break; 
	
      case '[': 
	itemcount++; 
	jint_count++; 
	while ((Bytes[pos] == '[' ||
		(Bytes[pos] >= '0' && Bytes[pos] <= '9')) &&
	       pos < len) { /* multiple arrays with sizes*/ 
	  pos++; 
	} 
	if (Bytes[pos] == 'L') { /* object signature */ 
	  while (Bytes[pos] != ';') { 
	    pos++; 
	  } 
	} 
	break; 
	
      default: 
	panic("illegal signature index %i, pos %i, itemcount %i, char is %c",
	      index, pos, itemcount, Bytes[pos]); 
      } //end second switch 
    } //end else 
    
#ifdef CLASSFILE_TRACE 
    eprintf("int32 argument len: %d\n", jint_count); 
#endif 
  } 
  
  return NULL; 
} 


jclass CLASS_GetPrimitiveClass(JNIEnv* env, char type)
{
  return java_lang_VMClassLoader_getPrimitiveClass(env, NULL, type);
}


jobjectArray CLASS_GetClassArrayFromSig(JNIEnv* env, char* sig,
					tClassLoaderTuple* loadingClass)
{
  int count;
  jobjectArray parameters;
  int j = 0;

  count = CLASSFILE_CalcNumArguments( sig );

  parameters = 
    (*env)->NewObjectArray(env, count, 
			   (*env)->FindClass(env, "java/lang/Class"), 
			   NULL);
  
  for (j = 0; j < (count); j++) {
    jclass type;
    type = CLASS_GetClassFromSig(env, sig, loadingClass, j);
    
    if (type == NULL) {
      (*env)->ThrowNew(env, (*env)->FindClass(env, "java/lang/NoClassDefFoundError"), sig);
      return NULL;
    }
    
    (*env)->SetObjectArrayElement(env, parameters, j, type);
  }
  return parameters;
}


jobject constructFieldObject(JNIEnv* env, jclass declaringClazz, 
			     Uid fieldUid, jint slot, jboolean isStatic,
			     int* failure)
{ 
  jclass fieldClazz; 
  jmethodID consID; 
  jobject fieldObject;

  fieldObject = INTERP_NewObjectFromName(env, "java/lang/reflect/Field", 
					 failure);
  assert(fieldObject);
  (*env)->NewGlobalRef(env, fieldObject);

  fieldClazz = (*env)->GetObjectClass(env, fieldObject); 
  assert(fieldClazz); 
  
  consID = (*env)->GetMethodID(env, fieldClazz, "<init>", 
			       "(Ljava/lang/Class;Ljava/lang/String;II)V"); 
  assert(consID); 

  (*env)->CallNonvirtualVoidMethod(env, fieldObject, fieldClazz, consID, 
				   declaringClazz, 
				   INTERP_NewStringFromAsciz(env, fieldUid),
				   slot, isStatic); 
  return fieldObject;
} 


static jobject doGetField(JNIEnv* env, jclass clazz, Uid fieldUid, 
			  int declared)
{
  int i, slot, failure;
  jfieldID fieldID; 
  tClassLoaderTuple* pstClass;

  assert(clazz);
  pstClass = CLASS_GetClassStruct(env, clazz);

  /* First look for a static field of this class */
  fieldID = CLASSFILE_UidStatFieldOffset(env, pstClass, fieldUid);
  if (fieldID != NULL) {
    slot = fieldID - pstClass->pstClass->pstStatOffsets;
    if (declared || 
	pstClass->pstClass->pstStatOffsets[slot].u16AccessFlags & ACC_PUBLIC) {
      return constructFieldObject(env, clazz, fieldUid, slot, 1, &failure);
    }
  }

  /* Then for an instance field for this class */
  fieldID = (*env)->GetFieldID(env, clazz, fieldUid, NULL); 
  if (fieldID != NULL) {
    jclass declaringClazz = 
      CLASSLOADER_TUPLE_GetClassObject(env, fieldID->pstClass);
    
    slot = fieldID - fieldID->pstClass->pstClass->pstInstOffsets;
    if (declared || 
	(fieldID->pstClass->pstClass->pstInstOffsets[slot].u16AccessFlags &
	 ACC_PUBLIC)) {
      return constructFieldObject(env, declaringClazz, fieldUid, slot, 
				  0, &failure);
    }
  }
  if (!declared) {
    /* If that fails, recursively search the class's interfaces */
    for (i = 0; i < pstClass->pstClass->u16InterfacesCount; i++) {
      tClassLoaderTuple* pstTuple = pstClass->pstClass->ppstInterfaces[i];
      jclass pstInterface = CLASSLOADER_TUPLE_GetClassObject(env, pstTuple);
      jobject field = doGetField(env, pstInterface, fieldUid, declared);

      if (field != NULL) {
	return field;
      }
    }
    /* Finally, recursively search the class's supertype */
    if (pstClass->pstClass->pstSuperClass) {
      tClassLoaderTuple* pstTuple = pstClass->pstClass->pstSuperClass;
      jclass pstSupertype = CLASSLOADER_TUPLE_GetClassObject(env, pstTuple);
      jobject field = doGetField(env, pstSupertype, fieldUid, declared);

      if (field != NULL) {
	return field;
      }
    }
  }
  /* We failed to find the field */
  return NULL;
} 


jobjectArray Java_java_lang_VMClass_getDeclaredFields(JNIEnv* env, 
						      jobject vmClass,
						      jboolean publicOnly)
{ 
  //just duplicating the code from getFields
  tClassLoaderTuple* tuple; 
  jclass clazz, fieldClazz; 
  int numInst = 0, numStatic = 0, i = 0;
  int j = 0; 
  jarray retArray; 
  int failure;
  Uid fieldUid;
  tOBREF fieldObject;

  tuple = CLASS_GetVMClassStruct(env, vmClass); 
  clazz = tuple->classObject;
  
  fieldClazz = (*env)->FindClass(env, "java/lang/reflect/Field");
  assert(fieldClazz);

  if (publicOnly) {
    for (i = 0; i < tuple->pstClass->u16InstCount; i++) {
      if (tuple->pstClass->pstInstOffsets[i].u16AccessFlags & ACC_PUBLIC) {
	numInst++;
      }
    }
    for (i = 0; i < tuple->pstClass->u16StatCount; i++) {
      if (tuple->pstClass->pstStatOffsets[i].u16AccessFlags & ACC_PUBLIC) {
	numStatic++;
      }
    }
  }
  else {
    numInst = tuple->pstClass->u16InstCount; 
    numStatic = tuple->pstClass->u16StatCount; 
  }

  retArray = (*env)->NewObjectArray(env, numInst + numStatic,
				    fieldClazz, NULL);
  assert(retArray); 
  
  i = 0;
  while (numInst > 0) { 
    assert(i < tuple->pstClass->u16InstCount); 
    if (!publicOnly ||
	tuple->pstClass->pstInstOffsets[i].u16AccessFlags & ACC_PUBLIC) {
      fieldUid = tuple->pstClass->pstInstOffsets[i].uidFieldName; 
      fieldObject = constructFieldObject(env, clazz, fieldUid,
					 i, 0 /* not static */,
					 &failure); 
      ((tOBREF*) (ADEREF(retArray)->pvElements))[j++] = fieldObject; 
      numInst--; 
    }
    i++; 
  } 
  
  i = 0; 
  while (numStatic > 0) { 
    assert(i < tuple->pstClass->u16StatCount); 
    if (!publicOnly ||
	tuple->pstClass->pstStatOffsets[i].u16AccessFlags & ACC_PUBLIC) {
      fieldUid = tuple->pstClass->pstStatOffsets[i].uidFieldName;
      fieldObject = constructFieldObject(env, clazz, fieldUid, 
					 i, 1 /* static */, 
					 &failure); 
      ((tOBREF*) (ADEREF(retArray)->pvElements))[j++] = fieldObject; 
      numStatic--; 
    }
    i++; 
  } 
  
  return retArray; 
} 
 
 
/* 
 * Helper methods for forName
 */
static jclass 
VMClass_forNameHelperFromStruct(JNIEnv* env,
				tClassLoaderTuple* pstOurClassType)
{
  tOBREF pstClassObject; 
  
  if (pstOurClassType == NULL) { 
    int failure;
    (*env)->Throw(env, INTERP_NewObjectFromName(env, "java/lang/ClassNotFoundException", &failure)); 
    return NULL; 
  } 

  if (pstOurClassType->classObject == NULL) {
    CLASSLOADER_TUPLE_MakeClassObject(env, pstOurClassType);
  }               
  return pstOurClassType->classObject; 
}


static jclass VMClass_forNameHelperFromName(JNIEnv* env, char* pszClassName,
					    tClassLoaderTuple* loadingClass)
{
  tClassLoaderTuple* pstOurClassType;
  uint32 error = 0;
  
  pstOurClassType =
    CLASSFILE_FindOrLoadWithError(env, pszClassName, 
				  loadingClass, &error); //XXX tuple
  
  if (pstOurClassType != NULL) {
    return VMClass_forNameHelperFromStruct(env, pstOurClassType);
  }
  else { 
    (*env)->ThrowNew(env, (*env)->FindClass(env, "java/lang/ClassNotFoundException"), pszClassName); 
    return NULL; 
  } 
}


 
/* 
 * @doc NATFUNC 
 * @func 
 * Implementation of: public static native Class forName(String); 
 * 
 * Returns the Class object associated with the class with the given 
 * string name. 
 *
 * When a primitive type is given as "I" or "Z" or "int" or "void", we
 * look for a user-defined class of that name. (Use the .TYPE field to
 * get these classes, eg java.lang.Boolean.TYPE)
 *
 * If the string starts with a "[" it may contain a type of the form Lsometype;
 * If it doesn't start with a "[", anything of that form is rejected.
 *  
 */
jclass Java_java_lang_VMClass_forName(JNIEnv* env, jclass clazz, 
				      jobject stringObj) 
{ 
  char* pszClassName; 

  jclass result;
  int failure;
  //NB don't use the class var here, it may be NULL 
  
  if (stringObj == NULL) { 
    (*env)->Throw(env, INTERP_NewObjectFromName(env, "java/lang/ClassNotFoundException", &failure)); 
    return NULL; 
  } 
  
  pszClassName = INTERP_AscizFromString(env, stringObj); 
  
  if (strcmp(pszClassName, "") == 0) {
    (*env)->ThrowNew(env, (*env)->FindClass(env, "java/lang/ClassNotFoundException"), pszClassName); 
    result = NULL;
  }
  else {
    tClassLoaderTuple* loadingClass = NULL;
    tStackFrame* f;

    /* The loading class is the class that called Class.forName().  Look
       for the closest call to a method that isn't a Class or VMClass 
       method.  (XXX - is this right?) */
    f = JNI_getJNIData(sys_current_thread())->pstTopFrame;
    while (f != NULL) {
      if (f->pstCurrMethod->pstClass != pstClassType &&
	  f->pstCurrMethod->pstClass != pstVMClassType) {
	loadingClass = f->pstCurrMethod->pstClass;
	break;
      }
      f = f->pstPrevFrame;
    }

    result = VMClass_forName(env, pszClassName, loadingClass);
  }
  sys_free(pszClassName);
  return result;
}


jclass VMClass_forName(JNIEnv* env, char* pszClassName, 
		       tClassLoaderTuple* loadingClass) 
{
  tJNIData* pstJNIData = JNI_getJNIData(sys_current_thread());
  
  if (pszClassName[0] == '[') {
    // For an array type, the type name should be encoded as per the 
    // classfile encoding.
    int typeIndex = 0;
    int len = strlen(pszClassName);

    while (typeIndex < len && pszClassName[typeIndex] == '[') {
      typeIndex++;
    }
    if (typeIndex == len) {
      (*env)->ThrowNew(env, (*env)->FindClass(env, "java/lang/ClassNotFoundException"), pszClassName); 
      return NULL;
    }
    else if (pszClassName[typeIndex] == 'L') {
      if (strchr(pszClassName, ';') == NULL) {
	(*env)->ThrowNew(env, (*env)->FindClass(env, "java/lang/ClassNotFoundException"), pszClassName); 
	return NULL;
      }
      return VMClass_forNameHelperFromName(env, pszClassName, loadingClass);
    }
    else {
      tClassLoaderTuple* pstOurClassType;

      if (strchr(pszClassName, ';') != NULL) {
	(*env)->ThrowNew(env, (*env)->FindClass(env, "java/lang/ClassNotFoundException"), pszClassName); 
	return NULL;
      }
      //just check that the next character is that of a primitive array
      if (!((pszClassName[typeIndex] == 'V') ||
	    (pszClassName[typeIndex] == 'B') ||
	    (pszClassName[typeIndex] == 'C') ||
	    (pszClassName[typeIndex] == 'F') ||
	    (pszClassName[typeIndex] == 'S') ||
	    (pszClassName[typeIndex] == 'Z') ||
	    (pszClassName[typeIndex] == 'I') ||
	    (pszClassName[typeIndex] == 'D') ||
	    (pszClassName[typeIndex] == 'J'))) {
	(*env)->ThrowNew( env, (*env)->FindClass(env, "java/lang/ClassNotFoundException"), pszClassName); 
	return NULL;
      }
      //a primitive array ... use the bootstrap class loader.
      pstOurClassType =	CLASSFILE_FindOrLoad(env, pszClassName, NULL);
      return VMClass_forNameHelperFromStruct(env, pstOurClassType);
    }
  }
  else {
    // For a non-array class, it should be a regular class name, or
    // a Java primitive type keyword.
    if (strchr(pszClassName, ';') != NULL) {
      (*env)->ThrowNew(env, (*env)->FindClass(env, "java/lang/ClassNotFoundException"), pszClassName); 
      return NULL;
    }
    return VMClass_forNameHelperFromName(env, pszClassName, loadingClass);
  }
} 


jobjectArray 
  Java_java_lang_VMClass_getDeclaredConstructors(JNIEnv* env,  
						 jobject vmClass,
						 jboolean publicOnly)
{ 
  tClassLoaderTuple* tuple = CLASS_GetVMClassStruct(env, vmClass);
  jclass clazz = tuple->classObject;
  jclass consClazz = (*env)->FindClass(env, "java/lang/reflect/Constructor");
  jobjectArray retval;
  tMethod* method;
  
  int i = 0;
  int j = 0;
  int numConstructors = 0;
  
  /* first count them */
  if (Java_java_lang_VMClass_isInterface(env, vmClass) ||
      Java_java_lang_VMClass_isPrimitive(env, vmClass) ||
      Java_java_lang_VMClass_isArray(env, vmClass)) {
    // No constructors for interfaces, primitive types or arrays
    numConstructors = 0; 
  }
  else {
    for (i = 0; i < tuple->pstClass->u16MethodsCount; i++) {
      method = tuple->pstClass->pstMethods + i;
      if (strcmp(method->uidName, "<init>") == 0) {
	if (!publicOnly || (method->u16AccessFlags & ACC_PUBLIC)) {     
	  numConstructors++;
	}
      }
    }
  }
  
  /* create our return array */
  retval = (*env)->NewObjectArray(env, numConstructors, consClazz, NULL);
  if (retval == NULL) { 
    (*env)->Throw(env, OOMExceptionObject);
    return NULL;
  }

  if (numConstructors == 0) {
    return retval;
  }
  
  //now make objects for each one
  for (i = 0; i < tuple->pstClass->u16MethodsCount; i++) {
    int k = 0;
    jobjectArray exceptionTypeArray;
    
    method = tuple->pstClass->pstMethods + i;
    if (strcmp(method->uidName, "<init>") == 0) { 
      if (!publicOnly || (method->u16AccessFlags & ACC_PUBLIC)) {
	jfieldID clazzField;
	jfieldID slotField;
	jfieldID parametersField;
	jfieldID exceptionsField;
	
	jobject consObject;
        jobjectArray sigArgs;
	
	int failure;
	
	clazzField = (*env)->GetFieldID(env, consClazz,
					"clazz", "Ljava/lang/Class;");
	slotField = (*env)->GetFieldID(env, consClazz, "slot", "I");
	parametersField = (*env)->GetFieldID(env, consClazz, "parameterTypes",
					     "[Ljava/lang/Class;");
	exceptionsField = (*env)->GetFieldID(env, consClazz, "exceptionTypes",
					     "[Ljava/lang/Class;");
	
	if ((clazzField == NULL) || (slotField == NULL) || 
	    (parametersField == NULL) || (exceptionsField == NULL)) {
	  (*env)->Throw(env, INTERP_ExceptionObjectFromNameAndMessage(env, "java/lang/InternalError", "Couldn't get field for java.lang.reflect.Constructor"));
	  return NULL;
	}
	
	consObject = INTERP_NewObject(env, 
				      CLASS_GetClassStruct(env, consClazz),
				      &failure);

	if (consObject == NULL) { 
	  (*env)->Throw(env, OOMExceptionObject);
	  return NULL; 
	}
	
        sigArgs = CLASS_GetClassArrayFromSig(env, method->uidSignature, 
					     method->pstClass);
	if (sigArgs == NULL) {
	  return NULL; //exception has been thrown
	}

	(*env)->SetObjectField(env, consObject, clazzField, clazz);
	(*env)->SetIntField(env, consObject, slotField, i);
	(*env)->SetObjectField(env, consObject, parametersField, 
			       (jobject) sigArgs);
	
	if (method->bHaveResolvedExceptions != 1) {
	  if (CLASSFILE_ResolveMethodExceptions(env, method) != 0) {
	    return NULL; //exception has been thrown
	  }
	}
	assert(method->bHaveResolvedExceptions == 1);
	
	exceptionTypeArray =
	  (*env)->NewObjectArray(env, method->u16NumExceptions, 
				 pstClassType->classObject, NULL);
	for (k = 0; k < method->u16NumExceptions; k++) {
	  jclass exceptionClazz =
	    CLASSLOADER_TUPLE_GetClassObject(env, method->pstExceptions[k]);
	  (*env)->SetObjectArrayElement(env, exceptionTypeArray, k,
					exceptionClazz);
					
	}
	(*env)->SetObjectField(env, consObject, exceptionsField, 
			       (jobject) exceptionTypeArray);
	(*env)->SetObjectArrayElement(env, retval, j++, consObject);
      }
    }
  }
  return retval;
}


jobjectArray Java_java_lang_VMClass_getDeclaredMethods(JNIEnv* env,
						       jobject vmClass,
						       jboolean publicOnly)
{
  tClassLoaderTuple* theClass, *saveClass; 
  tMethod* method; 
  jobjectArray retval; 
  jclass methodClazz = (*env)->FindClass(env, "java/lang/reflect/Method");
  int i; 
  int numFound = 0; 
  int j = 0; 
  
  assert(methodClazz);
  saveClass = theClass = CLASS_GetVMClassStruct(env, vmClass);
  assert(theClass); 
  
  // XXX - need to test for the Class representing 'void' too 
  if (Java_java_lang_VMClass_isPrimitive(env, vmClass) ||
      Java_java_lang_VMClass_isArray(env, vmClass)) {
    // No methods for interfaces, primitive types oe array types
    numFound = 0;
  }
  else {
    while (theClass != NULL) { 
      for (i = 0; i < theClass->pstClass->u16MethodsCount; i++) { 
	method = (tMethod*) (&theClass->pstClass->pstMethods[i]); 
	if ((strcmp(method->uidName, "<init>") == 0) ||
	    (strcmp(method->uidName, "<clinit>") == 0)) { 
	  // Skip initialisers
	} 
	else { 
	  if (publicOnly) {
	    if (method->u16AccessFlags & ACC_PUBLIC) {
	      numFound++; 
	    } 
	  }
	  else {
	    if (method->pstClass == saveClass) {
	      numFound++; 
	    }
	  }
	} 
      }
      theClass = theClass->pstClass->pstSuperClass; 
    } 
  }
  
  theClass = saveClass;

  retval = (*env)->NewObjectArray(env, numFound, methodClazz, NULL);
  
  while (numFound > 0) { 
    while (theClass != NULL) { 
      for (i = 0; i < theClass->pstClass->u16MethodsCount; i++) { 
	method =
	  (tMethod*) (&theClass->pstClass->pstMethods[i]); 
	if ((strcmp(method->uidName, "<init>") == 0) ||
	    (strcmp(method->uidName, "<clinit>") == 0)) { 
	  // Skip initialisers
	} 
	else { 
	  int useMethod = 0;
	  if (publicOnly) {
	    if (method->u16AccessFlags & ACC_PUBLIC) {
	      useMethod = 1;
	    }
	  }
	  else {
	    if (method->pstClass == saveClass) {
	      useMethod = 1;
	    }
	  }
	  
	  if (useMethod) { 
	    jfieldID fieldID; 
	    int failure;
	    
	    jobject o = 
	      INTERP_NewObjectFromName(env, "java/lang/reflect/Method", 
				       &failure); 
	    jclass methodClazz = (*env)->GetObjectClass(env, o); 
	    
	    numFound --; 
	    
	    fieldID = (*env)->GetFieldID(env, methodClazz, "declaringClass",
					 "Ljava/lang/Class;"); 
	    assert(fieldID); 
	    (*env)->SetObjectField(env, o, fieldID,
				   CLASSLOADER_TUPLE_GetClassObject(env,
								    theClass));
	    
	    //Set the name 
	    fieldID = (*env)->GetFieldID(env, methodClazz, "name",
					 "Ljava/lang/String;"); 
	    assert(fieldID); 
	    
	    (*env)->SetObjectField(env, o, fieldID,
				   INTERP_NewStringFromAsciz(env,
							     method->uidName));
	    //Set the slot 
	    fieldID = (*env)->GetFieldID(env, methodClazz, "slot", "I"); 
	    assert(fieldID); 
	    
	    (*env)->SetIntField(env, o, fieldID, i); 
	    
	    (*env)->SetObjectArrayElement(env, retval, j, o);
	    j++; 
	    
	    //we call this to make sure the exception types can be loaded
	    if (java_lang_reflect_Method_getExceptionTypes(env, o) == NULL) {
	      //an exception has occurred
	      return NULL;
	    }
	  }
	} //end else (init / /clinit)
      }
      theClass = theClass->pstClass->pstSuperClass; 
      
    }
  } 
  
  assert(retval != NULL); 
  return retval;    
}
