#include "config.h"
/* 
 * @doc MODULE 
 * @module disass.c | 
 * 
 * Procedures for disassembling Java bytecode instructions and printing 
 * them out. 
 * 
 */ 

#ifdef KISSME_LINUX_USER 
#include <stdio.h> 
#endif

#include "vm/stdtypes.h" 
#include "vm/opcodes.h" 
 
#include "vm/global.h"
#include "vm/interp.h"
 
static char* pszHex1Byte = "\t0x%02x "; 
static char* pszHex2Byte = "\t0x%04x "; 
static char* pszHex4Byte = "\t0x%08x "; 
 
/* 
 * @doc 
 * @globv ppszOpCodeNames | Array of strings containing opcode names 
 * 
 */ 
 
static char* ppszOpCodeNames[] = 
{ 
  "nop", 
  "aconst_null", 
  "iconst_m1", 
  "iconst_0", 
  "iconst_1", 
  "iconst_2", 
  "iconst_3", 
  "iconst_4", 
  "iconst_5", 
  "lconst_0", 
  "lconst_1", 
  "fconst_0", 
  "fconst_1", 
  "fconst_2", 
  "dconst_0", 
  "dconst_1", 
  "bipush", 
  "sipush", 
  "ldc", 
  "ldc_w", 
  "ldc2_w", 
  "iload", 
  "lload", 
  "fload", 
  "dload", 
  "aload", 
  "iload_0", 
  "iload_1", 
  "iload_2", 
  "iload_3", 
  "lload_0", 
  "lload_1", 
  "lload_2", 
  "lload_3", 
  "fload_0", 
  "fload_1", 
  "fload_2", 
  "fload_3", 
  "dload_0", 
  "dload_1", 
  "dload_2", 
  "dload_3", 
  "aload_0", 
  "aload_1", 
  "aload_2", 
  "aload_3", 
  "iaload", 
  "laload", 
  "faload", 
  "daload", 
  "aaload", 
  "baload", 
  "caload", 
  "saload", 
  "istore", 
  "lstore", 
  "fstore", 
  "dstore", 
  "astore", 
  "istore_0", 
  "istore_1", 
  "istore_2", 
  "istore_3", 
  "lstore_0", 
  "lstore_1", 
  "lstore_2", 
  "lstore_3", 
  "fstore_0", 
  "fstore_1", 
  "fstore_2", 
  "fstore_3", 
  "dstore_0", 
  "dstore_1", 
  "dstore_2", 
  "dstore_3", 
  "astore_0", 
  "astore_1", 
  "astore_2", 
  "astore_3", 
  "iastore", 
  "lastore", 
  "fastore", 
  "dastore", 
  "aastore", 
  "bastore", 
  "castore", 
  "sastore", 
  "pop", 
  "pop2", 
  "dup", 
  "dup_x1", 
  "dup_x2", 
  "dup2", 
  "dup2_x1", 
  "dup2_x2", 
  "swap", 
  "iadd", 
  "ladd", 
  "fadd", 
  "dadd", 
  "isub", 
  "lsub", 
  "fsub", 
  "dsub", 
  "imul", 
  "lmul", 
  "fmul", 
  "dmul", 
  "idiv", 
  "ldiv", 
  "fdiv", 
  "ddiv", 
  "imod", 
  "lmod", 
  "fmod", 
  "dmod", 
  "ineg", 
  "lneg", 
  "fneg", 
  "dneg", 
  "ishl", 
  "lshl", 
  "ishr", 
  "lshr", 
  "iushr", 
  "lushr", 
  "iand", 
  "land", 
  "ior", 
  "lor", 
  "ixor", 
  "lxor", 
  "iinc", 
  "i2l", 
  "i2f", 
  "i2d", 
  "l2i", 
  "l2f", 
  "l2d", 
  "f2i", 
  "f2l", 
  "f2d", 
  "d2i", 
  "d2l", 
  "d2f", 
  "int2byte", 
  "int2char", 
  "int2short", 
  "lcmp", 
  "fcmpl", 
  "fcmpg", 
  "dcmpl", 
  "dcmpg", 
  "ifeq", 
  "ifne", 
  "iflt", 
  "ifge", 
  "ifgt", 
  "ifle", 
  "if_icmpeq", 
  "if_icmpne", 
  "if_icmplt", 
  "if_icmpge", 
  "if_icmpgt", 
  "if_icmple", 
  "if_acmpeq", 
  "if_acmpne", 
  "goto", 
  "jsr", 
  "ret", 
  "tableswitch", 
  "lookupswitch", 
  "ireturn", 
  "lreturn", 
  "freturn", 
  "dreturn", 
  "areturn", 
  "return", 
  "getstatic", 
  "putstatic", 
  "getfield", 
  "putfield", 
  "invokevirtual", 
  "invokespecial", 
  "invokestatic", 
  "invokeinterface", 
  "newfromname", 
  "new", 
  "newarray", 
  "anewarray", 
  "arraylength", 
  "athrow", 
  "checkcast", 
  "instanceof", 
  "monitorenter", 
  "monitorexit", 
  "wide", 
/*  "breakpoint",*/ 
  "multianewarray", 
  "ifnull", 
  "ifnonnull", 
 
  /* optimisations */ 
  "ldc1_quick", 
  "ldc2_quick", 
  "ldc2w_quick", 
 
  "getfield_quick", 
  "putfield_quick", 
  "getfield2_quick", 
  "putfield2_quick", 
 
  "getstatic_quick", 
  "putstatic_quick", 
  "getstatic2_quick", 
  "putstatic2_quick", 
 
  "invokevirtual_quick", 
  "invokenonvirtual_quick", 
  "invokestatic_quick", 
  "invokeinterface_quick", 
  "invokevirtualobject_quick",     /* ???? I can't understand this !!!! */ 
 
  "new_quick", 
  "anewarray_quick", 
  "checkcast_quick", 
  "instanceof_quick", 
 
  "putstaticcurrclass_quick", 
  "putstatic2currclass_quick", 
  "getstaticcurrclass_quick", 
  "getstatic2currclass_quick", 
 
  "invokenonvirtualcurrclass_quick", 
  "invokestaticcurrclass_quick", 
 
  "invokeinterface_noguess", 
  "invokevirtual_quick_optimised" 
}; 
 
 
 
/* 
 * @doc FUNC 
 * @func 
 * This function is used to print out a bytecode instruction by name and 
 * print out its parameters. 
 * 
 * @rdesc Returns the total number of bytes that the instruction and its 
 *        parameters use. 
 * 
 */ 
 
int DISASS_PrintInstruction 
( 
  byte* pbInstr,     /* @parm Pointer to current instruction in code */ 
  byte* pbCodeStart  /* @parm Pointer to beginning of code */ 
) 
{ 
  int iNumBytes; /* total number of bytes used by instruction */ 
 
  eprintf("%s", ppszOpCodeNames[pbInstr[0]]); 
 
  switch (pbInstr[0]) 
  { 
    case nop: 
    { 
      iNumBytes = 1; 
      break; 
    } 
    case aconst_null: 
    { 
      iNumBytes = 1; 
      break; 
    } 
    case iconst_m1: 
    { 
      iNumBytes = 1; 
      break; 
    } 
    case iconst_0: 
    { 
      iNumBytes = 1; 
      break; 
    } 
    case iconst_1: 
    { 
      iNumBytes = 1; 
      break; 
    } 
    case iconst_2: 
    { 
      iNumBytes = 1; 
      break; 
    } 
    case iconst_3: 
    { 
      iNumBytes = 1; 
      break; 
    } 
    case iconst_4: 
    { 
      iNumBytes = 1; 
      break; 
    } 
    case iconst_5: 
    { 
      iNumBytes = 1; 
      break; 
    } 
    case lconst_0: 
    { 
      iNumBytes = 1; 
      break; 
    } 
    case lconst_1: 
    { 
      iNumBytes = 1; 
      break; 
    } 
    case fconst_0: 
    { 
      iNumBytes = 1; 
      break; 
    } 
    case fconst_1: 
    { 
      iNumBytes = 1; 
      break; 
    } 
    case fconst_2: 
    { 
      iNumBytes = 1; 
      break; 
    } 
    case dconst_0: 
    { 
      iNumBytes = 1; 
      break; 
    } 
    case dconst_1: 
    { 
      iNumBytes = 1; 
      break; 
    } 
    case bipush: 
    { 
      eprintf( pszHex1Byte, pbInstr[1]); 
      iNumBytes = 2; 
      break; 
    } 
    case sipush: 
    { 
      eprintf( pszHex2Byte, pbInstr[1] * 256 + pbInstr[2]); 
      iNumBytes = 3; 
      break; 
    } 
    case ldc: 
    { 
      eprintf( pszHex1Byte, pbInstr[1]); 
      iNumBytes = 2; 
      break; 
    } 
    case ldc_w: 
    { 
      eprintf( pszHex2Byte, pbInstr[1] * 256 + pbInstr[2]); 
      iNumBytes = 3; 
      break; 
    } 
    case ldc2_w: 
    { 
      eprintf( pszHex2Byte, pbInstr[1] * 256 + pbInstr[2]); 
      iNumBytes = 3; 
      break; 
    } 
    case iload: 
    { 
      eprintf( pszHex1Byte, pbInstr[1]); 
      iNumBytes = 2; 
      break; 
    } 
    case lload: 
    { 
      eprintf( pszHex1Byte, pbInstr[1]); 
      iNumBytes = 2; 
      break; 
    } 
    case fload: 
    { 
      eprintf( pszHex1Byte, pbInstr[1]); 
      iNumBytes = 2; 
      break; 
    } 
    case dload: 
    { 
      eprintf( pszHex1Byte, pbInstr[1]); 
      iNumBytes = 2; 
      break; 
    } 
    case aload: 
    { 
      eprintf( pszHex1Byte, pbInstr[1]); 
      iNumBytes = 2; 
      break; 
    } 
    case iload_0: 
    { 
      iNumBytes = 1; 
      break; 
    } 
    case iload_1: 
    { 
      iNumBytes = 1; 
      break; 
    } 
    case iload_2: 
    { 
      iNumBytes = 1; 
      break; 
    } 
    case iload_3: 
    { 
      iNumBytes = 1; 
      break; 
    } 
    case lload_0: 
    { 
      iNumBytes = 1; 
      break; 
    } 
    case lload_1: 
    { 
      iNumBytes = 1; 
      break; 
    } 
    case lload_2: 
    { 
      iNumBytes = 1; 
      break; 
    } 
    case lload_3: 
    { 
      iNumBytes = 1; 
      break; 
    } 
    case fload_0: 
    { 
      iNumBytes = 1; 
      break; 
    } 
    case fload_1: 
    { 
      iNumBytes = 1; 
      break; 
    } 
    case fload_2: 
    { 
      iNumBytes = 1; 
      break; 
    } 
    case fload_3: 
    { 
      iNumBytes = 1; 
      break; 
    } 
    case dload_0: 
    { 
      iNumBytes = 1; 
      break; 
    } 
    case dload_1: 
    { 
      iNumBytes = 1; 
      break; 
    } 
    case dload_2: 
    { 
      iNumBytes = 1; 
      break; 
    } 
    case dload_3: 
    { 
      iNumBytes = 1; 
      break; 
    } 
    case aload_0: 
    { 
      iNumBytes = 1; 
      break; 
    } 
    case aload_1: 
    { 
      iNumBytes = 1; 
      break; 
    } 
    case aload_2: 
    { 
      iNumBytes = 1; 
      break; 
    } 
    case aload_3: 
    { 
      iNumBytes = 1; 
      break; 
    } 
    case iaload: 
    { 
      iNumBytes = 1; 
      break; 
    } 
    case laload: 
    { 
      iNumBytes = 1; 
      break; 
    } 
    case faload: 
    { 
      iNumBytes = 1; 
      break; 
    } 
    case daload: 
    { 
      iNumBytes = 1; 
      break; 
    } 
    case aaload: 
    { 
      iNumBytes = 1; 
      break; 
    } 
    case baload: 
    { 
      iNumBytes = 1; 
      break; 
    } 
    case caload: 
    { 
      iNumBytes = 1; 
      break; 
    } 
    case saload: 
    { 
      iNumBytes = 1; 
      break; 
    } 
    case istore: 
    { 
      eprintf( pszHex1Byte, pbInstr[1]); 
      iNumBytes = 2; 
      break; 
    } 
    case lstore: 
    { 
      eprintf( pszHex1Byte, pbInstr[1]); 
      iNumBytes = 2; 
      break; 
    } 
    case fstore: 
    { 
      eprintf( pszHex1Byte, pbInstr[1]); 
      iNumBytes = 2; 
      break; 
    } 
    case dstore: 
    { 
      eprintf( pszHex1Byte, pbInstr[1]); 
      iNumBytes = 2; 
      break; 
    } 
    case astore: 
    { 
      eprintf( pszHex1Byte, pbInstr[1]); 
      iNumBytes = 2; 
      break; 
    } 
    case istore_0: 
    { 
      iNumBytes = 1; 
      break; 
    } 
    case istore_1: 
    { 
      iNumBytes = 1; 
      break; 
    } 
    case istore_2: 
    { 
      iNumBytes = 1; 
      break; 
    } 
    case istore_3: 
    { 
      iNumBytes = 1; 
      break; 
    } 
    case lstore_0: 
    { 
      iNumBytes = 1; 
      break; 
    } 
    case lstore_1: 
    { 
      iNumBytes = 1; 
      break; 
    } 
    case lstore_2: 
    { 
      iNumBytes = 1; 
      break; 
    } 
    case lstore_3: 
    { 
      iNumBytes = 1; 
      break; 
    } 
    case fstore_0: 
    { 
      iNumBytes = 1; 
      break; 
    } 
    case fstore_1: 
    { 
      iNumBytes = 1; 
      break; 
    } 
    case fstore_2: 
    { 
      iNumBytes = 1; 
      break; 
    } 
    case fstore_3: 
    { 
      iNumBytes = 1; 
      break; 
    } 
    case dstore_0: 
    { 
      iNumBytes = 1; 
      break; 
    } 
    case dstore_1: 
    { 
      iNumBytes = 1; 
      break; 
    } 
    case dstore_2: 
    { 
      iNumBytes = 1; 
      break; 
    } 
    case dstore_3: 
    { 
      iNumBytes = 1; 
      break; 
    } 
    case astore_0: 
    { 
      iNumBytes = 1; 
      break; 
    } 
    case astore_1: 
    { 
      iNumBytes = 1; 
      break; 
    } 
    case astore_2: 
    { 
      iNumBytes = 1; 
      break; 
    } 
    case astore_3: 
    { 
      iNumBytes = 1; 
      break; 
    } 
    case iastore: 
    { 
      iNumBytes = 1; 
      break; 
    } 
    case lastore: 
    { 
      iNumBytes = 1; 
      break; 
    } 
    case fastore: 
    { 
      iNumBytes = 1; 
      break; 
    } 
    case dastore: 
    { 
      iNumBytes = 1; 
      break; 
    } 
    case aastore: 
    { 
      iNumBytes = 1; 
      break; 
    } 
    case bastore: 
    { 
      iNumBytes = 1; 
      break; 
    } 
    case castore: 
    { 
      iNumBytes = 1; 
      break; 
    } 
    case sastore: 
    { 
      iNumBytes = 1; 
      break; 
    } 
    case pop: 
    { 
      iNumBytes = 1; 
      break; 
    } 
    case pop2: 
    { 
      iNumBytes = 1; 
      break; 
    } 
    case dup_: 
    { 
      iNumBytes = 1; 
      break; 
    } 
    case dup_x1: 
    { 
      iNumBytes = 1; 
      break; 
    } 
    case dup_x2: 
    { 
      iNumBytes = 1; 
      break; 
    } 
    case dup2_: 
    { 
      iNumBytes = 1; 
      break; 
    } 
    case dup2_x1: 
    { 
      iNumBytes = 1; 
      break; 
    } 
    case dup2_x2: 
    { 
      iNumBytes = 1; 
      break; 
    } 
    case swap: 
    { 
      iNumBytes = 1; 
      break; 
    } 
    case iadd: 
    { 
      iNumBytes = 1; 
      break; 
    } 
    case ladd: 
    { 
      iNumBytes = 1; 
      break; 
    } 
    case fadd: 
    { 
      iNumBytes = 1; 
      break; 
    } 
    case dadd: 
    { 
      iNumBytes = 1; 
      break; 
    } 
    case isub: 
    { 
      iNumBytes = 1; 
      break; 
    } 
    case lsub: 
    { 
      iNumBytes = 1; 
      break; 
    } 
    case fsub: 
    { 
      iNumBytes = 1; 
      break; 
    } 
    case dsub: 
    { 
      iNumBytes = 1; 
      break; 
    } 
    case imul: 
    { 
      iNumBytes = 1; 
      break; 
    } 
    case lmul: 
    { 
      iNumBytes = 1; 
      break; 
    } 
    case fmul: 
    { 
      iNumBytes = 1; 
      break; 
    } 
    case dmul: 
    { 
      iNumBytes = 1; 
      break; 
    } 
    case idiv: 
    { 
      iNumBytes = 1; 
      break; 
    } 
    case ldiv_: 
    { 
      iNumBytes = 1; 
      break; 
    } 
    case fdiv: 
    { 
      iNumBytes = 1; 
      break; 
    } 
    case ddiv: 
    { 
      iNumBytes = 1; 
      break; 
    } 
    case imod: 
    { 
      iNumBytes = 1; 
      break; 
    } 
    case lmod: 
    { 
      iNumBytes = 1; 
      break; 
    } 
    case fmod_: 
    { 
      iNumBytes = 1; 
      break; 
    } 
    case dmod: 
    { 
      iNumBytes = 1; 
      break; 
    } 
    case ineg: 
    { 
      iNumBytes = 1; 
      break; 
    } 
    case lneg: 
    { 
      iNumBytes = 1; 
      break; 
    } 
    case fneg: 
    { 
      iNumBytes = 1; 
      break; 
    } 
    case dneg: 
    { 
      iNumBytes = 1; 
      break; 
    } 
    case ishl: 
    { 
      iNumBytes = 1; 
      break; 
    } 
    case lshl: 
    { 
      iNumBytes = 1; 
      break; 
    } 
    case ishr: 
    { 
      iNumBytes = 1; 
      break; 
    } 
    case lshr: 
    { 
      iNumBytes = 1; 
      break; 
    } 
    case iushr: 
    { 
      iNumBytes = 1; 
      break; 
    } 
    case lushr: 
    { 
      iNumBytes = 1; 
      break; 
    } 
    case iand: 
    { 
      iNumBytes = 1; 
      break; 
    } 
    case land: 
    { 
      iNumBytes = 1; 
      break; 
    } 
    case ior: 
    { 
      iNumBytes = 1; 
      break; 
    } 
    case lor: 
    { 
      iNumBytes = 1; 
      break; 
    } 
    case ixor: 
    { 
      iNumBytes = 1; 
      break; 
    } 
    case lxor: 
    { 
      iNumBytes = 1; 
      break; 
    } 
    case iinc: 
    { 
      eprintf( pszHex1Byte, pbInstr[1]); 
      eprintf( pszHex1Byte, pbInstr[2]); 
      iNumBytes = 3; 
      break; 
    } 
    case i2l: 
    { 
      iNumBytes = 1; 
      break; 
    } 
    case i2f: 
    { 
      iNumBytes = 1; 
      break; 
    } 
    case i2d: 
    { 
      iNumBytes = 1; 
      break; 
    } 
    case l2i: 
    { 
      iNumBytes = 1; 
      break; 
    } 
    case l2f: 
    { 
      iNumBytes = 1; 
      break; 
    } 
    case l2d: 
    { 
      iNumBytes = 1; 
      break; 
    } 
    case f2i: 
    { 
      iNumBytes = 1; 
      break; 
    } 
    case f2l: 
    { 
      iNumBytes = 1; 
      break; 
    } 
    case f2d: 
    { 
      iNumBytes = 1; 
      break; 
    } 
    case d2i: 
    { 
      iNumBytes = 1; 
      break; 
    } 
    case d2l: 
    { 
      iNumBytes = 1; 
      break; 
    } 
    case d2f: 
    { 
      iNumBytes = 1; 
      break; 
    } 
    case int2byte: 
    { 
      iNumBytes = 1; 
      break; 
    } 
    case int2char: 
    { 
      iNumBytes = 1; 
      break; 
    } 
    case int2short: 
    { 
      iNumBytes = 1; 
      break; 
    } 
    case lcmp: 
    { 
      iNumBytes = 1; 
      break; 
    } 
    case fcmpl: 
    { 
      iNumBytes = 1; 
      break; 
    } 
    case fcmpg: 
    { 
      iNumBytes = 1; 
      break; 
    } 
    case dcmpl: 
    { 
      iNumBytes = 1; 
      break; 
    } 
    case dcmpg: 
    { 
      iNumBytes = 1; 
      break; 
    } 
    case ifeq: 
    { 
      eprintf( pszHex2Byte, pbInstr[1] * 256 + pbInstr[2]); 
      iNumBytes = 3; 
      break; 
    } 
    case ifne: 
    { 
      eprintf( pszHex2Byte, pbInstr[1] * 256 + pbInstr[2]); 
      iNumBytes = 3; 
      break; 
    } 
    case iflt: 
    { 
      eprintf( pszHex2Byte, pbInstr[1] * 256 + pbInstr[2]); 
      iNumBytes = 3; 
      break; 
    } 
    case ifge: 
    { 
      eprintf( pszHex2Byte, pbInstr[1] * 256 + pbInstr[2]); 
      iNumBytes = 3; 
      break; 
    } 
    case ifgt: 
    { 
      eprintf( pszHex2Byte, pbInstr[1] * 256 + pbInstr[2]); 
      iNumBytes = 3; 
      break; 
    } 
    case ifle: 
    { 
      eprintf( pszHex2Byte, pbInstr[1] * 256 + pbInstr[2]); 
      iNumBytes = 3; 
      break; 
    } 
    case if_icmpeq: 
    { 
      eprintf( pszHex2Byte, pbInstr[1] * 256 + pbInstr[2]); 
      iNumBytes = 3; 
      break; 
    } 
    case if_icmpne: 
    { 
      eprintf( pszHex2Byte, pbInstr[1] * 256 + pbInstr[2]); 
      iNumBytes = 3; 
      break; 
    } 
    case if_icmplt: 
    { 
      eprintf( pszHex2Byte, pbInstr[1] * 256 + pbInstr[2]); 
      iNumBytes = 3; 
      break; 
    } 
    case if_icmpge: 
    { 
      eprintf( pszHex2Byte, pbInstr[1] * 256 + pbInstr[2]); 
      iNumBytes = 3; 
      break; 
    } 
    case if_icmpgt: 
    { 
      eprintf( pszHex2Byte, pbInstr[1] * 256 + pbInstr[2]); 
      iNumBytes = 3; 
      break; 
    } 
    case if_icmple: 
    { 
      eprintf( pszHex2Byte, pbInstr[1] * 256 + pbInstr[2]); 
      iNumBytes = 3; 
      break; 
    } 
    case if_acmpeq: 
    { 
      eprintf( pszHex2Byte, pbInstr[1] * 256 + pbInstr[2]); 
      iNumBytes = 3; 
      break; 
    } 
    case if_acmpne: 
    { 
      eprintf( pszHex2Byte, pbInstr[1] * 256 + pbInstr[2]); 
      iNumBytes = 3; 
      break; 
    } 
    case goto_: 
    { 
      eprintf( pszHex2Byte, pbInstr[1] * 256 + pbInstr[2]); 
      iNumBytes = 3; 
      break; 
    } 
    case jsr: 
    { 
      eprintf( pszHex2Byte, pbInstr[1] * 256 + pbInstr[2]); 
      iNumBytes = 3; 
      break; 
    } 
    case ret: 
    { 
      eprintf( pszHex1Byte, pbInstr[1]); 
      iNumBytes = 2; 
      break; 
    } 
    case tableswitch: 
    { 
				int32 i32Temp; 
				int32 i32DefaultOff; 
				int32 i32Low; 
				int32 i32High; 
        int   i; 
        byte* pbPC; 
 
        /* remember PC at beginning of instruction */ 
        pbPC = pbInstr + 1; 
 
        eprintf(" ("); 
        while ((pbPC - pbCodeStart) % 4) 
        { 
			    eprintf( pszHex1Byte, *(pbPC++)); 
        } 
        eprintf( ") "); 
 
        i32DefaultOff = (((int32)pbPC[0]) << 24) | (((int32)pbPC[1]) << 16) | (((int32)pbPC[2]) << 8) | pbPC[3]; 
        i32Low = (((int32)pbPC[4]) << 24) | (((int32)pbPC[5]) << 16) | (((int32)pbPC[6]) << 8) | pbPC[7]; 
        i32High = (((int32)pbPC[8]) << 24) | (((int32)pbPC[9]) << 16) | (((int32)pbPC[10]) << 8) | (pbPC[11]); 
        pbPC += 12; 
        eprintf( pszHex4Byte, i32DefaultOff); 
        eprintf( pszHex4Byte, i32Low); 
        eprintf( pszHex4Byte, i32High); 
 
        for (i = 0; i < i32High - i32Low + 1; i++) 
        { 
          i32Temp = (((int32)pbPC[0]) << 24) | (((int32)pbPC[1]) << 16) | (((int32)pbPC[2]) << 8) | (pbPC[3]); 
          pbPC += 4; 
          eprintf( pszHex4Byte, i32Temp); 
        } 
        iNumBytes = pbPC - pbInstr; 
        break; 
    } 
    case lookupswitch: 
    { 
			int32 i32Temp; 
			int32 i; 
			int32 i32DefaultOff; 
			int32 i32NoPairs; 
      byte* pbPC; 
 
      /* remember PC at beginning of instruction */ 
      pbPC = pbInstr + 1; 
 
      eprintf( " ("); 
      while ((pbPC - pbCodeStart) % 4) 
      { 
			  eprintf( pszHex1Byte, *(pbPC++)); 
      } 
      eprintf( ") "); 
 
      i32DefaultOff = (((int32)pbPC[0]) << 24) | (((int32)pbPC[1]) << 16) | (((int32)pbPC[2]) << 8) | pbPC[3]; 
      i32NoPairs = (((int32)pbPC[4]) << 24) | (((int32)pbPC[5]) << 16) | (((int32)pbPC[6]) << 8) | pbPC[7]; 
      pbPC += 8; 
      eprintf( pszHex4Byte, i32DefaultOff); 
      eprintf( pszHex4Byte, i32NoPairs); 
 
			for (i = 0; i < i32NoPairs; i++) 
			{ 
        i32Temp = (((int32)pbPC[0]) << 24) | (((int32)pbPC[1]) << 16) | (((int32)pbPC[2]) << 8) | (pbPC[3]); 
        pbPC += 4; 
        eprintf( pszHex4Byte, i32Temp); 
        i32Temp = (((int32)pbPC[0]) << 24) | (((int32)pbPC[1]) << 16) | (((int32)pbPC[2]) << 8) | (pbPC[3]); 
        pbPC += 4; 
        eprintf( pszHex4Byte, i32Temp); 
      } 
      iNumBytes = pbPC - pbInstr; 
      break; 
    } 
    case ireturn: 
    { 
      iNumBytes = 1; 
      break; 
    } 
    case lreturn: 
    { 
      iNumBytes = 1; 
      break; 
    } 
    case freturn: 
    { 
      iNumBytes = 1; 
      break; 
    } 
    case dreturn: 
    { 
      iNumBytes = 1; 
      break; 
    } 
    case areturn: 
    { 
      iNumBytes = 1; 
      break; 
    } 
    case return_: 
    { 
      iNumBytes = 1; 
      break; 
    } 
    case getstatic: 
    { 
      eprintf( pszHex2Byte, pbInstr[1] * 256 + pbInstr[2]); 
      iNumBytes = 3; 
      break; 
    } 
    case putstatic: 
    { 
      eprintf( pszHex2Byte, pbInstr[1] * 256 + pbInstr[2]); 
      iNumBytes = 3; 
      break; 
    } 
    case getfield: 
    { 
      eprintf( pszHex2Byte, pbInstr[1] * 256 + pbInstr[2]); 
      iNumBytes = 3; 
      break; 
    } 
    case putfield: 
    { 
      eprintf( pszHex2Byte, pbInstr[1] * 256 + pbInstr[2]); 
      iNumBytes = 3; 
      break; 
    } 
    case invokevirtual: 
    { 
      eprintf( pszHex2Byte, pbInstr[1] * 256 + pbInstr[2]); 
      iNumBytes = 3; 
      break; 
    } 
    case invokespecial: 
    { 
      eprintf( pszHex2Byte, pbInstr[1] * 256 + pbInstr[2]); 
      iNumBytes = 3; 
      break; 
    } 
    case invokestatic: 
    { 
      eprintf(pszHex2Byte, pbInstr[1] * 256 + pbInstr[2]); 
      iNumBytes = 3; 
      break; 
    } 
    case invokeinterface: 
    { 
      eprintf(pszHex2Byte, pbInstr[1] * 256 + pbInstr[2]); 
      iNumBytes = 3; 
      break; 
    } 
    case newfromname: 
    { 
      iNumBytes = 1; 
      break; 
    } 
    case new_: 
    { 
      eprintf( pszHex2Byte, pbInstr[1] * 256 + pbInstr[2]); 
      iNumBytes = 3; 
      break; 
    } 
    case newarray: 
    { 
      eprintf( pszHex1Byte, pbInstr[1]); 
      iNumBytes = 2; 
      break; 
    } 
    case anewarray: 
    { 
      eprintf( pszHex2Byte, pbInstr[1] * 256 + pbInstr[2]); 
      iNumBytes = 3; 
      break; 
    } 
    case arraylength: 
    { 
      iNumBytes = 1; 
      break; 
    } 
    case athrow: 
    { 
      iNumBytes = 1; 
      break; 
    } 
    case checkcast: 
    { 
      eprintf( pszHex2Byte, pbInstr[1] * 256 + pbInstr[2]); 
      iNumBytes = 3; 
      break; 
    } 
    case instanceof: 
    { 
      eprintf( pszHex2Byte, pbInstr[1] * 256 + pbInstr[2]); 
      iNumBytes = 3; 
      break; 
    } 
    case monitorenter: 
    { 
      iNumBytes = 1; 
      break; 
    } 
    case monitorexit: 
    { 
      iNumBytes = 1; 
      break; 
    } 
  case wide:
    { 
      eprintf("ERROR, encountered wide and don't know how many bytes follow\n");
      iNumBytes = 2; 
      break; 
    } 
/*    case breakpoint: 
    { 
      iNumBytes = 1; 
      break; 
    }*/ 
    case multianewarray: 
    { 
      eprintf( pszHex2Byte, pbInstr[1], pbInstr[2]); 
      eprintf( pszHex1Byte, pbInstr[3]); 
      iNumBytes = 4; 
      break; 
    } 
    case ifnull: 
    { 
      eprintf( pszHex2Byte, pbInstr[1] * 256 + pbInstr[2]); 
      iNumBytes = 3; 
      break; 
    } 
    case ifnonnull: 
    { 
     eprintf( pszHex2Byte, pbInstr[1] * 256 + pbInstr[2]); 
      iNumBytes = 3; 
      break; 
    } 
    #ifdef QUICK 
    case ldc1_quick: 
    { 
      eprintf( pszHex1Byte, pbInstr[1]); 
      iNumBytes = 2; 
      break; 
    } 
    case ldc2_quick: 
    { 
      eprintf( pszHex2Byte, pbInstr[1] * 256 + pbInstr[2]); 
      iNumBytes = 3; 
      break; 
    } 
    case ldc2w_quick: 
    { 
      eprintf( pszHex2Byte, pbInstr[1] * 256 + pbInstr[2]); 
      iNumBytes = 3; 
      break; 
    } 
    case getfield_quick: 
    { 
      eprintf( pszHex1Byte, pbInstr[1]); 
      iNumBytes = 3; 
      break; 
    } 
    case putfield_quick: 
    { 
      eprintf( pszHex1Byte, pbInstr[1]); 
      iNumBytes = 3; 
      break; 
    } 
    case getfield2_quick: 
    { 
      eprintf( pszHex1Byte, pbInstr[1]); 
      iNumBytes = 3; 
      break; 
    } 
    case putfield2_quick: 
    { 
      eprintf( pszHex1Byte, pbInstr[1]); 
      iNumBytes = 3; 
      break; 
    } 
    #ifdef QUICK2 
    case getstatic_quick: 
    { 
      eprintf( pszHex1Byte, pbInstr[1]); 
      eprintf( pszHex1Byte, pbInstr[2]); 
      iNumBytes = 3; 
      break; 
    } 
    case putstatic_quick: 
    { 
      eprintf( pszHex1Byte, pbInstr[1]); 
      eprintf( pszHex1Byte, pbInstr[2]); 
      iNumBytes = 3; 
      break; 
    } 
    case getstatic2_quick: 
    { 
      eprintf( pszHex1Byte, pbInstr[1]); 
      eprintf( pszHex1Byte, pbInstr[2]); 
      iNumBytes = 3; 
      break; 
    } 
    case putstatic2_quick: 
    { 
      eprintf( pszHex1Byte, pbInstr[1]); 
      eprintf( pszHex1Byte, pbInstr[2]); 
      iNumBytes = 3; 
      break; 
    } 
    #else 
    case getstatic_quick: 
    { 
      eprintf( pszHex2Byte, pbInstr[1] * 256 + pbInstr[2]); 
      iNumBytes = 3; 
      break; 
    } 
    case putstatic_quick: 
    { 
      eprintf( pszHex2Byte, pbInstr[1] * 256 + pbInstr[2]); 
      iNumBytes = 3; 
      break; 
    } 
    case getstatic2_quick: 
    { 
      eprintf( pszHex2Byte, pbInstr[1] * 256 + pbInstr[2]); 
      iNumBytes = 3; 
      break; 
    } 
    case putstatic2_quick: 
    { 
      eprintf(pszHex2Byte, pbInstr[1] * 256 + pbInstr[2]); 
      iNumBytes = 3; 
      break; 
    } 
    case putstaticcurrclass_quick: 
    { 
      eprintf( pszHex1Byte, pbInstr[1]); 
      iNumBytes = 3; 
      break; 
    } 
    case putstatic2currclass_quick: 
    { 
      eprintf( pszHex1Byte, pbInstr[1]); 
      iNumBytes = 3; 
      break; 
    } 
    case getstaticcurrclass_quick: 
    { 
      eprintf( pszHex1Byte, pbInstr[1]); 
      iNumBytes = 3; 
      break; 
    } 
    case getstatic2currclass_quick: 
    { 
      eprintf( pszHex1Byte, pbInstr[1]); 
      iNumBytes = 3; 
      break; 
    } 
    #endif 
    case invokevirtual_quick: 
    { 
      eprintf(pszHex1Byte, pbInstr[1]); 
      eprintf(pszHex1Byte, pbInstr[2]); 
      iNumBytes = 3; 
      break; 
    } 
    #ifdef QUICK2 
    case invokespecial_quick: 
    { 
      eprintf( pszHex1Byte, pbInstr[1]); 
      eprintf(pszHex1Byte, pbInstr[2]); 
      iNumBytes = 3; 
      break; 
    } 
    case invokestatic_quick: 
    { 
      eprintf( pszHex1Byte, pbInstr[1]); 
      eprintf( pszHex1Byte, pbInstr[2]); 
      iNumBytes = 3; 
      break; 
    } 
    #else 
    case invokespecial_quick: 
    { 
      eprintf( pszHex2Byte, pbInstr[1] * 256 + pbInstr[2]); 
      iNumBytes = 3; 
      break; 
    } 
    case invokestatic_quick: 
    { 
      eprintf( pszHex2Byte, pbInstr[1] * 256 + pbInstr[2]); 
      iNumBytes = 3; 
      break; 
    } 
    case invokespecialcurrclass_quick: 
    { 
      eprintf( pszHex1Byte, pbInstr[1]); 
      iNumBytes = 3; 
      break; 
    } 
    case invokestaticcurrclass_quick: 
    { 
      eprintf( pszHex1Byte, pbInstr[1]); 
      iNumBytes = 3; 
      break; 
    } 
    #endif 
    case invokeinterface_quick: 
    { 
     eprintf( pszHex2Byte, pbInstr[1] * 256 + pbInstr[2]); 
      iNumBytes = 3; 
      break; 
    } 
    case invokeinterface_noguess: 
    { 
      eprintf( pszHex2Byte, pbInstr[1] * 256 + pbInstr[2]); 
      iNumBytes = 3; 
      break; 
    } 
/*    case invokevirtualobject_quick: 
    { 
      break; 
    }*/ 
    case new_quick: 
    { 
      eprintf( pszHex2Byte, pbInstr[1] * 256 + pbInstr[2]); 
      iNumBytes = 3; 
      break; 
    } 
    case anewarray_quick: 
    { 
      eprintf(pszHex2Byte, pbInstr[1] * 256 + pbInstr[2]); 
      iNumBytes = 3; 
      break; 
    } 
    case checkcast_quick: 
    { 
      eprintf(pszHex2Byte, pbInstr[1] * 256 + pbInstr[2]); 
      iNumBytes = 3; 
      break; 
    } 
    case instanceof_quick: 
    { 
      eprintf(pszHex2Byte, pbInstr[1] * 256 + pbInstr[2]); 
      iNumBytes = 3; 
      break; 
    } 
    #endif 
    default: 
    { 
      eprintf("Unknown assembler instruction: opcode %d\n", pbInstr[0]); 
    } 
  } 
 
  eprintf("\n"); 
  return iNumBytes; 
} 
 
/* end of disass.c */ 
