/*
 * Copyright (C) 2001, John Leuner.
 *
 * This file is part of the kissme/teaseme project, which in turn is part
 * of the JOS project.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2,
 * or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "config.h"


/*
 * @doc MODULE
 * @module interp.c |
 *
 * This module has functions for calling the interpreter from native methods.
 * The main() function is  also defined in this file.
 *
 */


#define DEBUG_TRACE 

#include "vm/wrappers.h"

#ifdef KISSME_LINUX_USER
#include <pthread.h>
#include <stdio.h>
#endif

#include <stdarg.h>

#include <math.h>
#include <string.h>

#include "vm/uid.h"
#include "vm/interp.h"

#include "vm/classfil.h"

#include "vm/jni.h"

#include "vm/opcodes.h"
#include "vm/stdtypes.h"
#include "vm/disass.h"

#include "vm/interp_methods.h"
#include "vm/classfile_methods.h"
#include "vm/cptuplelist.h"
#include "vm/jutils.h"

#include "vm/threadex.h"

#include "lib/indigenous/java.lang/Class.h"
#include "vm/garbage2.h"
#include "vm/newobject.h" 
#include "vm/interp/methodstacks.h"

#include "vm/kni.h"
#include "vm/kni_methods.h"

#include "vm/threads_native.h"
#include "vm/llist.h"

#include "vm/garbage.h"

#include "vm/global.h"
#include "vm/hash.h"

#include "vm/initialize.h"

#ifdef PERSIST
#include "vm/rpot.h"
#endif


/* global variables ********************************************************/

extern HASH_tstHashTable UID_sstUidHashTable; 

extern tAllocatorHeap* system_heap;

#ifdef DEBUG_TRACE
extern int INTERP_DebugOn;
#endif

extern tOBREF OOMExceptionObject;


//This can be set by the ProcessArgs routine 
int INTERP_InitialStackSize = 2048; 
int canStartOptimising = 0; //For the run-time-optimiser



/* 
 * Constructs a thread object with the name "main" 
 *
 * For use in the kissme main() method.
 */

tOBREF INTERP_ConstructMainThread(JNIEnv* env)
{
  tOBREF hMainThread;
  jmethodID methodID;
  jclass thread_cls = (*env)->FindClass(env, "java/lang/Thread");  

  assert(thread_cls);

  methodID = (*env)->GetMethodID(env, thread_cls, "<init>", 
				 "(Ljava/lang/String;)");
  assert(methodID);

  traceStartup0("Creating the main Thread object");
  hMainThread = (*env)->NewObject(env, thread_cls, methodID, 
				  INTERP_NewStringFromAsciz(env, "main"));
  traceStartup("Main Thread object is %p", hMainThread);
  if ((*env)->ExceptionOccurred(env) == NULL) {
    return hMainThread;
  }    
  traceStartup("Exception %s occurred while creating main Thread",
	       DEREF((*env)->ExceptionOccurred(env))->pstType->uidName);
  return NULL;
}


/*
 * @doc FUNC
 * @func
 * This function is used to check to see if one class can be cast to another
 * class.  This will be true pstClassCastTo is a superclass or interface class
 * of pstClassCastFrom.
 *
 * @rdesc Returns one of the following:
 *
 * @flag 0 | If the cast failed
 * @flag 1 | If the cast was successful
 *
 */
int INTERP_CheckCast(JNIEnv* env,
		     tClassLoaderTuple* pstClassCastTo,  
		     /* @parm The target class to try to cast to */
		     tClassLoaderTuple* pstClassCastFrom 
		     /* @parm The class that must be cast */)
{
  uint16 u16Temp;
  tClassLoaderTuple *pstClassTemp;
  
  if (pstClassCastFrom == pstClassCastTo) {
    return 1;
  }

  if (strcmp(pstClassCastFrom->uidName, pstClassCastTo->uidName) == 0 &&
      pstClassCastFrom->classLoader == pstClassCastTo->classLoader) {
    // XXX - restructure
    assert(pstClassCastFrom == pstClassCastTo);
  }
  
  /* i.e. if it's a cast to an interface */
  /* check if this class supports that interface */
    
  if (pstClassCastTo->pstClass->u16AccessFlags & ACC_INTERFACE) {
    if (pstClassCastFrom->uidName[0] == '[') {
      if (strcmp(pstClassCastTo->uidName, "java/lang/Cloneable") == 0 ||
	  strcmp(pstClassCastTo->uidName, "java/io/Serializable") == 0) {
	return 1;
      }
      // XXX - return 0 here?
    }
    if (pstClassCastFrom->pstClass->u16InterfacesCount == 0) {
      /* if this is an interface and supports no further interfaces
	 .. no go */
      if (pstClassCastFrom->pstClass->u16AccessFlags & ACC_INTERFACE) {
	return 0;
      }
      /* drop through to pick up superclass */
    }
    else {
      
      for (u16Temp = 0; 
	   u16Temp < pstClassCastFrom->pstClass->u16InterfacesCount;
	   u16Temp++) {
	pstClassTemp = pstClassCastFrom->pstClass->ppstInterfaces[u16Temp];
#ifdef DEBUG
	assert(pstClassTemp);
#endif
	if (INTERP_CheckCast(env, pstClassCastTo, pstClassTemp) == 1) {
	  return 1;
	}
      }
	
      /* if we didn't return, and we have an object, we must check
	 it's superclass, if we have an interface, we have already
	 checked the interfaces it supports, and turned up empty */
      if (pstClassCastFrom->pstClass->u16AccessFlags & ACC_INTERFACE) {
	return 0;
      }
      /* else drop through to superclass */
    }
  }
    
  // What about arrays?
  if (pstClassCastFrom->uidName[0] == '[') {
    if (pstClassCastTo->uidName[0] == '[') {
      // XXX - can we really assume that a class name has < 256 chars?
      char bufa[256];
      char bufb[256];
	
      int bothLobjects = 0;
      int oneArray = 0;
	
      // XXX - simplify
      if (!((pstClassCastFrom->uidName[1] == '[') && 
	    (pstClassCastTo->uidName[1] == '['))) {
	if (pstClassCastFrom->uidName[1] == '[' ||
	    pstClassCastTo->uidName[1] == '[') {
	  oneArray = 1;
	}
      }
	
      if ((pstClassCastFrom->uidName[1] == 'L') && 
	  (pstClassCastTo->uidName[1] == 'L')) {
	bothLobjects = 1;
      }
	
      if (!bothLobjects && !oneArray && 
	  pstClassCastFrom->uidName[1] != 'L'
	  && pstClassCastFrom->uidName[1] != '[') {
	// if they are arrays of primitives, compare the primitive types
	return strcmp(pstClassCastFrom->uidName, pstClassCastTo->uidName) == 0;
      } //XXX check this brace
	
      // if one is primitive and the other reference
      if ((pstClassCastFrom->uidName[1] == 'L') && 
	  (pstClassCastTo->uidName[1] != 'L') && !oneArray) {
	return 0;
      }
      // if one is primitive and the other reference (other case)
      if ((pstClassCastFrom->uidName[1] != 'L') && 
	  (pstClassCastTo->uidName[1] == 'L') && !oneArray) {
	return 0;
      }
	
      // only case left is that both are reference arrays, or one is
      // an object
      if (pstClassCastFrom->uidName[1] == '[') {
	strcpy(bufa, pstClassCastFrom->uidName + 1);
      }
      else {
	strcpy(bufa, pstClassCastFrom->uidName + 2);
	  
	if (bufa[strlen(bufa) - 1] == ';') {
	  bufa[strlen(bufa) - 1] = 0;
	}
      }
	
      if (pstClassCastTo->uidName[1] == '[') {
	strcpy(bufb, pstClassCastTo->uidName + 1);
      } 		      
      else {
	strcpy(bufb, pstClassCastTo->uidName + 2);
	  
	if (bufb[strlen(bufb) - 1] == ';') {
	  bufb[strlen(bufb) - 1] = 0;
	}
      }
	
      return INTERP_CheckCast(env,
			      CLASSFILE_FindOrLoad(env, bufb, 
						   pstClassCastFrom), 
			      CLASSFILE_FindOrLoad(env, bufa, 
						   pstClassCastFrom)); //check this tuple XXX
    }
    else {
      if (strcmp(pstClassCastTo->uidName, "java/lang/Object") == 0) {
	return 1; //an array is an object
      }
    }
    return 0;
  }
    
  /* get superclass */
  if (pstClassCastFrom->pstClass->pstSuperClass == NULL) {
    return 0;
  }
    
  pstClassCastFrom = pstClassCastFrom->pstClass->pstSuperClass;
#ifdef DEBUG
  assert(pstClassCastFrom);
#endif
    
  return INTERP_CheckCast(env, pstClassCastTo, pstClassCastFrom);
}


tOBREF INTERP_ExceptionObjectFromNameAndMessage(JNIEnv* env,  
						char* pszExName,           
						/* @parm Name of exception */
						char* message              
						/* @parm Message for the 
						   exception */)
{
  tClassLoaderTuple* pstExClass;
  tOBREF exceptionObject;
  int failure;

  //check this tuple XXX
  pstExClass = CLASSFILE_FindOrLoad(env, pszExName, NULL);
  
  // this method should only called from JVM code, so we don't need to
  // fail gracefully
  assert(pstExClass); 
  
  exceptionObject = INTERP_NewObject(env, pstExClass, &failure);
  if(failure)
    return exceptionObject;
  
  if (message == NULL) {
    // Construct the throwable object 
    tMethod* method = CLASSFILE_FindMethod(env, pstExClass, "<init>", "()V");
    if (method) {
      int32 args[1];
      args[0] = (int32) exceptionObject;
      INTERP_RunSpecialMethodFromPtr(env, method, args);
    }
    else {
      panic("Failed to find noarg constructor for exception %s", pszExName);
    }
  }
  else {
    // Construct the throwable object 
    tMethod* method = CLASSFILE_FindMethod(env, pstExClass, "<init>", 
					   "(Ljava/lang/String;)V");
    int32 args[2];
    args[0] = (int32) exceptionObject;
    args[1] = (int32) INTERP_NewStringFromAsciz(env, message);
    if (method) {
      INTERP_RunSpecialMethodFromPtr(env, method, args);
    }
    else {
      panic("Failed to find (Ljava/lang/String;) constructor for exception %s", 
	    pszExName);
    }
  }
  return exceptionObject;
}


tStackFrame* ThrowNamedException(JNIEnv* env,
				 tStackFrame* pstOldFrame, 
				 /* @parm Environment stack frame to use */
				 char* pszExName           
				 /* @parm Name of exception to throw */)
{
  return ThrowNamedExceptionWithMessage(env, pstOldFrame, pszExName, NULL);
}


/*
 * Throw an OutOfMemory exception is thrown using a preallocated
 * OutOfMemoryException object with no stack trace information.  This
 * avoids further OOM due to allocating the object and the stack trace.
 */
tStackFrame* ThrowOutOfMemory(JNIEnv* env,
			       tStackFrame* pstOldFrame 
			       /* @parm Environment stack frame to use */)
{
  assert(OOMExceptionObject);
  *(++(pstOldFrame->pi32OpTop)) = (int32) OOMExceptionObject;
  return CatchException(env, pstOldFrame);
}


tStackFrame* ThrowNullPointer(JNIEnv* env, tStackFrame* pstOldFrame)
{
  return ThrowNamedException(env, pstOldFrame,
			     "java/lang/NullPointerException");
}


tStackFrame* ThrowClassNotFound(JNIEnv* env, tStackFrame* pstOldFrame)
{
  return ThrowNamedException(env, pstOldFrame, 
			     "java/lang/ClassNotFoundException");
}


tStackFrame* ThrowNegativeArraySize(JNIEnv* env, tStackFrame* pstOldFrame)
{
  return ThrowNamedException(env, pstOldFrame, 
			     "java/lang/NegativeArraySizeException");
}


tStackFrame* ThrowArrayIndex(JNIEnv* env, tStackFrame* pstOldFrame, int32 index)
{
  char buf[20];
  sprintInt(buf, index);
  return ThrowNamedExceptionWithMessage(env, pstOldFrame, 
					"java/lang/ArrayIndexOutOfBoundsException", 
					buf);
}


tStackFrame* ThrowClassCast(JNIEnv* env, tStackFrame* pstOldFrame,
			    char *fromName, char *toName)
{
  tStackFrame* pstNewFrame;
  char* errMessage;

  errMessage = sys_malloc(100 + strlen(fromName) + strlen(toName));
  if (errMessage == NULL) { 
    return CatchExternalException(env, pstOldFrame, OOMExceptionObject); 
  }
  sprintf(errMessage, "Could not cast object to type %s from type %s", 
	  toName, fromName);
  pstNewFrame = ThrowNamedExceptionWithMessage(env, pstOldFrame, 
					       "java/lang/ClassCastException",
					       errMessage);
  sys_free(errMessage);
  return pstNewFrame;
}


tStackFrame* ThrowArithmetic(JNIEnv* env, tStackFrame* pstOldFrame)
{
  return ThrowNamedException(env, pstOldFrame, "java/lang/ArithmeticException");
}


tStackFrame* ThrowRuntimeException(JNIEnv* env, tStackFrame* pstOldFrame)
{

}


tStackFrame* ThrowInternalError(JNIEnv* env, tStackFrame* pstOldFrame,
				char *message)
{
  return ThrowNamedExceptionWithMessage(env, pstOldFrame, 
					"java/lang/InternalError", message);
}


tStackFrame* ThrowNoClassDefError(JNIEnv* env, tStackFrame* pstOldFrame,
				  char *message)
{
  return ThrowNamedExceptionWithMessage(env, pstOldFrame, 
					"java/lang/NoClassDefFoundError",
					message);
}


tStackFrame* ThrowNoSuchMethodError(JNIEnv* env, tStackFrame* pstOldFrame,
				    char *message)
{
  return ThrowNamedExceptionWithMessage(env, pstOldFrame, 
					"java/lang/NoSuchMethodError",
					message);
}

tStackFrame* ThrowNoSuchFieldError(JNIEnv* env, tStackFrame* pstOldFrame,
				   char *message)
{
  return ThrowNamedExceptionWithMessage(env, pstOldFrame, 
					"java/lang/NoSuchFieldError",
					message);
}


tStackFrame* ThrowClassChangeError(JNIEnv* env, tStackFrame* pstOldFrame,
				   char *format, ...)
{
  va_list ap;
  char buffer[256];
  tStackFrame* pstNewFrame;
  
  va_start(ap, format);
  vsnprintf(buffer, 255, format, ap);
  pstNewFrame = 
    ThrowNamedExceptionWithMessage(env, pstOldFrame, 
				   "java/lang/IncompatibleClassChangeError",
				   buffer);
  va_end(ap);
  return pstNewFrame;
}


/*
 * @doc FUNC
 * @func
 * This function is called to throw a particular exception.  The name of the
 * exception is passed.  This function pushes it onto the operand stack of
 * the current environment frame and then calls CatchException() to catch
 * the exception.  Always pass the current environment stack frame to it. The
 * environment registers should be saved before calling this function and
 * restored after calling it.
 *
 * @rdesc Returns the new environment stack frame to use after the exception
 *        has been caught.
 *
 * @ex Example of throwing a specific exception.  pstCurrFrame is a pointer
 *     to the current environment frame.  In this example it is overwritten
 *     by the new frame that is returned. |
 *
 * SAVEENV();
 * pstCurrFrame = ThrowNamedException(pstCurrFrame, "java.lang.Exception");
 * LOADENV();
 *
 */
tStackFrame* ThrowNamedExceptionWithMessage(JNIEnv* env,
					    tStackFrame* pstOldFrame,
					    /* @parm Environment stack frame
					       to use */
					    char* pszExName,          
					    /* @parm Name of exc to throw */
					    char* message              
					    /* @parm Message for the exc */)
{
  tOBREF exceptionObject = 
    INTERP_ExceptionObjectFromNameAndMessage(env, pszExName, message);
  *(++(pstOldFrame->pi32OpTop)) = (int32) exceptionObject;
  return CatchException(env, pstOldFrame);
}

#ifdef OLD_PRINTSTACKTRACE

/*
 * This method is called when CatchException fails to find a handler.
 *
 * It prints out the exception type, followed by it's 'message'.
 * Then it prints out the stack trace from a tList structure
 *
 * The arguments are the list, and a pointer to the exception object
 *
 */
static void printStackTrace(JNIEnv* env, 
			    char** stackTraceList, tOBREF exceptionObject)
{
  tClassLoaderTuple* pstExceptionClass;
  tField* pstField;
  
  if (exceptionObject == NULL) {
    panic0("printStackTrace: NULL exception");
  }

  pstExceptionClass = ODEREF(exceptionObject)->pstType;
  pstField = CLASSFILE_InstFieldOffset(env, pstExceptionClass, "message");
  
  if (pstField == NULL) {
    panic("printStackTrace: no 'message' field for exception (%p) %s",
	  exceptionObject, pstExceptionClass->uidName);
  }
  else {
    char* str;
    if ((ODEREF(exceptionObject)->pi32Vars[pstField->u16Offset]) == 0) {
      eprintf("%s with NULL message\n", pstExceptionClass->uidName);
    }
    else {
      tOBREF o = (tOBREF) 
	(ODEREF(exceptionObject)->pi32Vars[pstField->u16Offset]);
      str = INTERP_AscizFromString(env, o);
      eprintf("%s: %s\n", pstExceptionClass->uidName, str);
      sys_free(str);
    }
  }
  
  {
    int entryIndex = 0;
    while (stackTraceList[entryIndex]) {
      eprintf("- %s", stackTraceList[entryIndex++]);
    }
  }  
}

#else

/*
 * Prints the stack trace by calling the exception object's 
 * printStackTrace() method
 */
void INTERP_printStackTrace(JNIEnv* env, tOBREF exceptionObject)
{
  jclass clazz;
  jmethodID mid;

  if (exceptionObject == NULL) {
    panic0("INTERP_printStackTrace: NULL exception");
  }
  else {
    clazz = (*env)->GetObjectClass(env, exceptionObject);
    mid = (*env)->GetMethodID(env, clazz, "printStackTrace", "()V");
    if (mid == NULL) {
      tClassLoaderTuple* tuple;
      tuple = CLASS_GetClassStruct(env, clazz);
      panic("INTERP_printStackTrace: no 'void printStackTrace()' method "
	    "for exception (%p) %s", exceptionObject, tuple->uidName);
    }
    else {
      (*env)->CallVoidMethod(env, exceptionObject, mid);
    }
  }
}

#endif

/*
 * @doc FUNC
 * @func
 * This function is called to catch an exception.  This exception is expected
 * to be on the operand stack of the current environment frame.  If a matching
 * catch clause is found, then a pointer to the stack frame in which it was
 * found is returned.  Always pass the current environment stack frame to it.
 * The environment registers should be saved before calling this function and
 * restored after calling it.
 *
 * @rdesc Returns the new environment stack frame to use after the exception
 *        has been caught.
 *
 * @ex Example of throwing an exception that is on the operand stack.
 *     pstCurrFrame is a pointer to the current environment frame.  In this
 *     example it is overwritten by the new frame that is returned. |
 *
 * SAVEENV();
 * pstCurrFrame = CatchException(pstCurrFrame);
 * LOADENV();
 *
 */
tStackFrame* CatchException(JNIEnv* env, 
			    tStackFrame* pstOldFrame  
			    /* @parm Environment stack frame to use */)
{
  return CatchExceptionHelper(env, pstOldFrame, 
			      (tOBREF) *(pstOldFrame->pi32OpTop));
}


tStackFrame* CatchExternalException(JNIEnv* env, 
				    tStackFrame* pstOldFrame  
				    /* @parm Environment stack frame to use */,
				    tOBREF pstExOb)
{
  return CatchExceptionHelper(env, pstOldFrame, pstExOb);
}


int INTERP_FigureOutLineNumber(tStackFrame* pstFrame)
{
  if (pstFrame->pstCurrMethod->u16AccessFlags & ACC_NATIVE) {
    return -1;
  }
  else {
    int instruction = 
      pstFrame->pbPC - pstFrame->pstCurrMethod->pstCode->pbCode;
    int i;
    int retval = -1;
    
    tLineNumberTable* table = 
      pstFrame->pstCurrMethod->pstCode->pstLineNumberTable;
    if (table == NULL) {
      return retval;
    }
      
    for (i = table->u16LineNumberTableLength - 1; i >= 0; i--) {
      if ( instruction >= table->pstLineNumberTable[i].u16StartPC) {
	retval = table->pstLineNumberTable[i].u16LineNumber;
	break;
      }
    }
    return retval;
  }
}


/* potential bug:
 *
 * The pstOldFrame is accessed to get the exception object, even
 * thought that frame has already been freed.
 *
 * If we zero frames when popping them, this would make the exception
 * object NULL
 */
tStackFrame* CatchExceptionHelper(JNIEnv* env,
				  tStackFrame* pstOldFrame,  
				  /* @parm Environment stack frame to use */
				  tOBREF pstExOb)
{
  tStackFrame* pstCurrFrame = pstOldFrame;
  tClassLoaderTuple* pstEx;
  tClassLoaderTuple* pstCurrEx;
  tCode* pstCurrCode;
  uint32 u32CurrPC;
  uint16 u16Num;
  //  static Uid uidThreadDeath = NULL;
  int iFinished = 0;
#ifdef OLD_PRINTSTACKTRACE
  char** stackTraceList = sys_malloc(sizeof(char*) * 150);
  int stackEntries = 0;

  memset(stackTraceList, 0, sizeof(char*) * 150);
#endif

  if (pstExOb == NULL) {
    panic0("CatchExceptionHelper: NULL exception object");
  }
  pstEx = DEREF(pstExOb)->pstType;

  traceExceptions("Looking for exception handler for %s", pstEx->uidName);
  
  /* keep going up the call stack until we reach the top */
  while (!iFinished)
  {

#ifdef OLD_PRINTSTACKTRACE
    int lineNumber;
    char* stackTraceEntry = malloc(512);
    
    lineNumber = INTERP_FigureOutLineNumber(pstCurrFrame);
    if (lineNumber == -1)
      snprintf(stackTraceEntry, 512, 
	       "        at %s.%s:(no line numbers)\n",
	       pstCurrFrame->pstCurrMethod->pstClass->uidName, 
	       pstCurrFrame->pstCurrMethod->uidName);
    else
      snprintf(stackTraceEntry, 512, 
	       "        at %s.%s:%i\n", 
	       pstCurrFrame->pstCurrMethod->pstClass->uidName, 
	       pstCurrFrame->pstCurrMethod->uidName, lineNumber);
    if (stackEntries < 150) {
      stackTraceList[stackEntries++] = stackTraceEntry;
    }
    else {
      eprintf("Limiting stack trace to 150 entries\n");
    }
#endif
    
    /* find numeric value of PC in this stack frame */
    
    u32CurrPC = pstCurrFrame->pbPC - pstCurrFrame->pbCode;

    /* go through the class's exception table */
    pstCurrCode = pstCurrFrame->pstCurrMethod->pstCode;
    for (u16Num = 0; 
	 u16Num < pstCurrCode->u16ExceptionTableLength; 
	 u16Num++) {
      /* check to see if the currect exception catch range contains
         the current PC */
      if (pstCurrCode->pstExceptionTable[u16Num].u16EndPC >= u32CurrPC &&
          pstCurrCode->pstExceptionTable[u16Num].u16StartPC < u32CurrPC) {
        /* if the catch type is undefined then it will catch anything */
        if (pstCurrCode->pstExceptionTable[u16Num].pstCatchType == NULL) {
	  /* caught it! */
          /* set the PC in the frame we're in to the handler PC for
             the exception we've caught */
          pstCurrFrame->pbPC = pstCurrFrame->pbCode + 
	    pstCurrCode->pstExceptionTable[u16Num].u16HandlerPC;
	  
          /* put the exception on the stack of the new frame */
          if (pstCurrFrame != pstOldFrame) {
            *(++(pstCurrFrame->pi32OpTop)) = 
	      (int32) ((tOBREF) *(pstOldFrame->pi32OpTop));
          }
	  
#ifdef OLD_PRINTSTACKTRACE
	  while (stackEntries > -1) {
	    free(stackTraceList[stackEntries--]);
	  }
	  free(stackTraceList);
#endif
	  
          /* return new frame pointer */
          return pstCurrFrame;
        }

        /* check if the thrown object is the same as or is a child of
           the type currently being considered */
        for (pstCurrEx = pstEx; 
	     pstCurrEx; 
	     pstCurrEx = pstCurrEx->pstClass->pstSuperClass) {
          /* compare */
          if (pstCurrEx ==
	      pstCurrCode->pstExceptionTable[u16Num].pstCatchType) {
            /* caught it! */
            /* set the PC in the frame we're in to the handler PC for
               the exception we've caught */
            pstCurrFrame->pbPC = pstCurrFrame->pbCode +
	      pstCurrCode->pstExceptionTable[u16Num].u16HandlerPC;
	    
            /* put the exception on the stack of the new frame */
            if (pstCurrFrame != pstOldFrame) {
              *(++(pstCurrFrame->pi32OpTop)) = 
		(int32) ((tOBREF) *(pstOldFrame->pi32OpTop));
            }
	    
#ifdef OLD_PRINTSTACKTRACE
	    while( stackEntries > -1)  {
	      free(stackTraceList[ stackEntries --]);
	    }
	    free(stackTraceList);
#endif
	    
	    //clear the exception 
	    (*env)->ExceptionClear(env);
            
	    /* return new frame pointer */
            return pstCurrFrame;
          }
        }
      }

    } //for each method

    /* move to previous stack frame if not the last one */
    if (pstCurrFrame->pstPrevFrame && (!pstCurrFrame->bIsOrphan)) {
      pstCurrFrame = PopStackFrame(pstCurrFrame);
    }
    else {
      iFinished = 1;
    }
  }

  traceExceptions("Dealing with an uncaught exception (%s)", pstEx->uidName);

  /* 
   * if we get here we've reached the top of the call stack without
   * success.  if this is an orphan method (it was called via JNI), we
   * don't print out the stack trace, but return the exception
   *
   * XXX - should we print a stack trace for threads other than the
   * main thread?
   */
  pstCurrFrame->pbPC = pstCurrCode->pbCode + pstCurrCode->u32CodeLength - 1;

  if (pstCurrFrame->bIsOrphan == 0) {
    pstCurrFrame->pstException = NULL;
    // Print stack trace 
#ifdef OLD_PRINTSTACKTRACE
    printStackTrace(env, stackTraceList, pstExOb);
#else
    INTERP_printStackTrace(env, pstExOb);
#endif
  }

#ifdef OLD_PRINTSTACKTRACE
  while (stackEntries > -1) {
    free(stackTraceList[ stackEntries --]);
  }
  free(stackTraceList);
#endif

  if (pstCurrFrame->bIsOrphan == 0) {
    pstCurrFrame->pstException = NULL; // Just let the (main) thread stop
  }
  else {
    pstCurrFrame->pstException = pstExOb;
  }
  return pstCurrFrame;
}


/*
 * @doc FUNC
 * @func
 * This function runs a non-virtual method from a tClass pointer and a tMethod
 * pointer.  This function is used for running non-virtual Java methods from
 * native methods.  INTERP_RunNonVirtualMethod() can be used if the method
 * needs to be specified by name.
 *
 * @ex Example showing invocation of a non-virtual method from native method.
 *     In the example, pi32Args is assumed to point to the top of the operand
 *     stack with the parameters on it. |
 *
 * tClassLoaderTuple*  pstClass;
 * tMethod* pstMethod;
 *
 * pstClass = CPLIST_Find("MyClass");
 * pstMethod = CLASSFILE_FindMethod(pstClass, "MyMethod", "(I)I");
 * INTERP_RunNonVirtualMethodFromPtr(pstMethod, pi32Args);
 *
 */
tOBREF INTERP_RunSpecialMethodFromPtr(JNIEnv* env,
				      tMethod* pstTheMethod,   
				      /* @parm Pointer to Java method to
					 execute */
				      int32*   pi32Args        
				      /* @parm Pointer to arguments to use */)
{
  tStackFrame* pstCurrFrame;
  tOBREF pstExOb;
  KNIApi *kniPtr = KNI_getKNIApiPtr();

  /* invoke method... */
  pstCurrFrame = InitialStackFrameHelper(pstTheMethod, (tOBREF) *pi32Args, 
					 pi32Args, kniPtr->GetHeap(env), 1);
  
  THREADINFO_addInitialStackFrameToNode(THREAD_FindThread(), pstCurrFrame);
  pstExOb = Interpret(env, pstCurrFrame, 1);
  THREADINFO_removeInitialStackFrameFromNode(THREAD_FindThread(), 
					     pstCurrFrame);
  if (pstExOb) {
    PopStackFrame(pstCurrFrame);
    return pstExOb;
  }
  
  if (pstTheMethod->u16RetSize == 1) {
    *pi32Args = (int32) *pstCurrFrame->pi32OpTop;
  }
  else if (pstTheMethod->u16RetSize == 2) {
    *pi32Args = (int32) (*pstCurrFrame->pi32OpTop - 1); /* high */
    *(pi32Args + 1) = (int32) *pstCurrFrame->pi32OpTop; /* low */
  }
  
  PopStackFrame(pstCurrFrame);
  
  return NULL;
}


#ifdef not_used
/*
 * @doc FUNC
 * @func
 * This function runs a non-virtual method from a tClass pointer and a tMethod
 * pointer.  This function is used for running non-virtual Java methods from
 * native methods.  INTERP_RunNonVirtualMethod() can be used if the method
 * needs to be specified by name.
 *
 * This variation uses a native call stack (reversed)

 * @ex Example showing invocation of a non-virtual method from native method.
 *     In the example, pi32Args is assumed to point to the top of the operand
 *     stack with the parameters on it. |
 *
 * tClassLoaderTuple*  pstClass;
 * tMethod* pstMethod;
 *
 * pstClass = CPLIST_Find("MyClass");
 * pstMethod = CLASSFILE_FindMethod(pstClass, "MyMethod", "(I)I");
 * INTERP_RunNonVirtualMethodFromPtr(pstMethod, pi32Args);
 *
 */
tOBREF INTERP_RunSpecialMethodFromPtr_with_native_stack(JNIEnv* env,
							tMethod* pstTheMethod,
							int32*   pi32Args)
{
  tStackFrame* pstCurrFrame;
  KNIApi *kniPtr = KNI_getKNIApiPtr();
  int i;
  tOBREF pstExOb;
  
  /* Ok now we need to flip the arguments around, but this doesn't
     work for longs and doubles -- shit */
  for (i = 0; i < (pstTheMethod->u16ArgSize / 2); i++) {
    int temp = pi32Args[i];
    pi32Args[i] = pi32Args[ pstTheMethod->u16ArgSize - 1 - i];
    pi32Args[ pstTheMethod->u16ArgSize - 1 - i] = temp;
  }
  
  /* invoke method... */
  pstCurrFrame = InitialStackFrameHelper(pstTheMethod, (tOBREF) *pi32Args, 
					 pi32Args, kniPtr->GetHeap(env), 1);
  
  assert(THREADINFO_addInitialStackFrameToNode(THREAD_FindThread(),
					       pstCurrFrame) == 0);
  pstExOb = Interpret(env, pstCurrFrame, 1);
  assert(THREADINFO_removeInitialStackFrameFromNode(THREAD_FindThread(), 
						    pstCurrFrame) == 0);
  
  if (pstExOb) {
    PopStackFrame(pstCurrFrame);
    return pstExOb;
  }
  
  if (pstTheMethod->u16RetSize == 1) {
    *pi32Args = (int32) *pstCurrFrame->pi32OpTop;
  }
  else if (pstTheMethod->u16RetSize == 2)  {
    *pi32Args = (int32) (*pstCurrFrame->pi32OpTop - 1); /* high */
    *(pi32Args + 1) = (int32) *pstCurrFrame->pi32OpTop;   /* low */
  }
  
  PopStackFrame(pstCurrFrame);
  
  return NULL;
}
#endif


/*
 * @doc FUNC
 * @func
 * This function runs a non-virtual method from a tClass pointer and a tMethod
 * pointer.  This function is used for running non-virtual Java methods from
 * native methods.  INTERP_RunNonVirtualMethod() can be used if the method
 * needs to be specified by name.
 *
 * @ex Example showing invocation of a non-virtual method from native method.
 *     In the example, pi32Args is assumed to point to the top of the operand
 *     stack with the parameters on it. |
 *
 * tClassLoaderTuple*  pstClass;
 * tMethod* pstMethod;
 *
 * pstClass = CPLIST_Find("MyClass");
 * pstMethod = CLASSFILE_FindMethod(pstClass, "MyMethod", "(I)I");
 * INTERP_RunStaticMethodFromPtr(pstMethod, pi32Args);
 *
 */
tOBREF INTERP_RunStaticMethodFromPtr(JNIEnv* env,
				     tMethod* pstTheMethod,   
				     int32* pi32Args)
{
  KNIApi *kniPtr = KNI_getKNIApiPtr();
  tStackFrame* pstCurrFrame;
  tOBREF pstExOb;

  if (pstTheMethod->pstClass->pstClass->bInitialised == 0) {
    pstExOb = INITIALIZE_InitializeClass(env, 
					 pstTheMethod->pstClass->classObject);
    if (pstExOb) {
      return pstExOb;
    }
  }
#ifdef HASH_INTEGRITY_CHECK
  HASH_IntegrityCheck(&UID_sstUidHashTable);
#endif
  
  /* invoke method... */
  /* We make this an orphan frame, which means an exception will be
     returned to us if it isn't caught (and not printed out) */
  pstCurrFrame = InitialStackFrameHelper(pstTheMethod, NULL, pi32Args, 
					 kniPtr->GetHeap(env), 1);

  THREADINFO_addInitialStackFrameToNode(THREAD_FindThread(), pstCurrFrame);
  pstExOb = Interpret(env, pstCurrFrame, 1);
  THREADINFO_removeInitialStackFrameFromNode(THREAD_FindThread(), 
					     pstCurrFrame);
  
  if (pstExOb) {
    PopStackFrame(pstCurrFrame);
    return pstExOb;
  }
  
  if (pstTheMethod->u16RetSize == 1) {
    *pi32Args = (int32) *pstCurrFrame->pi32OpTop;
  }
  else if (pstTheMethod->u16RetSize == 2) {
    *pi32Args = (int32) (*pstCurrFrame->pi32OpTop - 1); /* high */
    *(pi32Args + 1) = (int32) *pstCurrFrame->pi32OpTop;   /* low */
  }
  
  PopStackFrame(pstCurrFrame);
  
  return NULL;
}


#ifdef not_used
/*
 * @doc FUNC
 * @func

 * This variation uses a reversed stack (called from C)

 * This function runs a non-virtual method from a tClass pointer and a tMethod
 * pointer.  This function is used for running non-virtual Java methods from
 * native methods.  INTERP_RunNonVirtualMethod() can be used if the method
 * needs to be specified by name.
 *
 * @ex Example showing invocation of a non-virtual method from native method.
 *     In the example, pi32Args is assumed to point to the top of the operand
 *     stack with the parameters on it. |
 *
 * tClassLoaderTuple*  pstClass;
 * tMethod* pstMethod;
 *
 * pstClass = CPLIST_Find("MyClass");
 * pstMethod = CLASSFILE_FindMethod(pstClass, "MyMethod", "(I)I");
 * INTERP_RunStaticMethodFromPtr(pstMethod, pi32Args);
 *
 */
tOBREF INTERP_RunStaticMethodFromPtr_with_native_stack(JNIEnv* env,  
						       tMethod* pstTheMethod,
						       int32*   pi32Args)
{
  KNIApi *kniPtr = KNI_getKNIApiPtr();
  tStackFrame* pstCurrFrame;
  tOBREF pstExOb;
  int i ;

  /* Ok now we need to flip the arguments around, but this doesn't
     work for longs and doubles -- shit */
   for (i = 0; i < (pstTheMethod->u16ArgSize / 2); i++) {
     int temp = pi32Args[i];
     pi32Args[i] = pi32Args[ pstTheMethod->u16ArgSize - 1 - i];
     pi32Args[ pstTheMethod->u16ArgSize - 1 - i] = temp;
   }
   
   /* invoke method... */
   pstCurrFrame = InitialStackFrameHelper(pstTheMethod, NULL, pi32Args, 
					  kniPtr->GetHeap(env), 1);
   
   assert(THREADINFO_addInitialStackFrameToNode(THREAD_FindThread(), 
						pstCurrFrame) == 0);
   pstExOb = Interpret(env, pstCurrFrame, 1);
   assert(THREADINFO_removeInitialStackFrameFromNode(THREAD_FindThread(), 
						     pstCurrFrame ) == 0);
   if (pstExOb) {
     PopStackFrame(pstCurrFrame);
     return pstExOb;
   }
   
   if (pstTheMethod->u16RetSize == 1) {
     *pi32Args = (int32) *pstCurrFrame->pi32OpTop;
   }
   else if (pstTheMethod->u16RetSize == 2) {
     *pi32Args = (int32) (*pstCurrFrame->pi32OpTop - 1); /* high */
     *(pi32Args + 1) = (int32) *pstCurrFrame->pi32OpTop;   /* low */
   }
   
   PopStackFrame(pstCurrFrame);
   
   return NULL;
}
#endif


/*
 * @doc FUNC
 * @func
 * This function runs a non-virtual method that is specified by the name of
 * its class and the name and signature of the method.  This function is used
 * for running non-virtual Java methods from native methods.  It finds the
 * correct method and then calls INTERP_RunNonVirtualMethodFromPtr().
 *
 * @ex Example showing invocation of a non-virtual method from native method.
 *     In the example, pi32Args is assumed to point to the top of the operand
 *     stack with the parameters on it. |
 *
 * INTERP_RunNonVirtualMethod("MyClass", "MyMethod", "(I)I", pi32Args);
 *
 */
tOBREF INTERP_RunNonVirtualMethod
(
  JNIEnv* env,
  tClassLoaderTuple* loadingClass,
  char* pszClassName,   /* @parm Name of class to use */
  char* pszMethodName,  /* @parm Name of method to use */
  char* pszMethodSig,   /* @parm Signature of method to use */
  int32 *pi32Args       /* @parm Pointer to stack containing arguments */
)
{
  tClassLoaderTuple* pstTheClass;
  tMethod *pstTheMethod;
  
  showLoading("INTERP_RunNonVirtualMethod %s", pszClassName);
  pstTheClass = CLASSFILE_FindOrLoad(env, pszClassName, loadingClass);
  
  if (pstTheClass == NULL) {
    panic("INTERP_RunNonVirtualMethod: class not found: %s", pszClassName);
  }
  
  pstTheMethod = CLASSFILE_FindMethod(env, pstTheClass, pszMethodName,
				      pszMethodSig);
  
  if (pstTheMethod == NULL) {
    panic("INTERP_RunNonVirtualMethod: method not found: %s %s %s",
	  pszClassName, pszMethodName, pszMethodSig);
  }
  
  return INTERP_RunSpecialMethodFromPtr(env, pstTheMethod, pi32Args);
}


/*
 * @doc FUNC
 * @func
 * This function runs a virtual method that is specified by the name of
 * its class and the name and signature of the method.  This function is used
 * for running virtual Java methods from native methods.
 *
 * @ex Example showing invocation of a virtual method from native method. In
 *     the example, pi32Args is assumed to point to the top of the operand
 *     stack with the parameters on it. |
 *
 * INTERP_RunVirtualMethod("MyClass", "MyMethod", "(I)I", pi32Args);
 *
 */
tOBREF INTERP_RunVirtualMethod(JNIEnv* env,
			       tClassLoaderTuple* loadingClass,
			       char*  pszClassName,  
			       /* @parm Name of class to use */
			       char*  pszMethodName, 
			       /* @parm Name of method to use */
			       char*  pszMethodSig,  
			       /* @parm Signature of method to use */
			       int32* pi32Args       
			       /* @parm Pointer to stack containing args */)
{
  KNIApi *kniPtr = KNI_getKNIApiPtr();
  tClassLoaderTuple* pstTheClass;
  tClassLoaderTuple* pstSuperClass;
  tMethod* pstTheMethod;
  tStackFrame* pstCurrFrame;
  tOBREF pstExOb; // seems to be the exception raised during method execution
  
  /* find the superclass */
  pstSuperClass = CLASSFILE_FindOrLoad(env, pszClassName, loadingClass);
  if (!pstSuperClass) {
    panic("INTERP_RunVirtualMethod: class %s not found", pszClassName);
  }
  /* find method in superclass */
  pstTheMethod = CLASSFILE_FindMethod(env, pstSuperClass, pszMethodName, 
				      pszMethodSig);
  if (!pstTheMethod) {
    panic("INTERP_RunVirtualMethod: method %s%s not found in class %s",
	  pszMethodName, pszMethodSig, pszClassName);
  }
  
  /* retrieve class pointer through handle on stack */
  pstTheClass = DEREF(((tOBREF) (*pi32Args)))->pstType;
  /* get real method pointer */
  pstTheMethod = pstTheClass->pstClass->ppstVT[pstTheMethod->u16VTIndex];
  
  /* invoke method */

  pstCurrFrame = InitialStackFrameHelper(pstTheMethod, (tOBREF) *pi32Args,
					 pi32Args, kniPtr->GetHeap(env), 1);
  
  THREADINFO_addInitialStackFrameToNode(THREAD_FindThread(), pstCurrFrame);
  pstExOb = Interpret(env, pstCurrFrame, 1);
  THREADINFO_removeInitialStackFrameFromNode(THREAD_FindThread(), 
					     pstCurrFrame);

  if (pstExOb) {
    PopStackFrame(pstCurrFrame);
    return pstExOb;
  }
  
  if (pstTheMethod->u16RetSize == 1) {
    *pi32Args = (int32) *pstCurrFrame->pi32OpTop;
  }
  else if (pstTheMethod->u16RetSize == 2) {
    *pi32Args = (int32) (*pstCurrFrame->pi32OpTop - 1); /* high */
    *(pi32Args + 1) = (int32) *pstCurrFrame->pi32OpTop;   /* low */
  }
  
  PopStackFrame(pstCurrFrame);
  
  return NULL;
}


tOBREF INTERP_RunVirtualMethodFromPtr(JNIEnv* env,
				      tMethod* pstMethod,
				      int32* pi32Args)
{
  KNIApi *kniPtr = KNI_getKNIApiPtr();
  tMethod* pstRightMethod;
  tStackFrame* pstFrame;
  tOBREF pstExOb;
  
  assert(!(pstMethod->u16AccessFlags & ACC_STATIC));
  assert(pi32Args[0] != 0);  // XXX - fix this
  // if (pi32Args[0] == 0) {
  //  char* errMessage = sys_malloc(1024);
  //  if (errMessage == NULL) {
  //    return OOMExceptionObject;
  //  }
  //  sprintf(errMessage, "null pointer encountered in INTERP_RunVirtualMethodPromPointer, running method %s.%s", pstMethod->pstClass->uidName, pstMethod->uidName);
  //  pstExOb = INTERP_ExceptionObjectFromNameAndMessage("java/lang/NullPointerException", errMessage);
  //  sys_free(errMessage);
  //  return pstExOb;
  // }
  pstRightMethod = DEREF((tOBREF) pi32Args[0])->
    pstType->pstClass->ppstVT[pstMethod->u16VTIndex];

  pstFrame = InitialStackFrameHelper(pstRightMethod, (tOBREF) pi32Args[0],
				     pi32Args, kniPtr->GetHeap(env), 1);
  THREADINFO_addInitialStackFrameToNode(THREAD_FindThread(), pstFrame);
  if (pstRightMethod->u16AccessFlags & ACC_SYNCHRONISED) {
    THREAD_SynchroniseEnter((tOBREF) pi32Args[0]);
  }

  pstExOb = Interpret(env, pstFrame, 1);

  THREADINFO_removeInitialStackFrameFromNode(THREAD_FindThread(), pstFrame);

  if (pstExOb) {
    PopStackFrame(pstFrame);
    return pstExOb;
  }
  
  if (pstRightMethod->u16RetSize == 1) {
    pi32Args[0] = pstFrame->pi32OpTop[0];
  }
  else if (pstRightMethod->u16RetSize == 2) {
    pi32Args[0] = pstFrame->pi32OpTop[-1];
    pi32Args[1] = pstFrame->pi32OpTop[0];
  }
  PopStackFrame(pstFrame);
  return NULL;
}


#ifdef not_used
/*
 * Same as above except now we have a native stack, so all the
 * arugments are the other way round (!)  
*/
tOBREF INTERP_RunVirtualMethodFromPtr_with_native_stack(JNIEnv* env,
							tMethod* pstMethod,
							int32* pi32Args)
{
  KNIApi *kniPtr = KNI_getKNIApiPtr();
  tMethod* pstRightMethod;
  tStackFrame* pstFrame;
  tOBREF pstExOb;
  int i = 0;

  /* Ok now we need to flip the arguments around, but this doesn't
     work for longs and doubles -- shit */
  for (i = 0; i < (pstMethod->u16ArgSize / 2); i++) {
    int temp = pi32Args[i];
    pi32Args[i] = pi32Args[ pstMethod->u16ArgSize - 1 - i];
    pi32Args[ pstMethod->u16ArgSize - 1 - i] = temp;
  }
  pstRightMethod = DEREF((tOBREF) pi32Args[0])->
    pstType->pstClass->ppstVT[pstMethod->u16VTIndex];
  
  pstFrame = InitialStackFrameHelper(pstRightMethod, (tOBREF) pi32Args[0],
				     pi32Args, kniPtr->GetHeap(env), 1);
  assert(THREADINFO_addInitialStackFrameToNode(THREAD_FindThread(), 
					       pstFrame) == 0);
  pstExOb = Interpret(env, pstFrame, 1);
  assert(THREADINFO_removeInitialStackFrameFromNode(THREAD_FindThread(),
						    pstFrame) == 0);
  
  if (pstExOb) {
    PopStackFrame(pstFrame);
    return pstExOb;
  }
  
  if (pstRightMethod->u16RetSize == 1) {
    pi32Args[0 /*pstMethod->u16ArgSize - 1*/ ] = pstFrame->pi32OpTop[0];
  }
  else if (pstRightMethod->u16RetSize == 2) {
    pi32Args[ 1 /*pstMethod->u16ArgSize - 2*/] = pstFrame->pi32OpTop[-1];
    pi32Args[0 /*pstMethod->u16ArgSize - 1*/] = pstFrame->pi32OpTop[0];
  }
  PopStackFrame(pstFrame);
  return NULL;
}
#endif


/*
 * @doc FUNC
 * @func
 * This function runs a static method that is specified by the name of
 * its class and the name and signature of the method.  This function is used
 * for running virtual Java methods from native methods.
 *
 * @ex Example showing invocation of a static method from native method. In
 *     the example, pi32Args is assumed to point to the top of the operand
 *     stack with the parameters on it. |
 *
 * INTERP_RunStaticMethod("MyClass", "MyMethod", "(I)I", pi32Args);
 *
 */
tOBREF INTERP_RunStaticMethod(JNIEnv* env,
			      tClassLoaderTuple* loadingClass,
			      char*  pszClassName,   
			      /* @parm Name of class to use */
			      char*  pszMethodName,  
			      /* @parm Name of method to use */
			      char*  pszMethodSig,   
			      /* @parm Signature of method to use */
			      int32* pi32Args        
			      /* @parm Pointer to stack containing args */)
{
  KNIApi *kniPtr = KNI_getKNIApiPtr();
  tClassLoaderTuple* pstTheClass;
  tMethod* pstTheMethod;
  tStackFrame* pstCurrFrame;
  tOBREF pstExOb;
  
  pstTheClass = CLASSFILE_FindOrLoad(env,pszClassName, loadingClass);
  if (pstTheClass == NULL) {
    panic("INTERP_RunStaticMethod: class not found: %s", pszClassName);
  }
  
  pstTheMethod = CLASSFILE_FindMethod(env,pstTheClass, pszMethodName, 
				      pszMethodSig);
  
  if (pstTheMethod == NULL) {
    panic("INTERP_RunStaticMethod: method not found: %s %s %s", 
	  pszClassName, pszMethodName, pszMethodSig);
  }
  
  /* invoke method */
  
  /* We make this an orphan frame, which means an exception will be
     returned to us if it isn't caught (and not printed out) */
  
  pstCurrFrame = InitialStackFrameHelper(pstTheMethod, NULL, pi32Args,
					 kniPtr->GetHeap(env), 1); 
  
  THREADINFO_addInitialStackFrameToNode(THREAD_FindThread(), pstCurrFrame);
  pstExOb = Interpret(env, pstCurrFrame, 1);
  THREADINFO_removeInitialStackFrameFromNode(THREAD_FindThread(), 
					     pstCurrFrame);
  
  if (pstExOb) {
    PopStackFrame(pstCurrFrame);
    return pstExOb;
  }
  
  PopStackFrame(pstCurrFrame);
  
  return NULL;
}


int isJavaFloatNaN(float num) 
{
  union {float f; uint32 i;} ui32fConv;

  ui32fConv.f = num;
  return (ui32fConv.i & 0x7f800000) == 0x7f800000 &&
    (ui32fConv.i & 0x7fffff) != 0;
}


int isJavaDoubleNaN(double num) 
{
  union {double d; long long i;} uint64dConv;
  
  uint64dConv.d = num;
  return (uint64dConv.i & 0x7ff0000000000000) == 0x7ff0000000000000 &&
    (uint64dConv.i & 0xfffffffffffff) != 0;
}
