/*
 * Copyright (C) 2001, John Leuner.
 *
 * This file is part of the kissme project.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2,
 * or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "config.h"


#include <string.h>

#include "vm/jni.h"
#include "vm/jni_data.h"
#include "vm/newobject.h"
#include "vm/interp_methods.h"
#include "vm/initialize.h"

#include "lib/indigenous/java.lang/VMClass.h"  
#include "lib/indigenous/java.lang/Class_Reflection.h"
#include "lib/indigenous/java.lang/VMClassLoader.h" 
#include "vm/garbage.h"


#define VOIDSIG (void (*) (JNIEnv*, ...))
#define BYTESIG (jbyte (*) (JNIEnv*, ...))
#define SHORTSIG (jshort (*) (JNIEnv*, ...))
#define INTSIG (jint (*) (JNIEnv*, ...))
#define LONGSIG (jlong (*) (JNIEnv*, ...))
#define FLOATSIG (jfloat (*) (JNIEnv*, ...))
#define DOUBLESIG (jdouble (*) (JNIEnv*, ...))

#ifdef DLOPEN
#include <dlfcn.h>
#endif


/*
 * Globals
 */

extern int INTERP_FirstThreadRunning;

// We keep this around to have JNI data while the first thread hasn't
// started yet
extern tJNIData* tempJNIData;  

// This can be globally used to get the type for object
extern tClassLoaderTuple* pstObjectType; 
extern tAllocatorHeap* system_heap;


// The "add_char" and "mangle_name" methods stolen for gcj for
// creating the method names

// Add a character to the buffer, encoding properly.
static void add_char (char *buf, jchar c, int *here)
{
  int i ;

  if (c == '_') {
    buf[(*here)++] = '_';
    buf[(*here)++] = '1';
  }
  else if (c == ';') {
    buf[(*here)++] = '_';
    buf[(*here)++] = '2';
  }
  else if (c == '[') {
    buf[(*here)++] = '_';
    buf[(*here)++] = '3';
  }
  
  // Also check for `.' here because we might be passed an internal
  // qualified class name like `foo.bar'.
  else if (c == '/' || c == '.') {
    buf[(*here)++] = '_';
  }
  else if ((c >= '0' && c <= '9') ||
           (c >= 'a' && c <= 'z') ||
           (c >= 'A' && c <= 'Z')) {
    buf[(*here)++] = (char) c;
  }
  else {
    // "Unicode" character.
    buf[(*here)++] = '_';
    buf[(*here)++] = '0';
    for (i = 0; i < 4; ++i) {
      int val = c & 0x0f;
      buf[(*here) + 3 - i] = (val > 10) ? ('a' + val - 10) : ('0' + val);
      c >>= 4;
    }
    *here += 4;
  }
}
  


// Compute a mangled name for a native function.  This computes the
// long name, and also returns an index which indicates where a NUL
// can be placed to create the short name.  This function assumes that
// the buffer is large enough for its results.
static void mangled_name (char* buf, char* chars, char* func_name,
			  char* signature, long* long_start)
{
  int here = 5;
  int i = 0;

  const unsigned char *fn = (const unsigned char *) func_name;
  const unsigned char *limit = fn + strlen(func_name); 
  const unsigned char *sig = (const unsigned char *) signature;

  jint len = strlen(chars);

  strcpy (buf, "Java_");
  for (i = 0; i < len; ++i) {
    add_char (buf, chars[i], &here);
  }

  // Don't use add_char because we need a literal `_'.
  buf[here++] = '_';

  for (i = 0; ; ++i) {
    int ch = fn[i]; //UTF8_GET (fn, limit);
    if (ch <= 0)
      break;
    add_char(buf, ch, &here);
  }

  // This is where the long signature begins.
  *long_start = here;

  if ((buf[here - 1]  == '(') && (buf[here] == ')')) {
    buf[here + 1] = '\0';
    return;
  }
  
  buf[here++] = '_';
  buf[here++] = '_';

  limit = sig + strlen(signature);
  assert(sig[0] == '(');
  ++sig;
  while (1) {
    int ch = sig[0];
    if (ch == ')' || ch <= 0) {
      break;
    }
    add_char (buf, ch, &here);
    sig++;
  }

  buf[here] = '\0';
}

static void JNI_pushThrownException(tStackFrame* pstFrame, tJNIData*pstJNIData)
{
#ifdef DEBUG_EXCEPTIONS
  eprintf("JNI_CallNativeMethod is returning -1 with pstJNIData %x "
	  "and exception %x(%s)\n",
	  pstJNIData, pstJNIData->pstException,
	  DEREF(pstJNIData->pstException)->pstType->uidName);
#endif
  pstFrame->pi32OpTop[1] = (int32) pstJNIData->pstException;
  pstFrame->pi32OpTop += 1;
  pstJNIData->pstException = NULL;
}


extern tClassLoaderTuple* pstClassType;

/*
 * @doc FUNC
 * @func
 * This function is called when a native method needs to be invoked. The
 * tMethod pointer is passed to it, along with the current stack frame. A
 * search is done to find the method is the class's native methods. When it
 * has been found it has to be cast to the correct type (return type and
 * number of parameters). The parameters that are passed to it are taken from
 * the operand stack (accessed via the current stack frame). The method is
 * invoked and its return value pushed back onto the operand stack. If an
 * exception is thrown, the function returns -1 and the interpreter catches
 * the exception.
 *
 */
int JNI_CallNativeMethod(tStackFrame* pstFrame,  
			 /* @parm Current Java stack frame */
			 tMethod* pstMethod     
			 /* @parm Method to invoke */)
{
  void* pFunction = NULL;
  void (*pVoidFunc) (JNIEnv*, ...);
  jbyte (*pByteFunc) (JNIEnv*, ...);
  jshort (*pShortFunc) (JNIEnv*, ...);
  jint (*pIntFunc) (JNIEnv*, ...);
  jlong (*pLongFunc) (JNIEnv*, ...);
  jfloat (*pFloatFunc) (JNIEnv*, ...);
  jdouble (*pDoubleFunc) (JNIEnv*, ...);
  int i, j, k;
  int iFound = 0;
  int iSafe = 0;
  int jniiter = 0;
  char* pszClassName = pstMethod->pstClass->uidName;
  jvalue jvRetVal;
  tJNIData* pstJNIData = JNI_getJNIData(sys_current_thread());
  int32* ot;
  char* pszRetChar;
  tClassLoaderTuple* theClass;
  JNIEnv* env;
  tStackFrame* pstOldTopFrame;
  
  traceJNI("native call to %s.%s", pszClassName, pstMethod->uidName);

  assert(pstJNIData);
  env = &pstJNIData->functions;
  
  /* record stack frame */
  pstOldTopFrame = pstJNIData->pstTopFrame;
  pstJNIData->pstTopFrame = pstFrame;
#ifdef DEBUG
  INTERP_CheckStackFrame(pstFrame);
  if (pstOldTopFrame) {
    INTERP_CheckStackFrame(pstOldTopFrame);
  }
#endif
  
  theClass = pstMethod->pstClass;
  
  if (pszClassName == NULL) {
    int failure;
    (*env)->Throw(env, INTERP_NewObjectFromName(env, "java/lang/NoSuchMethodException", &failure));
    JNI_pushThrownException(pstFrame, pstJNIData);
    pstJNIData->pstTopFrame = pstOldTopFrame;
    return -1;
  }
  
  if (pszClassName[0] == '[') {
    theClass = pstObjectType;
    pszClassName = "java/lang/Object";
  }

  if (theClass->pstClass->bInitialised == 0) {
    jobject pstExOb;

    traceJNI("Initialising native call target class %s ", pszClassName);
    pstExOb = INITIALIZE_InitializeClass(env, theClass->classObject);
    if (pstExOb) {
      traceJNI("Initialisation of class %s threw exception %s", pszClassName, 
	       (DEREF(pstJNIData->pstException))->pstType->uidName);
      pstFrame->pi32OpTop[1] = (int32) pstJNIData->pstException;
      pstFrame->pi32OpTop += 1;
      pstJNIData->pstException = NULL;
      pstJNIData->pstTopFrame = pstOldTopFrame;
      return -1;
    }
    traceJNI("Initialised native call target class %s ", pszClassName);
  }

  /* first find the method... use UID comparisons */
  // XXX - should throw an exception if there are no native methods.
  assert(theClass->pstClass->pstNativeMethods != NULL);
  traceJNI("Searching for native method %s.%s : %s",
	   pszClassName, pstMethod->uidName, pstMethod->uidSignature);
  for (i = 0; i < theClass->pstClass->u16NumNatives; i++) {
    if (theClass->pstClass->pstNativeMethods[i].fnPtr) {
      if (theClass->pstClass->pstNativeMethods[i].name == pstMethod->uidName) {
        if (theClass->pstClass->pstNativeMethods[i].signature ==
	    pstMethod->uidSignature) {
          iFound = 1;
          pFunction = theClass->pstClass->pstNativeMethods[i].fnPtr;
	  iSafe = theClass->pstClass->pstNativeMethods[i].isSafe;
          break;
        }
      }
    }
  }
  
  if (!iFound) {
#ifdef DLOPEN
    int loop = 0;
    int jFound = 0;
    long offset = 0;
    char symbolName[256];
    
    memset(symbolName, 0, 256);
    
    if (pstMethod->pstClass->uidName[0] == '[') {
      mangled_name(symbolName, "java/lang/Object", pstMethod->uidName, 
		   pstMethod->uidSignature, &offset);
    }
    else {
      mangled_name(symbolName, pstMethod->pstClass->uidName, 
		   pstMethod->uidName, pstMethod->uidSignature, &offset);
    }

    traceJNI("Searching for dynamically loaded native method %s", symbolName);
    for (loop = 0; loop < JNI_NumberLoadedLibraries; loop++)  {
      void* result = dlsym(JNI_DynLoadedLibraries[loop], symbolName);
      if (result != NULL) {
	jFound = 1;
	pFunction = result;
	iSafe = 1;
	break;
      }
    }
    
    if (!jFound) {
      //try the short name
      symbolName[offset] = 0;
      traceJNI("Searching for dynamically loaded native method %s", 
	       symbolName);
      for (loop = 0; loop < JNI_NumberLoadedLibraries; loop++) {
	void* result = dlsym(JNI_DynLoadedLibraries[loop], symbolName);
	if (result != NULL) {
	  jFound = 1;
	  pFunction = result;
	  iSafe = 1;
	  break;
	}
      } 
      
      if (!jFound) {
#ifdef DEBUG
	panic("Unknown native method called: %s.%s%s %s",
	      pszClassName, pstMethod->uidName,
	      pstMethod->uidSignature, 
	      pstMethod->pstClass->uidName);
#endif
	nonfatalError("Unknown native method called: %s.%s%s %s",
		      pszClassName, pstMethod->uidName,
		      pstMethod->uidSignature, 
		      pstMethod->pstClass->uidName);
	traceJNI("There were %i methods for this class\n", 
		 pstMethod->pstClass->pstClass->u16NumNatives);
	(*env)->Throw(env, INTERP_ExceptionObjectFromNameAndMessage(env, "java/lang/LinkageError", "Could not link to native method"));
	JNI_pushThrownException(pstFrame, pstJNIData);
	pstJNIData->pstTopFrame = pstOldTopFrame;
	return -1;
      }
    }
#else
    (*env)->Throw(env, INTERP_NewObjectFromName(env, "java/lang/NoSuchMethodException"));
    JNI_pushThrownException(pstFrame, pstJNIData);
    pstJNIData->pstTopFrame = pstOldTopFrame;
    return -1;
#endif
  }

  traceJNI("Found native method");

  /* reset optop to just before params */
  pstFrame->pi32OpTop -= pstMethod->u16ArgSize;
  ot = pstFrame->pi32OpTop;

#ifdef DEBUG
  INTERP_CheckStackFrame(pstFrame);
#endif 

  if (pstJNIData) {
    /* reserve local reference space for the args + the 'this' object
       or the class object */
    JNI_ReserveLocalRefs(pstJNIData, pstMethod->u16ArgSize + 1);

    /* put 'this' or the class object into the local reference table */
    if (pstMethod->u16AccessFlags & ACC_STATIC) {
      /* ake sure we have a reference to the object's class object */
      if (pstMethod->pstClass->classObject == NULL) {
	if ((pstMethod->pstClass == pstClassType) &&
	    (pstMethod->pstClass->classLoader == NULL)) {
	  //don't make a class object
	}
	else {
	  panic0("No reference to the pstClassObject in native call");
	}
      }
      JNI_CreateLocalRef(pstJNIData, pstMethod->pstClass->classObject);
    }
    else {
      /* put the "this" pointer into the local reference table */
      JNI_CreateLocalRef(pstJNIData, (tOBREF) ot[1]);
    }

    /* add reference arguments to the LocalRefs table*/
    if (pstMethod->u16AccessFlags & ACC_STATIC) {
      j = 1; 
    }
    else {
      j = 2;
    }
    
    for (i = 1; pstMethod->uidSignature[i] != ')'; i++) {
      switch (pstMethod->uidSignature[i]) {
      case 'J':
      case 'D':
	j++;
	break;

      case '[':
	JNI_CreateLocalRef(pstJNIData, (tOBREF) ot[j]);
	/* skip all the ['s */
	while (pstMethod->uidSignature[i] == '[') {
	  i++;
	}
	/* skip the type - either object or a single letter */
	if (pstMethod->uidSignature[i] == 'L') {
	  while (pstMethod->uidSignature[i] != ';') {
	    i++;
	  }
	}
	break;
	
      case 'L':
	JNI_CreateLocalRef(pstJNIData, (tOBREF) ot[j]);
	while (pstMethod->uidSignature[i] != ';') {
	  i++;
	}
	break;

      default:
	break;
      } //end switch

      j++;
    }
  }

  /* find return type char */
  pszRetChar = strchr(pstMethod->uidSignature, ')');
  pszRetChar += 1;
  
  if (iSafe) {
    GARBAGE_EnterGCRegion(JNI_getCurrentHeap());
  }

  traceJNI("Dispatching native method call");
#ifdef DEBUG
  INTERP_CheckStackFrame(pstFrame);
#endif
  switch (pstMethod->u16ArgSize) {
  case 0:
    {
      switch (*pszRetChar) {
      case 'V':
	pVoidFunc = VOIDSIG pFunction;
	if (pstMethod->u16AccessFlags & ACC_STATIC) {
	  (*pVoidFunc) (env, pstMethod->pstClass->classObject);
	}
	else {
	  (*pVoidFunc) (env);
	}
	break;
	
      case 'Z':
      case 'B':
	pByteFunc = BYTESIG pFunction;
	if (pstMethod->u16AccessFlags & ACC_STATIC) {
	  jvRetVal.b = (*pByteFunc) (env, pstMethod->pstClass->classObject);
	}
	else {
	  jvRetVal.b = (*pByteFunc) (env);
	}
	break;
	
      case 'C':
      case 'S':
	pShortFunc = SHORTSIG pFunction;
	if (pstMethod->u16AccessFlags & ACC_STATIC) {
	  jvRetVal.s = (*pShortFunc) (env, pstMethod->pstClass->classObject);
	}
	else {
	  jvRetVal.s = (*pShortFunc) (env);
	}
	break;

      case 'I':
      case 'L':
      case '[':
	pIntFunc = INTSIG pFunction;
	if (pstMethod->u16AccessFlags & ACC_STATIC) {
	  jvRetVal.i = (*pIntFunc) (env, pstMethod->pstClass->classObject);
	}
	else {
	  jvRetVal.i = (*pIntFunc) (env);
	}
	break;

      case 'F':
	pFloatFunc = FLOATSIG pFunction;
	if (pstMethod->u16AccessFlags & ACC_STATIC) {
	  jvRetVal.f = (*pFloatFunc) (env, pstMethod->pstClass->classObject);
	}
	else {
	  jvRetVal.f = (*pFloatFunc) (env);
	}
	break;

      case 'J':
	pLongFunc = LONGSIG pFunction;
	if (pstMethod->u16AccessFlags & ACC_STATIC) {
	  jvRetVal.j = (*pLongFunc) (env, pstJNIData,
				     pstMethod->pstClass->classObject);
	  
	}
	else {
          jvRetVal.j = (*pLongFunc) (env);
	}
	break;

      case 'D':
	pDoubleFunc = DOUBLESIG pFunction;
	if (pstMethod->u16AccessFlags & ACC_STATIC) {
	  panic("not implemented");
	}
	jvRetVal.d = (*pDoubleFunc) (env);
	break;
      }
      break;
    }

  case 1:
    {
      switch (*pszRetChar) {
      case 'V':
	pVoidFunc = VOIDSIG pFunction;
	if (pstMethod->u16AccessFlags & ACC_STATIC) {
	  (*pVoidFunc) (env, pstMethod->pstClass->classObject, ot[1]);
	}
	else {
	  (*pVoidFunc) (env, ot[1]);
	}
	break;
	
      case 'Z':
      case 'B':
	pByteFunc = BYTESIG pFunction;
	if (pstMethod->u16AccessFlags & ACC_STATIC) {
	  jvRetVal.b = (*pByteFunc) (env, pstMethod->pstClass->classObject, 
				     ot[1]);
	}
	else {
          jvRetVal.b = (*pByteFunc) (env, ot[1]);
	}
	break;

      case 'C':
      case 'S':
	pShortFunc = SHORTSIG pFunction;
	if (pstMethod->u16AccessFlags & ACC_STATIC) {
	  jvRetVal.s = (*pShortFunc) (env, pstMethod->pstClass->classObject, 
				      ot[1]);
	}
	else {
	  jvRetVal.s = (*pShortFunc) (env, ot[1]);
	}
	break;
	
      case 'I':
      case 'L':
      case '[':
          pIntFunc = INTSIG pFunction;
	  // Some experimentation here
	  if (pstMethod->u16AccessFlags & ACC_STATIC) {
	    jvRetVal.i = (*pIntFunc) (env, pstMethod->pstClass->classObject, 
				      ot[1]);
	  }
	  else {
	    jvRetVal.i = (*pIntFunc) (env, ot[1]);
	  }
          break;
	  
      case 'F':
	pFloatFunc = FLOATSIG pFunction;
	
	if (pstMethod->u16AccessFlags & ACC_STATIC) {
	  jvRetVal.f = (*pFloatFunc) (env, pstMethod->pstClass->classObject, 
				      ot[1]);
	}
	else {
	  jvRetVal.f = (*pFloatFunc) (env, ot[1]);
	}
	break;

      case 'J':
	pLongFunc = LONGSIG pFunction;
	if (pstMethod->u16AccessFlags & ACC_STATIC) {
	  jvRetVal.j = (*pLongFunc) (env, pstMethod->pstClass->classObject, 
				     ot[1]);
	}
	else {
	  jvRetVal.j = (*pLongFunc) (env, ot[1]);
	}
	break;
	
      case 'D':
	pDoubleFunc = DOUBLESIG pFunction;
	if (pstMethod->u16AccessFlags & ACC_STATIC) {
	  jvRetVal.d = (*pDoubleFunc) (env, pstMethod->pstClass->classObject, 
				       ot[1]);
	}
	else {
	  jvRetVal.d = (*pDoubleFunc) (env, ot[1]);
	}
	break;
      }
      break;
    }

  case 2:
    {
      switch (*pszRetChar) {
      case 'V':
	pVoidFunc = VOIDSIG pFunction;
	if (pstMethod->u16AccessFlags & ACC_STATIC) {
	  (*pVoidFunc) (env, pstMethod->pstClass->classObject, ot[1], ot[2]);
	}
	else {
	  (*pVoidFunc) (env, ot[1], ot[2]);
	}	
	break;
	
      case 'Z':
      case 'B':
	pByteFunc = BYTESIG pFunction;
	if (pstMethod->u16AccessFlags & ACC_STATIC) {
	  jvRetVal.b = (*pByteFunc) (env, pstMethod->pstClass->classObject, 
				     ot[1], ot[2]);
	}
	else {
	  jvRetVal.b = (*pByteFunc) (env, ot[1], ot[2]);
	}
	break;
	
      case 'C':
      case 'S':
	pShortFunc = SHORTSIG pFunction;
	if (pstMethod->u16AccessFlags & ACC_STATIC) {
	  jvRetVal.s = (*pShortFunc) (env, pstMethod->pstClass->classObject, 
				      ot[1], ot[2]);
	}
	else {
	  jvRetVal.s = (*pShortFunc) (env, ot[1], ot[2]);
	}
	break;
	
      case 'I':
      case 'L':
      case '[':
	pIntFunc =  INTSIG pFunction;
	if (pstMethod->u16AccessFlags & ACC_STATIC) {
	  jvRetVal.i = (*pIntFunc) (env, pstMethod->pstClass->classObject, 
				    ot[1], ot[2]);
	}
	else {
	  jvRetVal.i = (*pIntFunc) (env, ot[1], ot[2]);
	}
	break;
	
      case 'F':
	pFloatFunc =  FLOATSIG pFunction;
	if (pstMethod->u16AccessFlags & ACC_STATIC) {
	  jvRetVal.f = (*pFloatFunc) (env, pstMethod->pstClass->classObject, 
				      ot[1], ot[2]);
	}
	else {
	  jvRetVal.f = (*pFloatFunc) (env, ot[1], ot[2]);
	}
	break;

      case 'J':
	pLongFunc = LONGSIG pFunction;
	if (pstMethod->u16AccessFlags & ACC_STATIC) {
	  jvRetVal.j = (*pLongFunc) (env, pstMethod->pstClass->classObject, 
				     ot[1], ot[2]);
	}
	else {
	  jvRetVal.j = (*pLongFunc) (env, ot[1], ot[2]);
	}
	break;

      case 'D':
	pDoubleFunc = DOUBLESIG pFunction;
	if (pstMethod->u16AccessFlags & ACC_STATIC) {
	  jvRetVal.d = (*pDoubleFunc) (env, pstMethod->pstClass->classObject, 
				       ot[1], ot[2]);
	}
	else {
	  jvRetVal.d = (*pDoubleFunc) (env, ot[1], ot[2]);
	}
	break;
      }
      break;
    }

  case 3:
    {
      switch (*pszRetChar) {
      case 'V':
	pVoidFunc = VOIDSIG pFunction;
	if (pstMethod->u16AccessFlags & ACC_STATIC) {
	  (*pVoidFunc) (env, pstMethod->pstClass->classObject, 
			ot[1], ot[2], ot[3]);
	}
	else {
	  (*pVoidFunc) (env, ot[1], ot[2], ot[3]);
	}
	break;
	
      case 'Z':
      case 'B':
	pByteFunc = BYTESIG pFunction;
	if (pstMethod->u16AccessFlags & ACC_STATIC) {
	  jvRetVal.b = (*pByteFunc) (env, pstMethod->pstClass->classObject, 
				     ot[1], ot[2], ot[3]);
	}
	else {
	  jvRetVal.b = (*pByteFunc) (env, ot[1], ot[2], ot[3]);
	}
	break;
	
      case 'C':
      case 'S':
	pShortFunc = SHORTSIG pFunction;
	if (pstMethod->u16AccessFlags & ACC_STATIC) {
	  jvRetVal.s = (*pShortFunc) (env, pstMethod->pstClass->classObject,
				      ot[1], ot[2], ot[3]);
	}
	else {
	  jvRetVal.s = (*pShortFunc) (env, ot[1], ot[2], ot[3]);
	}
	break;
	
      case 'I':
      case 'L':
      case '[':
	pIntFunc = INTSIG pFunction;
	if (pstMethod->u16AccessFlags & ACC_STATIC) {
	  jvRetVal.i = (*pIntFunc) (env, pstMethod->pstClass->classObject, 
				    ot[1], ot[2], ot[3]);
	}
	else {
	  jvRetVal.i = (*pIntFunc) (env, ot[1], ot[2], ot[3]);
	}
	break;
	  
      case 'F':
	pFloatFunc = FLOATSIG pFunction;
	if (pstMethod->u16AccessFlags & ACC_STATIC) {
	  jvRetVal.f = (*pFloatFunc) (env, pstMethod->pstClass->classObject, 
				      ot[1], ot[2], ot[3]);
	}
	else {
	  jvRetVal.f = (*pFloatFunc) (env, ot[1], ot[2], ot[3]);
	}
	break;

      case 'J':
	pLongFunc = LONGSIG pFunction;
	if (pstMethod->u16AccessFlags & ACC_STATIC) {
	  panic("not implemented");
	}
	else {
	  jvRetVal.j = (*pLongFunc) (env, ot[1], ot[2], ot[3]);
	}
	break;
	
      case 'D':
	pDoubleFunc = DOUBLESIG pFunction;
	if (pstMethod->u16AccessFlags & ACC_STATIC) {
	  jvRetVal.d = (*pDoubleFunc) (env, pstMethod->pstClass->classObject,
				       ot[1], ot[2], ot[3]);
	}
	else {
	  jvRetVal.d = (*pDoubleFunc) (env, ot[1], ot[2], ot[3]);
	}
	break;
      }
      break;
    }
    case 4:
    {
      switch (*pszRetChar) {
        case 'V':
          pVoidFunc = VOIDSIG pFunction;
	  if (pstMethod->u16AccessFlags & ACC_STATIC) {
	    (*pVoidFunc) (env, pstMethod->pstClass->classObject, 
			  ot[1], ot[2], ot[3], ot[4]);
	  }
	  else {
	    (*pVoidFunc) (env, ot[1], ot[2], ot[3], ot[4]);
	  }
          break;

      case 'Z':
      case 'B':
	pByteFunc = BYTESIG pFunction;
	if (pstMethod->u16AccessFlags & ACC_STATIC) {
	  jvRetVal.b = (*pByteFunc) (env, pstMethod->pstClass->classObject, 
				     ot[1], ot[2], ot[3], ot[4]);
	}
	else {
	  jvRetVal.b = (*pByteFunc) (env, ot[1], ot[2], ot[3], ot[4]);
	}
	break;
	
      case 'C':
      case 'S':
	pShortFunc = SHORTSIG pFunction;
	if (pstMethod->u16AccessFlags & ACC_STATIC) {
	  jvRetVal.s = (*pShortFunc) (env, pstMethod->pstClass->classObject, 
				      ot[1], ot[2], ot[3], ot[4]);
	}
	else { 
	  jvRetVal.s = (*pShortFunc) (env, ot[1], ot[2], ot[3], ot[4]);
	}
	break;

      case 'I':
      case 'L':
      case '[':
	pIntFunc =  INTSIG pFunction;
	if (pstMethod->u16AccessFlags & ACC_STATIC) {
	  jvRetVal.i = (*pIntFunc) (env, pstMethod->pstClass->classObject, 
				    ot[1], ot[2], ot[3], ot[4]);
	}
	else {
	  jvRetVal.i = (*pIntFunc) (env, ot[1], ot[2], ot[3], ot[4]);
	}
	break;
	
      case 'F':
	pFloatFunc =  FLOATSIG pFunction;
	if (pstMethod->u16AccessFlags & ACC_STATIC) {
	  jvRetVal.f = (*pFloatFunc) (env, pstMethod->pstClass->classObject, 
				      ot[1], ot[2], ot[3], ot[4]);
	}
	else {
	  jvRetVal.f = (*pFloatFunc) (env, ot[1], ot[2], ot[3], ot[4]);
	}
	break;
	
      case 'J':
	pLongFunc = LONGSIG pFunction;
	if (pstMethod->u16AccessFlags & ACC_STATIC) {
	  jvRetVal.j = (*pLongFunc) (env, pstMethod->pstClass->classObject, 
				     ot[1], ot[2], ot[3], ot[4]);
	}
	else {
	  jvRetVal.j = (*pLongFunc) (env, ot[1], ot[2], ot[3], ot[4]);
	}
	break;
	
      case 'D':
	pDoubleFunc = DOUBLESIG pFunction;
	if (pstMethod->u16AccessFlags & ACC_STATIC) {
	  jvRetVal.d = (*pDoubleFunc) (env, pstMethod->pstClass->classObject, 
				       ot[1], ot[2], ot[3], ot[4]);
	}
	else {
	  jvRetVal.d = (*pDoubleFunc) (env, ot[1], ot[2], ot[3], ot[4]);
	}
	break;
      }
      break;
    }

  case 5:
    {
      switch (pstMethod->u16RetSize) {
      case 0:
	pVoidFunc = VOIDSIG pFunction;
	if (pstMethod->u16AccessFlags & ACC_STATIC) {
	  (*pVoidFunc) (env, pstMethod->pstClass->classObject, 
			ot[1], ot[2], ot[3], ot[4], ot[5]);
	}
	else {
	  (*pVoidFunc) (env, ot[1], ot[2], ot[3], ot[4], ot[5]);
	}
	break;

      case 1:
	pIntFunc = INTSIG pFunction;
	if (pstMethod->u16AccessFlags & ACC_STATIC) {
	  jvRetVal.i = (*pIntFunc) (env, pstMethod->pstClass->classObject,
				    ot[1], ot[2], ot[3], ot[4], ot[5]);
	}
	else {
	  jvRetVal.i = (*pIntFunc) (env, ot[1], ot[2], ot[3], ot[4], ot[5]);
	}
	break;
	
      case 2:
	pLongFunc = LONGSIG pFunction;
	if (pstMethod->u16AccessFlags & ACC_STATIC) {
	  jvRetVal.j = (*pLongFunc) (env, pstMethod->pstClass->classObject, 
				     ot[1], ot[2], ot[3], ot[4], ot[5]);
	}
	else {
	  jvRetVal.j = (*pLongFunc) (env, ot[1], ot[2], ot[3], ot[4], ot[5]);
	}
	break;
      }
      break;
    }

  case 6:
    {
      switch (pstMethod->u16RetSize) {
      case 0:
	pVoidFunc = VOIDSIG pFunction;
	if (pstMethod->u16AccessFlags & ACC_STATIC) {
	  (*pVoidFunc) (env, pstMethod->pstClass->classObject, 
			ot[1], ot[2], ot[3], ot[4], ot[5], ot[6]);
	}
	else {
	  (*pVoidFunc) (env, ot[1], ot[2], ot[3], ot[4], ot[5], ot[6]);
	}
	break;
	
      case 1:
	pIntFunc = INTSIG pFunction;
	if (pstMethod->u16AccessFlags & ACC_STATIC) {
	  jvRetVal.i = (*pIntFunc) (env, pstMethod->pstClass->classObject, 
				    ot[1], ot[2], ot[3], ot[4], ot[5], ot[6]);
	}
	else {
	  jvRetVal.i = (*pIntFunc) (env, ot[1], ot[2], ot[3], ot[4], 
				    ot[5], ot[6]);
	}
	break;
	
      case 2:
	pLongFunc = LONGSIG pFunction;
	if (pstMethod->u16AccessFlags & ACC_STATIC) {
	  jvRetVal.j = (*pLongFunc) (env, pstMethod->pstClass->classObject, 
				     ot[1], ot[2], ot[3], ot[4], ot[5], ot[6]);
	}
	else {
	  jvRetVal.j = (*pLongFunc) (env, ot[1], ot[2], ot[3], ot[4], 
				     ot[5], ot[6]);
	}
	break;
      }
      break;
    }

  case 7:
    {
      switch (pstMethod->u16RetSize) {
      case 0:
	pVoidFunc = VOIDSIG pFunction;
	if (pstMethod->u16AccessFlags & ACC_STATIC) {
	  (*pVoidFunc) (env, pstMethod->pstClass->classObject, 
			ot[1], ot[2], ot[3], ot[4], ot[5], ot[6], ot[7]);
	}
	else {
          (*pVoidFunc) (env, ot[1], ot[2], ot[3], ot[4], ot[5], ot[6], ot[7]);
	}
	break;
	
      case 1:
	pIntFunc = INTSIG pFunction;
	if (pstMethod->u16AccessFlags & ACC_STATIC) {
	  jvRetVal.i = (*pIntFunc) (env, pstMethod->pstClass->classObject, 
				    ot[1], ot[2], ot[3], ot[4], ot[5], 
				    ot[6], ot[7]);
	}
	else {
	  jvRetVal.i = (*pIntFunc) (env, ot[1], ot[2], ot[3], ot[4],
				    ot[5], ot[6], ot[7]);
	}
	break;
	
      case 2:
	pLongFunc = LONGSIG pFunction;
	if (pstMethod->u16AccessFlags & ACC_STATIC) {
	  jvRetVal.j = (*pLongFunc) (env, pstMethod->pstClass->classObject, 
				     ot[1], ot[2], ot[3], ot[4], ot[5], 
				     ot[6], ot[7]);
	}
	else { 
	  jvRetVal.j = (*pLongFunc) (env, ot[1], ot[2], ot[3], ot[4], 
				     ot[5], ot[6], ot[7]);
	}
	break;
      }
      break;
    }

  default:
    {
      switch (pstMethod->u16RetSize) {
      case 0:
	pVoidFunc = VOIDSIG pFunction;
	if (pstMethod->u16AccessFlags & ACC_STATIC) {
	  (*pVoidFunc) (env, pstMethod->pstClass->classObject,
			ot[1], ot[2], ot[3], ot[4], ot[5], ot[6], ot[7], ot[8],
			ot[9], ot[10], ot[11], ot[12], ot[13], ot[14], ot[15],
			ot[16], ot[17], ot[18], ot[19], ot[20]);
	}
	else {
	  (*pVoidFunc) (env, ot[1], ot[2], ot[3], ot[4], ot[5], ot[6], ot[7], 
			ot[8], ot[9], ot[10], ot[11], ot[12], ot[13], ot[14], 
			ot[15], ot[16], ot[17], ot[18], ot[19], ot[20]);
	}
	break;

      case 1:
	pIntFunc = INTSIG pFunction;
	if (pstMethod->u16AccessFlags & ACC_STATIC) {
	  jvRetVal.i = (*pIntFunc) (env, pstMethod->pstClass->classObject,
				    ot[1], ot[2], ot[3], ot[4], ot[5], ot[6], 
				    ot[7], ot[8], ot[9], ot[10], ot[11], 
				    ot[12], ot[13], ot[14], ot[15], ot[16], 
				    ot[17], ot[18], ot[19], ot[20]);
	}
	else {
	  jvRetVal.i = (*pIntFunc) (env, ot[1], ot[2], ot[3], ot[4], ot[5],
				    ot[6], ot[7], ot[8], ot[9], ot[10], ot[11],
				    ot[12], ot[13], ot[14], ot[15], ot[16], 
				    ot[17], ot[18], ot[19], ot[20]);
	}
	break;
	
      case 2:
	pLongFunc = LONGSIG pFunction;
	if (pstMethod->u16AccessFlags & ACC_STATIC) {
	  jvRetVal.j = (*pLongFunc) (env, pstMethod->pstClass->classObject, 
				     ot[1], ot[2], ot[3], ot[4], ot[5], ot[6],
				     ot[7], ot[8], ot[9], ot[10], ot[11], 
				     ot[12], ot[13], ot[14], ot[15], ot[16], 
				     ot[17], ot[18], ot[19], ot[20]);
	}
	else {
	  jvRetVal.j = (*pLongFunc) (env, ot[1], ot[2], ot[3], ot[4], ot[5],
				     ot[6], ot[7], ot[8], ot[9], ot[10], 
				     ot[11], ot[12], ot[13], ot[14], ot[15], 
				     ot[16], ot[17], ot[18], ot[19], ot[20]);
	}
	break;
      }
    }
  }

  if (iSafe) {
    GARBAGE_ExitGCRegion(JNI_getCurrentHeap());
  }

#ifdef DEBUG
  INTERP_CheckStackFrame(pstFrame);
#endif

  switch (*pszRetChar) {
  case 'V':
    break;
    
  case 'Z':
  case 'B':
    pstFrame->pi32OpTop[1] = jvRetVal.b;
    pstFrame->pi32OpTop += 1;
    break;

  case 'C':
  case 'S':
    pstFrame->pi32OpTop[1] = jvRetVal.s;
    pstFrame->pi32OpTop += 1;
    break;
    
  case 'I':
  case 'F':
    pstFrame->pi32OpTop[1] = jvRetVal.i;
    pstFrame->pi32OpTop += 1;
    break;
    
  case 'L':
  case '[':
#ifdef PERSIST
#ifdef DEBUG
#ifdef EVICTIONS
    eprintf("Retval.i is %i, meth %s\n", jvRetVal.i, pstMethod->uidName);
    assert(ISPID((PID) jvRetVal.i) == 0);
#endif
#endif
#endif
    pstFrame->pi32OpTop[1] = jvRetVal.i;
    pstFrame->pi32OpTop += 1;
    break;

  case 'J':
#ifdef __powerpc__
    pstFrame->pi32OpTop[1] = jvRetVal.j >> 32;
    pstFrame->pi32OpTop[2] = jvRetVal.j & 0xffffffff;
#else
    pstFrame->pi32OpTop[1] = jvRetVal.j & 0xffffffff;
    pstFrame->pi32OpTop[2] = jvRetVal.j >>32;
#endif

    pstFrame->pi32OpTop += 2;
    break;
    
  case 'D':
    {
      tDConv dconv;
      dconv.d = jvRetVal.d;
      pstFrame->pi32OpTop[1] = dconv.i.hi;
      pstFrame->pi32OpTop[2] = dconv.i.lo;
      pstFrame->pi32OpTop += 2;
      break;
    }
  }
  
#ifdef DEBUG
  INTERP_CheckStackFrame(pstFrame);
#endif

  /* get rid of local references */
  pstJNIData->u16NumLocalRefsUsed = 0;

  /* check if we need to throw an exception */
  if (pstJNIData->pstException) {
    traceJNI("Native call to %s.%s has thrown exception %s",
	     pszClassName, pstMethod->uidName, 
	     DEREF(pstJNIData->pstException)->pstType->uidName);
    JNI_pushThrownException(pstFrame, pstJNIData);
    pstJNIData->pstTopFrame = pstOldTopFrame;
    return -1;
  }
  else {
    traceJNI("Native call to %s.%s is returning normally",
	     pszClassName, pstMethod->uidName);
    pstJNIData->pstTopFrame = pstOldTopFrame;
    return 0;
  }
}

