#include "config.h"
#include "vm/sys_linux_host/wrappers.h"
#include "vm/llist.h"

#ifdef NATIVE_MUTEX

/* List of all mutexes in the system */

tList* kernel_mutex_list;

/* Create an initial empty list of mutexes */

int sys_mutex_provider_init()
{
  traceLock0("Creating sys_linux_kernel mutex list");
  kernel_mutex_list = LLIST_Create_List();
  assert(kernel_mutex_list != NULL);
  return 0;
}

/* Create a new system mutex object 
 *
 * @param type indicates whether the mutex is recursive
 *
 *
 */

tMutex* sys_mutex_create(int type)
{
  tMutex* newmutex;
  newmutex = sys_malloc(sizeof(tMutex));
  if(newmutex == NULL)
    return NULL;

  assert((type == 0) || (type == 1));

  newmutex->spinlock = 0;
  newmutex->owner_pthread = 0;
  newmutex->type = type;
  newmutex->recursion_count = 0;
  newmutex->status = 0;

  newmutex->waitList = NULL;
  newmutex->waitListSize = -1;
  newmutex->waitListUsed = 0;

  traceLock("sys_mutex_create %p, type %i", newmutex, type);

  return newmutex;
}

/* Destroy a system mutex object */

int sys_mutex_destroy(tMutex* mutex)
{
  //  eprintf("debug: sys_mutex_destroy\n");
  kspinlock_lock(&mutex->spinlock);
  if(mutex->waitList)
    {
      sys_free(mutex->waitList);
    }
  kspinlock_unlock(&mutex->spinlock);
  sys_free(mutex);
  //  eprintf("debug: sys_mutex_destroy __\n");
  return 0;
}


//I'm going to use cheap 'n nasty spinlocking until I get time to do this properly
int sys_mutex_lock(tMutex* mutex)
{
  if(mutex != NULL)
    {
      if(mutex->type == 1)
	{
	   if(mutex->owner_pthread == sys_current_thread())
	    mutex->recursion_count++;
	  else
	  {
	    kspinlock_lock(&mutex->spinlock);
	    mutex->owner_pthread = sys_current_thread();
	    mutex->recursion_count++;
	  }
	}
      else
	{
	kspinlock_lock(&mutex->spinlock);
	mutex->owner_pthread = sys_current_thread();
	}
    }
  else
    return -1;
  return 0;
}

int sys_mutex_unlock(tMutex* mutex)
{
  if(mutex != NULL)
    {
      if(mutex->type == 1)
	{
	  if(mutex->recursion_count > 0)
	    {
	      mutex->recursion_count--; 
	      if(mutex->recursion_count == 0)
		{
		  mutex->owner_pthread = 0;
		  kspinlock_unlock(&mutex->spinlock);
		}
	    }
	  else
	    {
	      mutex->owner_pthread = 0;
	      kspinlock_unlock(&mutex->spinlock);
	    }
	}
      else
	{
	  mutex->owner_pthread = 0;
	  kspinlock_unlock(&mutex->spinlock);
	}
    }
  else
    return -1;
  return 0;
}

//Condition variables
int sys_condition_provider_init()
{
  return 0;
}
tCondVariable* sys_condition_create()
{
  tCondVariable* ret = (tCondVariable*) sys_malloc(sizeof(tCondVariable));
  if(ret == NULL)
    return NULL; 

  ret->spinlock = 0;
  ret->pstWaitingList = LLIST_Create_List();
  if(ret == NULL)
    return NULL;
  return ret;
}
int sys_condition_destroy(tCondVariable* cond)
{
  if(cond != NULL)
    {
      if(cond->pstWaitingList)
	{
	  LLIST_DestroyList( cond->pstWaitingList );
	  cond->pstWaitingList = NULL;
	}
      sys_free(cond);
    }
  return 0;
}

int sys_condition_signal(tCondVariable* cond)
{
  if(cond == NULL)
    return -1;

  //  eprintf("Signalling %p\n", cond);
  kspinlock_lock(& cond->spinlock );
  if(cond->pstWaitingList)
    {
      int entryIndex = 0;
      int entries = cond->pstWaitingList->count;
      for(entryIndex = 0; entryIndex < entries;entryIndex ++)
	{
	  tCondWaitEntry* entry = (tCondWaitEntry*) LLIST_ElementAt( cond->pstWaitingList, entryIndex);
	  if(entry != NULL)
	    {
	      if(entry->signalled == 1)
		{
		  //		  eprintf("entry signalled %p (%x %i) but not caught\n", entry, entry->thread_id, entry->thread_id);
		}
	      else
		{
		  entry->signalled = 1;
		  break;
		}
	    }
	}
    }
  kspinlock_unlock(& cond->spinlock );
#ifdef KISSME_LINUX_KERNEL
  schedule();
#else
  pthread_yield();
#endif
  return 0;
}
int sys_condition_broadcast(tCondVariable* cond)
{
  if(cond == NULL)
    return -1;

  eprintf("Signalling (broadcast) %p\n", cond);
  kspinlock_lock(& cond->spinlock );
  if(cond->pstWaitingList)
    {
      int entries = cond->pstWaitingList->count;
      int i;
      for( i = 0; i < entries; i++)
	{
	  tCondWaitEntry* entry = (tCondWaitEntry*) LLIST_ElementAt( cond->pstWaitingList, i );
	  if(entry != NULL)
	    {
	      entry->signalled = 1;
	    }
	}
    }
  kspinlock_unlock(& cond->spinlock );
#ifdef KISSME_LINUX_USER
  pthread_yield();
#endif
  //  schedule();
  return 0;
}
int sys_condition_wait(tCondVariable* cond, tMutex* mutex)
{
  tCondWaitEntry* entry;

  if(cond == NULL)
    return -1;

  //  eprintf("Waiting on %p thread %x\n", cond, (int) sys_current_thread());
  
  kspinlock_lock(& cond->spinlock );
  if(cond->pstWaitingList == NULL)
    {
      cond->pstWaitingList = LLIST_Create_List();
    }

  entry = (tCondWaitEntry*) sys_malloc(sizeof(tCondWaitEntry));
  entry->thread_id = sys_current_thread();
  entry->signalled = 0;
  
  LLIST_AddAtTail( cond->pstWaitingList, entry );
  kspinlock_unlock(& cond->spinlock );
  sys_mutex_unlock(mutex);

  //  eprintf("Waiting on %p thread %x going into loop\n", cond, (int) sys_current_thread());
  //  jiffies_start = jiffies;
  while(1)
    {
      kspinlock_lock(& cond->spinlock );
      if(entry->signalled == 1)
	{
	  //	  eprintf("Waiting on %p thread (%x %i) signalled in loop\n", cond, (int) sys_current_thread(), (int) sys_current_thread());
	    LLIST_Remove( cond->pstWaitingList, entry);
	    //	  eprintf("Condition %p has %i waiting\n", cond, cond->pstWaitingList->count);
	  kspinlock_unlock(& cond->spinlock );
	  break;
	}
      kspinlock_unlock(& cond->spinlock );
      /*      if(jiffies - jiffies_start > 10000)
	{
	  eprintf("Thread %x %i is STILL waiting on %p\n", (int) sys_current_thread(), (int) sys_current_thread(), cond);
	  jiffies_start += 10000;
	}
      schedule();
      */
#ifdef KISSME_LINUX_KERNEL
      schedule();
#else
      usleep(50000);
      pthread_yield();
#endif
    }
  
  sys_mutex_lock(mutex);
  return 0;
}

int sys_condition_timedwait(tCondVariable* cond, tMutex* mutex, tTimeSpec* spec)
{
  panic0("sys_condition_timedwait: not implemented");
  //  pthread_cond_timedwait(cond, mutex, spec);
  return 0;
}

#endif


