#include "config.h"
/* 
 * @doc MODULE 
 * @module uid.c | 
 * Unique id's for strings - 
 * Strings are replaced with unique id's so that psz equality's can be 
 * done efficiently as an id comparison. The Uid hash table is a hash table 
 * of strings. The unique id (Uid) of a psz is simply the address of the 
 * psz that matches it in the psz table. 
 */ 
 
#include "vm/uid.h" 
#include "vm/stdtypes.h"
#include "vm/hash.h" 

#ifdef KISSME_LINUX_USER
#include <string.h> 
#endif

#include "vm/global.h" 
 
#define  TRUE    (BOOL) (1) 
#define  FALSE   (BOOL) /*lint -e506 */ (!TRUE ) /*lint -restore */ 
 

 
 
/* 
 * The hash table below is used to keep track of all the Uids created 
 * so far. 
 */ 
HASH_tstHashTable UID_sstUidHashTable; 
static BOOL sbInitialised = FALSE; 


/* 
 * @doc FUNC 
 * @func 
 *	Given a psz, this procedure returns a unique identifier 
 *	for the psz. 
 * 
 * @rdesc 
 *	This procedure returns a Uid corresponding to the "psz" 
 *	argument.  The Uid has a psz value identical to psz 
 *	(strcmp will return 0), but it's guaranteed that any other 
 *	calls to this procedure with a psz equal to "psz" will 
 *	return exactly the same result (i.e. can compare Uid 
 *	*values* directly, without having to call strcmp on what they 
 *	point to). 
 */  
Uid UID_GetUid(char* psz /* @parm String to convert. */)
{ 
  int32 i32Dummy; 
  
  /* add it to the hash table */ 
  if (!sbInitialised) { 
    HASH_InitHashTable(&UID_sstUidHashTable, HASH_STRING_KEYS); 
    sbInitialised = TRUE; 
  } 
  return (Uid) HASH_GetHashKey(&UID_sstUidHashTable,
			       HASH_CreateHashEntry(&UID_sstUidHashTable,
						    psz, &i32Dummy)); 
} 
 

/* 
 * @doc FUNC 
 * @func 
 * Compares Uid strings.  We check for the buggy condition of uid's different, 
 * but strings the same. 
 * 
 * @rdesc Mimics strcmp() for equality checks and returns one of: 
 * 
 * @flag 0 | if the uids are identical 
 * @flag 1 | otherwise  
 */ 

int UID_UidCmp(Uid uid1,   /* @parm First Uid to compare */ 
	       Uid uid2    /* @parm Second Uid to compare */ 
	       ) 
{ 
  if (uid1 == uid2) {     
    return 0; 
  } 
 
  /* If the strings really are the same, then we have a bug! */ 
  if (strcmp(uid1, uid2) == 0) { 
    panic("UID_UidCmp: uid's different, but strings the same (%s)", uid1); 
  } 
  
  return 1; 
} 
 
