# Copyright (c) 2008 Hideki Ikemoto
#
# Permission is hereby granted, free of charge, to any person obtaining
# a copy of this software and associated documentation files (the
# "Software"), to deal in the Software without restriction, including
# without limitation the rights to use, copy, modify, merge, publish,
# distribute, sublicense, and/or sell copies of the Software, and to
# permit persons to whom the Software is furnished to do so, subject to
# the following conditions:
#
# The above copyright notice and this permission notice shall be
# included in all copies or substantial portions of the Software.
#
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
# EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
# MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
# NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE
# LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION
# OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION
# WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

require 'writedialogbase.rb'
require 'singleton'
require 'net/http'

class WriteDialog < WriteDialogBase
  include Singleton

  slots 'post()', 'set_sage(bool)', 'cancel()', 'check_body()'

  def initialize
    super

    connect(@ok_button, SIGNAL("clicked()"), self, SLOT("post()"))
    connect(@cancel_button, SIGNAL("clicked()"), self, SLOT("cancel()"))
    connect(@sage_checkbox, SIGNAL("toggled(bool)"), self, SLOT("set_sage(bool)"))
    connect(@body_edit, SIGNAL("textChanged()"), self, SLOT("check_body()"))
  end

  def show
    super

    @name_line.text = ""
    @mail_line.text = ""
    @body_edit.text = ""
    @sage_checkbox.setChecked(false)
    @ok_button.setEnabled(false)
  end

  def dat_url=(dat_url)
    @dat_url = dat_url
    @board_url = Util.dat_url_to_board_url(dat_url)
    @thread_id = Util.dat_url_to_thread_id(dat_url)
    @board_id = Util.dat_url_to_board_id(dat_url)
    @write_url = Util.dat_url_to_write_url(dat_url)
  end

  def board_name=(board_name)
    @board_name_label.text = board_name
  end

  def thread_name=(thread_name)
    @thread_name_label.text = thread_name
  end

  def post
    name = @name_line.text
    mail = @mail_line.text
    body = @body_edit.text
    server_time = Time::now

    post_str = Util.build_post_str(name, mail, body,
                                   @board_id, @thread_id, server_time,
                                   Account.instance.session_id)

    response_body = nil
    cookies = []
    Net::HTTP.start(URI.parse(@write_url).host, 80) {|http|
      header = {
        "Content-Type" => "application/x-www-form-urlencoded",
        "Referer" => @board_url
      }
      response = http.post(URI.parse(@write_url).path,
                           post_str, header)

      response_body = NKF.nkf("-Swx", response.body) # SJIS -> UTF-8 with JISX0201

      response.get_fields("Set-Cookie").each {|cookie|
        value = cookie[0...cookie.index(';')]
        cookies.push(value)
      }
    }

    return unless response_body && response_body.match(/<!-- 2ch_X:cookie -->/)

    Net::HTTP.start(URI.parse(@write_url).host, 80) {|http|
      header = {
        "Content-Type" => "application/x-www-form-urlencoded",
        "Referer" => @board_url,
        "Cookie" => cookies.join(";")
      }
      response = http.post(URI.parse(@write_url).path,
                           post_str, header)

      response_body = NKF.nkf("-Swx", response.body) # SJIS -> UTF-8 with JISX0201
    }

    accept
  end

  def set_sage(on)
    if on == true
      @mail_stack = @mail_line.text
      @mail_line.text = "sage"
      @mail_line.setEnabled(false)
    else
      @mail_line.text = @mail_stack
      @mail_line.setEnabled(true)
    end
  end

  def cancel
    if @body_edit.text.length > 0 then
      if Qt::MessageBox::warning(self,
                                 "Kita",
                                 "このダイアログを閉じると、テキストが破棄されます。\n" +
                                 "閉じてもいいですか？",
                                 Qt::MessageBox::Ok,
                                 Qt::MessageBox::Cancel | Qt::MessageBox::Default) ==
        Qt::MessageBox::Ok then
        reject
      end
    else
      reject
    end
  end

  def check_body
    body = @body_edit.text
    if body.empty? then
      @ok_button.setEnabled(false)
    else
      @ok_button.setEnabled(true)
    end
  end
end
