# Copyright (c) 2008 Hideki Ikemoto
#
# Permission is hereby granted, free of charge, to any person obtaining
# a copy of this software and associated documentation files (the
# "Software"), to deal in the Software without restriction, including
# without limitation the rights to use, copy, modify, merge, publish,
# distribute, sublicense, and/or sell copies of the Software, and to
# permit persons to whom the Software is furnished to do so, subject to
# the following conditions:
#
# The above copyright notice and this permission notice shall be
# included in all copies or substantial portions of the Software.
#
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
# EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
# MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
# NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE
# LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION
# OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION
# WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

class ThreadListView < Qt::Widget
  slots 'show_thread(QTreeWidgetItem*)'
  slots 'search()'
  slots 'filter(bool)'
  slots 'remove_self()'
  slots 'reload()'
  slots 'copy_url()', 'open_with_web_browser()', 'copy_title_and_url()'

  SUBJECT_TXT_REGEXP1 = /((\d+)\.cgi),(.*)\((\d+)\)/
  SUBJECT_TXT_REGEXP2 = /((\d+)\.dat)<>(.*)\((\d+)\)/
  COLUMN_MARK = 0
  COLUMN_NO = 1
  COLUMN_SEARCH = 2
  COLUMN_TITLE = 3
  COLUMN_RESNUM = 4
  COLUMN_READNUM = 5
  COLUMN_UNREADNUM = 6
  COLUMN_OPENTIME = 7
  COLUMN_SPEED = 8
  COLUMN_URL = 9
  COLUMN_MARKORDER = 10

  ORDER_UNREAD = "0"
  ORDER_READ = "1"
  ORDER_NORMAL = "2"

  NULL_ICON = Qt::Icon.new

  def initialize(parent)
    super(parent)

    main_layout = Qt::VBoxLayout.new(self)
    tool_layout = Qt::HBoxLayout.new
    main_layout.addLayout(tool_layout)
    @search_line = Qt::LineEdit.new(self)
    tool_layout.addWidget(@search_line)
    @search_button = Qt::ToolButton.new(self)
    @search_button.setIcon(KDE::Icon.new("edit-find"))
    @search_button.setEnabled(false)
    tool_layout.addWidget(@search_button)
    connect(@search_button, SIGNAL("clicked()"),
            self, SLOT("search()"))
    connect(@search_line, SIGNAL("textChanged(const QString&)"),
            self, SLOT("search()"))

    @filter_button = Qt::ToolButton.new(self)
    @filter_button.setIcon(KDE::Icon.new("view-filter"))
    @filter_button.setCheckable(true)
    @filter_button.setEnabled(false)
    tool_layout.addWidget(@filter_button)
    connect(@filter_button, SIGNAL("toggled(bool)"),
            self, SLOT("filter(bool)"))

    @reload_button = Qt::ToolButton.new(self)
    @reload_button.setIcon(KDE::Icon.new("vcs_update"))
    @reload_button.setEnabled(false)
    tool_layout.addWidget(@reload_button)
    connect(@reload_button, SIGNAL("clicked()"), self, SLOT("reload()"))

    @close_button = Qt::ToolButton.new(self)
    @close_button.setIcon(KDE::Icon.new("dialog-close"))
    @close_button.setEnabled(false)
    tool_layout.addWidget(@close_button)
    connect(@close_button, SIGNAL("clicked()"), self, SLOT("remove_self()"))

    @list = Qt::TreeWidget.new(self)
    main_layout.addWidget(@list)

    connect(@list, SIGNAL("itemClicked(QTreeWidgetItem*,int)"),
            self, SLOT("show_thread(QTreeWidgetItem*)"))

    @list.setHeaderLabels(["", "No.", " ", "Thread name", "レス", "既読", "未読", "Open time", "Thread speed"])
    @list.setAllColumnsShowFocus(true)
    @list.setRootIsDecorated(false)
    @list.setSortingEnabled(true)
#    @list.header.setStretchEnabled(true, COLUMN_TITLE)

    # search
    @prev_search_text = nil
    @hit_list = []
    @hit_list_index = 0
  end

  def show_board(url)
    return if url == nil

    @board_url = url
    subject_txt_url = url + "subject.txt"

    thread_list = []
    case Util.board_type(url)
    when Util::TYPE_2CH, Util::TYPE_MACHIBBS, Util::TYPE_UNKNOWN
      str = Downloader.download_subject_txt(subject_txt_url)
    when Util::TYPE_JBBS
      str = Downloader.download_subject_txt_jbbs(subject_txt_url)
    end

    if str.size == 0
      Qt::MessageBox::warning(self,
                              "警告",
                              "subject.txtが空です。板が移転した可能性が高いです。",
                              Qt::MessageBox::Ok, Qt::MessageBox::NoButton)
      return
    end

    str.each {|line|
      if SUBJECT_TXT_REGEXP1.match(line) || SUBJECT_TXT_REGEXP2.match(line)
        dat_file_name = Regexp.last_match(1)
        open_time = Time::at(Regexp.last_match(2).to_i)
        title = Regexp.last_match(3)
        num = Regexp.last_match(4).to_i
        speed = num.to_i / ((Time::now - open_time) / 86400.0)

        dat_url = Util.make_dat_url(@board_url, dat_file_name)
        thread = ThreadInfo.new(dat_url, title, num, open_time, speed)
        thread_list.push(thread)
      end
    }

    @list.clear

    prev_item = 0
    thread_list.each_with_index {|thread, i|
      item = ThreadListViewItem.new(@list, prev_item)
      item.setText(COLUMN_NO, "%4s" % (i+1))
      item.setText(COLUMN_TITLE, thread.title)
      item.setText(COLUMN_RESNUM, "%4s" % thread.res_num)
      item.setText(COLUMN_READNUM, "%4s" % thread.read_num) if thread.read_num != 0
      item.setText(COLUMN_UNREADNUM, "%4s" % (thread.res_num - thread.read_num)) if thread.read_num != 0
      item.setText(COLUMN_OPENTIME, thread.open_time.strftime("%y/%m/%d %H:%M"))
      item.setText(COLUMN_SPEED, thread.speed.to_s)
      item.setText(COLUMN_URL, thread.dat_url)

      if thread.res_num > thread.read_num && thread.read_num != 0 then
        item.setIcon(COLUMN_MARK, Qt::Icon.new("unread.png"))
        item.setText(COLUMN_MARKORDER, ORDER_UNREAD)
      elsif thread.res_num == thread.read_num then
        item.setIcon(COLUMN_MARK, Qt::Icon.new("read.png"))
        item.setText(COLUMN_MARKORDER, ORDER_READ)
      else
        item.setText(COLUMN_MARKORDER, ORDER_NORMAL)
      end

      prev_item = item
    }

    resize_columns
    ViewMediator.set_url(url)
    enable_buttons
  end

  def resize_columns
    @list.resizeColumnToContents(COLUMN_MARK)
    @list.resizeColumnToContents(COLUMN_NO)
    @list.resizeColumnToContents(COLUMN_SEARCH)
    @list.resizeColumnToContents(COLUMN_TITLE)
    @list.resizeColumnToContents(COLUMN_RESNUM)
    @list.resizeColumnToContents(COLUMN_READNUM)
    @list.resizeColumnToContents(COLUMN_UNREADNUM)
    @list.resizeColumnToContents(COLUMN_OPENTIME)
    @list.resizeColumnToContents(COLUMN_SPEED)
  end

  def show_thread(item)
    return if item == nil

    thread_url = item.text(COLUMN_URL)
    return if thread_url == nil

    ViewMediator.show_thread(thread_url)
  end

  def search
    str = @search_line.text
    if str == @prev_search_text
      search_next
    elsif str != ""
      search_new(str)
      @prev_search_text = str
      @filter_button.toggled(true)
      filter(true)
    else
      search_clear
      @filter_button.toggled(false)
    end
  end

  def filter(on)
    it = Qt::TreeWidgetItemIterator.new(@list)
    while it.current do
      item = it.current
      if on && item.icon(COLUMN_SEARCH).isNull then
        item.setHidden(true)
      else
        item.setHidden(false)
      end
      it += 1
    end
  end

  def search_clear
    @hit_list.clear
    @hit_list_index = 0
    @prev_search_text = nil

    it = Qt::TreeWidgetItemIterator.new(@list)
    while it.current do
      item = it.current
      item.setIcon(COLUMN_SEARCH, NULL_ICON)
      it += 1
    end
  end

  def search_new(str)
    @hit_list.clear
    @hit_list_index = 0

    it = Qt::TreeWidgetItemIterator.new(@list)
    regexp = Regexp.new(str, Regexp::IGNORECASE)
    while it.current do
      item = it.current
      item.setIcon(COLUMN_SEARCH, NULL_ICON)
      title = item.text(COLUMN_TITLE)

      if title.match(regexp) then
        @hit_list.push(item)
      end

      it += 1
    end

    @hit_list.each {|item|
      item.setIcon(COLUMN_SEARCH, KDE::Icon.new("edit-find"))
    }
  end

  def search_next
    item = @hit_list[@hit_list_index]
    @list.ensureItemVisible(item)
    @list.setSelected(item, true)

    @hit_list_index += 1
    if @hit_list_index >= @hit_list.size then
      @hit_list_index = 0
    end
  end

  def contextMenuEvent(event)
    popup = KDE::Menu.new
    popup.addAction("ウェブブラウザで開く", self, SLOT('open_with_web_browser()'))
    popup.addAction("Copy URL", self, SLOT('copy_url()'))
    popup.addAction("Copy title and URL", self, SLOT('copy_title_and_url()'))
    popup.exec(event.globalPos())
  end

  def copy_url
    dat_url = @list.currentItem.text(COLUMN_URL)
    thread_url = Util.dat_url_to_thread_url(dat_url)
    clipboard = Qt::Application.clipboard
    clipboard.setText(thread_url, Qt::Clipboard::Clipboard)
    clipboard.setText(thread_url, Qt::Clipboard::Selection)
  end

  def copy_title_and_url
    dat_url = @list.currentItem.text(COLUMN_URL)
    thread_url = Util.dat_url_to_thread_url(dat_url)
    thread_title = @list.currentItem.text(COLUMN_TITLE)
    clipboard = Qt::Application.clipboard
    cliptext = thread_title + "\n" + thread_url
    clipboard.setText(cliptext, Qt::Clipboard::Clipboard)
    clipboard.setText(cliptext, Qt::Clipboard::Selection)
  end

  def open_with_web_browser
    dat_url = @list.currentItem.text(COLUMN_URL)
    thread_url = Util.dat_url_to_thread_url(dat_url)
    KDE::ToolInvocation.invokeBrowser(thread_url)
  end

  def remove_self
    ViewMediator::remove_thread_list_view(self)
  end

  def board_url
    @board_url
  end

  def reload
    show_board(@board_url)
  end

  def enable_buttons
    @search_button.setEnabled(true)
    @filter_button.setEnabled(true)
    @reload_button.setEnabled(true)
    @close_button.setEnabled(true)
  end
end

class ThreadListViewItem < Qt::TreeWidgetItem
  def compare(item, col, ascending)
    case col
    when ThreadListView::COLUMN_MARK
      key1 = key(ThreadListView::COLUMN_MARKORDER, ascending)
      key2 = item.key(ThreadListView::COLUMN_MARKORDER, ascending)

      key1.to_i <=> key2.to_i
    else
      key(col, ascending) <=> item.key(col, ascending)
    end
  end
end
