/*
 * Callbacks for widgets events
 */
#include <string.h>
#include <stdlib.h>
#include <unistd.h>

#include "support.h"
#include "interface.h"
#include "main.h"
#include "translation.h"
#include "keyboard.h"
#include "tutor.h"
#include "cursor.h"
#include "plot.h"
#include "basic.h"
#include "velocity.h"
#include "fluidness.h"
#include "callbacks.h"

/*
 * Variables
 */
extern GtkWidget *window_main_;
extern GtkWidget *window_tutor_;
extern GtkWidget *window_keyboard_;
extern GtkWidget *window_hints_;
extern GtkWidget *window_stat_;

GtkWidget *popup_other_ = NULL;
GtkWidget *filechooser_tutor = NULL;
GtkWidget *fontselectiondialog_tutor = NULL;

gboolean callbacks_shield = FALSE;


/**********************************************************************
 * Part 1
 * Callbacks from the main window
 */
/**********************************************************************
 * Main window: initialization and destruction
 */
void
on_window_main_destroy (GtkObject * object, gpointer user_data)
{
  if (callbacks_shield)
    return;
  main_window_pass_away ();
}


/**********************************************************************
 * Info-dialog shared tasks
 */
/*
 * Help dialog
 */
void
on_button_help_clicked (GtkButton * button, gpointer user_data)
{
  gchar text[TEXTMAXLEN];
  GtkWidget *dialog;
  GtkLabel *wg_label;
  GtkTextView *wg_text;

  dialog = create_dialog_info ();

  wg_label = GTK_LABEL (lookup_widget (dialog, "label_info"));
  gtk_label_set_text (wg_label, _("Help?"));

  wg_text = GTK_TEXT_VIEW (lookup_widget (dialog, "text_info"));
  trans_read_text (text, TEXTMAXLEN, "help.txt");
  gtk_text_buffer_set_text (gtk_text_view_get_buffer (wg_text), text, -1);

  gtk_widget_hide (window_main_);
  gtk_widget_show (dialog);
}


/*
 * About dialog
 */
void
on_button_about_clicked (GtkButton * button, gpointer user_data)
{
  gchar text[TEXTMAXLEN];
  gchar *text1;
  GtkWidget *dialog;
  GtkLabel *wg_label;
  GtkTextView *wg_text;

  dialog = create_dialog_info ();

  wg_label = GTK_LABEL (lookup_widget (dialog, "label_info"));
  gtk_label_set_text (wg_label, _("About 'Klavaro'"));

  wg_text = GTK_TEXT_VIEW (lookup_widget (dialog, "text_info"));
  trans_read_text (text, TEXTMAXLEN, "about.txt");
  text1 = strstr (text, "(V)");
  if (text1 == NULL)
    text1 = g_strconcat ("\n ", _("Version:"), " ", PACKAGE_VERSION,
			 "\n", text, NULL);
  else
    {
      *text1 = '\0';
      text1 += 3;
      text1 = g_strconcat (text, _("Version:"), " ", PACKAGE_VERSION,
			   text1, NULL);
    }

  gtk_text_buffer_set_text (gtk_text_view_get_buffer (wg_text), text1, -1);
  g_free (text1);

  gtk_widget_hide (window_main_);
  gtk_widget_show (dialog);
}

/*
 * Exit button
 */
void
on_button_exit_clicked                 (GtkButton       *button,
                                        gpointer         user_data)
{
	main_window_pass_away ();
}


/**********************************************************************
 * Changing the keyboard layout
 */
void
on_button_keyboard_clicked (GtkButton * button, gpointer user_data)
{
  GtkWidget *wg;

  window_keyboard_ = create_window_keyboard ();

  wg = lookup_widget (window_main_, "entry_keyboard");
  keyb_set_name (gtk_entry_get_text (GTK_ENTRY (wg)));
  keyb_get_layout ();
  keyb_fill_layout_list (0);	/* fills all the entries */
  keyb_update_virtual_layout ();

  wg = lookup_widget (window_keyboard_, "notebook_kb");
  gtk_notebook_set_current_page (GTK_NOTEBOOK (wg), 0);

  keyb_set_modified_status (FALSE);

  hints_init ();

  gtk_widget_show (window_keyboard_);
  gtk_widget_hide (window_main_);
}


/**********************************************************************
 * Changing the language
 */
void
on_button_change_language_clicked (GtkButton * button, gpointer user_data)
{
  GtkWidget *dialog_confirm;

  dialog_confirm = create_dialog_confirm ();
  gtk_label_set_text (GTK_LABEL (lookup_widget
				 (dialog_confirm, "label_confirm_action")),
		      "CHANGE");
  gtk_widget_show (dialog_confirm);
}


/**********************************************************************
 * Main menu comands
 */
/**********************************************************************
 * More one shared info-dialog callback: introduction dialog
 */
void
on_button_intro_clicked (GtkButton * button, gpointer user_data)
{
  gchar text[TEXTMAXLEN];
  GtkWidget *dialog_intro;
  GtkTextView *wg_text;

  dialog_intro = create_dialog_info ();

  wg_text = GTK_TEXT_VIEW (lookup_widget (dialog_intro, "text_info"));
  trans_read_text (text, TEXTMAXLEN, "intro.txt");
  gtk_text_buffer_set_text (gtk_text_view_get_buffer (wg_text), text, -1);

  gtk_widget_hide (window_main_);
  gtk_widget_show (dialog_intro);
}


/**********************************************************************
 * Basic course
 */
void
on_button_basic_clicked (GtkButton * button, gpointer user_data)
{
  gboolean show_intro;

  show_intro = ! (g_key_file_has_key (main_preferences_get (), "tutor", "basic_lesson", NULL));
  basic_set_lesson (0);
  if (show_intro)
    g_key_file_remove_key (main_preferences_get (), "tutor", "basic_lesson", NULL);

  tutor_init (TT_BASIC);
}


/**********************************************************************
 * Random letters training (adaptability)
 */
void
on_button_adapt_clicked (GtkButton * button, gpointer user_data)
{
  tutor_init (TT_ADAPT);
}


/**********************************************************************
 * Random words training (velocity)
 */
void
on_button_velo_clicked (GtkButton * button, gpointer user_data)
{
  tutor_init (TT_VELO);
}


/**********************************************************************
 * Random texts training (fluidness)
 */
void
on_button_fluid_clicked (GtkButton * button, gpointer user_data)
{
  tutor_init (TT_FLUID);
}


/**********************************************************************
 * More two shared info-dialog callbacks
 */
void
on_text_info_realize (GtkWidget * widget, gpointer user_data)
{
  PangoFontDescription *font_desc;

  /* Change default font throughout the widget */
  font_desc = pango_font_description_from_string (NORMAL_FONT);
  gtk_widget_modify_font (widget, font_desc);
  pango_font_description_free (font_desc);
}


void
on_dialog_info_destroy (GtkObject * object, gpointer user_data)
{
  gtk_widget_show (window_main_);
}


void
on_dialog_info_response (GtkDialog * dialog,
			 gint response_id, gpointer user_data)
{
  gtk_widget_destroy (GTK_WIDGET (dialog));
}


/**********************************************************************
 * Part 2
 * Callbacks from the keyboard window
 */
/**********************************************************************
 * Keyboard window destruction
 */
void
on_window_keyboard_destroy (GtkObject * object, gpointer user_data)
{
  GtkWidget *wg;

  if (window_tutor_ != NULL)
    {
      gtk_widget_show (window_tutor_);
      wg = lookup_widget (window_keyboard_, "checkbutton_hints");
      if (GTK_TOGGLE_BUTTON (wg)->active)
	{
	  window_hints_ = create_window_hints ();
	  gtk_widget_show (window_hints_);
	  hints_update_from_char (cursor_get_char ());
	}
      return;
    }

  wg = lookup_widget (window_main_, "entry_keyboard");
  keyb_set_name (gtk_entry_get_text (GTK_ENTRY (wg)));
  keyb_get_layout ();
  gtk_widget_show (window_main_);

  window_keyboard_ = NULL;
}


gboolean
on_window_keyboard_delete_event (GtkWidget * widget,
				 GdkEvent * event, gpointer user_data)
{
  gtk_widget_destroy (widget);
  return FALSE;
}


/**********************************************************************
 * Changing the main keyboard layout
 */
void
on_entry_kb_main_insert_text (GtkEditable * editable,
			      gchar * new_text,
			      gint new_text_length,
			      gpointer position, gpointer user_data)
{
  gchar *tmp_str;

  if (callbacks_shield == TRUE)
    return;

  tmp_str = g_strndup (new_text, new_text_length);
  keyb_set_name (tmp_str);
  g_free (tmp_str);
  keyb_get_layout ();
  keyb_fill_layout_list (2);
  keyb_fill_layout_list (3);
  keyb_update_virtual_layout ();
}


/**********************************************************************
 * Changing the user keyboard layout
 */
void
on_entry_kb_user_insert_text (GtkEditable * editable,
			      gchar * new_text,
			      gint new_text_length,
			      gpointer position, gpointer user_data)
{
  gchar *tmp_str;

  if (callbacks_shield == TRUE)
    return;

  tmp_str = g_strndup (new_text, new_text_length);
  keyb_set_name (tmp_str);
  g_free (tmp_str);
  keyb_get_layout ();
  keyb_fill_layout_list (3);
  keyb_fill_layout_list (1);
  keyb_update_virtual_layout ();
}


/**********************************************************************
 * Saving the new keyboard layout
 */
void
on_button_kb_save_clicked (GtkButton * button, gpointer user_data)
{
  gchar *tmp_path;
  FILE *fh;
  GtkWidget *wg;
  GtkWidget *dialog_confirm;

  wg = lookup_widget (GTK_WIDGET (button), "entry_kb_edit");
  keyb_set_name (gtk_entry_get_text (GTK_ENTRY (wg)));

  if (keyb_get_name ()[0] == '\0')
    keyb_set_name (".tmp");
  else
    {
      tmp_path =
	g_strconcat (main_get_user_dir (), keyb_get_name (), ".kbd", NULL);
      fh = (FILE*) g_fopen (tmp_path, "r");
      g_free (tmp_path);
      if (fh)
	{
	  fclose (fh);
	  dialog_confirm = create_dialog_confirm ();
	  gtk_label_set_text (GTK_LABEL (lookup_widget
					 (dialog_confirm,
					  "label_confirm_action")),
			      "OVERWRITE");
	  gtk_widget_show (dialog_confirm);
	  return;
	}
    }
  keyb_save_new_layout ();
  keyb_fill_layout_list (0);
}


/**********************************************************************
 * Deleting some user created layout
 */
void
on_button_kb_remove_clicked (GtkButton * button, gpointer user_data)
{
  gchar *tmp_path = NULL;
  gchar *tmp_keyb = NULL;
  GtkWidget *dialog_confirm;
  GtkWidget *wg;
  FILE *fh;

  wg = lookup_widget (window_keyboard_, "entry_kb_edit");
  tmp_keyb = gtk_editable_get_chars (GTK_EDITABLE (wg), 0, -1);
  tmp_path = g_strconcat (main_get_user_dir (), tmp_keyb, ".kbd", NULL);
  fh = (FILE*) g_fopen (tmp_path, "r");
  g_free (tmp_path);
  if (tmp_keyb[0] == '\0' || fh == NULL)
    {
      gdk_beep ();
      g_message ("no valid keyboard to remove...");
      g_free (tmp_keyb);
      return;
    }
  fclose (fh);
  g_free (tmp_keyb);

  dialog_confirm = create_dialog_confirm ();
  gtk_label_set_text (GTK_LABEL
		      (lookup_widget
		       (dialog_confirm, "label_confirm_action")), "REMOVE");
  gtk_widget_show (dialog_confirm);
}


/**********************************************************************
 * Selecting a keyboard layout
 */
void
on_button_keyboard_select_clicked (GtkButton * button, gpointer user_data)
{
  GtkWidget *dialog;
  GtkWidget *wg;

  if (keyb_get_modified_status () == TRUE)
    {
      wg = lookup_widget (GTK_WIDGET (button), "notebook_kb");
      gtk_notebook_set_page (GTK_NOTEBOOK (wg), 1);
      gdk_beep ();
      dialog = create_popup_message_save ();
      gtk_widget_show (dialog);
      return;
    }

  g_key_file_set_string (main_preferences_get (), "tutor", "keyboard", keyb_get_name ());

  wg = lookup_widget (window_main_, "entry_keyboard");
  gtk_entry_set_text (GTK_ENTRY (wg), keyb_get_name ());
  gtk_widget_destroy (gtk_widget_get_toplevel (GTK_WIDGET (button)));
}


/**********************************************************************
 * Closing the keyboard editor window
 */
void
on_button_keyboard_cancel_clicked (GtkButton * button, gpointer user_data)
{
  gtk_widget_destroy (gtk_widget_get_toplevel (GTK_WIDGET (button)));
}


/**********************************************************************
 * Allowing or not to select "load" tab on notebook
 */
void
on_notebook_kb_switch_page (GtkNotebook * notebook,
			    GtkNotebookPage * page,
			    guint page_num, gpointer user_data)
{
  if (keyb_get_modified_status () == FALSE)
    return;

  if (page_num == 1)
    return;

  g_timeout_add (150, &keyb_force_edit_tab, NULL);
  gdk_beep ();
}


/**********************************************************************
 * Editing the keyboard.
 */
/**********************************************************************
 * Shift keys
 */
void
on_toggle_shift1_toggled (GtkToggleButton * togglebutton, gpointer user_data)
{
  gboolean tog_state;
  GtkWidget *tog;

  tog = lookup_widget (GTK_WIDGET (togglebutton), "toggle_shift2");
  tog_state = gtk_toggle_button_get_active (togglebutton);
  gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (tog), tog_state);
  keyb_update_virtual_layout ();

  if (window_tutor_ != NULL)
    keyb_edit_key (GTK_BUTTON (togglebutton), "hands_0.xpm");
}


void
on_toggle_shift2_toggled (GtkToggleButton * togglebutton, gpointer user_data)
{
  gboolean tog_state;
  GtkWidget *tog;

  tog = lookup_widget (GTK_WIDGET (togglebutton), "toggle_shift1");
  tog_state = gtk_toggle_button_get_active (togglebutton);
  gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (tog), tog_state);
  keyb_update_virtual_layout ();

  if (window_tutor_ != NULL)
    keyb_edit_key (GTK_BUTTON (togglebutton), "hands_0.xpm");
}


/**********************************************************************
 * Space bar
 */
void
on_but_space_clicked (GtkButton * button, gpointer user_data)
{
  keyb_edit_key (button, "hands_5.xpm");
}


/**********************************************************************
 * All the other keys
 */
void
on_but1_clicked (GtkButton * button, gpointer user_data)
{
  keyb_edit_key (button, "hands_1.xpm");
}

void
on_but2_clicked (GtkButton * button, gpointer user_data)
{
  keyb_edit_key (button, "hands_1.xpm");
}

void
on_but3_clicked (GtkButton * button, gpointer user_data)
{
  keyb_edit_key (button, "hands_2.xpm");
}

void
on_but4_clicked (GtkButton * button, gpointer user_data)
{
  keyb_edit_key (button, "hands_3.xpm");
}

void
on_but5_clicked (GtkButton * button, gpointer user_data)
{
  keyb_edit_key (button, "hands_4.xpm");
}

void
on_but6_clicked (GtkButton * button, gpointer user_data)
{
  keyb_edit_key (button, "hands_4.xpm");
}

void
on_but7_clicked (GtkButton * button, gpointer user_data)
{
  keyb_edit_key (button, "hands_6.xpm");
}

void
on_but8_clicked (GtkButton * button, gpointer user_data)
{
  keyb_edit_key (button, "hands_6.xpm");
}

void
on_but9_clicked (GtkButton * button, gpointer user_data)
{
  keyb_edit_key (button, "hands_7.xpm");
}

void
on_but10_clicked (GtkButton * button, gpointer user_data)
{
  keyb_edit_key (button, "hands_8.xpm");
}

void
on_but11_clicked (GtkButton * button, gpointer user_data)
{
  keyb_edit_key (button, "hands_9.xpm");
}

void
on_but12_clicked (GtkButton * button, gpointer user_data)
{
  keyb_edit_key (button, "hands_9.xpm");
}

void
on_but13_clicked (GtkButton * button, gpointer user_data)
{
  keyb_edit_key (button, "hands_9.xpm");
}

void
on_but14_clicked (GtkButton * button, gpointer user_data)
{
  keyb_edit_key (button, "hands_9.xpm");
}

void
on_but_1_clicked (GtkButton * button, gpointer user_data)
{
  keyb_edit_key (button, "hands_1.xpm");
}

void
on_but_2_clicked (GtkButton * button, gpointer user_data)
{
  keyb_edit_key (button, "hands_2.xpm");
}

void
on_but_3_clicked (GtkButton * button, gpointer user_data)
{
  keyb_edit_key (button, "hands_3.xpm");
}

void
on_but_4_clicked (GtkButton * button, gpointer user_data)
{
  keyb_edit_key (button, "hands_4.xpm");
}

void
on_but_5_clicked (GtkButton * button, gpointer user_data)
{
  keyb_edit_key (button, "hands_4.xpm");
}

void
on_but_6_clicked (GtkButton * button, gpointer user_data)
{
  keyb_edit_key (button, "hands_6.xpm");
}

void
on_but_7_clicked (GtkButton * button, gpointer user_data)
{
  keyb_edit_key (button, "hands_6.xpm");
}

void
on_but_8_clicked (GtkButton * button, gpointer user_data)
{
  keyb_edit_key (button, "hands_7.xpm");
}

void
on_but_9_clicked (GtkButton * button, gpointer user_data)
{
  keyb_edit_key (button, "hands_8.xpm");
}

void
on_but_10_clicked (GtkButton * button, gpointer user_data)
{
  keyb_edit_key (button, "hands_9.xpm");
}

void
on_but_11_clicked (GtkButton * button, gpointer user_data)
{
  keyb_edit_key (button, "hands_9.xpm");
}

void
on_but_12_clicked (GtkButton * button, gpointer user_data)
{
  keyb_edit_key (button, "hands_9.xpm");
}

void
on_but__1_clicked (GtkButton * button, gpointer user_data)
{
  keyb_edit_key (button, "hands_1.xpm");
}

void
on_but__2_clicked (GtkButton * button, gpointer user_data)
{
  keyb_edit_key (button, "hands_2.xpm");
}

void
on_but__3_clicked (GtkButton * button, gpointer user_data)
{
  keyb_edit_key (button, "hands_3.xpm");
}

void
on_but__4_clicked (GtkButton * button, gpointer user_data)
{
  keyb_edit_key (button, "hands_4.xpm");
}

void
on_but__5_clicked (GtkButton * button, gpointer user_data)
{
  keyb_edit_key (button, "hands_4.xpm");
}

void
on_but__6_clicked (GtkButton * button, gpointer user_data)
{
  keyb_edit_key (button, "hands_6.xpm");
}

void
on_but__7_clicked (GtkButton * button, gpointer user_data)
{
  keyb_edit_key (button, "hands_6.xpm");
}

void
on_but__8_clicked (GtkButton * button, gpointer user_data)
{
  keyb_edit_key (button, "hands_7.xpm");
}

void
on_but__9_clicked (GtkButton * button, gpointer user_data)
{
  keyb_edit_key (button, "hands_8.xpm");
}

void
on_but__10_clicked (GtkButton * button, gpointer user_data)
{
  keyb_edit_key (button, "hands_9.xpm");
}

void
on_but__11_clicked (GtkButton * button, gpointer user_data)
{
  keyb_edit_key (button, "hands_9.xpm");
}

void
on_but__12_clicked (GtkButton * button, gpointer user_data)
{
  keyb_edit_key (button, "hands_9.xpm");
}

void
on_but___1_clicked (GtkButton * button, gpointer user_data)
{
  keyb_edit_key (button, "hands_1.xpm");
}

void
on_but___2_clicked (GtkButton * button, gpointer user_data)
{
  keyb_edit_key (button, "hands_1.xpm");
}

void
on_but___3_clicked (GtkButton * button, gpointer user_data)
{
  keyb_edit_key (button, "hands_2.xpm");
}

void
on_but___4_clicked (GtkButton * button, gpointer user_data)
{
  keyb_edit_key (button, "hands_3.xpm");
}

void
on_but___5_clicked (GtkButton * button, gpointer user_data)
{
  keyb_edit_key (button, "hands_4.xpm");
}

void
on_but___6_clicked (GtkButton * button, gpointer user_data)
{
  keyb_edit_key (button, "hands_4.xpm");
}

void
on_but___7_clicked (GtkButton * button, gpointer user_data)
{
  keyb_edit_key (button, "hands_6.xpm");
}

void
on_but___8_clicked (GtkButton * button, gpointer user_data)
{
  keyb_edit_key (button, "hands_6.xpm");
}

void
on_but___9_clicked (GtkButton * button, gpointer user_data)
{
  keyb_edit_key (button, "hands_7.xpm");
}

void
on_but___10_clicked (GtkButton * button, gpointer user_data)
{
  keyb_edit_key (button, "hands_8.xpm");
}

void
on_but___11_clicked (GtkButton * button, gpointer user_data)
{
  keyb_edit_key (button, "hands_9.xpm");
}

void
on_but___12_clicked (GtkButton * button, gpointer user_data)
{
  keyb_edit_key (button, "hands_9.xpm");
}


/**********************************************************************
 * Closing the keyboard editor window
 */
void
on_button_keyboard_close_clicked (GtkButton * button, gpointer user_data)
{
  gtk_widget_destroy (gtk_widget_get_toplevel (GTK_WIDGET (button)));
}


/**********************************************************************
 * Editing a key of the keyboard
 */
void
on_entry_popup_key_insert_text (GtkEditable * editable,
				gchar * new_text,
				gint new_text_length,
				gpointer position, gpointer user_data)
{
  if (callbacks_shield)
    return;

  keyb_change_key (g_utf8_get_char_validated (new_text, new_text_length));
  gtk_widget_hide (gtk_widget_get_toplevel (GTK_WIDGET (editable)));
}


/*******************************************************************************
 * Cancel the edition of a key
 */
void
on_button_popup_cancel_clicked (GtkButton * button, gpointer user_data)
{
  gtk_widget_hide (gtk_widget_get_toplevel (GTK_WIDGET (button)));
}


/*******************************************************************************
 * Closing the warning to save the created keyboard layout
 */
void
on_button_msg_save_clicked (GtkButton * button, gpointer user_data)
{
  gtk_widget_destroy (gtk_widget_get_toplevel (GTK_WIDGET (button)));
}


/**********************************************************************
 * Part 3
 * Callbacks from the shared tutor window
 */
/*******************************************************************************
 * Tutor events
 */
void
on_window_tutor_destroy (GtkObject * object, gpointer user_data)
{
  if (GTK_IS_WIDGET (window_hints_))
    gtk_widget_destroy (window_hints_);

  if (GTK_IS_WIDGET (filechooser_tutor))
    gtk_widget_destroy (filechooser_tutor);

  if (GTK_IS_WIDGET (fontselectiondialog_tutor))
	  gtk_widget_destroy (fontselectiondialog_tutor);

  gtk_widget_show (window_main_);
  window_tutor_ = NULL;
}


void
on_text_tutor_realize (GtkWidget * widget, gpointer user_data)
{
  gboolean beep;
  gchar *tmp_font;
  GtkWidget *wg;
  GtkTextBuffer *buf;
  PangoFontDescription *font_desc;
  static GdkColor color;

  /*
   * Colors of text on the tutor window
   */
  buf = gtk_text_view_get_buffer (GTK_TEXT_VIEW (widget));

  gtk_text_buffer_create_tag (buf, "char_untouched",
			      "background", TUTOR_GRAY_BLUE, NULL);

  gtk_text_buffer_create_tag (buf, "char_wrong",
			      "background", TUTOR_RED, NULL);

  gtk_text_buffer_create_tag (buf, "char_correct",
			      "background", TUTOR_GREEN, NULL);

  gtk_text_buffer_create_tag (buf, "char_retouched",
			      "background", TUTOR_YELLOW_GREEN, NULL);

  gtk_text_buffer_create_tag (buf, "cursor_blink",
			      "background", TUTOR_BLACK,
			      "foreground", TUTOR_WHITE, NULL);

  tmp_font = g_key_file_get_string (main_preferences_get (), "tutor", "lesson_font", NULL);
  if (tmp_font == NULL)
  {
	  tmp_font = g_strdup (LESSON_FONT);
	  g_key_file_set_string (main_preferences_get (), "tutor", "lesson_font", tmp_font);
  }
  gtk_text_buffer_create_tag (buf, "lesson_font", "font", tmp_font, NULL);
  g_free (tmp_font);

  /* Change default font throughout the widget */
  font_desc = pango_font_description_from_string (NORMAL_FONT);
  gtk_widget_modify_font (widget, font_desc);
  pango_font_description_free (font_desc);

  /* Change default text color throughout the widget */
  gdk_color_parse (TUTOR_BLACK, &color);
  gtk_widget_modify_text (widget, GTK_STATE_INSENSITIVE, &color);

  /* Change default background color throughout the widget */
  gdk_color_parse (TUTOR_CREAM, &color);
  gtk_widget_modify_base (widget, GTK_STATE_INSENSITIVE, &color);

  /*
   * Turns on/off the beeps according to last time
   */
  if (g_key_file_has_key (main_preferences_get (), "tutor", "tutor_beep", NULL))
      beep = g_key_file_get_boolean (main_preferences_get (), "tutor", "tutor_beep", NULL);
  else
      beep = TRUE;

  wg = lookup_widget (widget, "checkbutton_beep");
  gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (wg), beep);
  g_key_file_set_boolean (main_preferences_get (), "tutor", "tutor_beep", beep);
}

/**********************************************************************
 * Selecting the basic course lessons
 */
void
on_spinbutton_lesson_changed (GtkEditable * editable, gpointer user_data)
{
  gint tmp_int;
  gchar *tmp_str;

  if (callbacks_shield)
    return;

  tmp_str = gtk_editable_get_chars (editable, 0, -1);
  tmp_int = g_key_file_get_integer (main_preferences_get (), "tutor", "basic_lesson", NULL);
  basic_set_lesson (strtol (tmp_str, NULL, 0));
  g_key_file_set_integer (main_preferences_get (), "tutor", "basic_lesson", tmp_int);
  g_free (tmp_str);
  basic_init_char_set ();
  tutor_set_query (QUERY_INTRO);
  tutor_process_touch (L'\0');
}


/*
 * Reverting to the right lesson
 */
void
on_button_revert_lesson_clicked (GtkButton * button, gpointer user_data)
{
  gint tmp_int;

  tmp_int = g_key_file_get_integer (main_preferences_get (), "tutor", "basic_lesson", NULL);
  basic_set_lesson (0);
  g_key_file_set_integer (main_preferences_get (), "tutor", "basic_lesson", tmp_int);
  tutor_init (TT_BASIC);
}


/*
 * Resetting the basic course
 */
void
on_button_reset_lesson_clicked (GtkButton * button, gpointer user_data)
{
  GtkWidget *dialog_confirm;

  dialog_confirm = create_dialog_confirm ();
  gtk_label_set_text (GTK_LABEL
		      (lookup_widget
		       (dialog_confirm, "label_confirm_action")), "RESET");
  gtk_widget_show (dialog_confirm);
}


/**********************************************************************
 * Other exercises popup window
 */
void
on_button_tutor_other_clicked (GtkButton * button, gpointer user_data)
{
  popup_other_ = create_popup_other ();
  callbacks_shield = TRUE;
  gtk_widget_show (popup_other_);
}


/**********************************************************************
 * Progress charts window
 */
void
on_button_tutor_stat_clicked (GtkButton * button, gpointer user_data)
{
  GtkLabel *label_stat;

  window_stat_ = create_window_stat ();
  label_stat = GTK_LABEL (lookup_widget (window_stat_, "label_stat"));

  switch (tutor_get_type ())
    {
    case TT_BASIC:
      gtk_label_set_text (label_stat, _("basic course"));
      gtk_widget_show (lookup_widget (window_stat_, "label_stat_lesson"));
      gtk_widget_show (lookup_widget (window_stat_, "label_stat_lesson_n"));
      break;
    case TT_ADAPT:
      gtk_label_set_text (label_stat, _("adaptability exercises"));
      break;
    case TT_VELO:
      gtk_label_set_text (label_stat, _("velocity exercises"));
      break;
    case TT_FLUID:
      gtk_label_set_text (label_stat, _("fluidness exercises"));
      gtk_widget_show (lookup_widget (window_stat_, "toggle_stat_fluid"));
    }

  gtk_widget_show (window_stat_);
}


/**********************************************************************
 * Show the helping virtual keyboard
 */
void
on_button_tutor_show_keyb_clicked (GtkButton * button, gpointer user_data)
{
  if (window_hints_ != NULL)
    gtk_widget_destroy (window_hints_);

  window_keyboard_ = create_window_keyboard ();
  gtk_widget_hide (lookup_widget (window_keyboard_, "vbox_kb_left"));
  gtk_widget_show (lookup_widget (window_keyboard_, "hbox_kb_tutor"));
  gtk_widget_set_sensitive (lookup_widget (window_keyboard_, "but_space"),
			    TRUE);
  gtk_window_set_title (GTK_WINDOW (window_keyboard_), _("Virtual Keyboard"));
  gtk_label_set_text (GTK_LABEL
		      (lookup_widget (window_keyboard_, "label_virtual_kb")),
		      _("Relation between the fingers and the keys"));

  keyb_update_virtual_layout ();
  hints_init ();
  gtk_widget_show (window_keyboard_);
  gtk_widget_hide (window_tutor_);
}


/**********************************************************************
 * Go to the exercise's introduction
 */
void
on_button_tutor_intro_clicked (GtkButton * button, gpointer user_data)
{
  tutor_set_query (QUERY_INTRO);
  tutor_update ();
}


/*******************************************************************************
 * Returning to the main menu
 */
void
on_button_tutor_close_clicked (GtkButton * button, gpointer user_data)
{
  gtk_widget_destroy (gtk_widget_get_toplevel (GTK_WIDGET (button)));
}

/*******************************************************************************
 * Choosing the font for the exercises
 */
void
on_button_tutor_font_clicked           (GtkButton       *button,
                                        gpointer         user_data)
{
  if (fontselectiondialog_tutor == NULL)
    fontselectiondialog_tutor = create_fontselectiondialog_tutor ();

  gtk_widget_show (fontselectiondialog_tutor);
}

void
on_fontselectiondialog_tutor_response  (GtkDialog       *dialog,
                                        gint             response_id,
                                        gpointer         user_data)
{
  GtkWidget *wg;
  GtkTextBuffer *buf;
  GtkTextTagTable *ttab;
  GtkTextIter start;
  GtkTextIter end;
  gchar *tmp_font;

  if (response_id == GTK_RESPONSE_CANCEL)
	tmp_font = g_key_file_get_string (main_preferences_get (), "tutor", "lesson_font", NULL);
  else
  	tmp_font = g_strdup (gtk_font_selection_dialog_get_font_name (GTK_FONT_SELECTION_DIALOG (dialog)));

  if (tmp_font == NULL)
	  tmp_font = g_strdup (LESSON_FONT);

  if (response_id == GTK_RESPONSE_OK)
	g_key_file_set_string (main_preferences_get (), "tutor", "lesson_font", tmp_font);

  wg = lookup_widget (window_tutor_, "text_tutor");
  buf = gtk_text_view_get_buffer (GTK_TEXT_VIEW (wg));
  ttab = gtk_text_buffer_get_tag_table (buf);
  gtk_text_buffer_get_bounds (buf, &start, &end);

  gtk_text_tag_table_remove (ttab, gtk_text_tag_table_lookup (ttab, "lesson_font"));
  gtk_text_buffer_create_tag (buf, "lesson_font", "font", tmp_font, NULL);
  gtk_text_buffer_apply_tag_by_name (buf, "lesson_font", &start, &end);

  g_free (tmp_font);

  if (response_id != GTK_RESPONSE_APPLY)
	gtk_widget_hide (fontselectiondialog_tutor);
}

void
on_fontselectiondialog_tutor_destroy   (GtkObject       *object,
                                        gpointer         user_data)
{
  on_fontselectiondialog_tutor_response (GTK_DIALOG (object), GTK_RESPONSE_CANCEL, NULL);
  fontselectiondialog_tutor = NULL;
}


/*******************************************************************************
 * Enabling sound or not
 */
void
on_checkbutton_beep_toggled            (GtkToggleButton *togglebutton,
                                        gpointer         user_data)
{
  gboolean beep;

  beep = gtk_toggle_button_get_active (togglebutton);
  g_key_file_set_boolean (main_preferences_get (), "tutor", "tutor_beep", beep);
}


/*******************************************************************************
 * Managing the keyboard/messages entry of the tutor window
 */
gboolean
on_entry_mesg_focus_out_event (GtkWidget * widget,
			       GdkEventFocus * event, gpointer user_data)
{
  if (callbacks_shield)
    return (FALSE);

  gtk_widget_grab_focus (widget);
  return (FALSE);
}


void
on_entry_mesg_grab_focus (GtkWidget * widget, gpointer user_data)
{
  if (callbacks_shield)
    return;

  gtk_editable_select_region (GTK_EDITABLE (widget), 0, 0);
  gtk_editable_set_position (GTK_EDITABLE (widget), -1);
}


void
on_entry_mesg_activate (GtkEntry * entry, gpointer user_data)
{
  gchar *tmp1;
  gchar *tmp2;

  if (callbacks_shield)
    return;

  tmp1 = g_strdup (gtk_entry_get_text (entry));
  tmp2 = g_strconcat (tmp1, keyb_get_utf8_paragraph_symbol (), NULL);
  callbacks_shield = TRUE;
  gtk_entry_set_text (entry, tmp2);
  callbacks_shield = FALSE;
  g_free (tmp1);
  g_free (tmp2);

  gtk_editable_set_position (GTK_EDITABLE (entry), -1);

  tutor_process_touch (UPSYM);
}


void
on_entry_mesg_delete_text (GtkEditable * editable,
			   gint start_pos, gint end_pos, gpointer user_data)
{
  gint i;

  if (callbacks_shield)
    return;

  for (i=end_pos; i > start_pos; i--)
    tutor_process_touch (L'\b');
}


void
on_entry_mesg_insert_text (GtkEditable * editable,
			   gchar * new_text,
			   gint new_text_length,
			   gpointer position, gpointer user_data)
{
  if (callbacks_shield)
    return;

  tutor_process_touch (g_utf8_get_char_validated (new_text, new_text_length));
}


/**********************************************************************
 * Controlling the restart action during the exercises
 */

void
on_button_tutor_restart_clicked (GtkButton * button, gpointer user_data)
{
  if (tutor_get_query () == QUERY_END)
    return;

  tutor_set_query (QUERY_INTRO);
  tutor_process_touch (L'\0');
}


/**********************************************************************
 * Part 4
 * Callbacks from the "other exercises" popup window
 */
/**********************************************************************
 * Popup management
 */
gboolean
on_popup_other_enter_notify_event (GtkWidget * widget,
				   GdkEventCrossing * event,
				   gpointer user_data)
{
  if (callbacks_shield)
    {
      callbacks_shield = FALSE;
      return FALSE;
    }

  gtk_widget_destroy (widget);
  return FALSE;
}


gboolean
on_popup_other_leave_notify_event (GtkWidget * widget,
				   GdkEventCrossing * event,
				   gpointer user_data)
{
  if (callbacks_shield)
    return FALSE;

  gtk_widget_destroy (widget);
  return FALSE;
}


void
on_treeview_other_realize (GtkWidget * widget, gpointer user_data)
{
  GtkListStore *list;
  GtkCellRenderer *renderer;
  GtkTreeViewColumn *column;

  list = gtk_list_store_new (1, G_TYPE_STRING);
  if (tutor_get_type () == TT_VELO)
    tutor_load_list_other (".words", list);
  else if (tutor_get_type () == TT_FLUID)
    tutor_load_list_other (".paragraphs", list);

  gtk_tree_view_set_model (GTK_TREE_VIEW (widget), GTK_TREE_MODEL (list));

  renderer = gtk_cell_renderer_text_new ();
  column = gtk_tree_view_column_new_with_attributes (":-)",
						     renderer, "text", 0,
						     NULL);
  gtk_tree_view_append_column (GTK_TREE_VIEW (widget), column);
}


void
on_treeview_other_destroy (GtkObject * object, gpointer user_data)
{
  gchar *tmp_name;
  GtkTreeSelection *sel;
  GtkTreeModel *store;
  GtkTreeIter iter;

  sel = gtk_tree_view_get_selection (GTK_TREE_VIEW (object));
  if (gtk_tree_selection_get_selected (sel, &store, &iter) == FALSE)
    return;

  gtk_tree_model_get (store, &iter, 0, &tmp_name, -1);

  if (tutor_get_type () == TT_VELO)
    velo_init_dict (tmp_name);
  else if (tutor_get_type () == TT_FLUID)
    fluid_init_paragraph_list (tmp_name);

  tutor_set_query (QUERY_INTRO);
  tutor_process_touch ('\0');
}


void
on_button_other_apply_clicked (GtkButton * button, gpointer user_data)
{
  gtk_widget_destroy (gtk_widget_get_toplevel (GTK_WIDGET (button)));
}


/**********************************************************************
 * Get a new file for the exercises
 */
void
on_button_other_new_clicked (GtkButton * button, gpointer user_data)
{
  if (filechooser_tutor == NULL)
    filechooser_tutor = create_filechooser_tutor ();

  gtk_widget_destroy (gtk_widget_get_toplevel (GTK_WIDGET (button)));
  gtk_widget_show (filechooser_tutor);
}


/**********************************************************************
 * Remove some user exercise
 */
void
on_button_other_remove_clicked (GtkButton * button, gpointer user_data)
{
  gchar *tmp_str;
  gchar *tmp_name;
  GtkWidget *wg;
  GtkTreeSelection *sel;
  GtkTreeModel *store;
  GtkTreeIter iter;

  wg = lookup_widget (GTK_WIDGET (button), "treeview_other");
  sel = gtk_tree_view_get_selection (GTK_TREE_VIEW (wg));
  if (gtk_tree_selection_get_selected (sel, &store, &iter) == FALSE)
    {
      gdk_beep ();
      return;
    }

  gtk_tree_model_get (store, &iter, 0, &tmp_str, -1);

  if (strcmp (tmp_str, _("--> Default")) == 0)
    {
      gdk_beep ();
      return;
    }

  tmp_name = g_strconcat (main_get_user_dir (), tmp_str, NULL);
  if (tutor_get_type () == TT_VELO)
    tmp_name = g_strconcat (tmp_name, ".words", NULL);
  else if (tutor_get_type () == TT_FLUID)
    tmp_name = g_strconcat (tmp_name, ".paragraphs", NULL);
  g_unlink (tmp_name);
  g_free (tmp_name);

  gtk_list_store_remove (GTK_LIST_STORE (store), &iter);
}


/*******************************************************************************
 * Use chosen file to create a new exercise
 */
void
on_filechooser_tutor_response (GtkDialog * dialog,
			       gint response_id, gpointer user_data)
{
  gchar *tmp_path;

  gtk_widget_hide (filechooser_tutor);

  if (response_id != GTK_RESPONSE_OK)
    {
      gtk_widget_hide (filechooser_tutor);
      return;
    }

  tmp_path = gtk_file_chooser_get_filename (GTK_FILE_CHOOSER (dialog));
  if (tutor_get_type () == TT_VELO)
    velo_create_dict (tmp_path, TRUE);
  else if (tutor_get_type () == TT_FLUID)
    fluid_copy_text_file (tmp_path);
  g_free (tmp_path);
}


void
on_filechooser_tutor_destroy (GtkObject * object, gpointer user_data)
{
  filechooser_tutor = NULL;
}


/**********************************************************************
 * Part 5
 * Callbacks from the shared progress window
 */
/**********************************************************************
 * Initialization of the chart
 */
void
on_frame_stat_realize (GtkWidget * widget, gpointer user_data)
{
  plot_initialize (widget);
}


/**********************************************************************
 * Show accuracy chart
 */
void
on_toggle_stat_accur_toggled (GtkToggleButton * togglebutton,
			      gpointer user_data)
{
  GtkWidget *wg;

  wg = lookup_widget (GTK_WIDGET (togglebutton), "toggle_stat_velo");
  if (!gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (wg)))
    {
      wg = lookup_widget (GTK_WIDGET (togglebutton), "toggle_stat_fluid");
      if (!gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (wg)))
	{
	  if (!gtk_toggle_button_get_active (togglebutton))
	    gtk_toggle_button_set_active (togglebutton, TRUE);
	  return;
	}
      else if (!gtk_toggle_button_get_active (togglebutton))
	return;
      else
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (wg), FALSE);
    }
  else if (!gtk_toggle_button_get_active (togglebutton))
    return;
  else
    gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (wg), FALSE);

  plot_draw_chart (1);
}


/**********************************************************************
 * Show velocity chart
 */
void
on_toggle_stat_velo_toggled (GtkToggleButton * togglebutton,
			     gpointer user_data)
{
  GtkWidget *wg;

  wg = lookup_widget (GTK_WIDGET (togglebutton), "toggle_stat_accur");
  if (!gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (wg)))
    {
      wg = lookup_widget (GTK_WIDGET (togglebutton), "toggle_stat_fluid");
      if (!gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (wg)))
	{
	  if (!gtk_toggle_button_get_active (togglebutton))
	    gtk_toggle_button_set_active (togglebutton, TRUE);
	  return;
	}
      else if (!gtk_toggle_button_get_active (togglebutton))
	return;
      else
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (wg), FALSE);
    }
  else if (!gtk_toggle_button_get_active (togglebutton))
    return;
  else
    gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (wg), FALSE);

  plot_draw_chart (2);
}


/**********************************************************************
 * Show fluidness chart
 */
void
on_toggle_stat_fluid_toggled (GtkToggleButton * togglebutton,
			      gpointer user_data)
{
  GtkWidget *wg;

  wg = lookup_widget (GTK_WIDGET (togglebutton), "toggle_stat_accur");
  if (!gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (wg)))
    {
      wg = lookup_widget (GTK_WIDGET (togglebutton), "toggle_stat_velo");
      if (!gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (wg)))
	{
	  if (!gtk_toggle_button_get_active (togglebutton))
	    gtk_toggle_button_set_active (togglebutton, TRUE);
	  return;
	}
      else if (!gtk_toggle_button_get_active (togglebutton))
	return;
      else
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (wg), FALSE);
    }
  else if (!gtk_toggle_button_get_active (togglebutton))
    return;
  else
    gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (wg), FALSE);

  plot_draw_chart (3);
}


/*
 * Closing the progress window
 */
void
on_button_stat_close_clicked (GtkButton * button, gpointer user_data)
{
  gtk_widget_destroy (gtk_widget_get_toplevel (GTK_WIDGET (button)));
}


/**********************************************************************
 * Confirm-dialog shared tasks
 */
void
on_dialog_confirm_realize (GtkWidget * widget, gpointer user_data)
{
  gint nothing_to_do = 0;
  gchar *action;
  GtkLabel *wg_label;

  wg_label = GTK_LABEL (lookup_widget (widget, "label_confirm_action"));
  action = (gchar *) gtk_label_get_text (wg_label);

  wg_label = GTK_LABEL (lookup_widget (widget, "label_confirm"));

  if (strcmp (action, "CHANGE") == 0)
    nothing_to_do = 0;		/* Default values already applied on creation */

  else if (strcmp (action, "OVERWRITE") == 0)
    {
      gtk_window_set_title (GTK_WINDOW (widget), _("Overwrite user layout"));
      gtk_label_set_text (wg_label,
			  _
			  ("This will OVERWRITE an existent keyboard layout."));
    }

  else if (strcmp (action, "REMOVE") == 0)
    {
      gtk_window_set_title (GTK_WINDOW (widget), _("Remove user layout"));
      gtk_label_set_text (wg_label,
			  _("This will REMOVE an existent keyboard layout."));
    }

  else if (strcmp (action, "RESET") == 0)
    {
      gtk_window_set_title (GTK_WINDOW (widget), _("Reset lesson"));
      gtk_label_set_text (wg_label,
			  _("Confirming this, you will have to accomplish\n"
			    "all the lessons again, since the first one,\n"
			    "because its count will be RESET."));
    }

  else
    {
      g_warning ("No valid action for this confirm-dialog: %s", action);
      gtk_widget_destroy (widget);
    }

}


void
on_button_confirm_yes_clicked (GtkButton * button, gpointer user_data)
{
  gchar *action;
  GtkWidget *wg;

  wg = lookup_widget (GTK_WIDGET (button), "label_confirm_action");
  action = (gchar *) gtk_label_get_text (GTK_LABEL (wg));

  if (strcmp (action, "CHANGE") == 0)
    trans_change_language ();

  else if (strcmp (action, "OVERWRITE") == 0)
    {
      keyb_save_new_layout ();
      keyb_fill_layout_list (0);
    }

  else if (strcmp (action, "REMOVE") == 0)
    keyb_remove_user_layout ();

  else if (strcmp (action, "RESET") == 0)
    {
      basic_set_lesson (0);
      g_key_file_remove_key (main_preferences_get (), "tutor", "basic_lesson", NULL);
      tutor_init (TT_BASIC);
    }

  else
    g_warning ("No valid action selected for 'yes' confirm-button: %s",
	       action);

  gtk_widget_destroy (gtk_widget_get_toplevel (GTK_WIDGET (button)));
}


void
on_button_confirm_no_clicked (GtkButton * button, gpointer user_data)
{
  gchar *action;
  GtkWidget *wg;

  wg = lookup_widget (GTK_WIDGET (button), "label_confirm_action");
  action = (gchar *) gtk_label_get_text (GTK_LABEL (wg));

  if (strcmp (action, "OVERWRITE") == 0)
    {
      if (window_keyboard_ == NULL)
	return;
      wg = lookup_widget (window_keyboard_, "label_kb_name");
      keyb_set_name (gtk_label_get_text (GTK_LABEL (wg)));
    }

  gtk_widget_destroy (gtk_widget_get_toplevel (GTK_WIDGET (button)));
}

/*******************************************************************************
 * Window hints management on destruction
 */
void
on_window_hints_destroy (GtkObject * object, gpointer user_data)
{
  window_hints_ = NULL;
}



