/**************************************************************************/
/*  Klavaro - a flexible touch typing tutor                               */
/*  Copyright (C) 2005 - 2008  Felipe Castro                              */
/*                                                                        */
/*  This program is free software, licensed under the terms of the GNU    */
/*  General Public License as published by the Free Software Foundation,  */
/*  which is also included in this package, in a file named "COPYING".    */
/**************************************************************************/

/*
 * Velocity exercise
 */
#include <string.h>
#include <stdlib.h>
#include <stdio.h>
#include <glib.h>
#include <glib/gstdio.h>
#include <gtk/gtk.h>

#include "support.h"
#include "main.h"
#include "translation.h"
#include "tutor.h"
#include "velocity.h"

/*
 * Global variables
 */
extern GtkWidget *window_tutor_;

GList *word_list;

struct
{
	GList *list;
	gint len;
	gchar name[21];
} dict;

/*******************************************************************************
 * Interface functions
 */
gchar *
velo_get_dict_name ()
{
	return (dict.name);
}

void
velo_reset_dict ()
{
	g_list_free (dict.list);
	dict.list = NULL;
	dict.len = 0;
	dict.name[0] = '\0';
}

/**********************************************************************
 * Initialize the velo exercise window.
 */
void
velo_init ()
{
	gchar *tmp_name;
	gchar *wl_name;
	FILE *fh;

	if (dict.len == 0)
	{
		if (main_preferences_exist ("tutor", "word_list"))
		{
			wl_name = main_preferences_get_string ("tutor", "word_list");
			tmp_name = g_strconcat (main_get_user_dir (), wl_name, ".words", NULL);
			if ((fh = (FILE *) g_fopen (tmp_name, "r")))
			{
				fclose (fh);
				velo_init_dict (wl_name);
			}
			g_free (wl_name);
			g_free (tmp_name);
		}
	}
	if (dict.len == 0)
		velo_init_dict (NULL);
}

/**********************************************************************
 * Retrieves words from the dictionary.
 */
void
velo_init_dict (gchar * list_name)
{
	static gchar *word_str = NULL;	/* no doubt: this is initialized only at the first call */
	gchar *tmp_name;
	gchar *tmp_code;
	gchar str_32[33];
	guint nwords;
	FILE *fh;

	if (list_name && strcmp (list_name, _("--> Default")) != 0)
	{
		main_preferences_set_string ("tutor", "word_list", list_name);
		tmp_name = g_strconcat (main_get_user_dir (), list_name, ".words", NULL);
		g_message ("loading dictionary: %s.words", list_name);
		strncpy (dict.name, list_name, 20);
		dict.name[20] = '\0';
	}
	else
	{
		main_preferences_remove ("tutor", "word_list");
		tmp_code = main_preferences_get_string ("interface", "language");
		tmp_name = g_strconcat (main_get_data_path (), tmp_code, ".words", NULL);
		g_message ("loading dictionary: %s.words", tmp_code);
		strcpy (dict.name, "Default");
		g_free (tmp_code);
	}

	fh = (FILE *) g_fopen (tmp_name, "r");
	if (fh == NULL && strcmp (dict.name, "Default") == 0)
		fh = trans_lang_get_similar_file (".words");

	if (fh)
	{
		g_free (word_str);
		g_list_free (dict.list);
		dict.list = NULL;
		word_str = g_strdup ("");
		nwords = 0;
		g_print ("Tens of words:\n 0");
		while (fgets (str_32, 33, fh))
		{
			str_32[strlen (str_32) - 1] = '\0';
			word_str = g_strconcat (word_str, " ", str_32, NULL);
			dict.list = g_list_append (dict.list, strrchr (word_str, ' ') + 1);
			if (++nwords % 10 == 0)
				g_print (" - %u", nwords / 10);
		}
		fclose (fh);
		dict.len = g_list_length (dict.list);
		g_print ("\n");
		g_message ("Dictionary loaded!\n\n");
	}
	else
	{
		g_message ("could not open the file: %s", tmp_name);
		g_free (tmp_name);
		tmp_code = main_preferences_get_string ("interface", "language");
		if (strcmp (tmp_code, "C") == 0)
			g_error ("so, we must quit!");
		main_preferences_set_string ("interface", "language", "C");
		velo_init_dict (list_name);
		main_preferences_set_string ("interface", "language", tmp_code);
		g_free (tmp_code);
		return;
	}
	g_free (tmp_name);
}

/**********************************************************************
 * Draw random words selected from a 'discretionary'
 */
void
velo_draw_random_words ()
{
	gchar *text;
	gchar str_32[33];
	gint i, j;

	for (i = 0; i < 4; i++)
	{			/* 4 paragraphs per exercise */
		text = NULL;
		for (j = 0; j < 20; j++)
		{		/* 20 words per paragraph */
			velo_get_random_word (str_32);
			if (!j)
			{
				str_32[0] = g_ascii_toupper (str_32[0]);
				text = g_strconcat (str_32, NULL);
			}
			else
				text = g_strconcat (text, " ", str_32, NULL);
		}
		text = g_strconcat (text, ".\n", NULL);
		tutor_draw_paragraph (text);
		g_free (text);
	}
}

/**********************************************************************
 * Selects one word from a 'discretionary'
 */
void
velo_get_random_word (gchar str_32[33])
{
	strncpy (str_32, g_list_nth_data (dict.list, rand () % dict.len), 33);
	str_32[32] = '\0';
}

/**********************************************************************
 * Takes text and make a list of words, validating as UTF-8
 */
gchar *
velo_filter_utf8 (gchar * text)
{
	gunichar uch;
	gboolean is_searching_word;
	struct INPUT_TEXT
	{
		gchar *pt;
		gulong len;
		guint nwords;
	} raw;
	struct FILTERED_TEXT
	{
		gchar *txt;
		gulong i;
		gulong len;
	} flt;

	raw.len = strlen (text);

	/* Verify empty string
	 */
	if (raw.len == 0)
	{
		flt.txt = g_strdup ("01234\n56789\n43210\n98765\n:-)\n");
		return (flt.txt);
	}

	/* Allocate memory space for the result
	 */
	flt.i = 0;
	flt.len = raw.len + 100;
	flt.txt = g_malloc (flt.len);

	raw.pt = text;

	/* Filter
	 */
	raw.nwords = 0;
	is_searching_word = TRUE;
	while (raw.pt && raw.nwords < MAX_WORDS)
	{
		if (*raw.pt == '\0')
			break;

		/* Read valid utf8 char
		 */
		if ((uch = g_utf8_get_char_validated (raw.pt, 16)) == (gunichar) - 1
		    || uch == (gunichar) - 2)
			uch = L' ';

		/* Increase the pointer for the input text
		 */
		raw.pt = g_utf8_find_next_char (raw.pt, raw.pt + 16);

		/* Verify memory space of output buffer
		 */
		if (flt.i < flt.len - 7)
		{
			flt.len += 100;
			flt.txt = g_realloc (flt.txt, flt.len);
		}

		/* Test alphabetic char to form a word
		 */
		if (g_unichar_isalpha (uch))
		{
			flt.i += g_unichar_to_utf8 (uch, &flt.txt[flt.i]);
			is_searching_word = FALSE;
		}
		else if (!is_searching_word)
		{
			raw.nwords++;
			flt.txt[flt.i++] = '\n';
			is_searching_word = TRUE;
		}
	}
	flt.txt[flt.i++] = '\0';

	if (raw.nwords == 0)
	{
		g_free (flt.txt);
		flt.txt = g_strdup ("01234\n56789\n43210\n98765\n:-)\n");
		return (flt.txt);
	}

	return (flt.txt);
}

/**********************************************************************
 * Reads the text file "file_name" and write to the dictionary.
 * If overwrite is TRUE, then besides overwriting any .words file,
 * it loads a lesson with the new dictionary.
 */
void
velo_text_write_to_file (gchar * text_raw, gboolean overwrite)
{
	gchar *dict_path;
	gchar *dictio_name;
	gchar *text_filtered;
	FILE *fh_destiny;

	dictio_name = g_strdup (_("Pasted_or_dropped"));
	dict_path = g_strconcat (main_get_user_dir (), dictio_name, ".words", NULL);
	assert_user_dir ();
	if (!(fh_destiny = (FILE *) g_fopen (dict_path, "w")))
	{
		gdk_beep ();
		g_warning ("couldn't create the file:\n <%s>", dict_path);
		if (overwrite == FALSE)
		{
			g_free (dict_path);
			g_free (dictio_name);
			return;
		}
	}
	g_free (dict_path);

	/* Filter the text
	 */
	text_filtered = velo_filter_utf8 (text_raw);
	fwrite (text_filtered, sizeof (gchar), strlen (text_filtered), fh_destiny);
	fclose (fh_destiny);

	g_free (text_filtered);

	if (overwrite == TRUE)
	{
		velo_init_dict (dictio_name);
		tutor_set_query (QUERY_INTRO);
		tutor_process_touch ('\0');
	}
	g_free (dictio_name);
}

/**********************************************************************
 * Reads the text file "file_name" and write to the dictionary.
 * If overwrite is TRUE, then besides overwriting any .words file,
 * it loads a lesson with the new dictionary.
 */
void
velo_create_dict (gchar * file_name, gboolean overwrite)
{
	gchar *dict_path;
	gchar *dictio_name;
	gchar *text_raw;
	gchar *text_filtered;
	FILE *fh_destiny;

	if (!file_name)
	{
		gdk_beep ();
		g_warning ("velo_create_dict(): null file name as first argument.");
		return;
	}

	if (!g_file_get_contents (file_name, &text_raw, NULL, NULL))
	{
		gdk_beep ();
		g_warning ("couldn't read the file:\n <%s>", file_name);
		return;
	}

	dictio_name = g_strdup (strrchr (file_name, DIRSEP) + 1);
	dict_path = g_strconcat (main_get_user_dir (), dictio_name, ".words", NULL);
	assert_user_dir ();
	if (!(fh_destiny = (FILE *) g_fopen (dict_path, "w")))
	{
		gdk_beep ();
		g_warning ("couldn't create the file:\n <%s>", dict_path);
		if (overwrite == FALSE)
		{
			g_free (dict_path);
			g_free (dictio_name);
			return;
		}
	}
	g_free (dict_path);

	/* Filter the text
	 */
	text_filtered = velo_filter_utf8 (text_raw);
	fwrite (text_filtered, sizeof (gchar), strlen (text_filtered), fh_destiny);
	fclose (fh_destiny);

	g_free (text_raw);
	g_free (text_filtered);

	if (overwrite == TRUE)
	{
		velo_init_dict (dictio_name);
		tutor_set_query (QUERY_INTRO);
		tutor_process_touch ('\0');
	}
	g_free (dictio_name);
}

/**********************************************************************
 * Put on the screen the final comments
 */
void
velo_comment (gdouble accuracy, gdouble velocity)
{
	gchar *tmp_str;
	GtkWidget *wg;
	GtkTextBuffer *buf;

	/*
	 * Comments
	 */
	if (accuracy < 97)
		tmp_str = g_strdup (_(" Your accuracy rate is too low.\n"
				      " I only can rely on you if it is greater than 97 %.\n Please do it again.\n"));
	else if (velocity < 10)
		tmp_str =
			g_strdup (_
				  (" You must be kidding. You are so slow!\n Please do it again and again.\n"));
	else if (velocity < 20)
		tmp_str =
			g_strdup (_
				  (" Still away from the highway. You can make better...\n Try at least 20 WPM.\n"));
	else if (velocity < 30)
		tmp_str = g_strdup (_(" You are doing well, but need to go faster.\n"
				      " And don't forget the accuracy. Try to get 30 WPM.\n"));
	else if (velocity < 40)
		tmp_str =
			g_strdup (_
				  (" Fine. Now you need to start running.\n Can you reach 40 WPM?\n"));
	else if (velocity < 50)
		tmp_str =
			g_strdup (_
				  (" Very good. You are almost there.\n Can you finally reach 50 WPM?\n"));
	else if (velocity < 60)
		tmp_str =
			g_strdup (_
				  (" Excellent. For this course, that is enough.\n Try now the fluidness exercises, OK?\n"));
	else if (velocity < 70)
		tmp_str =
			g_strdup (_
				  (" Fast! Are you training for a competition?\n So, try to get 70 WPM!\n"));
	else if (velocity < 80)
		tmp_str = g_strdup (_(" Top of \"qwerty\"."
				      " Now it's time to change to the Dvorak mode.\n Are you afraid of reaching 80 WPM?\n"));
	else if (velocity < 90)
		tmp_str = g_strdup (_(" Dvorak mode dominated!\n Can you fly at 90 WPM?\n"));
	else
		tmp_str =
			g_strdup (_
				  (" Dvorak master!\n"
				   " I have no words to express my admiration!\n"));

	wg = lookup_widget (window_tutor_, "text_tutor");
	buf = gtk_text_view_get_buffer (GTK_TEXT_VIEW (wg));
	gtk_text_buffer_insert_at_cursor (buf, tmp_str, strlen (tmp_str));
	g_free (tmp_str);
}
