/**************************************************************************/
/*  Klavaro - a flexible touch typing tutor                               */
/*  Copyright (C) 2005 - 2008  Felipe Castro                              */
/*                                                                        */
/*  This program is free software, licensed under the terms of the GNU    */
/*  General Public License as published by the Free Software Foundation,  */
/*  which is also included in this package, in a file named "COPYING".    */
/**************************************************************************/

/*
 * Klavaro is a touch typing tutor and exercise program.
 * Copyright (C) 2007 - Felipe E. F. de Castro
 */
#include <string.h>
#include <stdio.h>
#include <stdlib.h>
#include <sys/stat.h>
#include <curl/curl.h>
#include <glib.h>
#include <glib/gstdio.h>
#include <gtk/gtk.h>

#include "support.h"
#include "interface.h"
#include "callbacks.h"
#include "translation.h"
#include "keyboard.h"
#include "tutor.h"
#include "basic.h"
#include "adaptability.h"
#include "velocity.h"
#include "fluidness.h"
#include "top10.h"
#include "main.h"

/*******************************************************************************
 * Variables
 */
GtkWidget *window_main_;

GKeyFile *preferences = NULL;

struct
{
	gchar *user_dir;
	gchar *data_path;
	gchar *score_path;
	gboolean curl_ok;
} general;


/*******************************************************************************
 * Interface functions
 */
gchar *
main_get_data_path ()
{
	return (general.data_path);
}

void
main_set_data_path (gchar * path)
{
	g_free (general.data_path);
	general.data_path = g_strdup (path);
}

gchar *
main_get_user_dir ()
{
	return (general.user_dir);
}

gchar *
main_get_score_path ()
{
	return (general.score_path);
}

gboolean
main_get_curl_ok ()
{
	return (general.curl_ok);
}

gboolean
main_preferences_exist (gchar * group, gchar * key)
{
	return (g_key_file_has_key (preferences, group, key, NULL));
}

void
main_preferences_remove (gchar * group, gchar * key)
{
	g_key_file_remove_key (preferences, group, key, NULL);
}

gchar *
main_preferences_get_string (gchar * group, gchar * key)
{
	return (g_key_file_get_string (preferences, group, key, NULL));
}

void
main_preferences_set_string (gchar * group, gchar * key, gchar * value)
{
	g_key_file_set_string (preferences, group, key, value);
}

gint
main_preferences_get_int (gchar * group, gchar * key)
{
	return (g_key_file_get_integer (preferences, group, key, NULL));
}

void
main_preferences_set_int (gchar * group, gchar * key, gint value)
{
	g_key_file_set_integer (preferences, group, key, value);
}

gboolean
main_preferences_get_boolean (gchar * group, gchar * key)
{
	return (g_key_file_get_boolean (preferences, group, key, NULL));
}

void
main_preferences_set_boolean (gchar * group, gchar * key, gboolean value)
{
	g_key_file_set_boolean (preferences, group, key, value);
}

void
main_preferences_save ()
{
	gchar *tmp_name;
	FILE *fh;

	/*
	 * Save preferences
	 */
	assert_user_dir ();
	tmp_name = g_strconcat (general.user_dir, "preferences.ini", NULL);
	fh = (FILE *) g_fopen (tmp_name, "w");
	if (fh == NULL)
		g_warning ("couldn't save your preferences at the user folder:\n %s", tmp_name);
	else
	{
		g_free (tmp_name);
		tmp_name = NULL;
		tmp_name = g_key_file_to_data (preferences, NULL, NULL);
		if (tmp_name != NULL)
			fputs (tmp_name, fh);
		fclose (fh);
	}
	g_free (tmp_name);
}

/*******************************************************************************
 * Program
 */
int
main (int argc, char *argv[])
{
#ifdef ENABLE_NLS
	bindtextdomain (GETTEXT_PACKAGE, PACKAGE_LOCALE_DIR);
	bind_textdomain_codeset (GETTEXT_PACKAGE, "UTF-8");
	textdomain (GETTEXT_PACKAGE);
#endif

	if (argc == 2)
	{
		if (g_str_equal (argv[1], "--versio") || g_str_equal (argv[1], "--version"))
		{
			g_print (PACKAGE " - " VERSION "\n");
			return 0;
		}
		if (g_str_equal (argv[1], "-a"))
		{
			main_initialize_global_variables ();
			general.curl_ok = curl_global_init (CURL_GLOBAL_WIN32) == CURLE_OK ? TRUE : FALSE;
			top10_global_rank ();
			if (general.curl_ok) curl_global_cleanup ();
			return 0;
		}
	}
	general.curl_ok = curl_global_init (CURL_GLOBAL_WIN32) == CURLE_OK ? TRUE : FALSE;

	gtk_disable_setlocale ();	/* The locale will be set according to user preference... */
	gtk_init (&argc, &argv);

	add_pixmap_directory (PACKAGE_DATA_DIR DIRSEP_S PACKAGE DIRSEP_S "pixmaps");
	add_pixmap_directory (".." DIRSEP_S "pixmaps");

	main_initialize_global_variables ();	/* Here the locale is set. */
	window_main_ = create_window_main ();	/* So we must postpone gtk function calls */
	gtk_widget_show (window_main_);
	main_window_init ();	/* and initialize its parameters only as a third step... */

	gtk_main ();

	return 0;
}

/**********************************************************************
 * Initialize the value of the global variables
 */
void
main_initialize_global_variables ()
{
	gchar *tmp_name;
	FILE *fh;

	/*
	 * Set the home user dir 
	 */
	general.user_dir = g_strconcat (g_get_home_dir (), DIRSEP_S ".klavaro" DIRSEP_S, NULL);

	/*
	 * Get a valid data path.
	 * First searches the path at the source directory.
	 */
	general.data_path = g_strconcat (".." DIRSEP_S "data" DIRSEP_S, NULL);
	tmp_name = g_strconcat (general.data_path, "basic_lessons.txt", NULL);
	fh = (FILE *) g_fopen (tmp_name, "r");
	if (fh == NULL)
	{
		g_free (general.data_path);
		g_free (tmp_name);
		general.data_path = g_strdup (PACKAGE_DATA_DIR DIRSEP_S PACKAGE DIRSEP_S);
		tmp_name = g_strconcat (general.data_path, "basic_lessons.txt", NULL);
		fh = (FILE *) g_fopen (tmp_name, "r");
	}
	if (fh == NULL)
		g_error ("couldn't find a data file at the data path:\n %s", tmp_name);
	fclose (fh);
	g_free (tmp_name);

	if (UNIX_OK)
		general.score_path = g_strdup ("/var/games/klavaro");
	else
		general.score_path =
			g_build_filename ((g_get_system_data_dirs ())[0], PACKAGE, NULL);

	/*
	 * Retrieve initial or saved preferences
	 */
	preferences = g_key_file_new ();
	tmp_name = g_strconcat (general.user_dir, "preferences.ini", NULL);
	if (g_file_test (tmp_name, G_FILE_TEST_EXISTS))
		g_key_file_load_from_file (preferences, tmp_name, G_KEY_FILE_NONE, NULL);
	else
		g_key_file_load_from_file (preferences, "/etc/klavaro_preferences.ini",
					   G_KEY_FILE_NONE, NULL);
	g_free (tmp_name);

	/*
	 * Apply the language environmental value
	 */
	trans_init_language_env ();

	/*
	 * Other initializations
	 */
	srand (time (0));

	tutor_init_timers ();
}

void
main_set_auto_state ()
{
	/*
	 * Set the autostart state
	 */
	if (!main_preferences_exist ("interface", "autostart"))
	{
		main_preferences_set_boolean ("interface", "autostart", FALSE);
		main_preferences_set_int ("interface", "exercise", 1);
	}
	else if (main_preferences_get_boolean ("interface", "autostart"))
		gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON
					      (lookup_widget (window_main_, "checkbutton_0")),
					      FALSE);

	if (!main_preferences_exist ("interface", "exercise"))
		main_preferences_set_int ("interface", "exercise", 1);

	switch (main_preferences_get_int ("interface", "exercise"))
	{
	case 2:
		gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON
					      (lookup_widget (window_main_, "radiobutton_2")),
					      TRUE);
		break;
	case 3:
		gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON
					      (lookup_widget (window_main_, "radiobutton_3")),
					      TRUE);
		break;
	case 4:
		gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON
					      (lookup_widget (window_main_, "radiobutton_4")),
					      TRUE);
		break;
	}
}

/*******************************************************************************
 * Initialize keyboard entry and language combo 
 */
void
main_window_init ()
{
	gchar *tmp_str;
	GtkWidget *wg;

	/*
	 * Set the initial keyboard to use
	 */
	wg = lookup_widget (window_main_, "entry_keyboard");
	if (!main_preferences_exist ("tutor", "keyboard"))
	{
		gtk_entry_set_text (GTK_ENTRY (wg), DEFAULT_KEYBOARD);
		g_key_file_set_string (preferences, "tutor", "keyboard", DEFAULT_KEYBOARD);
	}
	else
	{
		tmp_str = g_key_file_get_string (preferences, "tutor", "keyboard", NULL);
		gtk_entry_set_text (GTK_ENTRY (wg), tmp_str);
		g_free (tmp_str);
	}

	keyb_init_name (gtk_entry_get_text (GTK_ENTRY (wg)));
	keyb_get_layout ();

	/*
	 * Set the language
	 */
	trans_set_combo_language ();

	/* Run selected module, it if enabled
	 */
	main_set_auto_state ();
	if (main_preferences_get_boolean ("interface", "autostart"))
		switch (main_preferences_get_int ("interface", "exercise"))
		{
		case 1:
			on_button_basic_clicked (NULL, NULL);
			break;
		case 2:
			on_button_adapt_clicked (NULL, NULL);
			break;
		case 3:
			on_button_velo_clicked (NULL, NULL);
			break;
		case 4:
			on_button_fluid_clicked (NULL, NULL);
			break;
		}
}

/*******************************************************************************
 * Quit the application
 */
void
main_window_pass_away ()
{
	main_preferences_save ();

	g_rmdir ("tmp/klavaro");

	/*
	 * Good bye
	 */
	if (general.curl_ok) curl_global_cleanup ();
	g_print ("\nAdiaux!\n");
	gtk_exit (0);
}

/**********************************************************************
 * General functions
 */
/**********************************************************************
 * Search for the user directory and create it if not found
 */
void
assert_user_dir ()
{
	GDir *dh;

	dh = g_dir_open (main_get_user_dir (), 0, NULL);
	if (dh == NULL)
	{
		g_message ("creating an empty user folder:\n %s", main_get_user_dir ());

#ifdef G_OS_UNIX
		g_mkdir (main_get_user_dir (), S_IRWXU | S_IRGRP | S_IXGRP | S_IROTH | S_IXOTH);
#else
		g_mkdir (main_get_user_dir (), 0xFFFF);
#endif

		dh = g_dir_open (main_get_user_dir (), 0, NULL);
		if (dh == NULL)
			g_error ("could not creat a user folder, so we must quit!");
	}
	g_dir_close (dh);
}

/**********************************************************************
 * Compare two strings, so that it applies to other sorting functions.
 */
gint
comp_str (gconstpointer a, gconstpointer b)
{
	return (strcasecmp (a, b));
}
