/*****************************************************************************/
/*  Klavaro - a flexible touch typing tutor                                  */
/*  Copyright (C) 2005, 2006, 2007, 2008 Felipe Castro                       */
/*  Copyright (C) 2009 Free Software Foundation                              */
/*                                                                           */
/*  This program is free software, licensed under the terms of the GNU       */
/*  General Public License as published by the Free Software Foundation,     */
/*  either version 3 of the License, or (at your option) any later version.  */
/*  You should have received a copy of the GNU General Public License        */
/*  along with this program.  If not, see <http://www.gnu.org/licenses/>.    */
/*****************************************************************************/

/*
 * Klavaro is a touch typing tutor and exercise program.
 * Copyright (C) 2007 - Felipe E. F. de Castro
 */
#include <string.h>
#include <stdio.h>
#include <stdlib.h>
#include <sys/stat.h>
#include <glib.h>
#include <glib/gstdio.h>
#include <pango/pango-attributes.h>
#include <gtk/gtk.h>
#include <curl/curl.h>

#include "auxiliar.h"
#include "callbacks.h"
#include "translation.h"
#include "keyboard.h"
#include "tutor.h"
#include "basic.h"
#include "adaptability.h"
#include "velocity.h"
#include "fluidness.h"
#include "top10.h"
#include "main.h"

/*******************************************************************************
 * Variables
 */
GtkBuilder *gui;

static GKeyFile *preferences = NULL;

static struct
{
	struct
	{
		gchar *user;
		gchar *data;
		gchar *score;
	} path;	
	gboolean curl_ok;
} general;

gchar *KEYB_CUSTOM;
gchar *KEYB_EDIT;
gchar *OTHER_DEFAULT;

/*******************************************************************************
 * Internal functions
 */
void main_initialize_global_variables (void);

void main_window_init (void);

/*******************************************************************************
 * Interface functions
 */
gchar *
main_get_data_path ()
{
	return (general.path.data);
}

void
main_set_data_path (gchar *path)
{
	g_free (general.path.data);
	general.path.data = g_strdup (path);
}

gchar *
main_get_user_dir ()
{
	return (general.path.user);
}

gchar *
main_get_score_path ()
{
	return (general.path.score);
}

gboolean
main_get_curl_ok ()
{
	return (general.curl_ok);
}

gboolean
main_preferences_exist (gchar * group, gchar * key)
{
	return (g_key_file_has_key (preferences, group, key, NULL));
}

void
main_preferences_remove (gchar * group, gchar * key)
{
	g_key_file_remove_key (preferences, group, key, NULL);
}

gchar *
main_preferences_get_string (gchar *group, gchar *key)
{
	return (g_key_file_get_string (preferences, group, key, NULL));
}

void
main_preferences_set_string (gchar *group, gchar *key, gchar *value)
{
	g_key_file_set_string (preferences, group, key, value);
}

gint
main_preferences_get_int (gchar * group, gchar * key)
{
	return (g_key_file_get_integer (preferences, group, key, NULL));
}

void
main_preferences_set_int (gchar * group, gchar * key, gint value)
{
	g_key_file_set_integer (preferences, group, key, value);
}

gboolean
main_preferences_get_boolean (gchar * group, gchar * key)
{
	return (g_key_file_get_boolean (preferences, group, key, NULL));
}

void
main_preferences_set_boolean (gchar * group, gchar * key, gboolean value)
{
	g_key_file_set_boolean (preferences, group, key, value);
}

void
main_preferences_save ()
{
	gchar *tmp_name;
	FILE *fh;

	/*
	 * Save preferences
	 */
	assert_user_dir ();
	tmp_name = g_build_filename (general.path.user, "preferences.ini", NULL);
	fh = (FILE *) g_fopen (tmp_name, "w");
	if (fh == NULL)
		g_warning ("couldn't save your preferences at the user folder:\n %s", tmp_name);
	else
	{
		g_free (tmp_name);
		tmp_name = NULL;
		tmp_name = g_key_file_to_data (preferences, NULL, NULL);
		if (tmp_name != NULL)
			fputs (tmp_name, fh);
		else
			g_warning ("couldn't save your preferences at the user folder:\n %s", tmp_name);
		fclose (fh);
	}
	g_free (tmp_name);
}

/*******************************************************************************
 * Main program
 */
int
main (int argc, char *argv[])
{
	gchar *tmp;
	gboolean success = FALSE;

	/* Localization
	 */
#ifdef ENABLE_NLS
	bindtextdomain (GETTEXT_PACKAGE, PACKAGE_LOCALE_DIR);
	bind_textdomain_codeset (GETTEXT_PACKAGE, "UTF-8");
	textdomain (GETTEXT_PACKAGE);
#endif

	/* Command-line arguments
	 */
	if (argc == 2)
	{
		if (g_str_equal (argv[1], "--versio") || g_str_equal (argv[1], "--version"))
		{
			g_print (PACKAGE " - " VERSION "\n");
			return 0;
		}
#ifdef G_OS_UNIX
		if (g_str_equal (argv[1], "-a"))
		{
			main_initialize_global_variables ();
			general.curl_ok = curl_global_init (CURL_GLOBAL_WIN32) == CURLE_OK ? TRUE : FALSE;
			top10_global_rank ();
			if (general.curl_ok) curl_global_cleanup ();
			return 0;
		}
#endif
	}

#ifdef G_OS_UNIX
	general.curl_ok = curl_global_init (CURL_GLOBAL_WIN32) == CURLE_OK ? TRUE : FALSE;
#else
	general.curl_ok = 0;
#endif

	gtk_init (&argc, &argv);

	main_initialize_global_variables ();	/* Here the locale is got. */

	/* Create all the interface stuff
	 */
	gui = gtk_builder_new ();
	tmp = g_build_filename (main_get_data_path (), "klavaro.ui", NULL);
	if (g_file_test (tmp, G_FILE_TEST_IS_REGULAR))
		success = gtk_builder_add_from_file (gui, tmp, NULL);
	else
		g_error ("GUI file not found. Aborting.\n %s", tmp);
	if (!success)
		g_error ("GUI file found but couldn't create the GUI. Aborting.");
	if (!g_module_supported ())
		g_error ("GUI created but can't connect signals.");
	g_free (tmp);
	gtk_builder_connect_signals (gui, NULL);

	main_window_init ();	/* and initialize its parameters */

	gtk_main ();

	return 0;
}

/**********************************************************************
 * Initialize the value of the global variables
 */
void
main_initialize_global_variables ()
{
	gchar *tmp;
	FILE *fh;

	/* Set the home user dir (update to new config dir)
	 */
	tmp = g_build_filename (g_get_home_dir (), ".klavaro", NULL);
	general.path.user = g_build_filename (g_get_user_config_dir (), "klavaro", NULL);
	if (g_file_test (tmp, G_FILE_TEST_IS_DIR))
	{
#	ifdef G_OS_UNIX
		g_mkdir (g_get_user_config_dir (), S_IRWXU | S_IRGRP | S_IXGRP | S_IROTH | S_IXOTH);
#	else
		g_mkdir (g_get_user_config_dir (), 0xFFFF);
#	endif
		if (g_rename (tmp, general.path.user) != 0)
		{
			g_free (general.path.user);
			general.path.user = g_strdup (tmp);
		}
	}
	g_free (tmp);

	/* Get a valid data path.
	 * First searches the path at the source directory.
	 */
	general.path.data = g_build_filename ("..", "data", NULL);
	tmp = g_build_filename (general.path.data, "basic_lessons.txt", NULL);
	fh = (FILE *) g_fopen (tmp, "r");
	if (fh == NULL)
	{
		g_free (general.path.data);
		g_free (tmp);
		general.path.data = g_build_filename (PACKAGE_DATA_DIR, PACKAGE, NULL);
		tmp = g_build_filename (general.path.data, "basic_lessons.txt", NULL);
		fh = (FILE *) g_fopen (tmp, "r");
	}
	if (fh == NULL)
		g_error ("couldn't find a data file at the data path:\n %s", tmp);
	fclose (fh);
	g_free (tmp);

	/* Get a valid scoring path.
	 */
	if (UNIX_OK)
		general.path.score = g_strdup ("/var/games/klavaro");
	else
		general.path.score =
			g_build_filename ((g_get_system_data_dirs ())[0], PACKAGE, NULL);

	/* Retrieve initial or saved preferences
	 */
	preferences = g_key_file_new ();
	tmp = g_build_filename (general.path.user, "preferences.ini", NULL);
	if (! g_file_test (tmp, G_FILE_TEST_EXISTS))
	{
		g_free (tmp);
		tmp = g_strdup ("/etc/klavaro_preferences.ini");
	}
	g_key_file_load_from_file (preferences, tmp, G_KEY_FILE_NONE, NULL);
	g_free (tmp);

	/* Other initializations
	 */
	trans_init_lang_name_code ();
	trans_init_language_env ();
	srand (time (0));
	tutor_init_timers ();

	KEYB_CUSTOM = g_strdup (_("(Custom)"));
	KEYB_EDIT = g_strdup (_("(Edit custom)"));
	OTHER_DEFAULT = g_strdup (_("(Default)"));
}

void
main_set_pixmap (gchar *widget, gchar *image)
{
	gchar *tmp;
	GtkImage *img;

	tmp = g_build_filename (general.path.data, image, NULL);
	img = GTK_IMAGE (get_wg (widget));
	gtk_image_set_from_file (img, tmp);
	g_free (tmp);
}

/*******************************************************************************
 * Initialize some interface widgets
 */
void
main_window_init ()
{
	static GList *iconlist = NULL;
	static GdkPixbuf *icon[5];
	gint i;
	gchar *tmp;
	GtkWidget *wg;
	PangoAttrList *palist;

	/* Set the language
	 */
	trans_set_combo_language ();

	/* Set keyboard
	 */
	keyb_create_virtual_keys ();
	hints_init ();

	/* Set the initial keyboard to use
	 */
	keyb_set_combo_kbd ("combobox_kbd_country", "combobox_kbd_variant");
	keyb_set_combo_kbd ("combobox_keyboard_country", "combobox_keyboard_variant");

	/* Set window icons
	 */
	gtk_window_set_default_icon_name ("klavaro");

	/* Set pixmaps
	 */
	main_set_pixmap ("image_fluid", "fluid.png");
	main_set_pixmap ("image_keyboard", "key.png");
	main_set_pixmap ("image_beep", "beep.png");
	main_set_pixmap ("image_progress", "progress.png");
	main_set_pixmap ("image_other", "other.png");
	main_set_pixmap ("image_top10", "top10.png");

	/* Set Top10 TreeViews
	 */
	top10_init ();

	/* Set big and bold labels
	 */
	pango_parse_markup ("<big><b>----------------------------------------------------------------</b></big>",
		       	-1, 0, &palist, NULL, NULL, NULL);
	gtk_label_set_attributes (GTK_LABEL (get_wg ("label_keyboard_title")), palist);
	gtk_label_set_attributes (GTK_LABEL (get_wg ("label_main_intro")), palist);
	gtk_label_set_attributes (GTK_LABEL (get_wg ("label_main_basic")), palist);
	gtk_label_set_attributes (GTK_LABEL (get_wg ("label_main_adapt")), palist);
	gtk_label_set_attributes (GTK_LABEL (get_wg ("label_main_velo")), palist);
	gtk_label_set_attributes (GTK_LABEL (get_wg ("label_main_fluid")), palist);

	/* Set main labels (for translation)
	 */
	tmp = g_strdup_printf ("0 - %s", _("Introduction"));
	gtk_label_set_text (GTK_LABEL (get_wg ("label_main_intro")), tmp);
	g_free (tmp);
	tmp = g_strdup_printf ("1 - %s", _("Basic course"));
	gtk_label_set_text (GTK_LABEL (get_wg ("label_main_basic")), tmp);
	g_free (tmp);
	tmp = g_strdup_printf ("2 - %s", _("Adaptability"));
	gtk_label_set_text (GTK_LABEL (get_wg ("label_main_adapt")), tmp);
	g_free (tmp);
	tmp = g_strdup_printf ("3 - %s", _("Velocity"));
	gtk_label_set_text (GTK_LABEL (get_wg ("label_main_velo")), tmp);
	g_free (tmp);
	tmp = g_strdup_printf ("4 - %s", _("Fluidness"));
	gtk_label_set_text (GTK_LABEL (get_wg ("label_main_fluid")), tmp);
	g_free (tmp);

	/* Set Sexy Icon Entry...
	 */
	tutor_entry_mesg_init (); 

	/* Set version
	 */
	gtk_about_dialog_set_version (GTK_ABOUT_DIALOG (get_obj ("aboutdialog_klavaro")), VERSION);

	/* For remembering the window's position
	 */
	gtk_widget_show (get_wg ("window_main"));
	window_restore ("main");

	/* Run last used module
	 */
	if (!main_preferences_exist ("interface", "autostart"))
		main_preferences_set_boolean ("interface", "autostart", FALSE);
	if (!main_preferences_exist ("interface", "exercise"))
		main_preferences_set_int ("interface", "exercise", 1);
	if (main_preferences_get_boolean ("interface", "autostart"))
	{
		switch (main_preferences_get_int ("interface", "exercise"))
		{
		case TT_BASIC:
			on_button_basic_clicked (NULL, NULL);
			break;
		case TT_ADAPT:
			on_button_adapt_clicked (NULL, NULL);
			break;
		case TT_VELO:
			on_button_velo_clicked (NULL, NULL);
			break;
		case TT_FLUID:
			on_button_fluid_clicked (NULL, NULL);
			break;
		}
	}
}

/* Quit the application
 */
void
main_window_pass_away ()
{
	main_preferences_save ();

	g_rmdir ("tmp/klavaro");

	/*
	 * Good bye
	 */
#ifdef G_OS_UNIX
	if (general.curl_ok) curl_global_cleanup ();
#endif
	g_print ("\nAdiaux!\n");
	exit (0);
}

