/***************************************************************************
 *   Copyright (C) 2004 by Alessandro Bonometti                            *
 *   bauno@inwind.it                                                       *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/


#include "grouplist.h"
#include "binheader.h"
#include <kmessagebox.h>
#include <assert.h>
#include <kiconloader.h>
#include "progress.h"
#include <kprogress.h>
#include <kapplication.h>
#include "updopt.h"

GroupList::GroupList(QString _groupsDbName,  DbEnv *_dbenv, Servers *_servers, QWidget* parent, const char* name, WFlags fl)
	: m_groupWidget(parent,name,fl),  dbenv(_dbenv),  servers(_servers), groupsDbName(_groupsDbName)
{
	setCaption("SubScribed groups");
	setIcon(KGlobal::iconLoader()->loadIcon("icon_newsgroup",KIcon::Small, KIcon::SizeSmall, false));
	m_groupList->setColumnAlignment(Unread_Col, Qt::AlignRight);
	m_groupList->setColumnAlignment(Total_Col, Qt::AlignRight);
	m_groupList->setAllColumnsShowFocus(true);
	m_groupList->setRootIsDecorated(true);
	loadGroups();
	

	connect(m_groupList, SIGNAL(executed(QListViewItem*)), this, SLOT(slotExecuted(QListViewItem* )));
	connect(m_groupList, SIGNAL(selectionChanged()), this, SLOT(slotSelectionChanged()));
	connect(m_groupList, SIGNAL(contextMenuRequested(QListViewItem*, const QPoint&, int)), this, SLOT(slotGroupPopup(QListViewItem*, const QPoint& )));
	
	groups.setAutoDelete(true);
	categories.setAutoDelete(true);
	m_groupList->setColumnAlignment(1, Qt::AlignRight);
	
}


//Creates the widget without loading the groups...

GroupList::GroupList( QWidget * parent, const char * name, WFlags fl ) : m_groupWidget(parent,name,fl)
{
	setCaption("SubScribed groups");
	setIcon(KGlobal::iconLoader()->loadIcon("group",KIcon::User, 0, false));
	
	
	groupsDbName="newsgroups";
	dbenv=0;
	groupDb=0;
	

	connect(m_groupList, SIGNAL(executed(QListViewItem*)), this, SLOT(slotExecuted(QListViewItem* )));
	connect(m_groupList, SIGNAL(selectionChanged()), this, SLOT(slotSelectionChanged()));
	connect(m_groupList, SIGNAL(contextMenuRequested(QListViewItem*, const QPoint&, int)), this, SLOT(slotGroupPopup(QListViewItem*, const QPoint& )));
	
	groups.setAutoDelete(true);
	m_groupList->setColumnAlignment(1, Qt::AlignRight);
	
}



GroupList::~GroupList()
{
// 	QDictIterator<NewsGroup> it(groups);
// 	while (it.current()) {
// 		groups.remove(it.currentKey());
// 		
// 		
// 	}
	groups.clear();
	
	if (groupDb != NULL ) {
		groupDb->close(0);
// 		qDebug("Group db closed");
		delete groupDb;
		groupDb=0;
	}
	
}

void GroupList::loadGroups(DbEnv* dbe)
{
	if (dbe)
		dbenv=dbe;
	
	
	groupDb=new Db(dbenv,0);

	if (groupDb->open(NULL, groupsDbName.latin1() , NULL, DB_BTREE, DB_CREATE | DB_THREAD , 0644) != 0)
		qDebug("Error opening group db");
	
	char datamem[DATAMEM_SIZE];
	char keymem[KEYMEM_SIZE];
	
	Dbt key, data;
	key.set_flags(DB_DBT_USERMEM);
	key.set_ulen(KEYMEM_SIZE);
	key.set_data(keymem);
	
	data.set_flags(DB_DBT_USERMEM);
	data.set_ulen(DATAMEM_SIZE);
	data.set_data(datamem);
// 	groups.clear();
	
	Dbc *cursor;
	groupDb->cursor(0, &cursor, 0);
	
	while((cursor->get(&key, &data, DB_NEXT))==0) {

		NewsGroup *tempg=new NewsGroup(dbenv, datamem);
		groups.insert(tempg->getName(), tempg);
// 		tempg->setListItem(new KListViewItem(m_groupList, tempg->getName(), QString::number(tempg->getTotal())));
		//Create the category list...
		Category *cat;
		if ( tempg->getCategory() != "None" ) {
			if ((cat = categories.find(tempg->getCategory() )) != 0) {
				//Category exist, add the item as a child of the folder...
				cat->childs++;
				cat->listItem->setText(2, QString::number(cat->childs));
				tempg->setListItem(new NGListViewItem(cat->listItem, tempg));
				
			} else {
				//Create category
				cat = new Category;
				
				cat->childs = 1;
				cat->name=tempg->getCategory();
				cat->listItem= new KListViewItem(m_groupList, cat->name, NULL, QString::number(cat->childs));
				cat->listItem->setPixmap(0,BarIcon("folder", KIcon::SizeSmall));
// 				cat->listItem->setExpandable(true);
				categories.insert(cat->name, cat);
				//Now create the item
				tempg->setListItem(new NGListViewItem(cat->listItem, tempg));
				
				
			}
			
			
		} else tempg->setListItem(new NGListViewItem(m_groupList, tempg)); //No category
		checkAndCreateDir(tempg->getSaveDir());
		/*
		//begin debug...
		QMap<int,int>::iterator it;
		qDebug("loaded %s", (const char *) tempg->ngName);
		for (it=tempg->high.begin(); it != tempg->high.end(); ++it) {
			qDebug("server: %d, high: %d", it.key(), it.data());
			
		}
		//end debug
		*/
		
		
	}
	cursor->close();
	if (!groups.isEmpty())
		emit sigHaveGroups(true);
	
	
	
	
}


void GroupList::slotAddGroup( Group* g)
{
	QDictIterator<Category> git(categories);
	QStringList cats;
	while (git.current()) {
// 		qDebug("Found category: %s", (const char *) git.current()->getCategory());
		cats.append(git.current()->name);
		++git;
	}
	
// 	AddForm *af=new AddForm(ng->ngName, ng->saveDir, this);
	AddForm *af=new AddForm(g->name, Config().decodeDir , cats, g,this);
	connect(af, SIGNAL(newGroup(QStringList, Group* )), this, SLOT(slotNewGroup(QStringList, Group* )));
	
	connect(af, SIGNAL(addCat(QString )), this, SLOT(slotAddCategory(QString )));
	af->exec();
	
	
	
	
	
}

void GroupList::saveGroup( NewsGroup * ng )
{
	qDebug("Saving group");
	char datamem[DATAMEM_SIZE];
	char keymem[KEYMEM_SIZE];
	
	int ret;
	Dbt key, data;
	key.set_data(keymem);
	key.set_flags(DB_DBT_USERMEM);
	key.set_ulen(KEYMEM_SIZE);
	
	data.set_data(datamem);
	data.set_flags(DB_DBT_USERMEM);
	data.set_ulen(DATAMEM_SIZE);
	
	char *p=ng->data();
	
	memcpy(datamem, p, ng->getRecordSize());
	
	data.set_size(ng->getRecordSize());
	
	delete [] p;
	const char *i= (const char *) ng->getName();
	memcpy(keymem, i, ng->getName().length());
	key.set_size(ng->getName().length());

	
	ret=groupDb->put(NULL, &key, &data, 0);
	if (ret!=0) {
		qDebug("Ops...error saving group: %s", dbenv->strerror(ret));
		
	} 
	
	groupDb->sync(0);
// 	ng->listItem->setText(Unread_Col, QString::number(ng->unreadArticles));
// 	ng->listItem->setText(Total_Col, QString::number(ng->totalArticles));
	
}

void GroupList::slotExecuted( QListViewItem * item )
{
	if ( (item->rtti() != GITEM) ||  !item->isEnabled())
		return;
	NewsGroup *ng = ((NGListViewItem*) item)->getNg();
// 	if (groups[item->text(0)]->getView() == NULL)
	if ( ng->getView() == NULL)
		emit openNewsGroup(ng);
	else emit activateNewsGroup(ng->getView());
}

void GroupList::slotUpdateSelected( )
{
	QPtrList<QListViewItem> selection=m_groupList->selectedItems();
	QPtrListIterator<QListViewItem> it(selection);
	NGListViewItem *selected;
	
	while (it.current()) {
		
		if (it.current()->rtti() == GITEM) {
			selected = (NGListViewItem*) it.current();

			selected->setEnabled(false);
			emit updateNewsGroup(selected->getNg(),0 );
		}
		++it;
		
			
	}
	slotSelectionChanged();
	
}

void GroupList::slotUpdateSubscribed( )
{
	QDictIterator<NewsGroup> it(groups);
	while(it.current()) {
// 		emit updateNewsGroup(it.current());
// 		qDebug("Update: %s", it.current()->getName().latin1());
		if (! it.current()->isUpdating()) {
			it.current()->listItem->setEnabled(false);
			emit updateNewsGroup(it.current(), 0);
		} else qDebug("won't update group %s", (const char *) it.current()->ngName);
		++it;
	}
	slotSelectionChanged();
}

void GroupList::slotSelectionChanged( )
{
	QPtrList<QListViewItem> selection=m_groupList->selectedItems();
	if (selection.isEmpty())
		emit isSelected(false);
	else emit isSelected(true);
}

void GroupList::slotGroupPopup( QListViewItem *item, const QPoint & p)
{
	if (!item)
		return;
	if (item->rtti() == GITEM)
		emit popupMenu(p);
	else {
		m_groupList->selectAll(false);
		m_groupList->setSelected(item, true);
		emit popupCatMenu(p);
	}
}

void GroupList::slotDeleteSelected( )
{
	QPtrList<QListViewItem> selection=m_groupList->selectedItems();
	QPtrListIterator<QListViewItem> it(selection);
	NGListViewItem *selected;
	
	while (it.current()) {
		if (it.current()->rtti() != GITEM ) {
			++it;
			continue;
		}
		selected = (NGListViewItem*) it.current();
		NewsGroup *ng=selected->getNg() ;
		int result=KMessageBox::questionYesNoCancel(this, i18n("Do you want to unsubscribe from in\n%1\n?").arg( ng->ngName  ), i18n("question"));
		switch (result) {
			case KMessageBox::Yes:
				deleteGroup(ng);	
			case KMessageBox::No:
				break;
			case KMessageBox::Cancel:
				break;
		}
		if (result==KMessageBox::Cancel)
			break;
		
		
		++it;
	}
	if (groups.isEmpty())
		emit sigHaveGroups(false);
	
		
}

void GroupList::slotUpdateFinished( NewsGroup * ng)
{
// 	assert(ng);
	ng->listItem->setEnabled(true);
	ng->listItem->setText( Total_Col, QString::number(ng->totalArticles));
	ng->listItem->setText( Unread_Col, QString::number(ng->unreadArticles));
	saveGroup(ng);
	
}

void GroupList::slotZeroSelected( )
{
	if (KMessageBox::questionYesNo(this, i18n("This action will delete all the articles in the selected groups and reset the counters. Do you want to continue?"), i18n("question")) == KMessageBox::No )
		return;
	QPtrList<QListViewItem> selection=m_groupList->selectedItems();
	QPtrListIterator<QListViewItem> it(selection);
	NGListViewItem *selected;
	
	while (it.current()) {
		if (it.current()->rtti() != GITEM) {
			qDebug("No group selected");
			++it;
			continue;
		}
		selected= (NGListViewItem*) it.current();
		NewsGroup *ng=selected->getNg();
		uint *count=new uint;
		
		ng->getDb()->truncate(NULL, count, 0);
		qDebug("Truncated %d record(s)", *count);
		
		ng->low.clear();
		QMap<int, int>::iterator hit;
		
		for (hit = ng->high.begin(); hit != ng->high.end(); ++hit) {
			if (hit.data() != -1)
				hit.data()=0;
			
			
		}
		
			
		ng->totalArticles=0;
		ng->unreadArticles=0;
		
// 		ng->listItem->setText(Unread_Col, QString::number(ng->totalArticles));
// 		ng->listItem->setText(Total_Col, QString::number(ng->unreadArticles));
		saveGroup(ng);
		delete count;
		++it;
		
	}
	
}
/*
void GroupList::closeGroups( )
{
	QDictIterator<NewsGroup> it(groups);
	while (it.current()) {
		groups.remove(it.currentKey());
		
		
	}
	
	if (groupDb) {
		groupDb->close(0);
		qDebug("Group db closed");
		delete groupDb;
	}
}*/

void GroupList::enable( bool b)
{
	if (b) {
		m_groupList->setCursor(KCursor::arrowCursor());
		setCursor(KCursor::arrowCursor());
	} else {
		m_groupList->setCursor(KCursor::waitCursor());
		setCursor(KCursor::waitCursor());
	}
// 	m_groupList->setEnabled(b);
// 	setEnabled(b);
	
	
}

void GroupList::markAllGroupAsUpdating( )
{
	QDictIterator<NewsGroup> it(groups);
	while (it.current()) {
		it.current()->startUpdating();
		++it;
	}
	
}

void GroupList::slotDeleteAllArticles( int serverId)
{
// 	qDebug("GroupList::slotDeleteAllArticles()");
	QDictIterator<NewsGroup> it(groups);
	while (it.current()) {
		NewsGroup *ng=it.current();
		//Don't need to expire if the server doesn't have the group
		if (ng->high[serverId] != -1)
			deleteAllArticles( ng, serverId);
		qDebug("all articles deleted");
		ng->low.remove(serverId);
		ng->high.remove(serverId);
		ng->stopUpdating();
		//if it's the only server that carries the group, unsubscribe from group...
		//convoluted, will have to change the way NntpThreadSocket::slotAddUpdItem works...
		QMap<int, int>::iterator hit;
		for (hit=ng->high.begin(); hit!=ng->high.end(); ++hit) {
			if (ng->high[hit.key()] != -1)
				break;
		}
		if (hit == ng->high.end()) {
			// reached the end of the list->all other server don't carry the group, 
			// unsubscribe from group
			qDebug("Unsubscribing from %s", (const char *) ng->ngName);
			deleteGroup(ng);
		} else saveGroup(ng);
		
		++it;
	}
	qDebug("GroupList::slotDeleteAllArticles() done");
	
}

void GroupList::deleteAllArticles( NewsGroup *ng, int serverId )
{
	Dbc *cursor;
	
	
	qDebug("Reset server in group!");
	if ((ng->getDb()->cursor(0, &cursor, DB_WRITECURSOR))!= 0) {
		qDebug("Error creating cursor!!");
		return;
	}
    BinHeader *bh;
    int count =0, ret;
	Dbt key, data;
	
	
	
	while((ret=cursor->get
				(&key, &data, DB_NEXT)) != DB_NOTFOUND) {
		
		if (ret != 0) {
			qDebug("Error retrieving record: %d", ret);
			break;
		}
		
		count++;
		bh=new BinHeader((uchar*) data.get_data());
// 		bh=new BinHeader((uchar*)data->get_data());
		uchar *p;
// 		if (bh->expire(serverId, INT_MAX)) {
		switch (bh->expire(serverId, INT_MAX)) {
			case BinHeader::Delete_Unread:
				ng->decUnread();
			case BinHeader::Delete_Read:
				if ((ret=cursor->del(0))!=0)
					qDebug("Error deleting from db: %d", ret);
				else ng->decTotal();
				break;
			case BinHeader::No_Delete:
				p=bh->data();
				memset(&data, 0, sizeof(data));
				data.set_data(p);
				data.set_size(bh->getRecordSize());
				
				if ((ret=cursor->put(&key, &data, DB_CURRENT))!=0)
					qDebug("Error updating post: %d", ret);
				// 			else qDebug("Error! %d", ret);
				delete p;
				break;
			case BinHeader::No_Change:
				break;
			
		}

		delete bh;
		memset(&key, 0, sizeof(key));
		memset(&data, 0, sizeof(data));
		
	} 
	

	cursor->close();
	ng->high[serverId]=0;
	ng->low[serverId]=0;
	saveGroup(ng);

	
    qDebug("Elements in DB: %d", count);
}

void GroupList::deleteGroup( NewsGroup *ng )
{
		
		QString dbName=ng->ngName;
		QString groupName=ng->ngName;
		if (ng->getView())
			emit sigCloseView(ng->getView());
		//group has autoDelete enabled??
		groups.remove(groupName);
		
		//Delete from db
		Dbt *key=new Dbt;
		char *keymem=new char[dbName.length()];
		key->set_data(keymem);
		key->set_flags(DB_DBT_USERMEM);
		const char *p=(const char *) dbName;
		memcpy(keymem, p, dbName.length());
		key->set_size(dbName.length());
		int ret=groupDb->del(NULL, key, 0);
		if (ret != 0)
			qDebug("Error deleting key from groupsDb: %d", ret);
		QFile* dbFile=new QFile(dbPath + "/" + dbName);
		dbFile->remove();
		emit unsubscribe(groupName);

		//Delete db file...
		delete key;
		delete keymem;
		delete dbFile;
		
		
		
}

void GroupList::slotGroupProperties( )
{
	QPtrList<QListViewItem> selection=m_groupList->selectedItems();
	QPtrListIterator<QListViewItem> it(selection);
	if (it.current()->rtti() != GITEM)
		return;
	NGListViewItem *selected = (NGListViewItem*) it.current();
	
	
	NewsGroup *ng=selected->getNg();
	QStringList cats;
	//Iterate on the categories and create a QStringList to load in the combobox...
	QDictIterator<Category> git(categories);
	while (git.current()) {
// 		qDebug("Found category: %s", (const char *) git.current()->getCategory());
		cats.append(git.current()->name);
		++git;
	}
	
// 	AddForm *af=new AddForm(ng->ngName, ng->saveDir, this);
	AddForm *af=new AddForm(ng, servers, cats, this);
	connect(af, SIGNAL(newGroup(QStringList, Group* )), this, SLOT(slotModifyServerProperties(QStringList )));
	connect(af, SIGNAL(saveGroup(NewsGroup* )), this, SLOT(saveGroup(NewsGroup* )));
	connect(af, SIGNAL(resetServerInGroup(NewsGroup*, int )), this, SLOT(deleteAllArticles(NewsGroup*, int )));
	connect(af, SIGNAL(addCat(QString )), this, SLOT(slotAddCategory(QString )));
	af->exec();
}

void GroupList::slotNewGroup( QStringList entries, Group *g )
{
	//field 0 is name, field1 is the savedir, 2 is the alias, 3 is the category
	
	if (groups.find(entries[0]) != 0) {
		KMessageBox::error(this, i18n("Group is already subscribed"), i18n("Error"));
		return;
	}
	NewsGroup *tempNg = new NewsGroup(dbenv, entries[0], entries[1]);
	tempNg->setAlias(entries[2]);
	tempNg->setCategory(entries[3]);
	tempNg->setDeleteOlder(entries[4].toInt());
	groups.insert(tempNg->getName(), tempNg);
	Servers::iterator it;
	for (it =servers->begin(); it != servers->end(); ++it) {
		if ((!g->serverPresence.contains(it.key())) || (g->serverPresence[it.key()] == false)  )
			tempNg->high[it.key()] = -1;
	}
	//Display the new group...
	if (tempNg->getCategory() != "None") {
		//Add group to the category...
		tempNg->setListItem(new NGListViewItem(categories[tempNg->getCategory()]->listItem, tempNg));
		
	} else tempNg->setListItem(new NGListViewItem(m_groupList, tempNg));
	
	saveGroup(tempNg);
	emit sigHaveGroups(true);
	emit subscribed(g);
	
	
}



void GroupList::slotModifyServerProperties( QStringList entries)
{
	//field 0 is name, field1 is the savedir, 2 is the alias, 3 is the category
	NewsGroup *ng=groups[entries[0]];
	ng->saveDir=entries[1] + '/';
	ng->setAlias(entries[2]);
	//Move the item if needed, and save the category
	if ( ng->getCategory() != entries[3] ) {
		
		if (ng->getCategory() == "None") {
			//Item is top level, take it from the list.
			m_groupList->takeItem(ng->listItem);
		} else {
			//Take the item from the "old" parent...
			categories[ng->getCategory()]->listItem->takeItem(ng->listItem);
			categories[ng->getCategory()]->childs--;
			categories[ng->getCategory()]->listItem->setText(2, QString::number(categories[ng->getCategory()]->childs));
		}
		//Item is taken, insert in the new place...
		if (entries[3] == "None") {
			//insert toplevel
			m_groupList->insertItem(ng->listItem);
		} else {
			//Insert as child of category
			categories[entries[3]]->listItem->insertItem(ng->listItem);
			categories[entries[3]]->childs++;
			categories[entries[3]]->listItem->setText(2, QString::number(categories[entries[3]]->childs));
			
		}
					
		ng->setCategory(entries[3]);	
	}
	ng->setDeleteOlder(entries[4].toInt());
	
	
	saveGroup(ng);
	ng->listItem->setText(0, ng->getAlias());
	
}

void GroupList::slotAddCategory( QString catName)
{
	//Add the category to "categories" and to the listview...
	Category *cat = new Category;
	cat->childs=0;
	cat->name=catName;
	cat->listItem=new KListViewItem(m_groupList, cat->name, "0");
// 	cat->listItem->setExpandable(true);
	categories.insert(catName, cat);
	cat->listItem->setPixmap(0,BarIcon("folder", KIcon::SizeSmall));
}

uchar * GroupList::getBinHeader( QString index, QString group )
{
// 	qDebug("Getting header from %s", group.latin1());
	return groups[group]->getBinHeader(index);
}

NewsGroup * GroupList::getNg( QString group )
{
	return groups[group];
}

void GroupList::slotRemoveCategory( )
{
	QPtrList<QListViewItem> selection=m_groupList->selectedItems();
	QPtrListIterator<QListViewItem> it(selection);
	QListViewItem *item=it.current();
	
	if (item->childCount() == 0) {
		//Remove the item
// 		qDebug("About to remove %s", (const char *) item->text(0));
		delete item;
	} else KMessageBox::error(this, i18n("Can only remove empty categories!"), i18n("Error"));
		
}

void GroupList::checkGroups( )
{
	if (groups.isEmpty())
		emit sigHaveGroups(false);
	else emit sigHaveGroups(true);
}

void GroupList::slotCompactDbs( )
{
	//Try 1: Do not disable groups...use KProgressDlg
	KProgressDialog *kpd = new KProgressDialog(this , NULL, i18n("Compacting..."), NULL, true);
	kpd->showCancelButton(false);
	kpd->setAllowCancel(false);
	kpd->setMinimumDuration(1500);

	
	Dbt key, data;
	memset(&key, 0, sizeof(key));
	memset(&data, 0, sizeof(data));
	key.set_flags(DB_DBT_MALLOC);
	data.set_flags(DB_DBT_MALLOC);
	//For every newsgroup....
	NewsGroup *ng; //Current newsgroup
	QString tempName; //Name for temp newsgroup
	Db* newDb; //Handle for the new Db.
	Db* oldDb; //Copy of the "old" Db handle
	
	Dbc *cursor; //To scan the "old" newsgroup
	QTime current, previous;
	
	QDictIterator<NewsGroup> it(groups);
	kpd->show();
	int gCount=0;
	int totalNewsgroups=groups.count();
	
	while(it.current()) {
		ng=it.current();
		kdDebug() << "Compacting " << ng->getName() << endl;
		qDebug("Compacting %s", ng->getName().latin1());
		++it;
		gCount++;
		//-Create the new Db.
		newDb=new Db(dbenv,0);
		tempName=ng->getName() + ".new";
		if (newDb->open(NULL, tempName.latin1(), NULL, DB_BTREE, DB_THREAD | DB_CREATE, 0644) != 0) {
			qDebug("Error opening temp newsgroup");
			//Display a dialog box...
			return;
		}
		oldDb=ng->getDb();
		oldDb->cursor(0, &cursor,0);
		
		float total=(float)ng->getTotal();
		//Move data
		kpd->setLabel(i18n("Compacting %3 (%1/%2)").arg(gCount).arg(totalNewsgroups).arg(ng->getName()));
		uint unreadArticles=0;
		uint totalArticles=0;
		previous=QTime::currentTime();
		BinHeader *bh;
		while (cursor->get(&key, &data, DB_NEXT) != DB_NOTFOUND  ) {
			
			int size = data.get_size();
			int storedSize;
			uchar *p = (uchar*) data.get_data();
			memcpy(&storedSize, p + size-szInt, szInt);
			if (storedSize != size) {
				kdDebug() << "Error inside db: stored size different than actual key size!\n";
			} else {
				totalArticles++;
				bh = new BinHeader((uchar*)data.get_data());
				if (bh->getStatus() == BinHeader::bh_new)
					unreadArticles++;
				delete bh;
				if (newDb->put(0, &key, &data, 0) != 0)
					qDebug("Error putting data into new db!");
			}
			
			free(key.get_data());
			memset(&key, 0, sizeof(key));
			key.set_flags(DB_DBT_MALLOC);
			
			free(data.get_data());
			memset(&data, 0, sizeof(data));
			data.set_flags(DB_DBT_MALLOC);
			current=QTime::currentTime();
			if (previous.msecsTo(current) > 1000)  {
// 				pd->setProgress( uint((float(count)/float(ng->getTotal())) * 100));
				kpd->progressBar()->setProgress( uint((float(totalArticles)/float(total)) * 100));
				kapp->processEvents();
				
// 				previous=current;
				previous=QTime::currentTime();
			}
		}
		kdDebug() << ng->getName() << " Compacted\n";
		qDebug("%s Compacted", ng->getName().latin1());
		cursor->close();
		
		ng->setTotal(totalArticles);
		ng->listItem->setText(Total_Col, QString::number( totalArticles));
		
		ng->setUnread(unreadArticles);
		ng->listItem->setText(Unread_Col, QString::number(unreadArticles));
		
// 		qDebug("Moved %d records", count);
		//Rename db.
		newDb->sync(0);
		ng->close();
		
		Db *tempDb;
		tempDb=new Db(dbenv,0);
		tempDb->remove(ng->getName().latin1(),0, 0);
		delete tempDb;
		
		newDb->close(0);
		delete newDb;
		
		
		tempDb=new Db(dbenv,0);
		
		tempDb->rename(tempName.latin1(), 0, ng->getName(), 0);
		
		delete tempDb;
		ng->open();
		ng->stopUpdating();
		//Reassign newsgroup pointer
		
	}
	delete kpd;

	
	
	
	
}

void GroupList::slotSaveSettings( NewsGroup *ng, bool unread, bool complete)
{
// 	qDebug("Saving group");
	ng->showOnlyUnread=unread;
	ng->showOnlyComplete=complete;
	ng->listItem->setText(Unread_Col, QString::number(ng->unreadArticles));
	ng->listItem->setText(Total_Col, QString::number(ng->totalArticles));
// 	qDebug("Saving settings");
// 	qDebug("Unread: %d", ng->onlyUnread());
// 	qDebug("Complete: %d", ng->onlyComplete());
	saveGroup(ng);
}

void GroupList::updateCurrent( NewsGroup * ng )
{
	groups[ng->getName()]->listItem->setEnabled(false);
	
	///@todo disable action? What if the user starts clicking the update button like a mad monkey?
	
	emit updateNewsGroup(ng,0 );
	
}

void GroupList::slotUpdateWOptions( )
{
	kdDebug() << "Update with options\n";
	int headers;
	UpdOpt* upd = new UpdOpt(headers, this );
	if (upd->exec() == QDialog::Rejected) {
		kdDebug() << "Rejected\n";
	} else {
		
		kdDebug() << "Headers to download: " << headers << endl;
		QPtrList<QListViewItem> selection=m_groupList->selectedItems();
		QPtrListIterator<QListViewItem> it(selection);
		NGListViewItem *selected;
	
		while (it.current()) {
		
			if (it.current()->rtti() == GITEM) {
				selected = (NGListViewItem*) it.current();

				selected->setEnabled(false);
				emit updateNewsGroup(selected->getNg(), headers );
			}
			++it;
		
			
		}
		slotSelectionChanged();
	}
	
	
	
	delete upd;
}



void GroupList::slotValidateGroup( )
{
	qDebug("Validate selected newsgroup(s)");
	QPtrList<QListViewItem> selection=m_groupList->selectedItems();
	QPtrListIterator<QListViewItem> it(selection);
	NGListViewItem *selected;
	
	while (it.current()) {
		
		if (it.current()->rtti() == GITEM) {
			selected = (NGListViewItem*) it.current();

			selected->setEnabled(false);
#ifndef NDEBUG
			selected->getNg()->validate();
#endif
			selected->setEnabled(true);
		}
		++it;
		
			
	}
	slotSelectionChanged();
}






/*$SPECIALIZATION$*/


#include "grouplist.moc"

