/*************************************************/
/* methods for class Device                      */
/*                                               */
/* logical simulation for wired gates            */
/*                                               */
/* Andreas Rostin                                */
/* 15.03.99                                      */
/*************************************************/
#include <stdio.h>
#include <string.h>
#include <unistd.h>

#include <qstring.h>
#include <qregexp.h>

#include "klogic.h"
#include "device.h"
#include "tool.h"
#include "deviceTypes.h"
#include "busOutputArray.h"
#include "symbolOpStack.h"

/***************************************************/
/* static methods of Device                        */
/***************************************************/
int Device::STATcnt[] = {0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0};
const char * Device::STATtext[] = {"and", "or", "inv", "xor", "7s", "switch", "osc", "", "rs", "d",
			     "jk", "led", "in", "out", "net", "pwr", "text", "sub", "led", "led",
			     "led", "tri", "ram", ""};
int Device::STATdef_delay = Device::DEFAULT_DELAY;
int Device::STATdef_undefined = Device::DEFAULT_UNDEF;
int Device::STATdef_clock = CLK_FALLING_1EDGE;
int Device::STATreverse_trigger = 0;
bool Device::IMPORT_IGNORE_GLOBAL = false;
int Device::instance = 0;

// static method
void Device::resetCounter()
{	int i;

	for (i = 0; i < 20; i++)
		STATcnt[i] = 0;
}

// static method
int Device::defDelay()
{
	return STATdef_delay;
}

// static method
void Device::setDefDelay(int newi)
{
	if (newi > MAXDELAY) STATdef_delay = MAXDELAY;
	else if (newi < MINDELAY) STATdef_delay = MINDELAY;
	else STATdef_delay = newi;
}

// static method
int Device::defUndefined()
{
	return STATdef_undefined;
}

// static methods
void Device::setDefUndefined(int newi)
{
	if (newi >= 1) STATdef_undefined = 1;
	else STATdef_undefined = 0;
}

int Device::defClock()
{
	switch (STATdef_clock) {
		case CLK_RISING_2EDGE:
			STATdef_clock = CLK_RISING_1EDGE;
		case CLK_FALLING_2EDGE:
			STATdef_clock = CLK_FALLING_1EDGE;
	}
	return STATdef_clock;
}

void Device::setDefClock(int newi)
{
	if ((newi >= CLK_RISING_2EDGE) && (newi <= CLK_FALLING_1EDGE))
		STATdef_clock = newi;
	else STATdef_clock = CLK_FALLING_1EDGE;

	switch (STATdef_clock) {
		case CLK_RISING_2EDGE:
			STATdef_clock = CLK_RISING_1EDGE;
		case CLK_FALLING_2EDGE:
			STATdef_clock = CLK_FALLING_1EDGE;
	}
}

void Device::invertTrigger(int newi)
{
	STATreverse_trigger = newi;
}

int Device::triggerInverted()
{
	return STATreverse_trigger;
}

/***************************************************/
/* methods of Device                               */
/***************************************************/
// Constructor
Device::Device(int func, int del, int undef, int clock)
{
	instance++;

	id = uniqueID::getID();

	// these counters are only used to get a new name!
	char buf[101];
	sprintf(buf, "%s%d", STATtext[func - DeviceType::fBASE], ++STATcnt[func - DeviceType::fBASE]);
	text = buf;

	function = func;

	if (clock < 0) {
		if (function == DeviceType::fRS) clock_type = CLK_NONE;
		else clock_type = Device::STATdef_clock;
	} else {
		clock_type = clock;
	}

	// master-slave flipflop: master enabled
	if (function == DeviceType::fRS) master = 0;
	else master = 1;

	// detect changes of the output-value
	force_output_changed = 1;
	old_result = 0;
	new_result = 1;

	// all outputs have the same undefined output
	int undef_value;
	if (undef < 0) {
		if (Device::IMPORT_IGNORE_GLOBAL) {
			undef_value = Device::DEFAULT_UNDEF;
		} else {
			undef_value = Device::STATdef_undefined;
		}
	} else {
		undef_value = undef;
	}

	// all outputs have the same delay
	int delay;
	if (del < 0) {
		if (Device::IMPORT_IGNORE_GLOBAL) {
			delay = Device::DEFAULT_DELAY;
		} else {
			delay = Device::STATdef_delay;
		}
	} else {
		delay = del;
	}
	if (function == DeviceType::fNET ||
	    function == DeviceType::fLEDred ||
	    function == DeviceType::fLEDblue ||
	    function == DeviceType::fLEDgreen ||
	    function == DeviceType::fLEDyellow ||
	    function == DeviceType::fSS ||
            function == DeviceType::fSWI) delay = 0;

	// the first output in the list of outputs is an unnamed output, each device can use
	// it can be reached by "named_output.Get()->"
	opStack *op = new opStack(this, undef_value, delay);
	op->setPosition(-1);
	named_output.Append(op);

	static_input_value = 0;

	// no of active tristate outputs
	tristate = 0;
	tristate_active = false;

	// internally used input- or output-id
	output_bus = (BusOutputArray *)NULL;
	mq_id = 0;
	m_c1_1_id = 0;
	tristate_enable_id = 0;
}

// Destructor
Device::~Device()
{
	instance--;

	// destroy lists
	connection_list.Destroy();

	KlogicList<Value> *l = named_input.First();
	while(l) {
		delete l->Get();
		l = l->Next();
	}
	named_input.Destroy();

	KlogicList<opStack> *o = named_output.First();
	while(o) {
		delete o->Get();
		o = o->Next();
	}
	named_output.Destroy();
}

void * Device::getInstance()
{
	return (void *)this;
}

int Device::clock()
{
	return clock_type;
}

void Device::setClock(int newclock)
{
	// clock types added for JK in feb 2003
	// this is to avoid conflicts with other FF loaded from file
	if (type() != DeviceType::fJK) {
		if (newclock == CLK_RISING_2EDGE)
			newclock = CLK_RISING_1EDGE;
		if (newclock == CLK_FALLING_2EDGE)
			newclock = CLK_FALLING_1EDGE;
	}

	clock_type = newclock;

	setEquation();
	parseEquation();
}

int Device::hasMaster()
{
	return master;
}

// enable/disable master of a master-slave flipflop
void Device::setMaster(int newi)
{
	if (newi >= 1)
		master = 1;
	else
		master = 0;
	setEquation();
	parseEquation();
}

const char * Device::getText(int nospace)
{	
	static QString ret;

	if (!nospace) return (const char *)text;

	ret = text;
	ret.replace(QRegExp(" "), "_" );
	return (const char *)ret;
}

void Device::setText(const QString& new_text)
{
	text = new_text;
}

// return true if output is at low resistance
bool Device::outputIsActive(int output_id)
{
	KlogicList<opStack> *op = named_output.With(output_id);
	if (op->Get()->getResult() == TRISTATE) return false;
	return true;
}

// enable/disable tristate mode for a single output
// make sure there is an "EN" input available!
void Device::outputSetTristate(int output_id, bool flag)
{
	KlogicList<opStack> *op = named_output.With(output_id);
	if (op && op->Get()) {
		if (op->Get()->isInternal()) return;
		if (flag && op->Get()->isTristate()) return;
		if (!flag && !op->Get()->isTristate()) return;
		op->Get()->setTristate(flag);
	} else return;

	// check that device still has tristate outputs
	if (flag) tristate++;
	else tristate--;
}

// return tristate mode for a single output
bool Device::outputIsTristate(int output_id)
{
	KlogicList<opStack> *op = named_output.With(output_id);
	if (op) return op->Get()->isTristate();
	return false;
}

// enable/disable tristate mode for the complete device
void Device::setTristate(bool flag)
{
	KlogicList<opStack> *op = named_output.First();
	while(op) {
		outputSetTristate(op->getID1(), flag);
		op = op->Next();
	}
}

// returns wether device has tristate output or not
bool Device::isTristate()
{
	return tristate;
}

// returns wether input is the tristate control input or not
bool Device::isTristateControlInput(int input_id)
{
	if (!input_id || input_id != tristate_enable_id) return 0;
	return 1;
}

// connect two devices
//	out_dev:   connected device
//	input_id:  id of this named input
//	output_id: id of out_dev's named output
int Device::connectDevices(Device *out_dev, int input_id, int output_id)
{
	//fprintf(stdout, "connect %s(%d) --> %s(%d)\n", out_dev->getText(), output_id, getText(), input_id);
	if (!connection_list.Append(out_dev, input_id, output_id)) return 0;
	return 1;
}

// disconnect two devices
int Device::disconnectDevices(Device *out_dev, int input_id, int output_id)
{
	//fprintf(stdout, "disconnect %s(%d) --> %s(%d)\n", out_dev->getText(), output_id, getText(), input_id);
	if (!connection_list.Destroy(out_dev, input_id, output_id)) return 0;
	return 1;
}

// return type of device
int Device::type()
{	
	return function;
}

// change LED color by changing the type
void Device::changeLEDType(int new_type) {
	if (new_type != DeviceType::fLEDred &&
	    new_type != DeviceType::fLEDblue &&
	    new_type != DeviceType::fLEDgreen &&
	    new_type != DeviceType::fLEDyellow) return;
	function = new_type;
}

// static: devices with user interaction
int Device::isInteractive()
{	
	if (function == DeviceType::fSWI)
		return 1;
	return 0;
}

// check if name is unused
bool Device::inputNameIsUnique(QString& input_name)
{
	if (named_input.With(input_name)) return 0;
	return 1;
}

// add a new named input for this device
// pos contains the the position (GRID times related to device-image) of the named input device
// the input_id is used for import purposes!
int Device::addInputName(const QString& name, int pos, int input_id)
{
	Value *v = new Value;
	v->setPosition(pos);
	if (!input_id) input_id = uniqueID::getID();
	named_input.Append(v, input_id)->setText(name);
	return input_id;
}

// change the name of an input
int Device::changeInputName(int input_id, const QString& new_name)
{	KlogicList<Value> *v = named_input.With(input_id);
	if (v) v->setText(new_name);
	return input_id;
}

// remove an existing named input from this device
void Device::removeInputName(int input_id)
{	KlogicList<Value> *v = named_input.With(input_id);
	if (v) {
		delete v->Get();
		named_input.Destroy(input_id);
	}
}

// add a new named output for this device
// pos contains the the position (GRID times related to device-image) of the named output device
// the input_id is used for import purposes!
int Device::addOutputName(const QString& name, int pos, int output_id)
{
	opStack *op = new opStack(this, undef(), delay());
	op->setPosition(pos);
	if (!output_id) output_id = uniqueID::getID();
	named_output.Append(op, output_id)->setText(name);
	if (tristate) outputSetTristate(output_id, true);
	return output_id;
}

// change the name of an output
int Device::changeOutputName(int output_id, const QString& new_name)
{	
	KlogicList<opStack> *op = named_output.With(output_id);
	if (op) {
		op->setText(new_name);
	}
	return output_id;
}

// remove a named output from this device
void Device::removeOutputName(int output_id)
{
	KlogicList<opStack> *op = named_output.With(output_id);
	if (op && op != named_output.First()) {
		if (tristate) outputSetTristate(output_id, false);
		delete op->Get();
		named_output.Destroy(output_id);
	}
}

// return the name of an output id (equation devices: unique names!)
const char * Device::getOutputName(int output_id)
{	KlogicList<opStack> *lop = named_output.With(output_id);

	if (lop) return lop->getText();
	return (char *)NULL;
}

// return the id of an output name (equation devices: unique names!)
int Device::getInputID(QString& input_name)
{	KlogicList<Value> *lv = named_input.With(input_name);

	if (lv) return lv->getID1();
	return 0;
}

// return the name of an output id (equation devices: unique names!)
int Device::getOutputID(QString& output_name)
{	KlogicList<opStack> *lop = named_output.With(output_name);

	if (lop) return lop->getID1();
	return 0;
}

// change type of output
void Device::setOutputNameType(int output_id, int new_type)
{	KlogicList<opStack> *op = named_output.With(output_id);

	if (op && op->Get()) {
		if (new_type == INTERNAL_OUTPUT) {
			if (tristate) outputSetTristate(output_id, false);
			op->Get()->setInternal(true);
		} else {
			if (tristate) outputSetTristate(output_id, true);
			op->Get()->setInternal(false);
		}
	}
}

// add a new named internal output for this device
int Device::addInternalName(const QString& name)
{	int int_id = uniqueID::getID();

	opStack *op = new opStack(this, undef(), delay());
	op->setInternal(true);
	op->setPosition(0);
	named_output.Append(op, int_id)->setText(name);
	return int_id;
}

// returns wether id is an internal output name, or should be final or temporary
int Device::suggestedOutputType(int internal_id)
{	KlogicList<opStack> *op = named_output.With(internal_id);

	if (op && op->Get()->isInternal()) return INTERNAL_OUTPUT;
	if (type() == DeviceType::fLEDred ||
	    type() == DeviceType::fLEDblue ||
	    type() == DeviceType::fLEDgreen ||
	    type() == DeviceType::fLEDyellow ||
	    type() == DeviceType::fSS) return FINAL_OUTPUT;

	if (type() == DeviceType::fNET || type() == DeviceType::fEQU) {
		if (connection_list.With(internal_id)) return FINAL_OUTPUT;
	}

	return TMP_OUTPUT;
}

// remove a named internal output from this device
void Device::removeInternalName(int internal_id)
{
	removeOutputName(internal_id);
}

KlogicList<Value> * Device::getNamedIRef()
{
	return &named_input;
}

KlogicList<opStack> * Device::getNamedORef()
{
	return &named_output;
}

bool Device::hasNamedInput()
{
	if (function == DeviceType::fSS ||
	    function == DeviceType::fRS ||
	    function == DeviceType::fDFF ||
	    function == DeviceType::fNET ||
	    function == DeviceType::fEQU ||
	    function == DeviceType::fTRI)
		return true;
	return false;
}

bool Device::hasNamedOutput()
{
	if (function == DeviceType::fNET ||
	    function == DeviceType::fEQU ||
	    function == DeviceType::fTRI)
		return true;
	return false;
}

bool Device::hasStaticInput()
{
	if (function == DeviceType::fSWI || function == DeviceType::fPWR) return true;
	return false;
}

// set static output-value
void Device::toggleStaticInput()
{
	if (static_input_value) static_input_value = 0;
	else static_input_value = 1;
	named_output.Get()->flush(static_input_value);
	forceOutputChange();
}

// set static input-value
void Device::setStaticInput(int val)
{
	static_input_value = val;
	named_output.Get()->flush(static_input_value);
	forceOutputChange();
}

// set static output-value
void Device::setStaticOutput(int output_id, int val)
{	KlogicList<opStack> *lop = named_output.With(output_id);

	if (lop && lop->Get()) {
		if (lop->Get()->getResult() != val) {
			old_result = new_result;
			new_result -= lop->Get()->getResult() * lop->current();
			lop->Get()->flush(val);
			new_result += lop->Get()->getResult() * lop->current();
			return;
		}
		return;
	}
	lop = named_output.First();
	fatal("Device::setStaticOutput: unknown static output %d\n", output_id);
}

bool Device::hasClock()
{
	if (function == DeviceType::fDFF || function == DeviceType::fRS)
		return true;
	return false;
}

void Device::setID(int newid)
{
	id = newid;
}

// name of device
int Device::getID()
{
	return id;
}

// set value for undefined outputs
void Device::setUndef(int undef)
{
	KlogicList<opStack> *lo = named_output.First();
	while(lo) {
		lo->Get()->setUndefined(undef);
		lo = lo->Next();
	}
}

int Device::undef()
{
	return named_output.Get()->getUndefined();
}

// set signal delay
void Device::setDelay(int d)
{
	KlogicList<opStack> *lop = named_output.First();
	while(lop) {
		lop->Get()->setDelay(d);
		lop = lop->Next();
	}
}

int Device::delay()
{
	return named_output.Get()->delay();
}

// return input value
// tristateckeck is related to an output for which this method is called
// so it has nothing to do with the input value returned
// changed: using OR function when having more than one connection to a single input
int Device::input(int input_id)
{	bool tristate_found = false;
	bool value_found = false;
	int value = 0;

	KlogicList<Device> *l = connection_list.With(input_id);
	while(l) {
		int _value = l->Get()->output(l->getID2());
		if (_value == TRISTATE) {
			tristate_found = true;	// return another value if possible
		} else {
			value |= _value;
			value_found = true;
		}
		l = l->NextWith(input_id);
	}
	if (value_found) return value;
	if (tristate_found) return TRISTATE;

	return 0;	// no input value
}

// return output value
int Device::output(int output_id)
{
	// internal inverter device
	if (function == DeviceType::fINV_INTERNAL) {
		if (connection_list.First()) {
			int outval = connection_list.First()->Get()->output(connection_list.getID2());
			if (outval == 1) return 0;
			if (outval == 0) return 1;
			return TRISTATE;	// output is in third state
		}
		else {
			if (undef() >= 1) return 0;
			else return 1;
		}
	}

	return named_output.With(output_id)->Get()->getResult();
}

void Device::flush(int value)
{
	KlogicList<opStack> *lop = named_output.First();
	while(lop) {
		lop->Get()->flush(value);
		lop = lop->Next();
	}
}

void Device::forceOutputChange()
{
	force_output_changed = 1;
}

// return 1 if output value has changed since last calculation
int Device::outputChanged()
{
	if (force_output_changed) return 1;
	if (new_result != old_result) return 1;
	//if (function == fIN) return 1;
	return 0;
}

// calculate new output-values (old value at output visible)
void Device::Calculate(int)
{	KlogicList<opStack> *lop;
	opStack *op;
	KlogicList<Device> *l;
	int val;

	// no calculation
	if (function == DeviceType::fTXT || function == DeviceType::fIN) return;

	// check wether tristate input is active or not
	if (tristate) {
		KlogicList<Device> *l_en = connection_list.With(tristate_enable_id);
		if (!l_en || !l_en->Get()->output(l_en->getID2())) {
			// not connected or low: tristate output is at high resistance
			tristate_active = true;
		} else {
			// low resistance
			tristate_active = false;
		}
	} else tristate_active = false;

	// determine new output value(s)
	// calculate all stacks
	lop = named_output.First();
	while(lop) {
		op = lop->Get();
		// static operation devices: push all input values to the calculation stack
		if (op->hasStaticOP()) {
			if (function == DeviceType::fIN || function == DeviceType::fSWI || function == DeviceType::fPWR) {
				op->push(static_input_value);
			} else {
				// e.g. the AND gate
				l = connection_list.First();
				while(l) {
					// id2: id of the named output of the connected device
					val = l->Get()->output(l->getID2());
					op->push(val);
					l = l->Next();
				}
			}
		}

		// calculate new output value
		op->calculate(tristate_active);
		lop = lop->Next();
	}
}

// set output values of all devices
// most of it is detection of output changes .. (to be optimized?)
void Device::Propagate(int burst_step)
{
	// no calculation
	if (function == DeviceType::fTXT || function == DeviceType::fIN) return;

	// switch all outputs
	KlogicList<opStack> *lop = named_output.First();
	if (force_output_changed) {
		new_result = 1;
		old_result = 0;
		force_output_changed = 0;
		while(lop) {
			lop->Get()->shift();
			lop = lop->Next();
		}
	} else {
		int result_weight = 0;
		int val;
		if (burst_step) {
			new_result = 0;
			while(lop) {
				result_weight++;
				lop->Get()->shift();
				val = lop->Get()->getResult();
				if (val == TRISTATE) val = 2;
				new_result += val * result_weight;
				lop = lop->Next();
			}
		} else {
			// first burst_step
			new_result = 0;
			old_result = 0;
			while(lop) {
				result_weight++;

				val = lop->Get()->shift();
				if (val == TRISTATE) val = 2;
				old_result += val * result_weight;

				val = lop->Get()->getResult();
				if (val == TRISTATE) val = 2;
				new_result += val * result_weight;

				lop = lop->Next();
			}
		}
	}
}

// return equation string for an output
QString Device::getEquation(int output_id)
{
	KlogicList<opStack> *lo = named_output.With(output_id);
	if (lo) return lo->Get()->getEquation();
	else return QString("");
}

// set equation operation for predefined devices
void Device::setEquation()
{       int clk = clock();

	if (STATreverse_trigger) {
		switch(clk) {
			case CLK_RISING_2EDGE:
				clk = CLK_FALLING_2EDGE;
				break;
			case CLK_RISING_1EDGE:
				clk = CLK_FALLING_1EDGE;
				break;
			case CLK_FALLING_2EDGE:
				clk = CLK_RISING_2EDGE;
			case CLK_FALLING_1EDGE:
				clk = CLK_RISING_1EDGE;
				break;
			case CLK_HIGH_VALUE:
				clk = CLK_LOW_VALUE;
				break;
			case CLK_LOW_VALUE:
				clk = CLK_HIGH_VALUE;
				break;
		}
	}

	switch(function) {
	case DeviceType::fONE:
		setEquation(OP_NONE);
		break;
	case DeviceType::fTRI:
		setEquation("D0", (*output_bus)[0]);
		setEquation("D1", (*output_bus)[1]);
		setEquation("D2", (*output_bus)[2]);
		setEquation("D3", (*output_bus)[3]);
		setEquation("D4", (*output_bus)[4]);
		setEquation("D5", (*output_bus)[5]);
		setEquation("D6", (*output_bus)[6]);
		setEquation("D7", (*output_bus)[7]);
		break;
	case DeviceType::fPWR:
		setEquation(OP_NONE);
		break;
	case DeviceType::fSWI:
		setEquation(OP_NONE);
		break;
	case DeviceType::fTXT:
		setEquation(OP_NONE);
		break;
	case DeviceType::fIN:
		setEquation(OP_INTERFACE);
		break;
	case DeviceType::fOUT:
		setEquation(OP_INTERFACE);
		break;
	case DeviceType::fXOR:
		setEquation(OP_XOR);
		break;
	case DeviceType::fAND:
		setEquation(OP_AND);
		break;
	case DeviceType::fOR:
		setEquation(OP_OR);
		break;
	case DeviceType::fLEDred:
	case DeviceType::fLEDblue:
	case DeviceType::fLEDgreen:
	case DeviceType::fLEDyellow:
		setEquation(OP_OR);
		break;
	case DeviceType::fSS:
		setEquation("2^0  2^1  2^2  2^3");
		break;
	case DeviceType::fDFF:
		if (master) {
			switch(clk) {
			case CLK_RISING_1EDGE:
				// equations for internal outputs
				setEquation("/(C1 & /C1-1) & MQ + C1 & /C1-1 & 1D", mq_id);
				setEquation("C1", m_c1_1_id);
				// equation for unnamed output
				setEquation("/(/C1 & C1-1) & _this_ + /C1 & C1-1 & MQ");
				break;
			case CLK_FALLING_1EDGE:
				setEquation("/(/C1 & C1-1) & MQ + /C1 & C1-1 & 1D", mq_id);
				setEquation("C1", m_c1_1_id);
				setEquation("/(C1 & /C1-1) & _this_ + C1 & /C1-1 & MQ");
				break;
			case CLK_HIGH_VALUE:
				setEquation("/C1 & MQ + C1 & 1D", mq_id);
				setEquation((char *)NULL, m_c1_1_id);
				setEquation("C1 & _this_ + /C1 & MQ");
				break;
			case CLK_LOW_VALUE:
				setEquation("C1 & MQ + /C1 & 1D", mq_id);
				setEquation((char *)NULL, m_c1_1_id);
				setEquation("/C1 & _this_ + C1 & MQ");
				break;
			}
		} else {
			switch(clk) {
			case CLK_HIGH_VALUE:
				setEquation((char *)NULL, mq_id);
				setEquation((char *)NULL, m_c1_1_id);
				setEquation("/C1 & _this_ + C1 & 1D");
				break;
			case CLK_LOW_VALUE:
				setEquation((char *)NULL, mq_id);
				setEquation((char *)NULL, m_c1_1_id);
				setEquation("C1 & _this_ + /C1 & 1D");
				break;
			}
		}
		break;
	case DeviceType::fRS:
		if (master) {
			switch(clk) {
			case CLK_RISING_1EDGE:
				setEquation("/(C1 & /C1-1) & MQ + /1S & /1R & MQ + C1 & /C1-1 & 1S", mq_id);
				setEquation("C1", m_c1_1_id);
				setEquation("/(/C1 & C1-1) & _this_ + /C1 & C1-1 & MQ");
				break;
			case CLK_FALLING_1EDGE:
				setEquation("/(/C1 & C1-1) & MQ + /1S & /1R & MQ + /C1 & C1-1 & 1S", mq_id);
				setEquation("C1", m_c1_1_id);
				setEquation("/(C1 & /C1-1) & _this_ + C1 & /C1-1 & MQ");
				break;
			case CLK_HIGH_VALUE:
				setEquation("/C1 & MQ + /1S & /1R & MQ + C1 & 1S", mq_id);
				setEquation((char *)NULL, m_c1_1_id);
				setEquation("C1 & _this_ + /C1 & MQ");
				break;
			case CLK_LOW_VALUE:
				setEquation("C1 & MQ + /1S & /1R & MQ + /C1 & 1S", mq_id);
				setEquation((char *)NULL, m_c1_1_id);
				setEquation("/C1 & _this_ + C1 & MQ");
				break;
			}
		} else {
			switch(clk) {
			case CLK_NONE:
				setEquation((char *)NULL, mq_id);
				setEquation((char *)NULL, m_c1_1_id);
				setEquation("/S & /R & _this_ + S");
				break;
			case CLK_HIGH_VALUE:
				setEquation((char *)NULL, mq_id);
				setEquation((char *)NULL, m_c1_1_id);
				setEquation("/C1 & _this_ + /1S & /1R & _this_ + C1 & 1S");
				break;
			case CLK_LOW_VALUE:
				setEquation((char *)NULL, mq_id);
				setEquation((char *)NULL, m_c1_1_id);
				setEquation("C1 & _this_ + /1S & /1R & _this_ + /C1 & 1S");
				break;
			}
		}
		break;
	default:
		// no equation
		break;
	}
}

// set equation operation for the (unnamed) output
void Device::setEquation(char _new_equation)
{
	named_output.Get()->setEquation(_new_equation);
}

// set equation string for an output
void Device::setEquation(const char *_new_equation, int output_id)
{	KlogicList<opStack> *lo;

	if (!output_id) lo = named_output.First();
	else lo = named_output.With(output_id);

	if (lo) lo->Get()->setEquation(_new_equation);
}

// parse all equations, build the calculation stacks for this device
QueueInfo Device::parseEquation()
{	KlogicList<opStack> *lo = named_output.First();
	QueueInfo ret;

	while(lo) {
		ret = lo->Get()->parse(&named_input, &named_output);
		if (ret.no) {
			// error <no> occured at <position> and <length>
			ret.output_name = lo->getText();
			ret.output_id = lo->getID1();
			return ret;
		}
		lo = lo->Next();
	}
	return ret;
}

// clear all calculation stacks for this device
void Device::resetParsing()
{	KlogicList<opStack> *lo = named_output.First();

	while (lo) {
		lo->Get()->clear();
		lo = lo->Next();
	}
}

// deliveres a string like [/][qualifier_prefix.]devicename[.outputname]
// depends on the given device and output name
QString Device::qualifiedOutputName(Device *dev, const char *out_name, const char *qual_pref)
{	QString buf;
	const char *new_out_name;
	KlogicList<Device> *li;
	int dbl_inv = 0;

	const char *prefix=(char *)qual_pref;
	if (!prefix) prefix = "";
	buf[0] = 0;

	if (dev->type() == DeviceType::fINV_INTERNAL) {
		li = dev->connection_list.First();
		if (li) {
			if (li->Get()->type() == DeviceType::fINV_INTERNAL) {
				li = li->Get()->connection_list.First();
				dbl_inv = 1;
			}
		}
		if (li) {
			new_out_name = li->getText();
			if (new_out_name && strlen(new_out_name)) {
				if (dbl_inv) buf.sprintf("%s%s.%s", prefix, li->Get()->getText(NOSPACE), new_out_name);
				else buf.sprintf("/%s%s.%s", prefix, li->Get()->getText(NOSPACE), new_out_name);
			} else {
				if (dbl_inv) buf.sprintf("%s%s", prefix, li->Get()->getText(NOSPACE));
				else buf.sprintf("/%s%s", prefix, li->Get()->getText(NOSPACE));
			}
		}
	} else {
		if (out_name && strlen(out_name))
			buf.sprintf("%s%s.%s", prefix, dev->getText(NOSPACE), out_name);
		else
			buf.sprintf("%s%s", prefix, dev->getText(NOSPACE));
	}

	// remove all spaces from the used names!
	int pos;
	while ( -1 != (pos = buf.find(' ')))
		buf.replace(pos, 1, "_");

	return buf;
}

// append all equations of this device to the given list
// the qualifier will be appended to each name and equation
// isolate: just look at this device, not outside
void Device::getAllEquations(KlogicList<OutputInfo> *ls, const char *qualifier, bool isolate)
{	KlogicList<Device> *lconn;
	KlogicList<Value> *li;
	KlogicList<opStack> *lo;
	QString qual_pref;
	QString final;
	bool ffinal;

	if (!ls) fatal("Device::getAllEquations: no list??\n");

	if (qualifier) qual_pref = qualifier;
	else qual_pref = "";

	// remember the interface circuits
	if (!qualifier && function == DeviceType::fOUT ||
	                  function == DeviceType::fSS ||
			  function == DeviceType::fLEDred ||
			  function == DeviceType::fLEDblue ||
			  function == DeviceType::fLEDgreen ||
			  function == DeviceType::fLEDyellow) {
		final = "(final)";
		ffinal = true;
	} else {
		final = "";
		ffinal = false;
	}

	if (hasNamedInput()) {
		// device has named inputs

		if (!isolate) {
			// build artificial output for each input
			li = named_input.First();
			while(li) {
				// look for a connection (could be more than one)
				int connected = 0;
				QString eq;
				lconn = connection_list.First();
				while(lconn) {
					if (li->getID1() == lconn->getID1()) {
						if (!connected) {
							eq = qualifiedOutputName(lconn->Get(), lconn->Get()->getOutputName(lconn->getID2()), qual_pref);
							connected = 1;
						} else if (connected == 1) {
							eq = "(" + eq + ") + (";
							eq += qualifiedOutputName(lconn->Get(), lconn->Get()->getOutputName(lconn->getID2()), qual_pref);
							eq += ")";
							connected = 2;
						} else {
							eq += " + (";
							eq += qualifiedOutputName(lconn->Get(), lconn->Get()->getOutputName(lconn->getID2()), qual_pref);
							eq += ")";
						}

					}
					lconn = lconn->Next();
				}
				if (connected) {
					OutputInfo *oi = new OutputInfo();
					oi->device_id = this->getID();
					oi->output_id = 0;
					oi->output_name = qualifiedOutputName(this, li->getText(), qual_pref);
					oi->original_output_name = qualifiedOutputName(this, li->getText(), qual_pref);
					oi->output_type = INPUT;
					oi->equation = eq;
					oi->prefix = qual_pref;
					oi->suffix = final;
					ls->Append(oi)->setText(oi->output_name);
				}

				li = li->Next();
			}
		}
	}

	// named output equation handling
	lo = named_output.First();
	while(lo) {
		if (lo->Get()->hasEquation()) {
			QString qual;

			if (!isolate) qual = qual_pref + this->getText(NOSPACE) + ".";
			else qual = qual_pref;

			OutputInfo *oi = new OutputInfo();
			oi->device_id = getID();
			oi->output_id = lo->getID1();
			if (lo->Get()->hasStaticOP()) {
				// concatenate all inputs with the static operation
				lconn = connection_list.First();
				while(lconn) {
					oi->equation += qualifiedOutputName(lconn->Get(), lconn->Get()->getOutputName(lconn->getID2()), qual_pref);
					lconn = lconn->Next();
					if (lconn) {
						oi->equation += " ";
						oi->equation += lo->Get()->getEquation();	// the static operation
						oi->equation += " ";
					}
				}
			} else {
				// the opStack instance has the equation
				oi->equation = lo->Get()->getEquation(qual);
			}
			if (!isolate) {
				oi->output_name = qualifiedOutputName(this, lo->getText(), qual_pref);
				oi->original_output_name = qualifiedOutputName(this, lo->getText(), qual_pref);
			} else {
				oi->output_name = lo->getText();
				oi->original_output_name = lo->getText();
			}
			if (!isolate) {
				if (ffinal) oi->output_type = FINAL_OUTPUT;
				else oi->output_type = suggestedOutputType(oi->output_id);
			} else {
				if (ffinal) oi->output_type = FINAL_OUTPUT;
				else if (suggestedOutputType(oi->output_id) != INTERNAL_OUTPUT) oi->output_type = FINAL_OUTPUT;
				else oi->output_type = INTERNAL_OUTPUT;
			}
			oi->prefix = qual_pref;
			oi->suffix = final;
			//fprintf(stdout, "%s(%s)\n", (const char *)(oi->output_name), (const char *)(oi->suffix));
			ls->Append(oi)->setText(oi->output_name);
		}
		lo = lo->Next();
	}
}

// actualize all equations of this device with the given list
// add needed, and remove needless inputs and outputs
// change_size flag: change size to fit for all needed inputs and outputs
void Device::setAllEquations(KlogicList<OutputInfo> *equations, bool change_size)
{
	QString dummy;

/* debug code
	fprintf(stdout, "-----------------------------------\n");
	KlogicList<OutputInfo> *l = equations->First();
	while(l) {
		fprintf(stdout, "incoming: %s(%d) = %s\n", (const char *)l->Get()->output_name, l->Get()->output_id, (const char *)l->Get()->equation);
		l = l->Next();
	}

	KlogicList<opStack> *lo;
	lo = named_output.First();
	while(lo) {
		if (lo->Get()->hasEquation()) {
			fprintf(stdout, "having: %s(%d) = %s\n", lo->getText(), lo->getID1(), (const char *)lo->Get()->getEquation());
		}
		lo = lo->Next();
	}
	fprintf(stdout, "-----------------------------------\n");
*/

	// -------------------------------------------------------
	// create a list with new, and a list with all output vars
	// -------------------------------------------------------
	KlogicList<QString> all_outputs;
	KlogicList<QString> new_outputs;
	KlogicList<OutputInfo> *lout_new = equations->First();
	int output_id;
	QString output;
	while(lout_new) {
		all_outputs.Append((QString *)&dummy)->setText(lout_new->Get()->output_name);
		int id = getOutputID(lout_new->Get()->output_name);
		if (!id)
			new_outputs.Append((QString *)&dummy)->setText(lout_new->Get()->output_name);
		if (id != lout_new->Get()->output_id)
			lout_new->Get()->output_id = id;
		lout_new = lout_new->Next();
	}

	// -------------------------------------------------------
	// replace/remove output vars
	// -------------------------------------------------------
	KlogicList<opStack> *lop = named_output.First()->Next();
	KlogicList<QString> *lnew = new_outputs.First();
	OutputInfo *oi;
	while (lop) {
		output_id = lop->getID1();
		output = lop->getText();
		lop = lop->Next();

		if (!all_outputs.With(output)) {
			if (lnew) {
				// replace old output with a new one
				oi = equations->Get(lnew->getText());
				output_id = changeOutputName(output_id, oi->output_name);
				setEquation(oi->equation, output_id);
				setOutputNameType(output_id, oi->output_type);
				lnew = lnew->Next();
			} else {
				// nothing to replace: remove old output
				removeOutputName(output_id);
			}
		} else {
			oi = equations->Get(output);
			output_id = changeOutputName(output_id, oi->output_name);
			setEquation(oi->equation, output_id);
			setOutputNameType(output_id, oi->output_type);
		}
	}

	// -------------------------------------------------------
	// add new output vars left in the list
	// -------------------------------------------------------
	while(lnew) {
		if (change_size) setSize(neededSize() + 1);
		output = lnew->getText();
		oi = equations->Get(output);
		output_id = changeOutputName(0, oi->output_name);
		setEquation(oi->equation, output_id);
		setOutputNameType(output_id, oi->output_type);
		lnew = lnew->Next();
	}

	// -------------------------------------------------------
	// create a list with new, and a list with all input vars
	// -------------------------------------------------------
	lout_new = equations->First();
	QString input;
	KlogicList<QString> all_inputs;
	KlogicList<QString> new_inputs;
	while(lout_new) {
		// create a list of all symbols available in equation
		SymbolOpStack oop;
		oop.setEquation(lout_new->Get()->equation);
		oop.parse();

		int i = 0;
		while(NULL != (input = oop.getSymbol(i++))) {
			if (!opStack::isNumber(input)) {
				if (!new_inputs.With(input) && !equations->With(input) && inputNameIsUnique(input))
					new_inputs.Append((QString *)&dummy)->setText((const char *)input);
				if (!all_inputs.With(input))
					all_inputs.Append((QString *)&dummy)->setText((const char *)input);
			}
		}
		lout_new = lout_new->Next();
	}

	// -------------------------------------------------------
	// replace/remove input vars
	// remove input vars if there is an output with equal name
	// -------------------------------------------------------
	KlogicList<Value> *li = named_input.First();
	lnew = new_inputs.First();
	int input_id;
	while(li) {
		input_id = li->getID1();
		input = li->getText();
		li = li->Next();

		if (!isTristateControlInput(input_id)) {   // never change en control input!
			if (!all_inputs.With(input)) {
				if (lnew) {
					// replace old input with a new one
					changeInputName(input_id, lnew->getText());
					lnew = lnew->Next();
				} else {
					// nothing to replace: remove old input
					removeInputName(input_id);
				}
			} else {
				// if there is an output with that name, delete input
				if (all_outputs.With(input)) {
					removeInputName(input_id);
				}
			}
		}
	}

	// -------------------------------------------------------
	// add new input vars left in the list
	// -------------------------------------------------------
	while(lnew) {
		if (change_size) setSize(neededSize() + 1);
		addInputName(lnew->getText());
		lnew = lnew->Next();
	}

/* debug code
	fprintf(stdout, "-----------------------------------\n");
	lo = named_output.First();
	while(lo) {
		if (lo->Get()->hasEquation()) {
			fprintf(stdout, "now having: %s = %s\n", lo->getText(), (const char *)lo->Get()->getEquation());
		}
		lo = lo->Next();
	}
	fprintf(stdout, "-----------------------------------\n\n");
*/
}

