/*************************************************/
/* methods for class DlgPattern                  */
/*                                               */
/* dialog used to display all patterns of an     */
/* equation                                      */
/*                                               */
/* Andreas Rostin                                */
/* 27.10.00                                      */
/*************************************************/
#include <qdialog.h>
#include <qwidget.h>
#include <qscrollbar.h>
#include <qpushbutton.h>
#include <qbuttongroup.h>
#include <qradiobutton.h>
#include <qmessagebox.h>
#include <qpainter.h>
#include <qwmatrix.h>

#include <klogic.h>

#include <dlgPattern.h>
#include "dlgPattern.moc"

DlgPattern::DlgPattern(QWidget *parent, QString _caption, QString _equation)
	: QDialog(parent, _caption, TRUE, WStyle_DialogBorder)
{
	dframe = new QWidget(this);

	diagram = new KarnaughWidget(dframe, _equation);
	diagram->setGeometry(0, 0, 400, 400);
	diagram->setBackgroundMode(PaletteBase);
	diagram->setBackgroundColor(white);

	bgNType = new QButtonGroup(this);
	bgNType->setGeometry(10, 410, 110, 50);
	rbSubjunctive = new QRadioButton(bgNType);
	rbSubjunctive->setGeometry(5, 5, 100, 20);
	rbSubjunctive->setText(i18n("subjunctive"));
	rbSubjunctive->setChecked(TRUE);
	rbDisjunctive = new QRadioButton(bgNType);
	rbDisjunctive->setGeometry(5, 25, 100, 20);
	rbDisjunctive->setText(i18n("disjunctive"));

	bgNSecurity = new QButtonGroup(this);
	bgNSecurity->setGeometry(140, 410, 110, 50);
	rbSecure = new QRadioButton(bgNSecurity);
	rbSecure->setGeometry(5, 5, 100, 20);
	rbSecure->setText(i18n("secure"));
	rbSecure->setChecked(TRUE);
	rbInsecure = new QRadioButton(bgNSecurity);
	rbInsecure->setGeometry(5, 25, 100, 20);
	rbInsecure->setText(i18n("insecure"));

	bNormalize = new QPushButton(this);
	bNormalize->setText(i18n("Refresh"));
	bNormalize->setGeometry(260, 420, 60, 25);

	bOK = new QPushButton(this);
	bOK->setDefault(true);
	bOK->setText(i18n("OK"));

	setCaption(_caption);

	connect(bOK, SIGNAL(clicked()), SLOT(accept()));
	connect(bNormalize, SIGNAL(clicked()), SLOT(refresh()));

	// the size depends on the number of inputs of the equation
	if (diagram->getInputCounter() <= 4) {
		setFixedSize(410, 465);
		dframe->setGeometry(5, 5, 400, 400);
		diagram->setGeometry(0, 0, 400, 400);
		bOK->setGeometry(360, 420, 40, 25);
	} else if (diagram->getInputCounter() == 5) {
		setFixedSize(710, 465);
		diagram->setGeometry(0, 0, 700, 400);
		dframe->setGeometry(5, 5, 700, 400);
		bOK->setGeometry(660, 420, 40, 25);
	} else if (diagram->getInputCounter() == 6) {
		setFixedSize(710, 460);
		diagram->setGeometry(0, 0, 700, 700);
		dframe->setGeometry(5, 5, 700, 400);
		vscroll = new QScrollBar(0, diagram->height() - dframe->height(), 1, 80, 0, QScrollBar::Vertical, dframe);
		vscroll->setGeometry(dframe->width() - 10, 0, 10, dframe->height());
		connect(vscroll, SIGNAL(valueChanged(int)), SLOT(vScroll(int)));
		vScroll(0);
		vscroll->show();
		bOK->setGeometry(660, 420, 40, 25);
	} else {
		setFixedSize(410, 465);
		dframe->setGeometry(5, 5, 400, 400);
		diagram->setGeometry(0, 0, 400, 400);
		bOK->setGeometry(360, 420, 40, 25);
	}
}

// OK button pressed
void DlgPattern::done(int r)
{
	if (r == Accepted) {
	}
	QDialog::done(r);
}

// scroll vertically
void DlgPattern::vScroll(int newval)
{
	diagram->move(diagram->x(), -1 * newval);
}

// calculate and display again with changed parameters
void DlgPattern::refresh() {
	int security, type;

	if (rbSecure->isChecked()) security = 1;
	else security = 0;
	if (rbDisjunctive->isChecked()) type = 0;
	else type = 1;
	diagram->calculateNew(type, security);
}

KarnaughWidget::KarnaughWidget(QWidget *parent, QString _equation)
	: QWidget(parent)
{
	op.setEquation(_equation);
	op.parse();
	op.calculateSymbolicNormalized(1,0);
	op.getPatterns(&input_list, &input_cnt, &pattern, &pattern_cnt);
	op.getGroups(&group, &group_size, &group_cnt);
}

int KarnaughWidget::getInputCounter()
{
	return input_cnt;
}

void KarnaughWidget::calculateNew(int type, int secure)
{
	group = NULL;
	op.calculateSymbolicNormalized(type, secure);
	op.getGroups(&group, &group_size, &group_cnt);
	repaint(TRUE);
}

void KarnaughWidget::paintEvent(QPaintEvent *)
{	QPainter p;

	if (!group) return;

	if (input_cnt > 6) {
		p.begin(this);
		p.setPen(Qt::black);
		p.setFont( QFont( "helvetica", 10, QFont::Bold ) );
		p.drawText(50, 200, i18n("unable to display Karnaugh with more than 6 input variables ...\n"));
		p.end();
		return;
	}

	drawTable();
	drawPattern();
	drawGroups();
}

void KarnaughWidget::drawTable()
{	QPainter p;
	QWMatrix m_rot;
	m_rot.rotate(-90.0);

	switch (input_cnt) {
		case 1:
			p.begin(this);
			p.setPen(Qt::black);
			p.setFont( QFont( "helvetica", 10, QFont::Bold ) );
			p.drawRect(100,100,100,50);
			p.drawLine(150, 90, 150, 150);
			p.drawText(110, 94, input_list[0]);
			p.end();
			break;
		case 2:
			p.begin(this);
			p.setPen(Qt::black);
			p.setFont( QFont( "helvetica", 10, QFont::Bold ) );
			p.drawRect(100,100,100,100);
			p.drawLine(90, 150, 200, 150);
			p.drawLine(150, 90, 150, 200);
			p.drawText(110, 94, input_list[0]);
			p.setWorldMatrix(m_rot, TRUE);
			p.drawText(-140, 94, input_list[1]);
			p.end();
			break;
		case 3:
			p.begin(this);
			p.setPen(Qt::black);
			p.setFont( QFont( "helvetica", 10, QFont::Bold ) );
			p.drawRect(100,100,200,100);
			p.drawLine(90, 150, 300, 150);

			p.drawLine(150, 100, 150, 210);
			p.drawLine(200, 90, 200, 200);
			p.drawLine(250, 100, 250, 210);

			p.drawText(110, 94, input_list[0]);
			p.drawText(160, 210, input_list[1]);
			p.setWorldMatrix(m_rot, TRUE);
			p.drawText(-140, 94, input_list[2]);
			p.end();
			break;
		case 4:
			drawQuadTable(100, 100);
			break;
		case 5:
			drawQuadTable(100, 100);
			drawQuadTable(400, 100);
			break;
		case 6:
			drawQuadTable(100, 100);
			drawQuadTable(400, 100);
			drawQuadTable(100, 400);
			drawQuadTable(400, 400);
			break;
	}
}

void KarnaughWidget::drawQuadTable(int x, int y)
{
	QPainter p;
	QWMatrix m_rot;
	m_rot.rotate(-90.0);

	p.begin(this);
	p.setPen(Qt::black);
	p.setFont( QFont( "helvetica", 10, QFont::Bold ) );

	p.drawRect(x, y, 200, 200);
	p.drawLine(x, y + 50, x + 210, y + 50);
	p.drawLine(x - 10, y + 100, x + 200, y + 100);
	p.drawLine(x, y + 150, x + 210, y + 150);

	p.drawLine(x + 50, y, x + 50, y + 210);
	p.drawLine(x + 100, y - 10, x + 100, y + 200);
	p.drawLine(x + 150, y, x + 150, y + 210);

	if (x == 400 && y == 100) {
		p.drawLine(350, 20, 350, 380);
		QString s = input_list[4];
		s += " = 1";
		p.drawText(180, 20, s);
	}
	if (x == 400 && y == 400) {
		p.drawLine(350, 20, 350, 680);
		p.drawLine(20, 350, 680, 350);
	}

	p.drawText(x + 10, y - 6, input_list[0]);		// b Q1 top
	p.drawText(x + 60, y + 210, input_list[2]);		// a Q3 bottom

	p.setWorldMatrix(m_rot, TRUE);
	if (x == 400 && y == 400) {
		QString s = input_list[5];
		s += " = 1";
		p.drawText(-220, 20, s);
	}
	p.drawText(y * -1 - 90, x - 6, input_list[3]);		// c Q4 left
	p.drawText(y * -1 - 140, x + 212, input_list[1]);	// d Q2 right

	p.end();
}

// calculate the coordinates for the pattern at index i
void KarnaughWidget::getCoord(int *_pattern, int *_x, int *_y)
{
	int x = 125;
	int y = 130;
	if (input_cnt == 1) {
		if (_pattern[0] == 0) x+= 50;
	} else if (input_cnt == 2) {
		if (_pattern[0] == 0) x+= 50;
		if (_pattern[1] == 0) y+= 50;
	} else if (input_cnt == 3) {
		if (_pattern[0] == 1 && _pattern[1] == 1) x+= 50;
		if (_pattern[0] == 0 && _pattern[1] == 1) x+= 100;
		if (_pattern[0] == 0 && _pattern[1] == 0) x+= 150;
		if (_pattern[2] == 0) y+= 50;
	} else if (input_cnt >= 4) {
		if (_pattern[3] == 1 && _pattern[1] == 1) y += 50;
		if (_pattern[3] == 0 && _pattern[1] == 1) y += 100;
		if (_pattern[3] == 0 && _pattern[1] == 0) y += 150;

		if (_pattern[0] == 1 && _pattern[2] == 1) x += 50;
		if (_pattern[0] == 0 && _pattern[2] == 1) x += 100;
		if (_pattern[0] == 0 && _pattern[2] == 0) x += 150;

		if (input_cnt >= 5 && _pattern[4] == 0) x += 300;
		if (input_cnt >= 6 && _pattern[5] == 0) y += 300;
	}
	*_x = x;
	*_y = y;
}

void KarnaughWidget::drawPattern()
{	int i;
	int x, y;
	QPainter p;
	QString s;

	p.begin(this);
	p.setPen(Qt::black);
	p.setFont( QFont( "helvetica", 10, QFont::Bold ) );
	for(i = 0; i < pattern_cnt; i++) {
		getCoord(pattern[i], &x, &y);
		s.sprintf("%d", pattern[i][input_cnt]);
		p.drawText(x, y, s);
		
	}
	p.end();
}

void KarnaughWidget::drawGroups()
{	int i,j, k;
	QPainter p;
	int x, y;
	int maxhier = 100;
	int hier_cnt = 0;
	int hier;
	int hier_found;
	int minx[maxhier], miny[maxhier], maxx[maxhier], maxy[maxhier];
	int xrad, yrad;

	p.begin(this);
	for(i = 0; i < group_cnt; i++) {
		hier_cnt = 0;
		// change the color for each group
		j = i % 10;
		if (j == 0) p.setPen(Qt::darkMagenta);
		if (j == 1) p.setPen(Qt::red);
		if (j == 2) p.setPen(Qt::green);
		if (j == 3) p.setPen(Qt::blue);
		if (j == 4) p.setPen(Qt::cyan);
		if (j == 5) p.setPen(Qt::magenta);
		if (j == 6) p.setPen(Qt::gray);
		if (j == 7) p.setPen(Qt::darkYellow);
		if (j == 8) p.setPen(Qt::darkGreen);
		if (j == 9) p.setPen(Qt::darkRed);

		// determine coordinates of the rectangle
		for(j=0; j < maxhier; j++) {
			minx[j] = -1;
			maxx[j] = -1;
			miny[j] = -1;
			maxy[j] = -1;
		}
		for(j = 0; j < group_size[i]; j++) {
			getCoord(pattern[group[i][j]], &x, &y);

			// try find a matching hierarchy
			hier_found = 0;
			hier = 0;
			while (!hier_found && hier < hier_cnt) {
				if ((abs(minx[hier] - x) <= 50 ||
				     abs(maxx[hier] - x) <= 50) &&
				    (abs(miny[hier] - y) <= 50 ||
				     abs(maxy[hier] - y) <= 50)) {
					if (x < minx[hier]) minx[hier] = x;
					if (x > maxx[hier]) maxx[hier] = x;
					if (y < miny[hier]) miny[hier] = y;
					if (y > maxy[hier]) maxy[hier] = y;
					hier_found = 1;
				}
				hier++;
			}

			// new hierarchy
			if (!hier_found) {
				hier_cnt++;
				hier = hier_cnt - 1;
				minx[hier] = x;
				maxx[hier] = x;
				miny[hier] = y;
				maxy[hier] = y;
			}
		}

		// draw the rectangle(s)
		for(j=0; j < hier_cnt; j++) {
			// try to (visually) join the groups again before painting them
			hier = j + 1;
			while(hier < hier_cnt) {
				if ((abs(minx[hier] - minx[j]) <= 50 ||
				     abs(maxx[hier] - maxx[j]) <= 50) &&
				    (abs(miny[hier] - miny[j]) <= 50 ||
				     abs(maxy[hier] - maxy[j]) <= 50)) {
					if (minx[hier] < minx[j]) minx[j] = minx[hier];
					if (maxx[hier] > maxx[j]) maxx[j] = maxx[hier];
					if (miny[hier] < miny[j]) miny[j] = miny[hier];
					if (maxy[hier] > maxy[j]) maxy[j] = maxy[hier];
					hier_cnt--;
					for(k=hier; k < hier_cnt; k++) {
						minx[k] = minx[k + 1];
						maxx[k] = maxx[k + 1];
						miny[k] = miny[k + 1];
						maxy[k] = maxy[k + 1];
					}
					hier = j + 1;
				} else hier++;
			}
			// set corner radius
			if (maxx[j] - minx[j] < maxy[j] - miny[j]) {
				xrad = 50;
				yrad = 25;
			} else if (maxx[j] - minx[j] > maxy[j] - miny[j]) {
				xrad = 25;
				yrad = 50;
			} else {
				xrad = 30;
				yrad = 30;
			}

			// draw the rectangle
			p.drawRoundRect(minx[j] - 10,
					miny[j] - 20,
					maxx[j] - minx[j] + 25,
					maxy[j] - miny[j] + 30,
					xrad, yrad);
		}
	}
	p.end();
}
