/*************************************************/
/* member methods for class opStack              */
/*                                               */
/* equation analyzer and calculator              */
/*                                               */
/* Andreas Rostin                                */
/* 15.10.99                                      */
/*************************************************/
#include <opStack.h>

#include <qstring.h>
#include <qvalidator.h>

#include <device.h>
#include <tool.h>

int opStack::instance = 0;
QIntValidator opStack::intValidator(0, 64, 0);

opStack::opStack(Device *_dev, int _undef, int _delay)
	:DelayedValue(_delay)
{
	instance++;
	init();
	setUndefined(_undef);
	dev = _dev;
}

opStack::opStack()
	:DelayedValue(0)
{
	instance++;
	init();
}

void opStack::init()
{
	// no of variables used for the calculation of the queue
	total = 0;
	// max number of items for queue, stack and stack_info
	maxitem = 0;
	// number of terminal on the stack
	plane = 0;

	dev = (Device *)NULL;

	equation = (char *)NULL;
	queue = (char **)NULL;
	queue_info = (QueueInfo **)NULL;

	stack = (int *)NULL;

	staticOP = 0;
}

// virtual to avoid warnings only!
opStack::~opStack()
{
	instance--;

	clear();
	if (equation) free(equation);
	if (queue) free(queue);
	if (queue_info) {
		for(int i = 0; i < total; i++)
			delete queue_info[i];
		free(queue_info);
	}
	if (stack) free(stack);
}

// set a new equation
void opStack::setEquation(char _new_equation)
{
	clear();
	if (equation) free(equation);
	equation = (char *)malloc(sizeof(char) * 2);
	equation[0] = _new_equation;
	equation[1] = '\0';
}

// set a new equation
void opStack::setEquation(const char * _new_equation)
{
	setEquation((char *)_new_equation);
}

// set a new equation
void opStack::setEquation(char * _new_equation)
{
	clear();
	if (equation) free(equation);
	if (_new_equation && strlen(_new_equation))
		equation = strdup(_new_equation);
	else
		equation = (char *)NULL;

	// replace double-invertations by spaces
	if (equation) {
		char *buf;
		while (NULL != (buf = strstr(equation, "//"))) {
			*buf = ' ';
			*(buf + 1) = ' ';
		}
	}
}

// an empty equation has a content
int opStack::hasEquation()
{
	if (equation && equation[0] != OP_NONE) return 1;
	return 0;
}

QString opStack::getEquation()
{
	return equation;
}

// build equation with all non-terminal qualified by a string
// attention: allocates/returns uncontrolled memory
//            free char* returned by this method by yourself
QString opStack::getEquation(const char *qualifier)
{	char *buf;
	unsigned int cnt;
	QString new_equation;

	if (!qualifier || !strlen(qualifier)) {
		return QString(equation);
	}

	for(cnt = 0; cnt < strlen(equation); cnt++) {	// dirty loop!
		// copy string byte by byte
		if (equation[cnt] != ' ' && equation[cnt] != '(' && equation[cnt] != ')' && equation[cnt] != OP_AND &&
		    equation[cnt] != OP_OR && equation[cnt] != OP_XOR && equation[cnt] != OP_NOT &&
		    equation[cnt] != OP_BIN && equation[cnt] != OP_BIT) {
			new_equation += qualifier;
			// do not copy "_this_"
			buf = strstr(equation, "_this_");
			if (buf == equation + cnt) {
				cnt += 6;
				new_equation.truncate(new_equation.length() - 1);
			}
			while (cnt < strlen(equation) && equation[cnt] != ' ' && equation[cnt] != '(' && equation[cnt] != ')' &&
			       equation[cnt] != OP_AND && equation[cnt] != OP_OR && equation[cnt] != OP_XOR && equation[cnt] != OP_NOT &&
			       equation[cnt] != OP_BIN && equation[cnt] != OP_BIT) {
				new_equation += equation[cnt++];
			}
			cnt--;
		} else new_equation += equation[cnt];
	}

	return new_equation;
}

char opStack::hasStaticOP()
{
	return staticOP;
}

// debugging method!
void opStack::stackDump()
{
	printf("\n**********\n");
	printf("** Equation: %s\n", equation);
	printf("** Maxitem: %d\n", maxitem);
	for (int i = 0; i < total;i++)
		printf("<%s> at offset %d, length %d\n", queue[i], queue_info[i]->position, queue_info[i]->length);
	printf("**********\n");
}

// reset opStack to init state
void opStack::clear()
{
	if (queue) {
		for (int i = 0; i < maxitem; i++) free(queue[i]);
		free(queue);
		queue = (char **)NULL;
	}

	// modified 12/08/2000 start
	if (queue_info) {
		for(int i = 0; i < total; i++)
			delete queue_info[i];
		free(queue_info);
	}
	// modified 12/08/2000 end
	queue_info = (QueueInfo **)NULL;

	if (stack) free(stack);
	stack = (int *)NULL;

	maxitem = 0;
	total = 0;
	staticOP = 0;
}

// check stack if it is computational
// returns an offset to the suspicious nonterminal/operation contained in equation
// returns 0 on success (!)
// allocate memory for stack and queue, copy values
QueueInfo opStack::check(KlogicList<Value> *input, KlogicList<opStack> *output)
{	int cnt = 0;
	int i;
	QueueInfo ret;
	KlogicList<Value> *li;
	KlogicList<opStack> *lo;

	if (total == 0) return ret;

	// the number of nonterminal and operations must agree (make a test computation!)
	for (i = 0; i < total; i++) {
		switch (queue[i][0]) {
			case OP_AND:
			case OP_OR:
			case OP_XOR:
			case OP_BIT:
			case OP_BIN:
				if (cnt < 2) {
					queue_info[i]->no = STACK_NEG_ERROR;
					return *queue_info[i];
				}
				cnt--;
				break;
			case OP_NOT:
				if (cnt < 1) {
					queue_info[i]->no = STACK_NEG_ERROR;
					return *queue_info[i];
				}
				break;
			default:
				// check if nonterminal exists as an named input or output
				if (!strncmp(queue[i], "_this_", 6)) {
					queue_info[i]->output_name = dev->getText();
					queue_info[i]->output_id = dev->getID();
					queue_info[i]->type = THIS;	// special operand: current output
				} else if (input && NULL != (li = input->With(queue[i]))) {
					queue_info[i]->output_name = input->With(queue[i])->getText();
					queue_info[i]->output_id = li->getID1();
					queue_info[i]->type = Device::INPUT;
				} else if (output && NULL != (lo = output->With(queue[i]))) {
					queue_info[i]->output_name = output->With(queue[i])->getText();
					queue_info[i]->output_id = lo->getID1();
					queue_info[i]->type = Device::FINAL_OUTPUT;
				} else if (isNumber(queue[i])) {
					queue_info[i]->value = atoi(queue[i]);
					queue_info[i]->type = NUMBER;
					queue_info[i]->output_id = 0;
				} else {
					queue_info[i]->no = NTERM_ERROR;
					return *queue_info[i];
				}
				cnt++;
				break;
		}
	}
	if (cnt != 1) {
		ret.no = STACK_POS_ERROR;
	}
	return ret; 
}

QueueInfo opStack::parse()
{
	return parse((KlogicList<Value> *)NULL, (KlogicList<opStack> *)NULL);
}

bool opStack::isNumber(QString str)
{
	int i = 0;
	return QValidator::Acceptable == intValidator.validate(str, i);
}

// public parse method
// transform equation into reverse polish notation and create an operation stack
QueueInfo opStack::parse(KlogicList<Value> *input, KlogicList<opStack> *output)
{	QueueInfo rets;
	int ret;
	int i;

	if (!equation) return rets;
	if (strlen(equation) == 0) return rets;

	// set unique operation on all input values?
	if (strlen(equation) == 1 &&
	    (equation[0] == OP_AND ||
             equation[0] == OP_OR || 
	     equation[0] == OP_XOR ||
	     equation[0] == OP_BIN ||
             equation[0] == OP_NONE ||
             equation[0] == OP_INTERFACE ||
             equation[0] == OP_NOT)) {
		if (stack) free(stack);
		stack = (int *)malloc(sizeof(int) * OP_MAXINP);
		staticOP = equation[0];
		total = 0;
		plane = 0;
		return rets;
	}

	// *** count items, allocate needed memory ***
	clear();

	parse(equation, 0, 1);
	stack = (int *)malloc(sizeof(int) * total);
	queue = (char **)malloc(sizeof(char *) * total);
	queue_info = (QueueInfo **)malloc(sizeof(QueueInfo *) * total);
	for (i = 0; i < total; i++)
		queue_info[i] = new QueueInfo;
	maxitem = total;
	total = 0;

	// *** parse the equation, fillup queue ***
	ret = parse(equation, 0);

	// *** check the stack ***
	if (ret == 0) {
		// iput == output == NULL: do not check equation
		if (input || output) {
			rets = check(input, output);
		}
	} else {
		rets.no = PAR_ERROR;
		rets.position = ret;
		rets.length = 1;
	}

	return rets;
}

// parse the equation and resolve it
// push it to the computation stack in RPN
// if docnt is set, just count the items
int opStack::parse(char *_equation, int current_offset, int docnt)
{	char *bracket_start;	// parenthesis
	int bracket_length;	// len of parenthesis
	char *op;		// operation
	char *left;		// left operand
	int left_length;	// length of left operand
	char *right;		// right operand
	int right_length;	// length of right operand
	int new_offset;		// new current offset relative to the original equation
	int optype;		// type of operator
	int res;

	char *eq = strdup(_equation);

	// find the next operator/bracket
	while (NULL != (op = getOP(eq))) {
		optype = opType(*op);

		// parenthesis found
		if (optype == OP_PARENTHESIS) {
			bracket_start = op;
			bracket_length = getBracket(op);
			if (bracket_length < 0) return bracket_start - eq + 1 + current_offset;
			*(bracket_start + bracket_length) = '\0';

			// if parenthesis contains no operands, drop it
			if (!strchr(bracket_start + 1, OP_AND) &&
			    !strchr(bracket_start + 1, OP_OR) &&
			    !strchr(bracket_start + 1, OP_XOR) &&
			    !strchr(bracket_start + 1, OP_BIN) &&
			    !strchr(bracket_start + 1, OP_BIT) &&
			    !strchr(bracket_start + 1, OP_NOT)) {
				*bracket_start = ' ';
				*(bracket_start + bracket_length) = ' ';
				continue;
			}

			new_offset = current_offset + (bracket_start  + 1 - eq);
			res = parse(bracket_start + 1, new_offset, docnt);      // recursive call
			if (res != 0) return res;
			// replace the complete bracket with spaces
			memset(bracket_start, ' ', bracket_length + 1);
			continue;
		}

		// determine operands and their length
		if (optype == OP_LR) {
			left = op - 1;
			left_length = getLeft(eq, &left);
		} else left_length = 0;
		right = op + 1;
		right_length = getRight(eq + strlen(eq), &right);

		// ****************************
		// docnt: just count the items!
		// ****************************

		// push operands to the calculation stack
		if (left_length) {
			if (!docnt) push(left, left_length, current_offset + (left - eq));
			else total++;
		}
		if (right_length) {
			if (!docnt) push(right, right_length, current_offset + (right - eq));
			else total++;
		}

		// push operation to the calculation stack
		if (!docnt) push(op, 1,  current_offset + (op - eq));
		else total++;

		// replace the operation and its operands with spaces
		*op = ' ';
		if (right_length)
			memset(right, ' ', right_length);
		if (left_length)
			memset(left, ' ', left_length);
	}
	// find operand without operator
	right = eq;
	right_length = getRight(eq + strlen(eq), &right);
	if (right_length) {
		if (!docnt) push(right, right_length, current_offset + (right - eq));
		else total++;
	}
	free(eq);
	return 0;	// OK
}

// returns length of the bracket
int opStack::getBracket(char *__equation)
{	char *start = __equation;
	char *end = start;

	while (NULL != (end = strchr(end + 1, ')'))) {
		start = strchr(start + 1, '(');
		if (!start || start > end) {
			return end - __equation;
		}
	}
	return -1;
}

// return a pointer to the next operation
char * opStack::getOP(char *_equation)
{	char *pt0, *pt1, *pt2, *pt3, *pt4, *pt5, *pt6;
	char *bet;

	pt0 = strchr(_equation, OP_BRACKET);
	pt1 = strchr(_equation, OP_NOT);
	pt2 = strchr(_equation, OP_XOR);
	pt3 = strchr(_equation, OP_AND);
	pt4 = strchr(_equation, OP_OR);
	pt5 = strchr(_equation, OP_BIN);
	pt6 = strchr(_equation, OP_BIT);

	// find two operators to the left for which is valid: op1 left of op2 left of op0, then return op1
	// (numbers: precedency!)
	if (pt0) {
		if (pt1 && pt1 < pt0) {
			bet = strchr(pt1, OP_XOR);
			if (pt1 < bet && bet < pt0) return pt1;
			bet = strchr(pt1, OP_AND);
			if (pt1 < bet && bet < pt0) return pt1;
			bet = strchr(pt1, OP_OR);
			if (pt1 < bet && bet < pt0) return pt1;
		}
		if (pt2 && pt2 < pt0) {
			bet = strchr(pt2, OP_AND);
			if (pt2 < bet && bet < pt0) return pt2;
			bet = strchr(pt2, OP_OR);
			if (pt2 < bet && bet < pt0) return pt2;
		}
		if (pt3 && pt3 < pt0) {
			bet = strchr(pt3, OP_OR);
			if (pt3 < bet && bet < pt0) return pt3;
		}
		return pt0;
	}

	if (pt1) {
		if (pt2 && pt2 < pt1) {
			bet = strchr(pt2, OP_AND);
			if (pt2 < bet && bet < pt1) return pt2;
			bet = strchr(pt2, OP_OR);
			if (pt2 < bet && bet < pt1) return pt2;
		}
		if (pt3 && pt3 < pt1) {
			bet = strchr(pt3, OP_OR);
			if (pt3 < bet && bet < pt1) return pt3;
		}
		return pt1;
	}

	if (pt2) {
		if (pt3 && pt3 < pt2) {
			bet = strchr(pt3, OP_OR);
			if (pt3 < bet && bet < pt1) return pt3;
		}
		return pt2;
	}

	if (pt3) return pt3;
	if (pt4) return pt4;

	// special internal operators without interaction
	if (pt5) return pt5;
	return pt6;
}

// return type of operation:
//   OP_LR operator needs left and right operand
//   OP_R operator needs right operand
int opStack::opType(char op)
{
	switch(op) {
		case OP_AND:
		case OP_OR:
		case OP_XOR:
		case OP_BIN:
		case OP_BIT:
			return OP_LR;
			break;
		case OP_NOT:
			return OP_R;
			break;
	}
	return 0;
}

// sets pointer to the start of the left operand nonterminal
// returns length of the nonterminal
int opStack::getLeft(char *start, char **left)
{	int length = 0;

	// jump over spaces
	while (**left == ' ' && *left > start)
		(*left)--;

	// reverse find next space, operand, ( or )
	while (	*left >= start &&
	  **left != '(' &&
	  **left != ')' &&
	  **left != ' ' &&
	  **left != OP_AND &&
	  **left != OP_OR &&
	  **left != OP_XOR &&
	  **left != OP_BIN &&
	  **left != OP_BIT &&
	  **left != OP_NOT &&
	  **left != 0) {
		(*left)--;
		length++;
	}
	if (length) (*left)++;

	return length;
}

// sets pointer to the start of the right operand nonterminal
// returns length of the nonterminal
int opStack::getRight(char *end, char **right)
{	int length = 0;
	char *pt;

	// jump over spaces
	while (**right == ' ' && *right < end)
		(*right)++;

	pt = *right;
	// find next space, operand, ( or )
	while (	pt <= end &&
	  *pt != '(' &&
	  *pt != ')' &&
	  *pt != ' ' &&
	  *pt != OP_AND &&
	  *pt != OP_OR &&
	  *pt != OP_XOR &&
	  *pt != OP_BIN &&
	  *pt != OP_BIT &&
	  *pt != OP_NOT &&
	  *pt != 0) {
		pt++;
		length++;
	}

	return length;
}

// push something into the queue
void opStack::push(char *str, int length, int current_offset)
{
	// push string to the queue
	if (length) {
		if (total > maxitem) {
			fprintf(stderr, "too many items on queue??\n");
			exit(-1);
		}

		queue[total] = (char *)malloc(length * sizeof(char) + 1);
		strncpy(queue[total], str, length);
		*(queue[total] + length) = '\0';

		queue_info[total]->position = current_offset;
		queue_info[total]->length = length;

		total++;
	}
}

// push an input value to the stack
void opStack::push(int val)
{
	if (!staticOP) {
		fprintf(stderr, "pushing values to the stack??\n");
		exit(-1);
	}
	if (plane >= OP_MAXINP) {
		fprintf(stderr, "too many inputs (100)\n");
		exit(-1);
	}
	if (val != TRISTATE) {
		stack[total++] = val;
	} else {
		if (!total && staticOP == OP_INTERFACE)
			stack[total++] = val;
	}
}

// stack calculation
// returns the result
// changed in 12-02-2002: _NO_ tristate values at the stack anymore!!
int opStack::calculate(bool tristate_active)
{	int i;
	char currOP = 0;
	int bin_pow = 1;
	bool hadTristate = false;

	if (tristate_active && isTristate()) {
		setValue(TRISTATE);
		return TRISTATE;
	}

	setValue(undefined_result);
	if (!total) return undefined_result;		// nothing to do

	if (!staticOP) plane = -1;
	else {
		// pure AND/OR/..
		currOP = staticOP;
		plane = total - 1;			// values are push()ed from outside!
	}

	for (i = 0; i < total; i++) {
		if (!staticOP) currOP = queue[i][0];

		switch (currOP) {
			case OP_AND:
				if (plane > 0) {
					stack[plane - 1] = stack[plane - 1] & stack[plane];
					plane--;
				}
				break;
			case OP_OR:
				if (plane > 0) {
					stack[plane - 1] = stack[plane - 1] | stack[plane];
					plane--;
				}
				break;
			case OP_XOR:
				if (plane > 0) {
					if (stack[plane - 1] + stack[plane] == 1) stack[plane - 1] = 1;
					else stack[plane - 1] = 0;
					plane--;
				}
				break;
			case OP_BIN:
				if (plane > 0) {
					stack[plane - 1] += stack[plane] * Tool::pow2(bin_pow);
					bin_pow++;
					plane--;
				}
				break;
			case OP_BIT:
				if (plane > 0) {
					stack[plane - 1] = (stack[plane - 1] & Tool::pow2(stack[plane])) > 0;
					plane--;
				}
				break;
			case OP_NOT:
				if (plane >= 0) {
					if (stack[plane] == 1) stack[plane] = 0;
					else stack[plane] = 1;
				}
				break;
			case OP_NONE:
				break;
			case OP_INTERFACE:
				// use OR for interfaces; propagate TRISTATE
				// but do no calculation with TRISTATE values!
				if (plane > 0) {
					if (stack[plane - 1] == TRISTATE)
						stack[plane - 1] = stack[plane];
					if (stack[plane] != TRISTATE && stack[plane - 1] != TRISTATE)
						stack[plane - 1] |= stack[plane];
					else 
						hadTristate = true;
					plane--;
				}
				if (!plane && stack[plane] == TRISTATE)
					hadTristate = true;
				break;
			default:
				plane++;
				if (plane > maxitem) {
					fprintf(stderr, "opStack: stack overflow!\n");
					exit(-1);
				}
				if (!dev)
					stack[plane] = getPattern(i);	// virtual by SymbolOpStack
				else if (queue_info[i]->type == THIS)
					stack[plane] = dev->output();
				else if (queue_info[i]->type == Device::FINAL_OUTPUT)
					stack[plane] = dev->output(queue_info[i]->output_id);
				else if (queue_info[i]->type == Device::INPUT)
					stack[plane] = dev->input(queue_info[i]->output_id);
				else if (queue_info[i]->type == NUMBER)
					stack[plane] = queue_info[i]->value;
				if (stack[plane] == TRISTATE) {
					hadTristate = true;
					plane--;
				}
				break;
		}
	}
	if (plane != 0) {
		if (staticOP) total = 0;	// pure AND/OR/..
		if (plane > 0) {
			// should never happen
			warning("opStack::calculate: stack plane is not zero!\n");
			plane = 0;
			return undefined_result;
		} else if (hadTristate) {
			plane = 0;
			// special case: interface devices
			if (staticOP == OP_INTERFACE) {
				setValue(TRISTATE);
				return TRISTATE;
			}
			setValue(undefined_result);
			return undefined_result;
		} else {
			// should never happen
			warning("opStack::calculate: stack plane is not zero!\n");
			plane = 0;
			return undefined_result;
		}
	}

	if (staticOP) total = 0;	// pure AND/OR/..

	setValue(stack[plane]);
	return stack[plane];
}

char ** opStack::getQueue()
{
	return queue;
}

int opStack::getCounter()
{
	return total;
}

QueueInfo ** opStack::getInfo()
{
	return queue_info;
}


