/* creation and destruction of buffers */
#include <pthread.h>
#include <string.h>
#include <stdlib.h>
#include <sys/time.h>
#include <sys/types.h>
#include <sys/sysinfo.h>
#include <unistd.h>
#include <math.h>
#include "buffer.h"
#include "interpol.h"
#include "speciallist.h"



#define MAXPATHLEN 256

long sys_samplerate;

void buffer_lock(buffer_info_t* buf){
	pthread_mutex_lock (&buf->buffermutex);
}

void buffer_unlock(buffer_info_t* buf){
	pthread_mutex_unlock (&buf->buffermutex);
}


static int strnflst(const char *s1,const  char *s2,int n){
        /* sucht s2 in s1 bis maximal s1[n]*/
        int i;          
        int len2 = strlen(s2);
        int found = -1; 
                        
        if (n >  strlen(s1)){n = strlen(s1);}
        if (n < 0){ n = 0;}             
                
        for (i=0;i<n;i++){ 
                if(!strncasecmp((char*)(s1+i),s2,len2)){
                        found = i;
                }       
        }               
/*      printf("found=%d\n",found);*/
        return found;
}


buffer_info_t* buffer_new(int id){
	buffer_info_t* buf;

	buf = (buffer_info_t*) malloc (sizeof(buffer_info_t));

	buf->id = id;
	buf->status = BUFFER_STATUS_NEW;
	buf->pending = 0;
	buf->error = 0;
	buf->percent_filled = 0.;
	buf->normalized = 0;
	buf->size = 0;	
	buf->L = NULL;
	buf->R = NULL;
	buf->filename= NULL;
	buf->shortname = NULL;
	buf->channels = 2;
	buf->is_ds = 0;
	pthread_mutex_init (&buf->buffermutex, NULL);

	return buf;
}

int buffer_delete(buffer_info_t* buf){
	buf->pending = BUFFER_STATUS_DEAD;
	free (buf->R);
	buf->R = NULL;
	free (buf->L);
	buf->L = NULL;
	free (buf->filename);
	buf->filename = NULL;
	free (buf->shortname);
	buf->shortname = NULL;
		
	return 0;
}

void buffer_set_channels(buffer_info_t* buf, int channels){
	if (channels == 1){
/*		printf ("1 channel\n");*/
		free(buf->R);
		buf->R = NULL;
	}else if (channels == 2)
		buf->R = (float*) realloc(buf->R, sizeof(float) * buf->size);
	
	buf->channels = channels;
}

void buffer_resize(buffer_info_t* buf, long new_size){
	long l,i,max;
	max  = 44100; /* ~1 secs at once */
	float *tmp = NULL;

	if ((getpagesize() * sysconf (_SC_AVPHYS_PAGES)) < (new_size * sizeof(float) * buf->channels)){
		buf->error = BUFFER_ERROR_ALLOC;
		return;
	}
	
	for (i = 0;i < (new_size / max);i++){
		tmp = (float*) realloc(buf->L, sizeof(float) * (i + 1) * max);
		if (tmp != NULL){	
			buf->L = tmp;
			memset ((void*)(buf->L + (i  * max)),0,max);		
		}else{
			buf->size = 0;
			return;
		}
		if (buf->channels > 1){
			tmp = (float*) realloc(buf->R, sizeof(float) * (i + 1) * max);
			if (tmp){
				buf->R = tmp;
				memset ((void*)(buf->R + (i  * max)),0,max);
			}else{
				buf->size = 0;
				return;
			}
		}
		/* sleep a little bit to prevent stress */
/*                g_usleep(1000);*/
	}

	l = new_size % max;
        tmp = (float*) realloc(buf->L, sizeof(float) * (i * max + l));
	if (tmp){
                buf->L = tmp;
		memset ((void*)(buf->L + (i * max)),0,l);
        }else{
                buf->size = 0;
                return;
        }
        if (buf->channels > 1){
                tmp = (float*) realloc(buf->R, sizeof(float) * (i * max + l));
		if (tmp){
                        buf->R = tmp;
			memset ((void*)(buf->R + (i * max)),0,l);
                }else{
                        buf->size = 0;
                        return;
                }
        }
	buf->size = new_size;
/*	printf ("new buffer size:%ld\n",buf->size);*/
}

float buffer_readsample(buffer_info_t* buf, int channel, long index){
	if (!buf) return 0;

	if (buf->is_ds){
		return 0;
	}else{
		if (channel) 	{return *(buf->R + index);}
		else 		{return *(buf->L + index);}
	}
}

void buffer_writesample(buffer_info_t* buf, int channel, long index, float sample){
	if (!buf) return;
	if (buf->is_ds) return;  /* never writze to a discstream */

	if (channel) 	{*(buf->R + index) = sample;}
	else		{*(buf->L + index) = sample;}
}

float buffer_analyze(buffer_info_t* buf,long start,long end){
	/* analyze all if end < start or end == 0 */
	float max = .0;
	struct timeval tv;
	long  l;

	if (buf){
		if (end < 1)		{ end = buf->size;}
		if (start < 0)          { start = 0; }
                if (!(end > start))     { end = buf->size;}
                if (start > buf->size)  { start = 0; }
                if (end > buf->size)    { end = buf->size; }

/*		printf ("analyze: start:%ld, end:%ld\n",start,end);*/

		for (l=start; l < end;l++){
			if (fabs(*(buf->L + l)) > max){ max = fabs(*(buf->L + l));}
			if (buf->channels > 1){
				if (fabs(*(buf->R + l)) > max){ max = fabs(*(buf->R + l));}
			}
			if (!(l % 100000)){
/*				g_usleep(1000);*/
		                tv.tv_sec = 0;
                		tv.tv_usec = 100000;
                		select(0, NULL, NULL, NULL, &tv);
			}
		}
		return max;
	}else return 0;
}

void buffer_normalize(buffer_info_t* buf, float factor, long start, long end){
	 /* normalze all if end < start or end == 0 */
	long l, k, diff;
	long bsize = 44100; /* do 1sec at once i and sleep inbetween*/

/*	printf ("start normalize by factor %f, L:%ld, R:%ld, L==R:%d\n",factor,(long)buf->L,(long)buf->R,(int)(buf->L == buf->R));*/
	buffer_analyze(buf,0,0);
	if (buf){
		if (start < 0)		{ start = 0; }
		if (!(end > start))	{ end = buf->size;}
		if (start > buf->size)	{ start = 0; }
		if (end > buf->size)	{ end = buf->size; }

		diff = end - start;

		if (factor != 0.0){
			for (k = 0; k < (long)(diff / bsize); k++){	
				buffer_lock(buf);
				for (l = (k * bsize + start); l < ((k + 1) * bsize + start);l++){
                			*(buf->L + l) *= factor;
					if (buf->channels > 1) *(buf->R + l) *= factor;
        			}
				buffer_unlock(buf);
/*				g_usleep(1000);*/
			}
		
			/* the rest */
			buffer_lock(buf);
			for (l = (bsize * (long)(diff / bsize)) + start;l <= end;l++){
				*(buf->L + l) *= factor;
				if (buf->channels > 1) *(buf->R + l) *= factor;
			}
			buf->normalized = 1;
			buffer_unlock(buf);
		}else{
			buffer_lock(buf);
			memset ((void*)(buf->L + start),0,((end - start) * sizeof(float)));
			if (buf->channels > 1)
				memset ((void*)(buf->R + start),0,((end - start) * sizeof(float)));
			
			buf->normalized = 1;
			buffer_unlock(buf);
		}
	}
/*	printf ("normalized\n");*/
}

static void buffer_load_soundfile(buffer_info_t*  buf){
	float*		tmp;
	float*		tmpL = NULL;
	float*		tmpR = NULL;
	float		rateratio = 0;
	long 		l;
	SF_INFO*	sfinfo = (SF_INFO*) malloc(sizeof(SF_INFO));
	SNDFILE*	sf;	
	sf_count_t	frames_read = 1; 
	sf_count_t 	frames_done = 0;
	long 		blocksize = 32768;

/*	pthread_setcanceltype (PTHREAD_CANCEL_ASYNCHRONOUS, NULL);*/

	buffer_lock(buf);
	buf->pending = BUFFER_STATUS_LOADING;
	buffer_unlock(buf);

	sf = sf_open (buf->filename, SFM_READ, sfinfo);

	if (sf == NULL){
/*		fprintf(stderr,"could not open file:%s\n",buf->filename);*/
                free(sfinfo);
		buffer_lock(buf);
/*
		free(buf->filename);
                free(buf->shortname);
*/
                buf->pending = BUFFER_STATUS_DEAD;
		buf->error = BUFFER_ERROR_NOSF;
		buffer_unlock(buf);
                return;
        }else{
/*		printf ("soundfile: %d channels, %d hz\n",sfinfo->channels,sfinfo->samplerate);*/
	}


	
	rateratio = (float)sys_samplerate / (float)sfinfo->samplerate;
	if (!rateratio) rateratio = 1;
	if (sfinfo->channels == 1)
		buffer_set_channels(buf,sfinfo->channels);
	else 
		buffer_set_channels(buf,2);
	
	buffer_resize(buf,rateratio * sfinfo->frames);
	

	tmp = (float*) malloc (sizeof(float) * blocksize *  sfinfo->channels);
	if ((rateratio != 1.) && (sfinfo->channels > 1)){
		tmpL = (float*) malloc (sizeof(float) * blocksize);
		tmpR = (float*) malloc (sizeof(float) * blocksize);
	}

	if ((buf->L == NULL) || ((buf->channels > 1) && (buf->R == NULL)) || (tmp == NULL) || (!buf->size)){ 
		fprintf(stderr,"could not allocate enough memory for the buffer\n");
		buffer_lock(buf);
		buf->error = BUFFER_ERROR_ALLOC;
		free(buf->L);
		free(buf->R);
/*
		free(buf->filename);
		free(buf->shortname);
*/
		buf->pending = BUFFER_STATUS_DEAD;
		free(tmp);
		buffer_unlock(buf);
		
		return;
	}

	while ((frames_done < (long)(sfinfo->frames * rateratio)) && frames_read){
		frames_read = sf_readf_float(sf,tmp,(sf_count_t)blocksize);
		if (rateratio == 1.){
			/* files samplerate == jacks samplerate */
			buffer_lock(buf);
			for (l=0;l < frames_read;l++){
				*(buf->L + l + frames_done) = *(tmp + l * sfinfo->channels); 
				/* if file is mono we fill L and R with the same data */
				if (sfinfo->channels > 1){
					*(buf->R + l + frames_done) = *(tmp + (l * sfinfo->channels + 1));
				}
			}	
			frames_done += frames_read;
			buf->percent_filled = (double)(frames_done) / (double)(sfinfo->frames);
			buffer_unlock(buf);
		}else{
			/* we have to convert samplerates: */
			if (sfinfo->channels > 1){
				/* copy to seperate L and R tmp buffers for later interpolation */
				for (l=0;l < (long)(frames_read);l++){
					*(tmpL + l) = *(tmp + l * sfinfo->channels);
					*(tmpR + l) = *(tmp + l * sfinfo->channels + 1);
				}
			}
			buffer_lock(buf);
			for (l=0;l < (long)(frames_read * rateratio);l++){
				if (sfinfo->channels > 1){
                                        *(buf->L + l + frames_done) = inter4pol(tmpL, l / rateratio, frames_read);
                                        *(buf->R + l + frames_done) = inter4pol(tmpR, l / rateratio, frames_read);
                                }else {
                                        *(buf->L + l + frames_done) = 
						inter4pol (tmp, l * sfinfo->channels / rateratio, frames_read);
                                }
			}
			frames_done += (long)(frames_read * rateratio);
			buf->percent_filled = (double)(frames_done) / (double)(sfinfo->frames * rateratio);
			buffer_unlock(buf);
		}
	}

	sf_close(sf);
        free (tmp);
	if (tmpL){free(tmpL);}
	if (tmpR){free(tmpR);}

	buffer_lock(buf);
	buf->size = frames_done;
	buf->pending = BUFFER_STATUS_READY;
	buffer_unlock(buf);

/*	printf ("loading done\n");*/
	return;
}

void buffer_set_filename(buffer_info_t* buf, char* name){
        buf->filename = (char*) realloc(buf->filename,sizeof(char) * strlen(name) + 1);
	strcpy(buf->filename,name);
	return;
}

void buffer_set_shortname(buffer_info_t* buf, char* name){
        buf->shortname = (char*) realloc(buf->shortname,sizeof(char) * strlen(name) + 1);
        strcpy(buf->shortname,name);
        return;
}  

void buffer_set_ds(buffer_info_t* buf, int is_ds){
	if (is_ds){
		buf->is_ds = 1;
		buffer_resize (buf, 44100 * 60); /* one minute buffer for now */
	} else buf->is_ds = 0;
	return;
}

void  buffer_loadfile(buffer_info_t* buf, char* fname, long srate){
	/* create a reding thread, wait for join, return success*/
	int lp;
	sys_samplerate = srate;

	buf->filename = (char*) malloc (sizeof(char) * (strlen(fname) + 1));
	strcpy(buf->filename,fname);
/*	printf("filename now:%s\n",buf->filename);*/
	lp = strnflst	(fname,"/",-1);
	buf->shortname = (char*) malloc(sizeof(char) * (strlen(fname) - lp + 1));
	strncpy(buf->shortname,(char*)(buf->filename + lp + 1), strlen(fname) - lp);
	buf->pending = BUFFER_STATUS_TOLOAD;
/*	printf("shortname now:%s\n",buf->shortname);*/

	return;
}

static void* buffer_loadthread(void *ptr){
	speciallist_t *buffer_list = (speciallist_t*)ptr;
	buffer_info_t *buf;
        struct timeval tv;

	while(1){
		buf = speciallist_get_first(buffer_list);
		while (buf){
			if ((buf->status == BUFFER_STATUS_TOLOAD) && (!buf->pending))
				buffer_load_soundfile(buf);
			buf = speciallist_get_next(buffer_list,buf);
		}
	        tv.tv_sec = 0;
        	tv.tv_usec = 100000;
		select(0, NULL, NULL, NULL, &tv);
		
	}
	/* won't happen: */
	return NULL;
}

void buffer_loadthread_launch(speciallist_t *buffer_list){
	pthread_t lt;
	pthread_create (&lt, NULL, buffer_loadthread, buffer_list);
}


int  buffer_save (buffer_info_t* buf, char* fname, long srate, long start, long end){
	SF_INFO*        sfinfo = (SF_INFO*) malloc(sizeof(SF_INFO));
        SNDFILE*        sf;     
	sf_count_t	frames;
	sf_count_t      frames_done = 0;
	sf_count_t	frames_at_once = 16384;
                
	struct timeval tv;
	float		*tbuf;
	long i;

        tv.tv_sec = 0;
        tv.tv_usec = 10000;

	sfinfo->samplerate = srate;
	sfinfo->channels = buf->channels;
	/* LATER allow other formats */
	sfinfo->format = SF_FORMAT_WAV | SF_FORMAT_PCM_16;

/*
	if (strncasecmp (".wav",(char*)(fname + strlen(fname) - 4),4)){
		i = strlen(fname);
		fname = realloc (fname,(sizeof(char)*(i + 5)));
		memset((char*)(fname + i),0,4);
		strncpy((fname + i),".wav",4);
	}
*/


	sf = sf_open (fname, SFM_WRITE, sfinfo);

	if (!sf) return -1;
	
	tbuf = malloc (sizeof(float) * frames_at_once * buf->channels);	
	if (!tbuf) return -1;

	while (frames_done < (end - start)){
		frames = end - start - frames_done;
		if (frames > frames_at_once) frames = frames_at_once;

		for (i = 0; i < frames; i++){
			*(tbuf + i * buf->channels) = 
				*(buf->L + i + start + frames_done);
			if (buf->channels > 1)
				*(tbuf + i * buf->channels + 1) = 
					*(buf->R + i + start + frames_done);
		}
/*		printf ("i:%ld\n",i);*/
		frames = sf_writef_float(sf, tbuf, frames);
		if (frames < 1){
			free(buf);
			return -1;
		}
		frames_done += frames;
/*		printf ("frames:%ld, target:%ld\n",(long)frames_done,(long)(end - start));*/
/*		g_usleep(1000);*/
                tv.tv_sec = 0;
                tv.tv_usec = 100000;
                select(0, NULL, NULL, NULL, &tv);
	}

	sf_close(sf);
	free(tbuf);	
	return 1;
}
