#include <pthread.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>
#include "buffer.h"
#include "interpol.h"
#include "peakview.h"


peakdata_t *peakdata_new(){
	peakdata_t *pd = malloc(sizeof(peakdata_t));
	pd->start = 0;
	pd->end = 0;
	pd->pstart = 0;
	pd->pend = 0;
	pd->status = PD_STATUS_NEW;
	pd->Lmin = NULL;
	pd->Lmax = NULL;
	pd->Rmin = NULL;
	pd->Rmax = NULL;
	return pd;
}

static void peakdata_delete(peakdata_t *pd){
	if (!pd) return;
	if (pd->status != PD_STATUS_DEAD) return;

/*	printf ("delete : %p (%ld -%ld)\n",pd,pd->pstart,pd->pend);*/
	if (pd->Lmin) free(pd->Lmin);
	pd->Lmin = NULL;
	if (pd->Lmax) free(pd->Lmax);
        pd->Lmax = NULL;
	if (pd->Rmin) free(pd->Rmin);
	pd->Rmin = NULL;
	if (pd->Rmax) free(pd->Rmax);
        pd->Rmax = NULL;
}

void peakdata_set_status(peakdata_t *pd, int status){
	if (!pd) return;
	if (pd->status == PD_STATUS_DEAD) return;
/*	printf ("set status to dead %p\n",pd);*/
	pd->status = status;
}


static void* peakview_workthread(void *ptr){
	peakview_t *view = (peakview_t*)ptr;
	peakdata_t *pd;
	struct timeval tv;
	int dirty;
	long l;

	pthread_setcancelstate(PTHREAD_CANCEL_DISABLE,NULL);
	
	while (!view->dead){	
		dirty = 0;
		peakview_lock(view);
		pd = speciallist_get_first(view->returnlist);
		while (pd && !dirty){
			if (pd->status == PD_STATUS_NEW){
				long oldsample = 0;
				long newsample = 0;
				signed char Lmin = 0;
				signed char Lmax = 0;
				signed char Rmin = 0;
				signed char Rmax = 0;
				signed char val;
				long gstart = (long)((float)pd->start / view->samplesperpixel);
				oldsample = gstart;
				for (l = pd->start; l <= pd->end; l++){
					val = (signed char) (buffer_readsample_interpolated(
						view->buf,1,(float)l,view->group) * 127.);
					if (val < Lmin) Lmin = val;
					if (val > Lmax) Lmax = val;
					if (buffer_get_channels(view->buf) > 1){
						val = (signed char) (buffer_readsample_interpolated(
                                                view->buf,2,(float)l,view->group) * 127.);
						if (val < Rmin) Rmin = val;
                                        	if (val > Rmax) Rmax = val;
					}
					newsample = (long)((float)l / view->samplesperpixel);
					if (newsample != oldsample){
/*						printf ("%ldmin = %d\n",newsample - gstart,Lmin);*/
						*(pd->Lmin + oldsample - gstart) = Lmin;
						*(pd->Lmax + oldsample - gstart) = Lmax;
						if (buffer_get_channels(view->buf) > 1){
							*(pd->Rmin + oldsample - gstart) = Rmin;
							*(pd->Rmax + oldsample - gstart) = Rmax;
						}
						Lmin = 0;
						Lmax = 0;	
						Rmin = 0;
						Rmax = 0;
						oldsample = newsample;
					} 	
				}
/*				printf ("did work:%p\n",pd);*/
				pd->status = PD_STATUS_DONE;
				dirty = 1;
			} 
			pd  = speciallist_get_next(view->returnlist,pd);
		}

		pd = speciallist_get_first(view->returnlist);
                while (pd){
			if (pd->status == PD_STATUS_DEAD){
				peakdata_t *tmp =pd;
				pd  = speciallist_get_next(view->returnlist,pd);
				speciallist_delete(view->returnlist,tmp);
				peakdata_delete(tmp);
			}else pd  = speciallist_get_next(view->returnlist,pd);
		}
		peakview_unlock(view);
/*		if (!dirty){*/
                tv.tv_sec = 0;
                tv.tv_usec = 10000; 
                select(0, NULL, NULL, NULL, &tv);
/*                }*/
	}

        pd = speciallist_get_first(view->returnlist);
        while(pd){
/*		printf ("delete next\n");*/
                pd = speciallist_get_next(view->returnlist,pd);
		peakdata_set_status(pd,PD_STATUS_DEAD);
		peakdata_delete(pd);
        }
        speciallist_destroy(view->returnlist);
        free (view);
        view = NULL;
	printf ("peakview return\n");
	return NULL;
}


peakview_t* peakview_new(buffer_info_t* buf){
	pthread_t wt;
	peakview_t *view = (peakview_t*) malloc (sizeof(peakview_t));
	if (!buf) return NULL;

/*	printf ("peakview new\n");*/
	view->buf = buf;
	view->dead = 0;
	view->group = buffer_get_group(buf, DS_PRIORITY_RELAXED);
	view->returnlist = speciallist_new();
	pthread_mutex_init (&view->peakviewmutex, NULL);

	pthread_create (&wt, NULL, peakview_workthread, view);

	return view;
}

void peakview_delete(peakview_t* view){
	if (!view) return;

	peakview_lock(view);
	view->dead = 1;
	peakview_unlock(view);
}

void peakview_set_buffer(peakview_t* view,buffer_info_t* buf){
	peakdata_t* pd;
	if (!view) return;
	if (view->dead) return;

	peakview_lock(view);
       	pd = speciallist_get_first(view->returnlist);
        while(pd){
                peakdata_set_status(pd,PD_STATUS_DEAD);
                pd = speciallist_get_next(view->returnlist,pd);
        }
	view->buf = buf;
        view->start = 0;
        view->end = 0;
        view->pixels = 0;
        view->samplesperpixel = .0;
	if (buf) view->group = buffer_get_group(buf, DS_PRIORITY_RELAXED);
	peakview_unlock(view);
}

void peakview_lock(peakview_t* view){
	pthread_mutex_lock (&view->peakviewmutex);
}

void peakview_unlock(peakview_t* view){
	pthread_mutex_unlock (&view->peakviewmutex);
}

void peakview_set_area(peakview_t* view, long start, long end, long pixels){
/*	int ok = 0;*/
	peakdata_t* pd;
	if (!view) return;
	if (view->dead) return;
/*
	while (!ok){
		ok = 1;
		pd = speciallist_get_first(view->returnlist);
		while(pd){
			if (!pd->status != PD_STATUS_DEAD) ok = 0;
			pd = speciallist_get_next(view->returnlist,pd);
		}
	}
*/

	peakview_lock(view);
	/* delete old list blocks */
	pd = speciallist_get_first(view->returnlist);
	while(pd){
		peakdata_set_status(pd,PD_STATUS_DEAD);
                pd = speciallist_get_next(view->returnlist,pd);
        }
	view->start = start;
	view->end = end;
	view->pixels = pixels;
	view->samplesperpixel = (float)(end - start) / (float)pixels;
/*	printf ("set area: %ld, %ld = %ldpixels (%.2f spp)\n",start,end,pixels,view->samplesperpixel);*/
	peakview_unlock(view);
}

void peakview_calculate(peakview_t* view, long start, long end){
	long blocklength; 
	long l;
	if (!view) return;
	if (view->dead) return;
	if (!view->buf) return;
	if (!view->end) return;
	if (!view->returnlist) return;

	if (start < view->start) start = view->start;
        if (end >= view->end) end = view->end - 1;

	if (buffer_get_type(view->buf) != BUFFER_TYPE_DISCSTREAM){
		/* ~10 seconds at once */
		blocklength = 441000;
	}else{
		/* 200 pixels at once (max) */
		blocklength = (long)((float)(end - start) / (float)(view->end - view->start)
				* view->samplesperpixel * 200);
		if (blocklength > 88200) blocklength = 88200;
	}

/*
	printf ("calculate %ld - %ld of %ld\n	-> (%.2f samples/pixel, blocklength %ld)\n"
		,start,end,buffer_get_size(view->buf),view->samplesperpixel,blocklength);
*/

	peakview_lock(view);
	for (l = start; l <= end; l += blocklength){
                if (buffer_get_type(view->buf) == BUFFER_TYPE_DISCSTREAM){
                        /* this one is just empty to make waiting more comfortable */
                        peakdata_t* pd = peakdata_new();
                        pd->start = l;
                        pd->end = l + blocklength;
                        if (pd->end >= end) pd->end = end - 1;
                        if (pd->start < pd->end){
                                pd->pstart = (long)((float)(pd->start - view->start) / view->samplesperpixel);
                                pd->pend = (long)((float)(pd->end - view->start) / view->samplesperpixel);
                                pd->Lmin = calloc (sizeof(signed char), (pd->pend - pd->pstart + 1));
                                pd->Lmax = calloc (sizeof(signed char), (pd->pend - pd->pstart + 1));
                                if (!(pd->Lmin && pd->Lmax)) printf ("memory allocation error!");
                                if (buffer_get_channels(view->buf) > 1){
                                        pd->Rmin = calloc (sizeof(signed char), (pd->pend - pd->pstart + 1));
                                        pd->Rmax = calloc (sizeof(signed char), (pd->pend - pd->pstart + 1));
                                        if (!(pd->Rmin && pd->Rmax)) printf ("memory allocation error!");
                                }else {
                                        pd->Rmin = NULL;
                                        pd->Rmax = NULL;
                                }
                                pd->status = PD_STATUS_DONE;
                                speciallist_append(view->returnlist,pd);
                        }
                }

		peakdata_t* pd = peakdata_new();
		pd->start = l;
		pd->end = l + blocklength;
		if (pd->end >= end) pd->end = end - 1;
		if (pd->start < pd->end){
			pd->pstart = (long)((float)(pd->start - view->start) / view->samplesperpixel);
			pd->pend = (long)((float)(pd->end - view->start) / view->samplesperpixel);
/*			printf ("	%ld - %ld 	= %ld - %ld\n",pd->start,pd->end,pd->pstart,pd->pend);*/
                        pd->Lmin = malloc (sizeof(signed char) * (pd->pend - pd->pstart + 1));
                        pd->Lmax = malloc (sizeof(signed char) * (pd->pend - pd->pstart + 1));
                        if (!(pd->Lmin && pd->Lmax)) printf ("memory allocation error!");
                        if (buffer_get_channels(view->buf) > 1){
                                pd->Rmin = malloc (sizeof(signed char) * (pd->pend - pd->pstart + 1));
                                pd->Rmax = malloc (sizeof(signed char) * (pd->pend - pd->pstart + 1));
                                if (!(pd->Rmin && pd->Rmax)) printf ("memory allocation error!");
                        }else {
                                pd->Rmin = NULL;
                                pd->Rmax = NULL;
                        }
/*			printf ("new pd:%p, Lmin:%p\n",pd,pd->Lmin);*/
			speciallist_append(view->returnlist,pd);	
		}
	}
	peakview_unlock(view);

}

        
peakdata_t *peakview_get_next(peakview_t* view){
	if (!view) return NULL;
	if (view->dead) return NULL;
	peakdata_t* pd = speciallist_get_first(view->returnlist);

        while(pd){
		if (pd->status == PD_STATUS_DONE){	
/*                	speciallist_delete(view->returnlist,pd);*/
			return pd;
		}
		pd = speciallist_get_next(view->returnlist,pd);
        }
	return NULL;
}	
