/***************************************************************************
                                 qsprojection.cpp
                             -------------------                                         
    begin                : 01-January-2000
    copyright            : (C) 2000 by Kamil Dobkowski                         
    email                : kamildobk@poczta.onet.pl                                     
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   * 
 *                                                                         *
 ***************************************************************************/


#include "qsprojection.h"
#include "qsconsole.h"
#include <math.h>

const double QSProjection::pi = 3.141592653589793115997963468544185161590576171875;

#define QS_CLIP_BUFFER_SIZE  60 // QMAX number of vertices in clipped polygons clipPoly2 and clipPoly3

//-------------------------------------------------------------//

QSProjection::QSProjection()
 {
  m_light = false;
  m_ambient_light = 0;
  m_directed_light = 0;
 }

//-------------------------------------------------------------//

QSProjection::~QSProjection()
 {
 }
//-------------------------------------------------------------//

void QSProjection::setLight( bool enabled )
 {
  m_light = enabled;
 }

//-------------------------------------------------------------//

void QSProjection::setLightParameters( const QSPt3f& lightVector, int ambientLight, int directedLight )
 {
  m_light_vector = lightVector;
  m_ambient_light = ambientLight;
  m_directed_light = directedLight;
 }

//-------------------------------------------------------------//

bool QSProjection::pointInPoly( const QSPt2f& pos, const QSPt2f* pts, int npts )
// comp.graphics.algorithms.faq
    {
      int i;
      int j;
      bool result = false;
      for ( i=0, j=npts-1; i<npts; j=i++ ) {
        if ( ( (pts[i].y<=pos.y && pos.y<pts[j].y) ||
               (pts[j].y<=pos.y && pos.y<pts[i].y)  ) &&
             pos.x < (pts[j].x-pts[i].x) * (pos.y-pts[i].y) / (pts[j].y-pts[i].y) + pts[i].x )
             result = !result;
      }
      return result;
    }

//-------------------------------------------------------------//

void QSProjection::clipVertexNormals( const QSPt3f in[], int inpoints, const QSPt3f clip[], int clipoints, const QSPt3f innorm[], QSPt3f outnorm[] )
 {

  for ( int j=0; j<clipoints; j++ ) {
        int p1[2];
        int p2[2];
        double t1;
        double t2;
        double t3;

        get_interpolation_params( in, inpoints, clip[j], p1, p2, &t1, &t2, &t3 );

        QSPt3f p3[2];
        p3[0] = normalize( interpolation( innorm[p1[0]], innorm[p1[1]], t1 ) );
        p3[1] = normalize( interpolation( innorm[p2[0]], innorm[p2[1]], t2 ) );

        outnorm[j] = normalize( interpolation( p3[0], p3[1], t3 ) );
       }

 }

//-------------------------------------------------------------//

void QSProjection::clipVertexColors( const QSPt3f in[], int inpoints, const QSPt3f clip[], int clipoints, const QSGFill incol[], QSGFill outcol[] )
 {
  for ( int j=0; j<clipoints; j++ ) {
        int p1[2];
        int p2[2];
        double t1;
        double t2;
        double t3;

        get_interpolation_params( in, inpoints, clip[j], p1, p2, &t1, &t2, &t3 );

        QSGFill p3[2];
        p3[0] = interpolation( incol[p1[0]], incol[p1[1]], t1 );
        p3[1] = interpolation( incol[p2[0]], incol[p2[1]], t2 );

        outcol[j] = interpolation( p3[0], p3[1], t3 );
       }

 }

//-------------------------------------------------------------//

void QSProjection::get_interpolation_params(const QSPt3f in[], int inpoints, const QSPt3f& clip, int p1[2], int p2[2], double *t1, double *t2, double *t3 )
//
//      t1    1-t1
// p1[0]   |       p1[1]
//    --------------\            .
//   |   t3|         \           .
//   |     + clip     \          .
//   | 1-t3|           |         . interpolation straight line is always vertical
//   |     |          /          .
//   ----------------/           .
// p2[1]   |        p2[0]
//    t2-1     t2
//
 {
  int lnum = 0;
  int pass = 0;

  int p[2][2];
  double t[2];
  //
  // In maximum three passes.
  //
  while( lnum<2 && pass<3 ) {

        lnum = 0;

        int curr;
        int prev = inpoints-1;
        for( int i=0; i<inpoints; i++ ) {
                curr = i;

                if ( (pass == 0 && (in[prev].x<clip.x) != (in[curr].x<clip.x)) ||
                     (pass == 1 && (in[prev].y<clip.y) != (in[curr].y<clip.y)) ||
                     (pass == 2 && (in[prev].z<clip.z) != (in[curr].z<clip.z))  ) {

                     p[lnum][0] = prev;
                     p[lnum][1] = curr;

                     switch( pass ) {
                         case 0:  t[lnum] = (clip.x-in[prev].x)/(in[curr].x-in[prev].x); break;
                         case 1:  t[lnum] = (clip.y-in[prev].y)/(in[curr].y-in[prev].y); break;
                         case 2:  t[lnum] = (clip.z-in[prev].z)/(in[curr].z-in[prev].z); break;
                        }

                     if ( ++lnum == 2 ) break;

                   } else if ( in[curr].x == clip.x &&
                               in[curr].y == clip.y &&
                               in[curr].z == clip.z ) {

                                p[0][0] = curr;
                                p[0][1] = curr;
                                p[1][0] = curr;
                                p[1][1] = curr;
                                t[0] = 0.0;
                                t[1] = 0.0;
                                lnum = 2;

                                break;
                               }

                prev = curr;
               }

       pass++;
      }

  if ( lnum == 2 ) {

         p1[0] = p[0][0];
         p1[1] = p[0][1];
         p2[0] = p[1][0];
         p2[1] = p[1][1];

         *t1 = t[0];
         *t2 = t[1];

         QSPt3f p3[2];

         p3[0] = interpolation( in[p1[0]], in[p1[1]], *t1 );
         p3[1] = interpolation( in[p2[0]], in[p2[1]], *t2 );

         // Take care about round-off errors
         int    c = 0;
         double d = 0.0;

         if ( fabs(p3[0].x-p3[1].x) > d ) { d=fabs(p3[0].x-p3[1].x); c=1; }
         if ( fabs(p3[0].y-p3[1].y) > d ) { d=fabs(p3[0].y-p3[1].y); c=2; }
         if ( fabs(p3[0].z-p3[1].z) > d ) { d=fabs(p3[0].z-p3[1].z); c=3; }

         switch( c ) {
                 case 1: *t3 = (clip.x-p3[0].x)/(p3[1].x-p3[0].x); break;
                 case 2: *t3 = (clip.y-p3[0].y)/(p3[1].y-p3[0].y); break;
                 case 3: *t3 = (clip.z-p3[0].z)/(p3[1].z-p3[0].z); break;
                 default: *t3 = 0.0; break;
                }

        } else {

         QSConsole::write("ERROR: QSProjection::get_interpolation_params.");

         p1[0] = p1[1] = 0;
         p2[0] = p2[1] = 0;

         *t1 = 0.0;
         *t2 = 0.0;
         *t3 = 0.0;

        }
 }

//-------------------------------------------------------------//

QSPt3f QSProjection::interpolation( const QSPt3f& p1, const QSPt3f& p2, double t )
//
// For t=0.0 returns p1.
// For t=1.0 returns p2.
// For t =(0.0, 1.0) returns the point on the straigth line from p1 to p2.
//
 {
  QSPt3f result;
  result.x = p1.x + t*(p2.x-p1.x);
  result.y = p1.y + t*(p2.y-p1.y);
  result.z = p1.z + t*(p2.z-p1.z);
  return result;
 }

//-------------------------------------------------------------//

QSGFill QSProjection::interpolation( const QSGFill& f1, const QSGFill& f2, double t )
//
// For t=0.0 returns f1.
// For t=1.0 returns f2.
// For t =(0.0, 1.0) returns the fill between f1 and f2.
//
 {
  QSGFill result;

  result.style = f1.style;  // Style must be the same in f1 and f2

  result.color.r = (unsigned char)floor(f1.color.r + t*(f2.color.r-f1.color.r) + 0.5);
  result.color.g = (unsigned char)floor(f1.color.g + t*(f2.color.g-f1.color.g) + 0.5);
  result.color.b = (unsigned char)floor(f1.color.b + t*(f2.color.b-f1.color.b) + 0.5);
  result.color.a = (unsigned char)floor(f1.color.a + t*(f2.color.a-f1.color.a) + 0.5);
  return result;
 }

//-------------------------------------------------------------//

void QSProjection::getPoly3Cube( const QSPt3f in[], int inpoints, QSPt3f c[2] )
// r[0] - min coordinates
// r[1] - max coordinates
 {
  c[0].x = c[1].x = in[0].x;
  c[0].y = c[1].y = in[0].y;
  c[0].z = c[1].z = in[0].z;

  for( int i=1; i<inpoints; i++ ) {
        c[0].x = QMIN( in[i].x, c[0].x ); c[1].x = QMAX( in[i].x, c[1].x );
        c[0].y = QMIN( in[i].y, c[0].y ); c[1].y = QMAX( in[i].y, c[1].y );
        c[0].z = QMIN( in[i].z, c[0].z ); c[1].z = QMAX( in[i].z, c[1].z );
       }

 }

//-------------------------------------------------------------//


QSPt3f QSProjection::normal( const QSPt3f v[4], bool norm )
// Normal to the surface defined by its vertices.
// J.D. Foley, ... "Wstp do grafiki komputerowej"
// ( "Introduction to Computer Graphics" ).
// Chapter 9.1.2
 {
  QSPt3f result;

  // area of the projection of the mesh
  // on (y,z) surface .
  result.x =  (v[1].z+v[0].z)*(v[1].y-v[0].y) +
              (v[2].z+v[1].z)*(v[2].y-v[1].y) +
              (v[3].z+v[2].z)*(v[3].y-v[2].y) +
              (v[0].z+v[3].z)*(v[0].y-v[3].y) ;

  result.y =  (v[1].x+v[0].x)*(v[1].z-v[0].z) +
              (v[2].x+v[1].x)*(v[2].z-v[1].z) +
              (v[3].x+v[2].x)*(v[3].z-v[2].z) +
              (v[0].x+v[3].x)*(v[0].z-v[3].z) ;

  result.z =  (v[1].y+v[0].y)*(v[1].x-v[0].x) +
              (v[2].y+v[1].y)*(v[2].x-v[1].x) +
              (v[3].y+v[2].y)*(v[3].x-v[2].x) +
              (v[0].y+v[3].y)*(v[0].x-v[3].x) ;

  if ( norm ) result = normalize( result );
  return result;
 }

//-------------------------------------------------------------//

QSPt3f QSProjection::normal( const QSPt3f *v, int npoints, bool norm )
// Normal to the surface defined by its vertices.
// J.D. Foley, ... "Wstp do grafiki komputerowej"
// ( "Introduction to Computer Graphics" ).
// Chapter 9.1.2
 {
  QSPt3f result( 0.0, 0.0, 0.0 );

  int s = npoints - 1;
  int p = 0;

  while( p < npoints ) {

         result.x += (v[p].z+v[s].z)*(v[p].y-v[s].y);
         result.y += (v[p].x+v[s].x)*(v[p].z-v[s].z);
         result.z += (v[p].y+v[s].y)*(v[p].x-v[s].x);

         s = p;
         p = p+1;
        };

  if ( norm ) result = normalize( result );
  return result;
 }

//-------------------------------------------------------------//

QSPt3f QSProjection::normalize( QSPt3f vector )
 {
  double len = sqrt( vector.x*vector.x +
                     vector.y*vector.y +
                     vector.z*vector.z );
  if ( len ) {
        vector.x /= len;
        vector.y /= len;
        vector.z /= len;
       }

  return vector;
 }

//-------------------------------------------------------------//

double QSProjection::dotProduct( const QSPt3f& p1, const QSPt3f& p2 )
 {
  return p1.x*p2.x + p1.y*p2.y + p1.z*p2.z;
 }

//-------------------------------------------------------------//

QSPt3f QSProjection::vectorProduct( const QSPt3f& p1, const QSPt3f& p2 )
 {
  return QSPt3f( p1.y*p2.z - p1.z*p2.y, p1.z*p2.x - p1.x*p2.z, p1.x*p2.y - p1.y*p2.x );
 }

//-------------------------------------------------------------//

bool QSProjection::isBottom( const QSPt2f pts[], int npoints )
 {
  double area = 0; int p0 = npoints - 1; int p1 = 0;
  while( p1 < npoints ) {
         area += pts[p0].x*pts[p1].y - pts[p1].x*pts[p0].y;
         p0 = p1; p1 = p1+1;
        };
  return ( area < 0.0 );
 }

//-------------------------------------------------------------//

bool QSProjection::isBottom( const QSPt3f pts[], int npoints )
 {
  double area = 0; int p0 = npoints - 1; int p1 = 0;
  while( p1 < npoints ) {
         area += pts[p0].x*pts[p1].y - pts[p1].x*pts[p0].y;
         p0 = p1; p1 = p1+1;
        };
  return ( area < 0.0 );
 }

//-------------------------------------------------------------//

bool QSProjection::isFlat( const QSPt3f pts[], int npoints )
 {
  if ( npoints<4 ) return true;
  bool bottom = isBottom( pts[0], pts[1], pts[2] );
  for( int t=1;t<npoints-2;t++ )
	if ( bottom != isBottom(pts[0],pts[t+1],pts[t+2]) ) return false;
  return true;
 }

//-------------------------------------------------------------//

bool QSProjection::isCorrect( const double values[4] )
 {
  return !(QMAX(values[0],values[2]) < QMIN(values[1],values[3]) || QMAX(values[1],values[3]) < QMIN(values[0],values[2]));
 }

//-------------------------------------------------------------//

bool QSProjection::isCorrect( const QSPt3f pts[4] )
 {
  return !(QMAX(pts[0].z,pts[2].x) < QMIN(pts[1].z,pts[3].z) || QMAX(pts[1].z,pts[3].z) < QMIN(pts[0].z,pts[2].z));
 }

//-------------------------------------------------------------//

void QSProjection::triangulate1( int triangleNumber, const QSPt3f pts[], int npoints, QSPt3f triangle[3], bool edges[3] )
 {
  triangle[0] = pts[0];
  triangle[1] = pts[triangleNumber+1];
  triangle[2] = pts[triangleNumber+2];
  edges[0] = triangleNumber==npoints-3?true:false;
  edges[1] = triangleNumber==0?true:false;
  edges[2] = true;
 }


//-------------------------------------------------------------//
//-------------------------------------------------------------//
//-------------------------------------------------------------//
//-------------------------------------------------------------//
//-------------------------------------------------------------//



//-------------------------------------------------------------//

QSProjection::ClipResult QSProjection::clipPoint2( const QSPt2f& p1 ) const
 {
  static const double cmin_x = 0.0;
  static const double cmax_x = 1.0;
  static const double cmin_y = 0.0;
  static const double cmax_y = 1.0;

  if ( p1.x > cmax_x ||
       p1.x < cmin_x ||
       p1.y > cmax_y ||
       p1.y < cmin_y  ) return Rejected;
  return Accepted;
 }

//-------------------------------------------------------------//

QSProjection::ClipResult QSProjection::clipLine2( QSPt2f* p0, QSPt2f* p1 ) const
// Cohen-Sutherland's method.
// J. D. Foley, ...
// "Wprowadzenie do grafiki komputerowej" ( "Introduction to computer graphics" )
//  Chap 3.9
 {
  static const double cmin_x = 0.0;
  static const double cmax_x = 1.0;
  static const double cmin_y = 0.0;
  static const double cmax_y = 1.0;

  QSPt2f cp;

  union {
       unsigned int all;
       struct {
             unsigned int l : 1; // left
             unsigned int r : 1; // right
             unsigned int t : 1; // top
             unsigned int b : 1; // bottom
            } bit;
      } outcode0, outcode1, outcodeOut;

  outcode0.all = 0;
  outcode1.all = 0;

  if ( p0->y > cmax_y ) outcode0.bit.t = 1; else if ( p0->y < cmin_y ) outcode0.bit.b = 1;
  if ( p0->x > cmax_x ) outcode0.bit.r = 1; else if ( p0->x < cmin_x ) outcode0.bit.l = 1;

  if ( p1->y > cmax_y ) outcode1.bit.t = 1; else if ( p1->y < cmin_y ) outcode1.bit.b = 1;
  if ( p1->x > cmax_x ) outcode1.bit.r = 1; else if ( p1->x < cmin_x ) outcode1.bit.l = 1;

  if ( (outcode0.all | outcode1.all) == 0 ) return Accepted;

  while( 1 )
      if ( (outcode0.all | outcode1.all) == 0 ) return Clipped;
        else if ( (outcode0.all & outcode1.all) != 0 ) return Rejected;
        else {
              if ( outcode0.all != 0 ) outcodeOut = outcode0;
                                  else outcodeOut = outcode1;

              if ( outcodeOut.bit.t ) {
                   cp.y = cmax_y;
                   cp.x = p0->x+(p1->x-p0->x)*(cmax_y-p0->y)/(p1->y-p0->y);
                  } else if ( outcodeOut.bit.b ) {
                   cp.y = cmin_y;
                   cp.x = p0->x+(p1->x-p0->x)*(cmin_y-p0->y)/(p1->y-p0->y);
                  } else if ( outcodeOut.bit.r ) {
                   cp.x = cmax_x;
                   cp.y = p0->y+(p1->y-p0->y)*(cmax_x-p0->x)/(p1->x-p0->x);
                  } else if ( outcodeOut.bit.l ) {
                   cp.x = cmin_x;
                   cp.y = p0->y+(p1->y-p0->y)*(cmin_x-p0->x)/(p1->x-p0->x);
                  }

               if ( outcodeOut.all == outcode0.all ) {
                   p0->x = cp.x;
                   p0->y = cp.y;
                   outcode0.all = 0;
                   if ( p0->y > cmax_y ) outcode0.bit.t = 1; else if ( p0->y < cmin_y ) outcode0.bit.b = 1;
                   if ( p0->x > cmax_x ) outcode0.bit.r = 1; else if ( p0->x < cmin_x ) outcode0.bit.l = 1;
                  } else {
                   p1->x = cp.x;
                   p1->y = cp.y;
                   outcode1.all = 0;
                   if ( p1->y > cmax_y ) outcode1.bit.t = 1; else if ( p1->y < cmin_y ) outcode1.bit.b = 1;
                   if ( p1->x > cmax_x ) outcode1.bit.r = 1; else if ( p1->x < cmin_x ) outcode1.bit.l = 1;
                  }
              }
 }

//-------------------------------------------------------------//

QSProjection::ClipResult QSProjection::clipLine( QSPt2f* p0, QSPt2f* p1, const QSPt2f& clip_pos, const QSPt2f& clip_area )
// Cohen-Sutherland's method.
// J. D. Foley, ...
// "Wprowadzenie do grafiki komputerowej" ( "Introduction to computer graphics" )
//  Chap 3.9
 {
  const double cmin_x = clip_pos.x;
  const double cmin_y = clip_pos.y;
  const double cmax_x = clip_pos.x+clip_area.x;
  const double cmax_y = clip_pos.y+clip_area.y;

  QSPt2f cp;

  union {
       unsigned int all;
       struct {
             unsigned int l : 1; // left
             unsigned int r : 1; // right
             unsigned int t : 1; // top
             unsigned int b : 1; // bottom
            } bit;
      } outcode0, outcode1, outcodeOut;

  outcode0.all = 0;
  outcode1.all = 0;

  if ( p0->y > cmax_y ) outcode0.bit.t = 1; else if ( p0->y < cmin_y ) outcode0.bit.b = 1;
  if ( p0->x > cmax_x ) outcode0.bit.r = 1; else if ( p0->x < cmin_x ) outcode0.bit.l = 1;

  if ( p1->y > cmax_y ) outcode1.bit.t = 1; else if ( p1->y < cmin_y ) outcode1.bit.b = 1;
  if ( p1->x > cmax_x ) outcode1.bit.r = 1; else if ( p1->x < cmin_x ) outcode1.bit.l = 1;

  if ( (outcode0.all | outcode1.all) == 0 ) return Accepted;

  while( 1 )
      if ( (outcode0.all | outcode1.all) == 0 ) return Clipped;
        else if ( (outcode0.all & outcode1.all) != 0 ) return Rejected;
        else {
              if ( outcode0.all != 0 ) outcodeOut = outcode0;
                                  else outcodeOut = outcode1;

              if ( outcodeOut.bit.t ) {
                   cp.y = cmax_y;
                   cp.x = p0->x+(p1->x-p0->x)*(cmax_y-p0->y)/(p1->y-p0->y);
                  } else if ( outcodeOut.bit.b ) {
                   cp.y = cmin_y;
                   cp.x = p0->x+(p1->x-p0->x)*(cmin_y-p0->y)/(p1->y-p0->y);
                  } else if ( outcodeOut.bit.r ) {
                   cp.x = cmax_x;
                   cp.y = p0->y+(p1->y-p0->y)*(cmax_x-p0->x)/(p1->x-p0->x);
                  } else if ( outcodeOut.bit.l ) {
                   cp.x = cmin_x;
                   cp.y = p0->y+(p1->y-p0->y)*(cmin_x-p0->x)/(p1->x-p0->x);
                  }

               if ( outcodeOut.all == outcode0.all ) {
                   p0->x = cp.x;
                   p0->y = cp.y;
                   outcode0.all = 0;
                   if ( p0->y > cmax_y ) outcode0.bit.t = 1; else if ( p0->y < cmin_y ) outcode0.bit.b = 1;
                   if ( p0->x > cmax_x ) outcode0.bit.r = 1; else if ( p0->x < cmin_x ) outcode0.bit.l = 1;
                  } else {
                   p1->x = cp.x;
                   p1->y = cp.y;
                   outcode1.all = 0;
                   if ( p1->y > cmax_y ) outcode1.bit.t = 1; else if ( p1->y < cmin_y ) outcode1.bit.b = 1;
                   if ( p1->x > cmax_x ) outcode1.bit.r = 1; else if ( p1->x < cmin_x ) outcode1.bit.l = 1;
                  }
              }
 }

//-------------------------------------------------------------//

QSProjection::ClipResult QSProjection::clipPoly2( const QSPt2f in[], int inpoints, QSPt2f out[], int *outpoints, int maxpoints, bool outedges[] = NULL, const bool inedges[] = NULL ) const
 {
  static const double cmin_x = 0.0;
  static const double cmax_x = 1.0;
  static const double cmin_y = 0.0;
  static const double cmax_y = 1.0;

  int i;
  int nr;

  QSPt2f c[2];

  c[0].x = c[1].x = in[0].x;
  c[0].y = c[1].y = in[0].y;

  for( int i=1; i<inpoints; i++ ) {
        c[0].x = QMIN( in[i].x, c[0].x ); c[1].x = QMAX( in[i].x, c[1].x );
        c[0].y = QMIN( in[i].y, c[0].y ); c[1].y = QMAX( in[i].y, c[1].y );
       }

  if ( c[0].x >= cmin_x && c[1].x <= cmax_x &&
       c[0].y >= cmin_y && c[1].y <= cmax_y  ) return Accepted;

  if ( ( c[0].x > cmax_x || c[1].x < cmin_x ) &&
       ( c[0].y > cmax_y || c[1].y < cmin_y )  ) return Rejected;

  // init clipping
  QSPt2f buff[QS_CLIP_BUFFER_SIZE];
  bool edge_buff[QS_CLIP_BUFFER_SIZE];

  int opts = QMIN( QS_CLIP_BUFFER_SIZE, maxpoints );
  int ipts = QMIN( QS_CLIP_BUFFER_SIZE, inpoints  );

  for( i=0; i<ipts; i++ ) { buff[i] = in[i]; if ( outedges ) edge_buff[i] = inedges ? inedges[i] : true; }

  // clip poly to the rectangle ( by four half-planes ).
  for( int plane=0; plane<4; plane++ ) {

        QSPt2f p;
        QSPt2f s = buff[ipts-1];

        bool s_inside = true;    // no compiler warning
        switch( plane ) {
           case 0: s_inside = ( s.x >= cmin_x ); break;
           case 1: s_inside = ( s.x <= cmax_x ); break;
           case 2: s_inside = ( s.y >= cmin_y ); break;
           case 3: s_inside = ( s.y <= cmax_y ); break;
          }

        nr = 0;
        for( i=0; i<ipts; i++ ) {
                p = buff[i];
                bool p_inside = true;
                switch( plane ) {
                   case 0: p_inside = ( p.x >= cmin_x ); break;
                   case 1: p_inside = ( p.x <= cmax_x ); break;
                   case 2: p_inside = ( p.y >= cmin_y ); break;
                   case 3: p_inside = ( p.y <= cmax_y ); break;
                  }

		// edge inside the clip area - pass it to the output
		if ( p_inside && s_inside ) {
			if ( nr<opts ) {
				if ( outedges ) outedges[nr] = edge_buff[i];
				out[nr++] = p;					
				}
			}
		else
		// edge crosses the boundary of the clip area
		if ( p_inside != s_inside ) {   // line from s to p crosses boundary
			double t = 0.0;
                	switch( plane ) {
				case 0: t = double(cmin_x-s.x)/(p.x-s.x); break;
				case 1: t = double(cmax_x-s.x)/(p.x-s.x); break;
				case 2: t = double(cmin_y-s.y)/(p.y-s.y); break;
				case 3: t = double(cmax_y-s.y)/(p.y-s.y); break;
				}
			QSPt2f c;
			c.x = s.x + t*(p.x-s.x);
			c.y = s.y + t*(p.y-s.y);
			if ( nr<opts ) {
				if ( outedges ) if ( p_inside ) outedges[nr] = false; else outedges[nr] = edge_buff[i];
				out[nr++] = c;
				}
			if ( p_inside ) if ( nr<opts ) {
				if ( outedges ) outedges[nr] = edge_buff[i];
				out[nr++] = p;
				}
		}

                s = p;
                s_inside = p_inside;
               }  // for ( i=0; i<npoints ...

          for( i=0; i<nr; i++ )  { buff[i] = out[i]; if ( outedges ) edge_buff[i] = outedges[i]; }
          ipts = nr;
        } // for( plane=0;

  if ( outpoints ) *outpoints = nr;

  return ( nr ) ? Clipped : Rejected;
 }

//-------------------------------------------------------------//

QSProjection::ClipResult QSProjection::clipPoint3( const QSPt3f& pos ) const
 {
  static const double cmin_x = 0.0;
  static const double cmax_x = 1.0;
  static const double cmin_y = 0.0;
  static const double cmax_y = 1.0;
  static const double cmin_z = 0.0;
  static const double cmax_z = 1.0;

  if ( pos.x >= cmin_x && pos.x <= cmax_x &&
       pos.y >= cmin_y && pos.y <= cmax_y &&
       pos.z >= cmin_z && pos.z <= cmax_z ) return Accepted;

  return Rejected;
 }
//-------------------------------------------------------------//


QSProjection::ClipResult QSProjection::clipLine3( QSPt3f* p0, QSPt3f* p1 ) const
 {
  static const double cmin_x = 0.0;
  static const double cmax_x = 1.0;
  static const double cmin_y = 0.0;
  static const double cmax_y = 1.0;
  static const double cmin_z = 0.0;
  static const double cmax_z = 1.0;

  union {
       unsigned int all;
       struct {
             unsigned int l : 1; // left
             unsigned int r : 1; // right
             unsigned int t : 1; // top
             unsigned int b : 1; // bottom
	     unsigned int f : 1; // front
	     unsigned int k : 1; // back
            } bit;
      } outcode0, outcode1, outcodeOut;

  outcode0.all = 0;
  outcode1.all = 0;

  if ( p0->y > cmax_y ) outcode0.bit.t = 1; else if ( p0->y < cmin_y ) outcode0.bit.b = 1;
  if ( p0->x > cmax_x ) outcode0.bit.r = 1; else if ( p0->x < cmin_x ) outcode0.bit.l = 1;
  if ( p0->z > cmax_z ) outcode0.bit.k = 1; else if ( p0->z < cmin_z ) outcode0.bit.f = 1;

  if ( p1->y > cmax_y ) outcode1.bit.t = 1; else if ( p1->y < cmin_y ) outcode1.bit.b = 1;
  if ( p1->x > cmax_x ) outcode1.bit.r = 1; else if ( p1->x < cmin_x ) outcode1.bit.l = 1;
  if ( p1->z > cmax_z ) outcode0.bit.k = 1; else if ( p1->z < cmin_z ) outcode0.bit.f = 1;

  if ( (outcode0.all | outcode1.all) == 0 ) return Accepted;

  QSPt3f cp;
  double t;
  while( 1 )
      if ( (outcode0.all | outcode1.all) == 0 ) return Clipped;
        else if ( (outcode0.all & outcode1.all) != 0 ) return Rejected;
        else {
              if ( outcode0.all != 0 ) outcodeOut = outcode0;
                                  else outcodeOut = outcode1;

              if ( outcodeOut.bit.t ) {
		   t = (cmax_y-p0->y)/(p1->y-p0->y);
                   cp.y = cmax_y;
                   cp.x = p0->x+(p1->x-p0->x)*t;
                   cp.z = p0->z+(p1->z-p0->z)*t;
                  } else if ( outcodeOut.bit.b ) {
		   t = (cmin_y-p0->y)/(p1->y-p0->y);
                   cp.y = cmin_y;		
                   cp.x = p0->x+(p1->x-p0->x)*t;
                   cp.z = p0->z+(p1->z-p0->z)*t;
                  } else if ( outcodeOut.bit.r ) {
 		   t = (cmax_x-p0->x)/(p1->x-p0->x);
                   cp.x = cmax_x;
                   cp.y = p0->y+(p1->y-p0->y)*t;
		   cp.z = p0->z+(p1->z-p0->z)*t;
                  } else if ( outcodeOut.bit.l ) {
		   t = (cmin_x-p0->x)/(p1->x-p0->x);
                   cp.x = cmin_x;
                   cp.y = p0->y+(p1->y-p0->y)*t;
		   cp.z = p0->z+(p1->z-p0->z)*t;
                  } else if ( outcodeOut.bit.f ) {
		   t = (cmin_z-p0->z)/(p1->z-p0->z);
                   cp.z = cmin_z;
		   cp.x = p0->x+(p1->x-p0->x)*t;
                   cp.y = p0->y+(p1->y-p0->y)*t;	
		  } else if ( outcodeOut.bit.k ) {
		   t = (cmax_z-p0->z)/(p1->z-p0->z);
                   cp.z = cmax_z;
		   cp.x = p0->x+(p1->x-p0->x)*t;
                   cp.y = p0->y+(p1->y-p0->y)*t;		
                  }

               if ( outcodeOut.all == outcode0.all ) {
                   p0->x = cp.x;
                   p0->y = cp.y;
		   p0->z = cp.z;
                   outcode0.all = 0;
                   if ( p0->y > cmax_y ) outcode0.bit.t = 1; else if ( p0->y < cmin_y ) outcode0.bit.b = 1;
                   if ( p0->x > cmax_x ) outcode0.bit.r = 1; else if ( p0->x < cmin_x ) outcode0.bit.l = 1;
		   if ( p0->z > cmax_z ) outcode0.bit.k = 1; else if ( p0->z < cmin_z ) outcode0.bit.f = 1;
                  } else {
                   p1->x = cp.x;
                   p1->y = cp.y;
		   p1->z = cp.z;
                   outcode1.all = 0;
                   if ( p1->y > cmax_y ) outcode1.bit.t = 1; else if ( p1->y < cmin_y ) outcode1.bit.b = 1;
                   if ( p1->x > cmax_x ) outcode1.bit.r = 1; else if ( p1->x < cmin_x ) outcode1.bit.l = 1;
		   if ( p1->z > cmax_z ) outcode0.bit.k = 1; else if ( p1->z < cmin_z ) outcode0.bit.f = 1;
                  }
              }
 }

//-------------------------------------------------------------//

QSProjection::ClipResult QSProjection::clipPoly3( const QSPt3f in[], int inpoints, QSPt3f out[], int *outpoints, int maxpoints, const QSPt3f c[2], bool outedges[] = NULL, const bool inedges[] = NULL ) const
 {
  static const double cmin_x = 0.0;
  static const double cmax_x = 1.0;
  static const double cmin_y = 0.0;
  static const double cmax_y = 1.0;
  static const double cmin_z = 0.0;
  static const double cmax_z = 1.0;

  int i;
  int nr;

  // check if a mesh is inside of the axis box.
  if ( c[0].x >= cmin_x && c[1].x <= cmax_x &&
       c[0].y >= cmin_y && c[1].y <= cmax_y &&
       c[0].z >= cmin_z && c[1].z <= cmax_z )   return Accepted;

  // check if a mesh is outside of the axis box.
  if ( ( c[0].x > cmax_x || c[1].x < cmin_x ) &&
       ( c[0].y > cmax_y || c[1].y < cmin_y ) &&
       ( c[0].z > cmax_z || c[1].z < cmin_z ) ) return Rejected;

  // init clipping
  QSPt3f buff[QS_CLIP_BUFFER_SIZE];
  bool edge_buff[QS_CLIP_BUFFER_SIZE];

  int opts = QMIN( QS_CLIP_BUFFER_SIZE, maxpoints );
  int ipts = QMIN( QS_CLIP_BUFFER_SIZE, inpoints  );

  for( i=0; i<ipts; i++ ) { buff[i] = in[i]; if ( outedges ) edge_buff[i] = inedges ? inedges[i] : true; }

  // clip mesh to the axis box ( by six planes )
  for( int plane=0; plane<6; plane++ ) {

        QSPt3f p;
        QSPt3f s = buff[ipts-1];

        bool s_inside = true;    // no compiler warning
        switch( plane ) {
           case 0: s_inside = ( s.x >= cmin_x ); break;
           case 1: s_inside = ( s.x <= cmax_x ); break;
           case 2: s_inside = ( s.y >= cmin_y ); break;
           case 3: s_inside = ( s.y <= cmax_y ); break;
           case 4: s_inside = ( s.z >= cmin_z ); break;
           case 5: s_inside = ( s.z <= cmax_z ); break;
          }

        nr = 0;

        for( i=0; i<ipts; i++ ) {
                p = buff[i];

                bool p_inside = true;
                switch( plane ) {
                   case 0: p_inside = ( p.x >= cmin_x ); break;
                   case 1: p_inside = ( p.x <= cmax_x ); break;
                   case 2: p_inside = ( p.y >= cmin_y ); break;
                   case 3: p_inside = ( p.y <= cmax_y ); break;
                   case 4: p_inside = ( p.z >= cmin_z ); break;
                   case 5: p_inside = ( p.z <= cmax_z ); break;
                  }

		// edge inside the clip area
                if ( p_inside && s_inside ) {
			if ( nr<opts ) {
				if ( outedges ) outedges[nr] = edge_buff[i];
				out[nr++] = p;					
				}
			}
		else
		// line from s to p crosses the boundary of the clipping area
                if ( p_inside != s_inside ) {
			double t = 0.0;
			switch( plane ) {
				case 0: t = (cmin_x-s.x)/(p.x-s.x); break;
				case 1: t = (cmax_x-s.x)/(p.x-s.x); break;
				case 2: t = (cmin_y-s.y)/(p.y-s.y); break;
				case 3: t = (cmax_y-s.y)/(p.y-s.y); break;
				case 4: t = (cmin_z-s.z)/(p.z-s.z); break;
				case 5: t = (cmax_z-s.z)/(p.z-s.z); break;
				}

			QSPt3f c;
			c.x = s.x + t*(p.x-s.x);
			c.y = s.y + t*(p.y-s.y);
			c.z = s.z + t*(p.z-s.z);
			if ( nr<opts ) {
				if ( outedges ) if ( p_inside ) outedges[nr] = false; else outedges[nr] = edge_buff[i];
				out[nr++] = c;
				}
			if ( p_inside ) if ( nr<opts ) {
				if ( outedges ) outedges[nr] = edge_buff[i];
				out[nr++] = p;
				}
			}

                s = p;
                s_inside = p_inside;
               }  // for ( i=0; i<npoints ...

          for( i=0; i<nr; i++ ) { buff[i] = out[i]; if ( outedges ) edge_buff[i] = outedges[i]; }
          ipts = nr;
        } // for( plane=0; plane<5 ...

  if ( outpoints ) *outpoints = nr;

  return ( nr ) ? Clipped : Rejected;
 }

//			out[ nr<opts ? nr++ : nr-1 ] = p;
//			out[ nr<opts ? nr++ : nr-1 ] = c;
//			if ( p_inside ) out[ nr<opts ? nr++ : nr-1 ] = p;

//-------------------------------------------------------------//

void QSProjection::shade( QSGFill &fill, const QSPt3f& normal, const QSPt2f* canvas_pts, int npoints, QSGFill *bottomFill ) const
// Uff !
 {
  if ( fill.style == QSGFill::Transparent ) return;

  double dratio = 4.0;
  if ( bottomFill && isBottom( canvas_pts, npoints ) ) { fill = *bottomFill; dratio = -4.0; }

  if ( !m_light ) return;

  unsigned int r = fill.color.r;
  unsigned int g = fill.color.g;
  unsigned int b = fill.color.b;

  // reverse bottom
  double directed_light = dratio*dotProduct( m_light_vector, normal )*(m_directed_light+50.0);
  double ratio = 256.0 - (m_directed_light+50.0)*3.0 + QMAX(directed_light,0.0) + m_ambient_light*4.0;
  unsigned int shade = QMAX( 0, int(ratio) );

  // shade with distant light source
  r = (shade * r) >> 8;
  g = (shade * g) >> 8;
  b = (shade * b) >> 8;

  fill.color.r = (unsigned char )QMIN( 255U, QMAX( r, 0U ) );
  fill.color.g = (unsigned char )QMIN( 255U, QMAX( g, 0U ) );
  fill.color.b = (unsigned char )QMIN( 255U, QMAX( b, 0U ) );
 }

//-------------------------------------------------------------//

QSPt3f QSProjection::furthest() const
 {
  return QSPt3f(0.0,0.0,0.0);
 }

//-------------------------------------------------------------//

QSPt2f QSProjection::middle() const
 {
  return QSPt2f(0.0,0.0);
 }




       /**
         * Maps 'p' to the screen coordinates.
         *
      virtual QSPt2 world3DToScreen( const QSPt3f &p ) const ;


//-------------------------------------------------------------//

QSPt2 QSProjection::world3DToScreen( const QSPt3f &p ) const
 {
  return world2DToScreen( map(p) );
 }


      * Maps the point to the screen coordinates.

    virtual QSPt2 world2DToScreen( const QSPt2f& p ) const = 0;

         * Maps 'p' to the screen coordinates and returns the Z coordinate ( depth ).

//      virtual double world3DToScreenDepth( const QSPt3f &p ) const;

//-------------------------------------------------------------//

QSPt2f QSProjection::world3DToScreenF( const QSPt3f &p ) const
 {
  return world2DToScreenF( map(p) );
 }

//-------------------------------------------------------------//

double QSProjection::world3DToScreenDepth( const QSPt3f &p ) const
 {
  return (1.0-p.z);
 }
*/